(* Copyright 1992 by AT&T Bell Laboratories *)

structure ElabUtil:ELABUTIL = struct

  open Symbol Absyn Ast ErrorMsg PrintUtil AstUtil Types BasicTypes TyvarSet
  Access Modules EqTypes ModuleUtil TypesUtil Variables Misc

  structure SP = SymPath

val unitPat = RECORDpat{fields = nil, flex = false, typ = ref UNDEFty}
			
val unitExp = RECORDexp nil

val TRUEpat = CONpat(trueDcon,NONE)
val TRUEexp = CONexp(trueDcon,NONE)
val FALSEpat = CONpat(falseDcon,NONE)
val FALSEexp = CONexp(falseDcon,NONE)



(* These constants are never used; NONE may not be appropriate here (zsh) *)
val NILpat = CONpat(nilDcon,NONE)
val NILexp = CONexp(nilDcon,NONE)
val CONSpat = fn pat => APPpat(consDcon,NONE,pat)
val CONSexp = CONexp(consDcon,NONE)

(* Verifies that all the elements of a list are unic *)
fun checkUniq (err,message) l =
 let val l' = Sort.sort Symbol.symbolGt l
     fun f (x::y::rest) = (
	   if Symbol.eq(x,y)
	   then err COMPLAIN (message^ ": " ^ Symbol.name x) nullErrorBody
	   else ();
	   f(y::rest))
       | f _ = ()
 in f l'
 end

(* extract all the variables from a pattern
 * NOTE: the "freeOrVars" function in elabcore.sml should probably
 * be merged with this.
 *)
fun bindVARp (patlist,err) =
    let val vl = ref (nil: symbol list)
	val env = ref(Env.empty: Modules.env)
	fun f (VARpat(v as VALvar{path=SP.SPATH[name],access,...})) = (
		if Symbol.eq(name, EQUALsym)
		  then (case access 
                         of INLINE _ => ()
                          | _ => err WARN "rebinding =" nullErrorBody)
		  else ();
		env := Env.bind(name,VARbind v,!env); 
		vl := name :: !vl)
	  | f (RECORDpat{fields,...}) = app(fn(_,pat)=>f pat) fields
	  | f (VECTORpat(pats,_)) = app f pats
	  | f (APPpat(_,_,pat)) = f pat
	  | f (CONSTRAINTpat(pat,_)) = f pat
	  | f (LAYEREDpat(p1,p2)) = (f p1; f p2)
	  | f (ORpat(p1, p2)) = (f p1; bindVARp([p2], err); ())
	  | f _ = ()
     in app f patlist;
	checkUniq (err,"duplicate variable in pattern(s)") (!vl);
	!env
    end

(* sort the labels in a record the order is redefined to take the usual 
   ordering on numbers expressed by strings (tuples) *)

local 
  val sort = Sort.sort (fn ((a,_),(b,_)) => TypesUtil.gtLabel (a,b))
in fun sortRecord(l,err) =
     (checkUniq(err, "duplicate label in record") (map #1 l); sort l)
end

fun makeRECORDexp(fields,err) =
  let val fields' = map (fn(id,exp)=> (id,(exp,ref 0))) fields
      fun assign(i,(_,(_,r))::tl) = (r := i; assign(i+1,tl))
	| assign(_,nil) = ()
      fun f(i,(id,(exp,ref n))::r) = (LABEL{name=id,number=n},exp)::f(i+1,r)
        | f(_,nil) = nil
  in assign(0, sortRecord(fields',err)); RECORDexp(f(0,fields')) end

fun TUPLEexp l = 
  let fun addlabels(i,e::r) = 
	    (LABEL{number=i-1,name=(Tuples.numlabel i)},e) :: addlabels(i+1,r)
        | addlabels(_, nil) = nil
  in RECORDexp (addlabels(1,l)) end

(* Adds a default case to a list of rules. 
   If given list is marked, all ordinarily-marked expressions 
     in default case are also marked, using end of given list 
     as location.
   KLUDGE! The debugger distinguishes marks in the default case by
     the fact that start and end locations for these marks 
     are the same! *)
fun completeMatch'' rule [r as RULE(pat,MARKexp(_,(left,_)))] =
    [r, rule (fn exp => MARKexp(exp,(left,left)))]
  | completeMatch'' rule [r as RULE(pat,CONSTRAINTexp(MARKexp(_,(left,_)),_))] =
    [r, rule (fn exp => MARKexp(exp,(left,left)))]
  | completeMatch'' rule [r] = [r,rule (fn exp => exp)]
  | completeMatch'' rule (a::r) = a :: completeMatch'' rule r
  | completeMatch'' _ _ = impossible "completeMatch''"

fun completeMatch' (RULE(p,e)) =
    completeMatch'' (fn marker => RULE(p,marker e))

exception NoCore

fun completeMatch(coreEnv,name) =
 let val CONbind exnMatch = ModuleUtil.lookVARCON 
	 (coreEnv,SymPath.SPATH[strSymbol "Core", varSymbol name],
	  fn _ => fn s => fn _ => raise NoCore)
	 handle NoCore => CONbind Types.bogusEXN
  in  completeMatch'' 
      (fn marker =>
       RULE(WILDpat, 
	    marker(RAISEexp(CONexp(exnMatch,NONE), UNDEFty))))
 end

val trivialCompleteMatch = completeMatch(StaticEnv.empty,"Match")


(* Transform a while loop in a call to a recursive function *)
val whileSym = Symbol.varSymbol "while"

fun IFexp (a,b,c) =
    CASEexp(a, trivialCompleteMatch [RULE(TRUEpat,b), RULE(FALSEpat,c)])

fun TUPLEpat l =
  let fun addlabels(i,e::r) = (Tuples.numlabel i, e) :: addlabels(i+1, r)
	| addlabels(_, nil) = nil
  in
  RECORDpat{fields=addlabels(1,l), flex=false, typ=ref UNDEFty}
  end

val argName = [Symbol.varSymbol "arg"]

fun FUNdec (completeMatch,fbl,errorMatch,ln) =
    let fun fb2rvb (FB {var, clauses as (CLAUSE{pats,...}::_),tyvars}) =
	    let fun getvar _ =  VALvar{access=LVAR(mkLvar()),
				       path=SP.SPATH argName,
				       typ=ref UNDEFty}
		val vars = map getvar pats
		fun not1(f,[a]) = a
		  | not1(f,l) = f l
		fun dovar valvar = VARexp(ref(valvar),NONE)
		fun doclause (CLAUSE{pats,exp,resultty=NONE}) =
			      RULE(not1(TUPLEpat,pats), exp)
		  | doclause (CLAUSE{pats,exp,resultty=SOME ty}) =
			      RULE(not1(TUPLEpat,pats),CONSTRAINTexp(exp,ty))

	        fun last[x] = x | last (a::r) = last r
		val mark =  case (hd clauses, last clauses)
	                     of (CLAUSE{exp=MARKexp(_,(a,_)),...},
				 CLAUSE{exp=MARKexp(_,(_,b)),...}) =>
			         (fn e => MARKexp(e,(a,b)))
			      | _ => fn e => e
		fun makeexp [var] = 
                      FNexp(completeMatch (map doclause clauses),UNDEFty)
		  | makeexp vars = 
                      foldr (fn (w,e) => 
                             FNexp(completeMatch [RULE(VARpat w,mark e)],
                                   UNDEFty))
				(CASEexp(TUPLEexp(map dovar vars),
					 completeMatch (map doclause clauses)))
				vars
	     in RVB {var=var,
		     exp=makeexp vars,
		     resultty=NONE,
		     tyvars=tyvars}
	    end
          | fb2rvb _ = ErrorMsg.impossible "absyn.38"
    in VALRECdec (map fb2rvb fbl) end

fun WHILEexp (a,b) =
  let val fvar = mkVALvar whileSym
      val id = fn x => x
      val (markdec,markall,markend,markbody) =
	case (a,b)
	of (MARKexp(_,(a1,a2)), MARKexp(_,(b1,b2))) =>
	      (fn e => MARKdec(e,(a1,b2)), fn e => MARKexp(e,(a1,b2)),
	       fn e => MARKexp(e,(b2,b2)), fn e => MARKexp(e,(b1,b2)))
	 | _ => (id,id,id,id)
      val body = 
        markbody(SEQexp[b, APPexp(markend(VARexp(ref fvar,NONE)), 
                                  markend unitExp)])
      val loop = markall(IFexp(a,body, markend unitExp))
      val fnloop = markall(FNexp(trivialCompleteMatch 
				  [RULE(unitPat,loop)],UNDEFty))
  in 
  markall 
    (LETexp(
       markdec 
         (VALRECdec[RVB{var=fvar, exp=fnloop, resultty = NONE, 
			tyvars = ref nil}]),
       APPexp(markall(VARexp (ref fvar,NONE)), markend unitExp)))
  end

fun makeHANDLEexp(exp,rules) =
  let 
    val v = mkVALvar exnID
    val r = RULE(VARpat v, RAISEexp(VARexp(ref(v),NONE),UNDEFty))
    val rules = completeMatch' r rules 
  in 
    HANDLEexp(exp, HANDLER(FNexp(rules,UNDEFty))) 
  end

fun isBoundConstructor(env,var) =
  (case lookShortVARCON(env,var,fn _ => raise Env.Unbound)
   of CONbind _ => true
    | _ => false)
   handle Env.Unbound => false

fun checkBoundConstructor(env,var,err) =
  if isBoundConstructor(env,var)
		 then err COMPLAIN ("rebinding data constructor \""
		                    ^Symbol.name var^ "\" as variable")
					  nullErrorBody
		 else ()

(* transforme a VarPat in either a variable or a constructor. If we are given
   a long path (>1) then it has to be a constructor *)

fun pat_id (spath,env,err) = 
  case spath
  of SymPath.SPATH[id] =>
    ((case lookShortVARCON (env,id,fn _ => raise Env.Unbound)
     of CONbind c => CONpat(c,NONE) 
      | _ => VARpat(mkVALvar id))
     handle Env.Unbound => VARpat(mkVALvar id))
   | _ =>
    CONpat((case lookVARCON (env,spath,err)
             of VARbind c =>
		(err COMPLAIN 
		  (SymPath.makestring spath ^
		   " is a variable. It must be a constructor.")
		  nullErrorBody;
	         (bogusCON,NONE))
              | CONbind c => (c,NONE)
              | _ => ErrorMsg.impossible "CoreLang.qid_pat")
            handle Env.Unbound => impossible "unbound untrapped")

fun makeRECORDpat(l,flex,err) =
  RECORDpat{fields=sortRecord(l,err), flex=flex, typ=ref UNDEFty}

fun clean_pat err (CONpat(DATACON{const=false,name,...},_)) = 
      (err COMPLAIN ("data constructor "^Symbol.name name^
		     " used without argument in pattern")
         nullErrorBody;
       WILDpat)
  | clean_pat err p = p

fun pat_to_string WILDpat = "_"
  | pat_to_string (VARpat(VALvar{path,...})) = SP.makestring path
  | pat_to_string (CONpat(DATACON{name,...},_)) = Symbol.name name
  | pat_to_string (INTpat (i,_)) = i
  | pat_to_string (REALpat s) = s
  | pat_to_string (STRINGpat s) = s
  | pat_to_string (CHARpat s) = "#"^s
  | pat_to_string (RECORDpat _) = "<record>"
  | pat_to_string (APPpat _) = "<application>"
  | pat_to_string (CONSTRAINTpat _) = "<constraint pattern>"
  | pat_to_string (LAYEREDpat _) = "<layered pattern>"
  | pat_to_string (VECTORpat _) = "<vector pattern>"
  | pat_to_string (ORpat _) = "<or pattern>"
  | pat_to_string _ = "<illegal pattern>"

fun makeAPPpat err (CONpat(d as DATACON{const=false,...},t),p) = APPpat(d,t,p)
  | makeAPPpat err (CONpat(d as DATACON{name,...},_),_) = 
      (err COMPLAIN
        ("constant constructor applied to argument in pattern:"
	 ^ Symbol.name name)
         nullErrorBody;
       WILDpat)
  | makeAPPpat err (rator,_) = 
      (err COMPLAIN (concat["non-constructor applied to argument in pattern: ",
			     pat_to_string rator])
         nullErrorBody;
       WILDpat)

fun makeLAYEREDpat ((x as VARpat _), y, _) = LAYEREDpat(x,y)
  | makeLAYEREDpat (CONSTRAINTpat(x,t), y, err) = 
      makeLAYEREDpat(x,CONSTRAINTpat(y,t),err)
  | makeLAYEREDpat (x,y,err) =
      (err COMPLAIN "pattern to left of AS must be variable" nullErrorBody;
       y)

fun calc_strictness (arity, body) =
    let val argument_found = Array.array(arity,false)
	fun search(VARty(ref(INSTANTIATED ty))) = search ty
	  | search(IBOUND n) = Array.update(argument_found,n,true)
	  | search(CONty(tycon, args)) = app search args
	  | search _ = ()	(* for now... *)
    in
	search body;
	ArrayExt.listofarray argument_found
    end;

fun makeTB(args,name,(ty,tv),err) notwith (env,path) =
  let val _ = 
	(checkbound(tv,args,err); TypesUtil.bindTyvars args; compressTy ty)
      val arity = length args
      val binding = 
	DEFtyc{path=InvPath.extend(path,name), strict=calc_strictness(arity,ty),
	       tyfun=TYFUN{arity=arity, body=ty}}
  in ([TB{tyc=binding,def=ty}],
      if notwith then Env.empty else Env.bind(name,TYCbind binding,Env.empty))
  end

fun makeTYPEdec ((tbs,env),err) =
  let val _ =  
        checkUniq (err, "duplicate type definition") 
                  (map (fn TB{tyc=DEFtyc{path,...},...} =>
			     InvPath.last path
			 | _ => ErrorMsg.impossible "CoreLang.makeTYPEdec")
			tbs)
      val env' = ref env  
      fun bindtyc (TB{tyc as DEFtyc{path,...},...}) = 
	    env' := Env.bind(InvPath.last path,TYCbind tyc,!env')
        | bindtyc _ = ErrorMsg.impossible "makeTYPEdec.bindtyc"
  in app bindtyc tbs; (TYPEdec tbs, !env') end

end
