(* Copyright 1992 by AT&T Bell Laboratories *)

(****************************************************************************

  elabstr.sml: elaborate the AST representing structures

 ****************************************************************************)

structure ElabStr : ELABSTR =
struct

open Access Symbol Types Modules ModuleUtil ErrorMsg Extern
     TyvarSet Absyn Variables ElabSig Control Ast PrettyPrint

structure SP = SymPath
structure IP = InvPath

val say = Control.Print.say
fun next (ri as ref i) = (ri := i+1; i)

val debugging = ref false

fun debugmsg (msg: string) = if (!debugging) then (say msg; say "\n") else ()

fun debugPrint(msg: string, printfn: ppstream -> 'a -> unit, arg: 'a) =
    if (!debugging) then
       with_pp (ErrorMsg.defaultConsumer())
	(fn ppstrm =>
	  (begin_block ppstrm CONSISTENT 0;
	   add_string ppstrm msg;
	   add_newline ppstrm;
	   add_string ppstrm "  ";
	   begin_block ppstrm CONSISTENT 0;
	   printfn ppstrm arg;
	   end_block ppstrm;
	   add_newline ppstrm;
	   end_block ppstrm;
	   flush_ppstream ppstrm))
    else ()

(* TYPES FOR ELABORATING STRUCTURES *)

(* type strKind:
   This type describes the context in which an object is added.

   It must be seen as an abstraction of the structure built. It forgets all
   the useless fields but contains a specification of the skeleton for
   InstanceStr.  The structure produced has IRRELEVANT instead because the
   skeleton is built directly and so its specification is not needed.

   Top is used for the top level environment

   Invariants:
    (1) !strCount < length strArray
    (2) !fctCount < length fctArray
    (3) !typCount < length typArray
*)

datatype strKind 
  = SimpleStr of StaticEnv.staticEnv ref
  | InstanceStr of
     {myself: Structure,
      (* pointer to the environment (incremental updating) *)
      env: StaticEnv.staticEnv ref,
      (* the parent structure *)
      parent: Structure,
      (* test if a stamp has been generated in the parent or 
       * in the argument *)
      inFunctor: Stamps.stamp -> bool,
      inArgument: Stamps.stamp -> bool,
      (* normalizing path map to translate paths used for sharing
       * constraints in the specification of functor arguments *)
      normTotal: Normalize.normMap ref,
      (* counters to allocate slots in arrays for objects *)
      strCount: int ref, 
      fctCount: int ref,
      typCount: int ref,
      (* arrays to store mutable objects *)
      strArray: Structure array,
      fctArray: Functor array,
      typArray: Types.tycon array}
  | SuperStr of strKind
  | TopLevel

(* THE TYPE OF THE ELABORATION CONTEXT FOR STRUCTURES *)

type contextStr =
 {(* what kind of structure? *)
  kind: strKind,
  (* are we in the local part of a Local or Let? *)
  (* this controls typechecking of core decls, and top-level open decls *)
  top: bool, 
  (* the global environment *)
  fullenv: StaticEnv.staticEnv,
  (* inverse path to the current structure *)
  rpath: InvPath.path,
  (* last positions recorded in the source file *)
  region: region,
  (* scope of stamps *)
  scope: Stamps.scope}


(* special symbols *)

val functorId = Symbol.fctSymbol "<functor>"
val hiddenId = Symbol.strSymbol "<hidden>"
val localStrName = Symbol.strSymbol "a funny structure"
    (* used in makeOpenDec to build redeclaration of components *)

val BogusTy = UNDEFty (*** BogusTy is just a place holder ***)

fun externAccess(EXTERN _) = true
  | externAccess(PATH(_,p)) = externAccess p
  | externAccess _ = false


(* countStrFctTyp : Ast.dec -> int * int * int
   computes the number of elements defined in a structure
   it is used to preallocate the arrays of an INSTANCE structure
   before the real elaboration begins
*)

fun countStrFctTyp (strexp: Ast.strexp) : (int * int * int) =
    let fun getStr (Strb {def,...}) = def
	  | getStr (MarkStrb (d,_)) = getStr d
	fun getFct (Fctb {def,...}) = def
	  | getFct (MarkFctb (d,_)) = getFct d
	fun count(ldec, t as (strN,fctN,typN)) =
	    case ldec
	      of TypeDec tb => (strN,fctN,typN + length tb)
	       | DatatypeDec {datatycs,withtycs} => 
		   (strN,fctN,typN + length datatycs + length withtycs)
	       | AbstypeDec {abstycs,withtycs,body,...} => 
		   let val (strN,fctN,typN) = count (body,t)
		    in (strN,fctN,typN+length abstycs+length withtycs)
		   end
	       | OpenDec l => (strN + length l, fctN, typN)
	       | StrDec l => foldr countStr t (map getStr l)
	       | AbsDec l => foldr countStr t (map getStr l)
	       | FctDec l => foldr countFct t (map getFct l)
	       | LocalDec (l1,l2) =>
		   let val t1 = count (l1,t)
		       val t2 = count (l2,t1)
		    in t2
		   end
	       | SeqDec ldec => foldr count t ldec
	       | MarkDec (d,_) => count (d,t)
	       | _ => t
	and countStr(strexp, tr as (strN,fctN,typN)) =
	    case strexp
	      of VarStr _ => (strN+1,fctN,typN)
	       | StructStr dec => count(dec,(strN+1,fctN,typN))
	       | AppStr(_,l) =>
		   foldr (fn ((s,_),t) => countStr (s,t))
		     (strN+(length l)+1,fctN,typN) l
	       | LetStr(dec,str) =>
		   count(dec,countStr(str,(strN+1,fctN,typN)))
	       | MarkStr(s,_) => countStr(s,tr)
	and countFct(fctexp, tr as (strN,fctN,typN)) =
	    case fctexp
	      of VarFct _ => (strN,fctN+1,typN)
	       | FctFct _ => (strN,fctN+1,typN)
	       | AppFct(_,l,_) =>
		   foldr (fn ((s,_),t) => countStr (s,t))
		     (strN+(length l),fctN+1,typN) l
	       | LetFct(dec,fct) =>
		   count(dec,countFct(fct,(strN,fctN+1,typN)))
	       | MarkFct(fct,_) => countFct(fct,tr)
     in countStr (strexp,(0,0,0))
    end

(* Recompute paths after the elaboration of a structure *)

fun computePath (bind as VARbind(VALvar{access as INLINE _,typ,...}),
                 slotCount,_,_) = 
      (bind,[VALtrans(access,!typ,NONE)]) before (next slotCount)
  | computePath (VARbind(VALvar{path,typ,access}),slotCount,_,_) =
      let val access' = SLOT (next slotCount) 
       in (VARbind(VALvar{access=access',path=path,typ=typ}),
           [VALtrans(access,!typ,NONE)])
      end
  | computePath (CONbind(DATACON{name,const,sign,
                                 typ,rep,orig}),slotCount,_,_) =
      let val (rep',trans') = 
           case rep
            of VARIABLE access =>
                let val access' = SLOT(next slotCount)
                 in (VARIABLE access',[VALtrans(access,BogusTy,NONE)]) 
                end
             | VARIABLEc access =>
                let val access' = SLOT(next slotCount)
                 in (VARIABLEc access',[VALtrans(access,BogusTy,NONE)]) 
                end
             | _ => (rep,[])
       in (CONbind(DATACON{name=name,const=const,sign=sign,
                           typ=typ,orig=orig,rep=rep'}),
           trans')
      end
  | computePath (STRbind(STRvar{name,access,binding}),slotCount,strLty,_) =
      let val access' = if hidden name then SLOT 0 else SLOT(next slotCount)
          val trans' = 
            if hidden name then []
            else if externAccess(access) 
                 then [STRtrans(access,strLty binding)]
                 else [VALtrans(access,BogusTy,NONE)]
       in (STRbind(STRvar{name=name,access=access',binding=binding}),trans')
      end
  | computePath (FCTbind(FCTvar{name,access,binding}),slotCount,_,fctLty) =
      let val access' = SLOT(next slotCount)
          val trans' =
            if externAccess(access) 
            then [FCTtrans(access,fctLty binding,(NONE,NONE,NONE))]
            else [VALtrans(access,BogusTy,NONE)]
       in (FCTbind(FCTvar{name=name,access=access',binding=binding}),trans')
      end
  | computePath (bind,_,_,_) = (bind,[])


(* Recompute all the paths in an environment, suppress doubles and
   allocate slots. Objects are ordered so that slot allocation is independant
   from the way elaboration is done *)

fun recomputePath(origEnv,strLty,fctLty) =
    let val slotCount = ref 0
	val env = ref Env.empty
	val trans = ref ([]: trans list)
	val revEnv = sortEnvBindings (Env.consolidate origEnv) 
	val symbols = map #1 revEnv
     in app
	 (fn (name,bind) => 
	   let val (newbind,newtrans) = computePath (bind,slotCount,strLty,fctLty)
	    in trans := newtrans @ (!trans); env := Env.bind(name,newbind,!env) 
	   end)
	 revEnv;
	(Env.consolidate (!env),rev (!trans),symbols)
    end

(*** ADDING OBJECTS TO THE DIFFERENT ELABORATION ENVIRONMENTS ***)
(* adds various simple objects to the environment *)

fun addObject (sym, binding, SimpleStr env, rpath) =
      (env := Env.bind(sym,binding,!env); binding)
  | addObject (sym, binding as TYCbind def,
               InstanceStr{env,typCount,typArray,normTotal,...}, rpath) = 
      let val pos = next typCount
          val newbind = TYCbind(FORMtyc{pos=pos,name=sym,spec=def})
       in Array.update(typArray,pos,def) 
            handle Subscript => impossible "addObject";
	  normTotal := Normalize.extend(sym,InvPath.extend(rpath,sym),!normTotal);
	  env := Env.bind(sym,newbind,!env);
	  newbind
      end
  | addObject (sym, binding, InstanceStr{env,...},_) =
      (env := Env.bind(sym,binding,!env);binding)
  | addObject (sym, binding, _,_) = binding


(*
 * addStr1 is used to update the environment with the raw structure. If we
 * are not in a functor body, there is no reason to update the environment
 * so nothing is done. This means that the second call to addStr2 is
 * mandatory !
 *
 * Remark: We are building the full and true signature of the slot. It is 
 * necessary for elaborating functor arguments.
 *)

fun addStr1(sym,str,sgn,InstanceStr{env,strCount,strArray,normTotal,...},rpath)=
      let val access = if hidden sym then NO_ACCESS else LVAR(namedLvar sym)
	  val pos = next strCount
          val newStr = STR_FORMAL{pos=pos,spec=sgn}
          val newVar = STRvar{name=sym,access=access,binding=newStr}
	  val _ = debugmsg "addStr1 1"
       in Array.update(strArray,pos,str)
	   handle Subscript => (say "addStr1: pos = ";
				say(makestring pos); say ", |strArray| = ";
				say(makestring(Array.length strArray));
				say "\n";
				ErrorMsg.impossible "addStr1");
	  normTotal := Normalize.extend(sym,rpath,!normTotal);
	  env := Env.bind(sym,STRbind newVar,!env)
      end
  | addStr1 _ = ()


(*
 * addStr2 adds the structure to the environment if it is a simple and
 * updates it if it is an instance (it finds the slot and modifies it)
 * if it is a functor body, nothing is modified: the result of elaborate
 * will be used by the function building functors.
 *)

fun getEnvPos env (sym: Symbol.symbol) = 
    case Env.look(env,sym)
      of (STRbind(STRvar {binding=(STR_FORMAL {pos,...}),...})) => pos
       | _ => impossible "Instantiate:getStrPos"

fun addStr2 (sym,str,InstanceStr{env,strArray,...}) =
      let val (pos,access,symbbind)=
            case Env.look(!env,sym)
	      of (symbbind as STRbind(STRvar {binding=(STR_FORMAL {pos,...}),
					      access,...})) =>
		    (pos,access,symbbind)
	       | _ => impossible "Instantiate:getStrPos"
	  val strvar = STRvar{name=sym,access=access,binding=str}
       in Array.update(strArray,pos,str) 
	    handle Subscript => impossible "elaboration.2";
          (strvar,symbbind)
      end
  | addStr2 (sym,str,SimpleStr env) =
      let val access = if hidden sym then NO_ACCESS else LVAR(namedLvar sym)
          val newVar = STRvar{name=sym,access=access,binding=str}
       in env := Env.bind(sym,STRbind newVar, !env);
	  (newVar,STRbind newVar)
      end
  | addStr2 (sym,str,_ ) = 
      let val access = if hidden sym then NO_ACCESS else LVAR(namedLvar sym)
          val newVar = STRvar{name=sym,access=access,binding=str}
       in (newVar,STRbind newVar)
      end

(* Add a functor in the environment *)

fun addFct (sym,fct,SimpleStr env) =
      let val slot = namedLvar(sym)
          val fctVar = FCTvar{name=sym,access=LVAR slot,binding=fct}
       in env := Env.bind(sym,FCTbind fctVar,!env);
	  (fctVar,fctVar)
      end
  | addFct (sym,fct,InstanceStr{env,fctCount,fctArray,...}) =
      let val access = LVAR (namedLvar sym)
	  val pos = next fctCount
          val newFct = FCT_FORMAL{pos=pos,spec=FULL_FSIG}
          val fctVar = FCTvar{name=sym,access=access,binding=newFct}
          val fctVar' = FCTvar{name=sym,access=access,binding=fct}
       in Array.update(fctArray,pos,fct);
          env := Env.bind(sym,FCTbind fctVar,!env);
	  (fctVar',fctVar)
      end
  | addFct (sym,fct,_) =
      let val slot = namedLvar(sym)
          val fctVar = FCTvar{name=sym,access=LVAR slot,binding=fct}
       in (fctVar,fctVar)
      end

(* When we have the parent, first we need its signature to elaborate
   the signature of the functor argument
 *)

fun signatureOfStructure (INSTANCE{sign,...}) =sign
  | signatureOfStructure ERROR_STR = ERROR_SIG
  | signatureOfStructure _ = impossible "signatureOfStructure"


fun elaborateTop(lpdec,env,coreEnv,error,errorMatch,transform) = 
let 

(*** ELABORATE A STRUCTURE ***)
(* When we start, the context is the context of the embedding structure *)

fun elaborate_structure
      {context as {kind,top,fullenv,rpath,region,scope}: contextStr,
       abstract: bool, name, constraint, def} =
  (* elaborate the raw structure: without the signature constraint *)
  let fun elab (context as {kind,top,fullenv,rpath,region,scope}: contextStr)
	       (StructStr ldec) =
	    (* the list of the symbols defined: can't be known easily 
               because of patterns... *)
	    let val _ = debugmsg "elab:StructStr - start"
		val makestamp = Stamps.newStamp scope
		(* builds the kind of the new structure *)
		val newKind =
		  case kind 
		    of SimpleStr _ => SimpleStr(ref Env.empty)
		     | InstanceStr {normTotal,inFunctor,inArgument,parent,...} =>
			 let (* preallocation and building of the skeleton *)
			     val (strs,fcts,typs) = countStrFctTyp(StructStr ldec)
			     val env = ref (Env.empty:env)
			     val strArray = Array.array(strs+1,ERROR_STR)
			     val fctArray = Array.array(fcts,ERROR_FCT)
			     val typArray = Array.array(typs,ERRORtyc)
			     val kindinst = 
			       TOP{strcount=strs, fctcount=fcts, typecount=typs,
				   slotcount=0, sConstraints=[], tConstraints=[],
				   abbreviations=[]}
			     val sgn =
			       SIG{symbols=ref [], env=env, name=NONE,
				   stamp=makestamp (), kind=ref kindinst}
			     val str =
			       INSTANCE{
				 sign = sgn, path = rpath,
				 subStrs=strArray,
				 subFcts=fctArray, types=typArray,
				 origin = STAMP_ORIGIN(makestamp())}
			  in addStr1 (name,str,sgn,kind,rpath);
			     InstanceStr
			       {env=env, normTotal=normTotal,myself=str,
				strCount=ref 0, fctCount=ref 0,
				typCount=ref 0, parent=parent,
				inFunctor=inFunctor,strArray=strArray,
				inArgument=inArgument, fctArray=fctArray,
				typArray=typArray}
			 end
		     | SuperStr kind => kind
		     | TopLevel => SimpleStr(ref Env.empty)
		val _ = debugmsg "elab:StructStr - newKind defined"
	 	(* Builds the new context with the new kind *)
		val newContext = {kind=newKind, top=true, fullenv=fullenv,
				  rpath=rpath, region=region, scope=scope}
		(* elaborate the environment from the spec *)
		val (decStr,envStr) = elabDecl(newContext,ldec)
		 (* fix paths and thinnings, fixes the set of symbols *)
		val _ = debugmsg "elab:StructStr - ldec elaborated"
		val (resultat,locations) = 
		   case newKind 
		     of SimpleStr env =>
			 let val (nenv,loc,_)=
			       recomputePath(!env,TransBinding.transStrLty,
					     TransBinding.transFctLty)
			     val stamp = makestamp()
			  in (SIMPLE{stamp=stamp, path = rpath,
				     env=nenv, id=ModuleId.SIMPLEid stamp},
			      loc) 
			 end
		      | InstanceStr{myself as INSTANCE{sign=SIG{symbols,... },
						       ...}, env, ... } =>
			 let fun g(s) = TransBinding.transStrLty0(s,myself)
			     fun h(f) = TransBinding.transFctLty0(f,myself)
			     val (nenv,loc,symb)=recomputePath(!env,g,h)
			  in env:=nenv;
			     symbols:= symb;
			     (myself,loc) 
			 end
		      | _ => impossible "elaborate: resultat"
		val _ = debugmsg "elab:StructStr - end"
	     in (resultat,
		 STRUCTstr{body=[decStr],str=resultat,
			   locations=locations})
            end

	| elab (context as {kind,top,fullenv,rpath,region,scope})
	       (AppStr (funcName ,[(arg,b)])) =
	    let val _ = debugmsg "elab:AppStr(f,[arg]) - start"
		(* a newKind to build the argument *)
		val (inArgument,newKind) =
		  case kind
		    of SimpleStr env => (fn _ => false,SimpleStr(ref (!env)))
		     | TopLevel => (fn _ => false,kind)
		     | SuperStr(k as InstanceStr{inArgument,...}) => (inArgument,k)
		     | SuperStr _ => impossible "elaborate_structure: SuperStr"
		     | InstanceStr{env,normTotal,myself,strCount,fctCount,
				   typCount,parent,inFunctor,inArgument,strArray,
				   fctArray,typArray,...} =>
		       (inArgument,
			InstanceStr{env = ref(!env), normTotal = ref(!normTotal),
				    myself=myself,strCount = ref(!strCount),
				    fctCount = ref 0, typCount = ref 0,
				    parent=parent, inFunctor=inFunctor,
				    strArray = strArray, fctArray = fctArray,
				    typArray = typArray, inArgument=inArgument})
		(* the context that goes with it *)
		val localContext = {kind = newKind, top = top, fullenv = fullenv,
				    rpath = IP.IPATH [parameterId], region = region,
				    scope = scope}
		val _ = debugmsg "elab:AppStr(f,[arg]) - localContext defined"
		(* call elaborate on the argument *)
		val (argDec as STRB{strvar=STRvar{binding=argStr,...},
				    def=argExp,...},_) = 
		  elaborate_structure 
		    {context=localContext,abstract=false,
		     def=arg,constraint=NONE,name=parameterId}
		val _ = debugmsg "elab:AppStr(f,[arg]) - argument elaborated"
		(* look for the functor definition *)
		val (fctVar as FCTvar{binding=funcFct,...}) = 
		  lookFCT (fullenv,SP.SPATH funcName,error region) 
		val _ = debugPrint("elab(AppStr(f,[arg]): the functor",
				   PPBasics.ppFunctor, (fullenv,funcFct,20))
	        (* apply the functor to its argument *)
		val (res,argThin,realArg) =
		  ApplyFunctor.applyFunctorFull
                    (funcFct, argStr, scope, rpath, error region,
		     fullenv, SigMatch.matchFarg)
		val _ = debugmsg "elab:AppStr(f,[arg]) - functor applied"
                val realArg' = lookBindingSTR(realArg,SP.SPATH [parameterId])

		(* get rid of the info on the application if unnecessary *)
		val simpRes = 
		  if inArgument (getFctStamp funcFct) then res
		  else case res
			 of APPLY{res,...} => res
			  | ERROR_STR => res
			  | _ => impossible "strange result of APPLY" 

                (* remember the actual lambdaty instance of the functor *)
                val lambdaty = 
                      LambdaType.injARROW(TransBinding.transStrLty(realArg'),
                                          TransBinding.transStrLty(simpRes))

		val resDec =
		  APPstr {oper=fctVar,argexp=argExp,argthin=argThin,
			  instancelty=lambdaty,str=simpRes}
		val _ = debugmsg "elab:AppStr(f,[arg]) - end"
	     in addStr1(name,simpRes,FULL_SIG,kind,rpath);
		(simpRes,resDec)
	    end

        | elab context (AppStr(qid,arg :: larg)) =
	   (debugmsg "elab: AppStr(f,args)";
            elab context
		 (LetStr(StrDec[Strb{name=hiddenId,constraint=NONE,
				     def=AppStr(qid,[arg])}],
			 AppStr([hiddenId,functorId],larg))))

        | elab context (AppStr(qid,[])) =
            impossible "elabStrRaw.AppStr"

        | elab (context as {kind,top,fullenv,rpath,region,scope}) 
	      (VarStr name')=
            let val _ = debugmsg "elab:VarStr"
		val (var as STRvar{binding,...}) =
                  lookSTR (fullenv,SP.SPATH name',error region)
             in addStr1(name,binding,FULL_SIG,kind,rpath);
	        (binding,VARstr var)
	    end

	| elab (context as {kind,top,fullenv,rpath,region,scope})
	      (LetStr (ldec,str)) =
            let val _ = debugmsg "elab:LetStr - start"
		(* contexts with their own env reference must be restored
		   after elaborating local decls *)
		val after_local = 
		  case kind
		    of InstanceStr{env,...} => 
			 let val old_env = !env in fn () => env := old_env end
		     | SuperStr(InstanceStr{env,...}) =>
			 let val old_env = !env in fn () => env := old_env end
		     | SimpleStr env =>
			 let val old_env = !env in fn () => env := old_env end
		     | _ => (fn () => ())
		(* if we are in a functor, we must consider ourselves as
		 * a substructure of the result and so not as with the
		 * SuperStr level but with it already processed. this
		 * is a hack. !!?? [so what's the nonhack solution? -dbm] *)
		val c1 = {kind = case kind of SuperStr k => k | _ => kind,
			  top=false, fullenv=fullenv, rpath=rpath,
			  region=region, scope=scope}
		val (ld1,env1) =  elabDecl(c1,ldec)
		val _ = after_local ()
		(* context with augmented env *)
		val c2 = {kind=kind, top=top, fullenv=Env.atop(env1,fullenv),
			  rpath=rpath, region=region, scope=scope}
		val (binding,absstr) = elab c2 str
		val _ = debugmsg "elab:LetStr - end"
	     in (binding,LETstr(ld1,absstr))
	    end

        | elab {kind,top,fullenv,rpath,region,scope} 
	      (MarkStr(str,region')) = 
	    let val (binding,str') =
		     elab {kind=kind,top=top,fullenv=fullenv,rpath=rpath,
			   region=region',scope=scope} 
		          str
	     in (binding,MARKstr(str',region'))
	    end

      (* Build the uncoerced structure from its specification *)
      val (strRaw,strDec) = elab context def
      (* Coerce it if necessary and get back the thinning *)
      val ((coercedStr,absWrap,thin),sgnOption) = 
	case constraint
	  of NONE => ((strRaw,NONE,NONE),NONE)
	   | SOME sgn =>
	     let (* translate the signature in the new environment *)
		 val sgnElab = 
		   elabTOPsig (error,region)
			      (fullenv,Stamps.newStamp scope,NONE)
			      sgn
		 fun computeSelf (StructStr _) = true
		   | computeSelf (MarkStr(def,_)) = computeSelf def
		   | computeSelf _ = false
		 val self = computeSelf def
	      in (* translate the structure in a coerced one *)
	         (SigMatch.match
		   {abstract=abstract,arg_option=NONE,err=error region,scope=scope,
		    rpath=rpath, printEnv=fullenv, self=self, str=strRaw,
		    sign=sgnElab}, 
		  SOME sgnElab)
	     end
      (* Fix the version of the structure stored in the environments *)
      val (resVar,symbind) = addStr2(name,coercedStr,kind)
   in (STRB{strvar=resVar,abslty=absWrap,thin=thin,
	    constraint=sgnOption,def=strDec},
       symbind)
  end

(*** ELABORATE A FUNCTOR ***)

and elaborate_functor
      (context as {kind,top,fullenv,rpath,region,scope}: contextStr, name,
       VarFct(spath,constraint)) =
      let (* look for the functor *)
	  val fctvar as FCTvar{binding=fct,...} =
	        lookFCT(fullenv,SP.SPATH spath,error region)
	  (* coerce it if necessary *)
	  val ((fctcoerced,thin),constraint') =
	    case constraint
	      of NONE => ((fct,NONE),NONE)
	       | SOME fsig =>
		  let val (parent,normTotal) =
			  case kind
			    of InstanceStr{normTotal,parent,...} => 
			       (signatureOfStructure parent, ref(!normTotal))
			     | _ => (ERROR_SIG,ref Normalize.empty)
		      (* signature context to elaborate the functor signature *)
		      val context =
			  {strs=ref 0, fcts=ref 0, tycons=ref 0, slots=ref 0,
			   total=ref fullenv, makeStamp=Stamps.newStamp scope,
			   inner=ref Env.empty, parent=parent,normTotal=normTotal,
			   names = IP.IPATH[], t = ref [], s = ref [], a = ref []}
		      val fsig = 
			  ElabSig.make_fsigexp (error,region) NONE context fsig
		   in (* verifies that the signature matches the functor *)
		      (SigMatch.matchFct 
			{abstract=false, err=error region, rpath=rpath,
			 self=false, scope=scope, printEnv=fullenv, fsig=fsig,
			 fct=fct},
		       SOME fsig)
		  end
          val fctThinning = case thin of NONE => (NONE,NONE,NONE) 
                                       | SOME fctThin => fctThin 
 
	  (* add the definition obtained to the environments *)
	  val (realVar,formalVar) = addFct(name,fctcoerced,kind)
       in (FCTB{fctvar=realVar,
		def=VARfct{fctThin=fctThinning,
			   constraint=constraint',def=fctvar}},
	   FCTbind formalVar) 
      end

  | elaborate_functor 
      (context as {kind,top,fullenv,rpath,region,scope}, name,
       LetFct(ldec,fct)) =
      let (* contexts with their own env reference must be restored
	     after elaborating local decls *)
	  val after_local = 
	      case kind
		of InstanceStr{env,...} => 
		     let val old_env = !env in fn () => env := old_env end
		 | SimpleStr env =>
		     let val old_env = !env in fn () => env := old_env end
		 | _ => (fn () => ())
	  (* unlike the LetStr case, we don't strip off a SuperStr constructor
	     from the kind for the local context -- probably because it can't
	     be there because a functor expression cannot be the body of a
	     functor? *)
	  val c1 = {kind=kind, top=false, fullenv=fullenv, rpath=rpath,
		    region=region, scope=scope}
	  val (ld1,env1) =  elabDecl(c1,ldec)
	  val _ = after_local ()
	  (* context with augmented env *)
	  val c2 = 
	      {kind=kind, top=top, fullenv=Env.atop(env1,fullenv),
	       rpath=rpath, region=region, scope=scope}
	  val (FCTB{fctvar, def}, binding) = elaborate_functor(c2,name,fct)
	  val absfct = FCTB{fctvar=fctvar,def=LETfct(ld1,def)}
       in (absfct,binding)
      end

  | elaborate_functor
      (context as {kind,top,fullenv,rpath,region,scope}, name,
       MarkFct(fct,region')) =
      elaborate_functor 
        ({kind=kind, top=top, fullenv=fullenv, rpath=rpath,
	  scope=scope, region=region'},
         name,fct)

  | elaborate_functor (context, name, AppFct(qid,larg,constraint)) =
      elaborate_functor
        (context, name,
	 LetFct(StrDec[Strb{name=hiddenId,constraint=NONE,
			    def=AppStr(qid,larg)}],
		VarFct([hiddenId,functorId],constraint)))

  | elaborate_functor (context,_,FctFct{params=[],...}) =
      impossible "elaborate_functor.FctFct"

  | elaborate_functor
      (context as {kind,top,fullenv,rpath,region,scope}, name,
       FctFct{params=[param],body=def,constraint}) =
      let (* to identify what has been elaborated during the
	     treatment of the functor *)
	  val argScope = Stamps.newBoundScope ()
	  val inParam = Stamps.isBound argScope
	  val bodyScope = Stamps.newBoundScope ()
	  val inBody = Stamps.isBound bodyScope
	  val makestamp = Stamps.newStamp bodyScope
	  (* basic infos on the context of elaboration *)
	  val (inFunctor,inArgument,parent,normTotal) = 
	      case kind
		of InstanceStr{inFunctor,inArgument,parent,normTotal,...} =>
		     (inFunctor,inArgument,parent,!normTotal)
		 | _ => (fn _ => false,fn _ => false,ERROR_STR,Normalize.empty)
	  (* get back the signature of the parent *)
	  val sigParent = signatureOfStructure parent
	  val normParent = Normalize.prefix(parentId,normTotal)
	  val normTotalAll = ref normParent
	  val (nameParam,specParam) =
	      case param of (NONE,s) => (parameterId,s) | (SOME n,s) => (n,s)
	  val accessParam = LVAR(namedLvar nameParam)
	  (* elaborate the signature of the argument (pair parent + parameter) *)
	  val sgnArg = 
		elabPARTIALsig 
		  (error,region)
		  (parentId,sigParent,fullenv,normTotalAll,
		   Stamps.newStamp Stamps.freeScope)
		  (SigSig[StrSpec[(parameterId,specParam)]])
	  (* this signature is instantiated with the real parent to substitute
	     in place of the parent signature *)
	  val strArg = Instantiate.instantiate_argument
			 ((IP.IPATH [],argScope,error region),
			  nameParam,parent,sgnArg)
	  val _ = debugPrint("elab_func(FctFct([param]): parameter structure",
			     PPBasics.ppStructure,(fullenv,strArg,20))
	  (* define a structureVar for the argument (dummy name, access) *)
	  val varArg = STRvar{name=argumentId,access=NO_ACCESS,binding=strArg}
	  (* extract the instantiated parameter structure from strArg *)
  	  val strParam = lookBindingSTR (strArg,SP.SPATH [parameterId])
	  val varParam =
		STRvar{name=nameParam,access=accessParam,binding=strParam}
	  (* elaboration of the Body begins here *)
	  (* so the elaboration of the argument is what has been until now 
	     elaboration of parent is included *)
	  fun inArg s = 
	      (inFunctor s) orelse (inParam s)
	  (* for the body, identify all that has been done while
	     parsing parameters (their signature and their instantiation) *)
	  fun inFunctorBody s =
	      (inFunctor s) orelse (inParam s) orelse (inBody s)
	  fun inArgumentBody s = 
	      (inArgument s) orelse (inParam s)
	  (* adjust fullenv and normTotal with informations on the parameter *)
	  val (newenv,newNormTotal) =
	      case param 
		of (NONE,_) =>
		     (openStructureVar(fullenv,varParam),
		      Normalize.openSubStr(parameterId,sgnArg,normParent))
		 | (SOME _,_) => 
		     (Env.bind(nameParam, STRbind varParam, fullenv), 
		      Normalize.extend(nameParam,IP.IPATH[parameterId],
				       normParent))
	  (* a new kind and a new context for the body *)
	  val normTotal = ref (Normalize.prefix(argumentId,newNormTotal))
	  val newKind = 
	      let (* same as regular Instance + slot for argument *)
		  val (strs,fcts,typs) = countStrFctTyp def
		  val strArray = Array.array(strs+1,ERROR_STR)
		  val fctArray = Array.array(fcts,ERROR_FCT)
		  val typArray = Array.array(typs,ERRORtyc)
		  val argumentVar = 
		      STRvar
			{binding =
			   STR_FORMAL
			     {pos=0,
			      spec=signatureOfStructure strArg},
			 name = argumentId, access = NO_ACCESS}
		  val env = ref(Env.bind(argumentId,STRbind argumentVar,
					 Env.empty))
		  val kindinst = 
		    TOP{strcount=strs+1,fctcount=fcts,typecount=typs,
			slotcount=0, sConstraints=[], tConstraints=[],
			abbreviations=[]}
		  val sgn =
		    SIG{symbols=ref [], env=env, name=NONE,
			stamp=makestamp (), kind=ref kindinst}
		  val str =
		    INSTANCE{
		      sign = sgn, path = rpath,
		      subStrs=strArray, subFcts=fctArray,
		      types=typArray,
		      origin = STAMP_ORIGIN(makestamp ())}
		  val _ = Array.update(strArray,0,strArg)
	       in SuperStr(InstanceStr
		    {env = env, normTotal = normTotal,myself=str,
		     strCount = ref 1, fctCount = ref 0, typCount = ref 0,
		     parent = str,
		     inFunctor = inFunctorBody,
		     inArgument = inArgumentBody,
		     strArray = strArray,
		     fctArray = fctArray,
		     typArray = typArray})
	      end
	  val bodyContext: contextStr = 
		{kind=newKind,top=true,fullenv=newenv,rpath=IP.IPATH [],
		 region=region,scope=bodyScope}
	  val _ = debugmsg "elab_func:FctFct([param]) - start body"
	  (* the body is built *)
	  val (STRB{strvar,abslty,thin,constraint,def=defBody},bodyBind) =
	    elaborate_structure{context=bodyContext,abstract=false,
				name=bodyId,def=def,constraint=constraint}
	  val _ = debugmsg "elab_func:FctFct([param]) - body elaborated"
	  val STRvar{binding=strBody,...} = strvar

	  val _ = debugPrint("elab_func(FctFct([param]): body structure",
			     PPBasics.ppStructure,(newenv,strBody,20))

          val lambdaty=
               LambdaType.injARROW(TransBinding.transStrLty(strParam),
                                   TransBinding.transStrLty(strBody))

	  (* Then it is abstracted *)
	  val fctBody =
		AbstractFct.abstractBody (strBody,strArg,inBody,inArg)
	  (* So that the  functor can be built *)
	  val fct =
		FCT{stamp = Stamps.newStamp scope (),
                    lambdaty=lambdaty, 
		    parent = parent,
		    paramName = nameParam,
		    argument = sgnArg,
		    body = fctBody}
	  (* and added to the environments *)
	  val (realVar,formalVar) = addFct(name,fct,kind)
       in (FCTB{fctvar = realVar,
		def = FCTfct{param = varParam, def = defBody, thin = thin,
			     constraint = constraint}},
	   FCTbind formalVar)
      end

  | elaborate_functor
      (context, name, FctFct{params=param :: lparam,body,constraint}) =
      elaborate_functor
        (context,name,
	 FctFct{params=[param],
		body=StructStr(FctDec[Fctb
				      {name=functorId,
				       def=FctFct{params=lparam,body=body,
						  constraint=constraint}}]),
		constraint=NONE})

(*** ELABORATE A DECLARATION (STRUCTURE LEVEL) ***)

and elabDecl (context as {kind,top,fullenv,rpath,scope,region}, decl) =
case decl
  of OpenDec strs =>
       let fun makeOpenDecls (str, spath) =
	       let (* get a list of component names (symbols) from a structure *)
		   fun getSymbols(SIMPLE{env,...}) =
			 let val r = ref([]: Symbol.symbol list)
			  in Env.app (fn (s,_) => r := s::(!r)) env;
			     !r
			 end
		     | getSymbols(INSTANCE{sign=SIG{symbols,...},...}) = !symbols
		     | getSymbols ERROR_STR = []
		     | getSymbols (APPLY{res,...}) =   (* 103e: fix bug 833 *)
		         getSymbols res
		     | getSymbols _ = ErrorMsg.impossible "getSymbols"
		   fun makeDecl(comp: Symbol.symbol, decls : Ast.dec list)
			 : Ast.dec list =
		       case Symbol.nameSpace comp
			 of VALspace =>
			     (case lookBinding(str,SP.SPATH [comp],NO_ACCESS)
				of VARbind(VALvar _) =>
				     (* ignore OVLDvar variables *)
				     ValDec([Vb{pat=VarPat[comp],
						exp=VarExp([localStrName,comp])}])
				     :: decls
				 | CONbind(DATACON{rep=VARIABLE _,...}) =>
				     ExceptionDec([EbDef{exn=comp,
						      edef=([localStrName,comp])}])
				     :: decls
				 | CONbind(DATACON{rep=VARIABLEc _,...}) =>
				     ExceptionDec([EbDef{exn=comp,
						      edef=([localStrName,comp])}])
				     :: decls
				 | _ => decls) (* ordinary datacon *)
			  | STRspace =>
			      if Symbol.eq(comp, Extern.argumentId)
			      then decls (* ignore "<Argument>" *)
			      else StrDec[Strb{name=comp,
					       def=VarStr([localStrName,comp]),
					       constraint=NONE}]
				   :: decls
			  | FCTspace =>
			      FctDec[Fctb{name=comp,
					  def=VarFct([localStrName,comp],NONE)}]
			      :: decls
			  | _ => decls
		in LocalDec(StrDec[Strb{name=localStrName,
					def = VarStr(spath),
					constraint = NONE}],
			    SeqDec(foldr makeDecl [] (getSymbols str)))
	       end
	   (* get the definition of structures *)
	   val lstr = map (fn id => lookSTR(fullenv,SP.SPATH id,error region)) strs
	   (* open their environment *)
	   val openEnv =
	     foldl 
	       (fn (str,env) => openStructureVar(env,str))
	       Env.empty lstr
	   (* if the kind is Instance there is a lot of work to do:
	       - create a dummy that contains the defs obtained in openEnv
	       - build the indirection in the current environment
	       - add the dummy to the current structure
	      The reason for this: we must only use one slot per open *)
	   val kind' = case kind of SuperStr k => k | _ => kind
	   val _ =
	     case kind'
	       of InstanceStr{env,normTotal,strCount,strArray,...} =>
		  let val strC = ref 0 and fctC = ref 0 and typC = ref 0
		      val strA = ref [] and fctA = ref [] and typA = ref []
		      val auxEnv = ref Env.empty
		      val posOpen = next strCount
		      val openEnv' = ref Env.empty
		      (* openStruct builds the new dummy and records the new
			 position of each element *)
		      fun openStruct 
			      (name,
			       orig as STRbind(STRvar{binding,access,...})) =
			    let val pos = next strC
				val auxBind =
				  STRbind(STRvar{
				    name=name, access=access,
				    binding=
				      STR_FORMAL{pos=pos,spec=FULL_SIG}})
				val openBind =
				  STRbind(STRvar{
				    name=name, access=access,
				    binding=
				      STR_OPEN{pos=[posOpen,pos],
					       spec=FULL_SIG,
					       path=SP.SPATH [name]}})
			     in strA := binding :: (!strA);
				auxEnv := Env.bind(name,auxBind,!auxEnv);
				openBind
			    end
			| openStruct 
			      (name,
			       orig as FCTbind(FCTvar{binding,access,...}))=
			    let val pos = next fctC
				val auxBind =
				  FCTbind(FCTvar{
				    name=name,access=access,
				    binding=
				      FCT_FORMAL{pos=pos,spec=FULL_FSIG}})
				val openBind =
				  FCTbind(FCTvar{
				    name=name,
				    binding=
				      FCT_OPEN{pos=[posOpen,pos],
					       spec=FULL_FSIG,
					       path=SP.SPATH [name]},
				    access=access})
			     in fctA := binding :: (!fctA);
				auxEnv := Env.bind(name,auxBind,!auxEnv);
				openBind
			    end
			| openStruct (name,orig as TYCbind binding) =
			    let val pos = next typC
				val auxBind =
				  TYCbind(FORMtyc{name=name,pos = pos,
						  spec=binding})
				val openBind =
				  TYCbind(OPENFORMtyc{
				       pos=([posOpen],pos),spec=binding,
				       path=SP.SPATH [name]})
			     in typA := binding :: (!typA);
				auxEnv := Env.bind(name,auxBind,!auxEnv);
				openBind
			    end
			| openStruct (name,binding) = binding
		       (* apply openStruct on the whole environment *)
		      val _ = 
			Env.app
			  (fn binding as (name,_) => 
			    env := Env.bind(name,openStruct binding,!env))
			  openEnv;
		       (* the kind of the intermediate with counts coming
			  from the application of openStruct *)
		      val kindinst = 
			TOP{strcount= !strC,fctcount= !fctC,typecount= !typC,
			    slotcount=0, sConstraints=[], tConstraints=[],
			    abbreviations=[]}
		       (* the definition of the intermediate structure *)
		      val openStr =
			  INSTANCE{sign = SIG{symbols=ref [],env=auxEnv,
					      kind=ref kindinst,name = NONE,
					      stamp = Stamps.newStamp scope ()},
				   subStrs = Array.fromList(rev (!strA)),
				   subFcts = Array.fromList(rev (!fctA)),
				   types = Array.fromList(rev (!typA)),
				   origin = STAMP_ORIGIN(Stamps.newStamp scope ()),
				   path = IP.extend(rpath,openId)}
		   in Array.update(strArray,posOpen,openStr)
		  end
	       | SimpleStr env => env := Env.atop(openEnv,!env)
	       | TopLevel => ()
       	       | SuperStr _ => impossible "SuperStr 2"
	in case (kind,top,!Control.copyToplevelOpen)
	     of (TopLevel,true,true) =>
		 let val newDecs = 
			 foldr (fn ((str,spath),decs) =>
				    makeOpenDecls(str,spath) :: decs)
			   []
			   (List2.zip2(map (fn STRvar{binding,...} => binding)
					 lstr,
				     strs))
		     val (SEQdec ld1,e1) = elabDecl(context,(SeqDec newDecs))
		  in (SEQdec(OPENdec lstr :: ld1),  (* OPENdec is misplaced
						       but should be a no-op *)
		      Env.consolidate(Env.atop(e1,openEnv)))
		 end
	      | _ => (OPENdec lstr,openEnv)
       end

   | StrDec pstrl =>
       let fun elabStrb region (Strb{name,constraint,def},(ldec,env)) =
		 let val (dec as STRB{strvar,...},str) =
			 elaborate_structure 
			   {context={kind=kind,top=true,fullenv=fullenv,
				     rpath=IP.extend(rpath,name),scope=scope,
				     region=region},
			    abstract=false,name=name,constraint=constraint,
			    def=def}
		  in (dec::ldec, Env.bind(name,STRbind strvar,env))
		 end
	     | elabStrb _ (MarkStrb(s,region),ctx) = elabStrb region (s,ctx)
	   val (strbl,env) = foldl (elabStrb region) ([],Env.empty) pstrl
	in (STRdec(rev strbl), env)
       end

   | AbsDec pstrl =>
       let fun elabAbs region (Strb{name,constraint,def},(ldec,env)) =
		 let val (dec as STRB{strvar,...},_) =
			 elaborate_structure 
			   {context={kind=kind,top=true,fullenv=fullenv,
				     rpath=IP.extend(rpath,name), scope=scope,
				     region=region},
			    abstract=true,name=name,constraint=constraint,
			    def=def}
		  in  case constraint
 		       of NONE => error region ErrorMsg.COMPLAIN 
 			           "abstraction without signature constraint"
 				   ErrorMsg.nullErrorBody
 		        | _ => ();
                     (dec::ldec, Env.bind(name,STRbind strvar,env))
		 end
	     | elabAbs _ (MarkStrb(s,region),ctx) = elabAbs region (s,ctx)
	   val (strbl,env) = foldl (elabAbs region) ([],Env.empty) pstrl
	in (ABSdec(rev strbl), env)
       end

   | FctDec pfctl =>
       let fun elabFctb region (Fctb{name,def},(ldec,env)) =
		 let val (dec as FCTB{fctvar,...},_) =
			 elaborate_functor(context,name,def)
		  in (dec::ldec, Env.bind(name,FCTbind fctvar,env))
		 end
	     | elabFctb _ (MarkFctb(fctb,region),ctx) = 
		 elabFctb region (fctb,ctx)
	   val (fctbl,env) = foldl (elabFctb region) ([],Env.empty) pfctl
	in (FCTdec(rev fctbl),env)
       end

   | SigDec psigbl =>
       let fun elabSigb region (Sigb{name,def},(ldec,env)) =
		 let val sgndef = 
			 elabTOPsig 
			   (error,region) 
			   (fullenv,Stamps.newStamp scope,SOME name) def
		     val sgn =
		         SIGvar{name=name, binding= sgndef}
		     val sigbind = SIGbind sgn
		  in (sgn::ldec,Env.bind(name,sigbind,env))
		 end
	     | elabSigb _ (MarkSigb(s,region),ctx) = elabSigb region (s,ctx)
	   val (sigbl,env) = foldl (elabSigb region) ([],Env.empty) psigbl
        in (SIGdec (rev sigbl),env)
       end

   | FsigDec pfsigbl =>
       let fun elabFsigb region (Fsigb{name,def},(ldec,env)) =
		 let val (ldec',env') = 
		         elabFSIGB (error,region) (#fullenv context,scope) (name,def)
		  in (ldec'@ldec,Env.atop(env',env))
		 end
	     | elabFsigb _ (MarkFsigb(s,region),ctx) = 
		 elabFsigb region (s,ctx)
           val (fsigbl,env) = 
               foldl (elabFsigb region) ([],Env.empty) pfsigbl
        in (FSIGdec(rev fsigbl), env)
       end

   | LocalDec(ldec_in,ldec_out) =>
       let (* context kinds with their own env ref need to be restored
	      after local decls are elaborated.
	      Maybe local decls should have a different context kind? *)
	   val after_local = 
	       case kind
		 of InstanceStr{env,...} => 
		      let val old_env = !env in fn () => env := old_env end
		  | SimpleStr env =>
		      let val old_env = !env in fn () => env := old_env end
		  | _ => (fn () => ())
           val c_in = {kind=kind, top=false, fullenv=fullenv, rpath=rpath,
		       region=region, scope=scope}
           val (ld1,env1) =  elabDecl(c_in,ldec_in)
	   val _ = after_local ()
	   (* context with augmented env *)
	   val c_out = {kind=kind, top=top, fullenv= Env.atop(env1,fullenv),
			rpath=rpath, region=region, scope=scope}
	   val (ld2,env2) = elabDecl(c_out,ldec_out)
        in (LOCALdec(ld1,ld2),env2)
       end

   | SeqDec ldec => 
       let val (ldec,env) = 
	       foldl
		 (fn (decl,(ld,env)) =>
		    let val (ld',env') =
			  elabDecl({kind=kind, top=top,
				    fullenv=Env.atop(env,fullenv),
				    rpath=rpath, region=region, scope=scope},
				   decl)
		     in (ld' :: ld,Env.atop(env',env))
		    end)
		 ([],Env.empty) ldec
	in (SEQdec(rev ldec),env)
       end

   | MarkDec(dec,region) =>
       let val (dec',env) = 
	   elabDecl({kind=kind,top=top,fullenv=fullenv,rpath=rpath,
		     region=region,scope=scope},
		    dec)
	in (if !Control.markabsyn then MARKdec(dec',region) else dec', env)
       end

   | dec =>
        let val (decl',env',tyv,updt) =
		ElabCore.elabDec (coreEnv,error,errorMatch,region) 
				 (fullenv,rpath,scope) 
				 dec
	    val _ = updt tyv
	    val decl'' = transform decl'
            val _ = Env.app
		      (fn (name,obj) => (addObject(name,obj,kind,rpath); ()))
		      env'
            val ndecl = Typecheck.decType(Env.atop (env',fullenv), decl'',
                                          top, error,region)
	 in (ndecl,env')
	end

 (* body of elaborateTop *)
 in elabDecl({kind=TopLevel, top=true, fullenv=env, region=(0,0),
	      rpath=IP.IPATH[], scope = Stamps.freeScope},
             lpdec)

end (* elaborateTop *)

end (* structure ElabStr *)
