(* Copyright 1992 by AT&T Bell Laboratories *)
(***************************************************************************

	ELABCORE.SML translate AST to Absyn for the core language

 ***************************************************************************)

structure ElabCore: ELABCORE =
struct

open BasicTypes Symbol Absyn Ast PrintUtil AstUtil Types BasicTypes TyvarSet
     Modules EqTypes ModuleUtil TypesUtil Variables Misc ElabUtil ErrorMsg
     Access 
structure SP = SymPath
structure IP = InvPath


infix -->
type tyvUpdate = tyvarset -> unit
fun no_updt (_ : tyvarset) = ()

val union = union_tyvars

fun smash f l = 
    let fun h(a,(pl,oldl,newl)) =
	  let val (p,old,new) = f a
	   in (p::pl,old@oldl,new@newl)
	  end
     in foldr h (nil,nil,nil) l
    end

fun patproc pp0 =
    let val oldnew : (Absyn.pat * var) list ref = ref nil

	fun f (p as VARpat(VALvar{access=LVAR v,typ=ref typ',path})) =
	      let fun find ((VARpat(VALvar{access=LVAR w,...}),x)::rest) = 
			if v=w then x else find rest
		    | find nil =
		        let val x = VALvar{access=LVAR (dupLvar(v)),
					   typ=ref(typ'),path=path}
			 in oldnew := (p,x):: !oldnew;
			    x
			end
	       in VARpat(find(!oldnew))
	      end
	  | f (VARpat(VALvar{access,typ=ref typ',path})) =
		VARpat(VALvar{access=access,typ=ref(typ'),path=path})
	  | f (RECORDpat{fields,flex,typ}) =
		RECORDpat{fields=map (fn(l,p)=>(l,f p)) fields,flex=flex,typ=typ}
	  | f (VECTORpat(pats,t)) = VECTORpat(map f pats, t)
	  | f (APPpat(d,c,p)) = APPpat(d,c,f p)
	  | f (ORpat(a,b)) = ORpat(f a, f b)
	  | f (CONSTRAINTpat(p,t)) = CONSTRAINTpat(f p, t)
	  | f (LAYEREDpat(p,q)) = LAYEREDpat(f p, f q)
	  | f p = p

     in (f pp0, map #1 (!oldnew), map #2 (!oldnew))
    end

fun bindpat pp =
    case pp 
     of VARpat(VALvar{access=INLINE(_),...}) => false
      | CONSTRAINTpat(VARpat(VALvar{access=INLINE _,...}),_) => false
      | VARpat(VALvar{access=LVAR v,...}) => false
      | CONSTRAINTpat(VARpat(VALvar{access=LVAR v,...}),_) => false
      | _ => true

(**** TYPES ****)

fun elabTyv error (region:region) (tyv:Ast.tyvar) =
    case tyv
      of Tyv vt => mkTyvar(mkUBOUND(vt, error region))
       | MarkTyv(tyv,region) => elabTyv error region tyv

fun elabTyvList error region tyvars =
    let val tvs = map (elabTyv error region) tyvars
	val _ =  checkUniq ((error region),"duplicate type variable name")
			   (map (fn (ref(ubound as OPEN{kind=UBOUND name,...}))
					  => name
				  | _ => impossible "elabTyvList") tvs)
     in tvs
    end

type typeEnv = Modules.env * Normalize.normMap option

fun elabType error (region:region) (env:typeEnv) (ast:Ast.ty) : 
      (Types.ty * tyvarset) =
    case ast
      of VarTy vt => 
	   let val tyv = elabTyv error region vt
	    in (VARty tyv, singleton_tyvar tyv)
	   end
       | ConTy (co,ts) => 
	   let val co1 = 
		 if (Symbol.name (hd co)) = "->"
		 then BasicTypes.arrowTycon
		 else lookArTYC(env,SP.SPATH co,length ts,error region)
	       val (lts1,lvt1) = elabTypeList error region env ts
	    in (mkCONty (co1,lts1),lvt1)
	   end
       | RecordTy lbs => 
	   let val (lbs1,lvt1) = elabTLabel error region env lbs
	    in (recordTy(sortRecord(lbs1,error region)),lvt1)
	   end
       | TupleTy ts =>
	   let val (lts1,lvt1) = elabTypeList error region env ts
	    in (tupleTy lts1,lvt1)
	   end
       | MarkTy (ty,region) => elabType error region env ty

and elabTLabel error (region:region) (env:typeEnv) labs =
    foldr 
      (fn ((lb2,t2),(lts2,lvt2)) => 
	  let val (t3,lvt3) = elabType error region env t2
	  in ((lb2,t3) :: lts2, union(lvt3,lvt2,error region)) end)
      ([],no_tyvars) labs

and elabTypeList error (region:region) (env:typeEnv) ts =
    foldr 
      (fn (t2,(lts2,lvt2)) => 
	  let val (t3,lvt3) = elabType error region env t2
	  in (t3 :: lts2, union(lvt3,lvt2,error region)) end)
      ([],no_tyvars) ts


(**** DATATYPE DECLARATION ****)
exception ISREC

fun elabDB (envorig,env,rpath:IP.path,error) (args,name,def,region) =
    let val rhs = mkCONty(lookArTYC (env,SP.SPATH[name],length args,error region),
			  map VARty args)

	fun checkrec(_,NONE) = ()
	  | checkrec(_,SOME typ) = 
	      let fun findname(VarTy _) = ()
		    | findname(ConTy([co],ts)) = if co = name then (raise ISREC) 
						 else app findname ts
		    | findname(ConTy(_,ts)) = app findname ts
		    | findname(RecordTy lbs) = app (fn (_,t) => findname t) lbs
		    | findname(TupleTy ts) = app findname ts
		    | findname(MarkTy(t,_)) = findname t
	       in findname(typ)
	      end

	fun elabConstr (name,SOME ty) =
	      let val (t,tv) = elabType error region env ty
	       in ((name,false,(t --> rhs)),tv)
	      end
	  | elabConstr (name,NONE) = ((name,true,rhs),no_tyvars)

	val arity = length args
	val isrec = (app checkrec def; false) handle ISREC => true
	val (dcl,tvs) = 
	      foldr
		(fn (d,(dcl1,tvs1)) =>
		   let val (dc2,tv2) = elabConstr d
		   in (dc2::dcl1,union(tv2,tvs1,error region)) end)
		([],no_tyvars) def
	val _ = checkbound(tvs,args,error region)
	val _ = TypesUtil.bindTyvars args
	val sdcl = sort3 dcl
	val sign = ConRep.boxed isrec sdcl
	fun binddcons ((sym,const,typ),rep) =
	      let val _ = compressTy typ
		  val typ = 
		      if arity > 0
		      then POLYty {sign=mkPolySign arity,abs=0,
				   tyfun=TYFUN{arity=arity,body=typ}}
		      else typ
	       in DATACON{name=sym, const=const, rep=rep, 
                          sign=sign, typ=typ, orig=NONE}
	      end
	fun binddconslist ((r1 as (name,_,_))::l1,r2::l2) =
	      let val dcon = binddcons (r1,r2)
		  val (dcl,e2) = binddconslist (l1,l2)
	       in (dcon::dcl,Env.bind(name,CONbind dcon,e2))
	      end
	  | binddconslist ([],[]) = ([],envorig)
	  | binddconslist _ = impossible "elabDB.binddcons"
     in if length sdcl < length dcl  (* duplicate constructor names *)
	then let fun member(x:string,[]) = false
		   | member(x,y::r) = (x = y) orelse member(x,r)
		 fun dups([],l) = l
		   | dups(x::r,l) =
		       if member(x,r) andalso not(member(x,l))
		       then dups(r,x::l)
		       else dups(r,l)
		 fun add_commas [] = []
		   | add_commas (y as [_]) = y
		   | add_commas (s::r) = s :: "," :: add_commas(r)
		 val duplicates = dups(map (fn (n,_,_) => Symbol.name n) dcl,[])
	      in error region COMPLAIN
		   (concat["datatype ", Symbol.name name,
			    " has duplicate constructor name(s): ",
			    concat(add_commas(duplicates))])
		   nullErrorBody
	     end
	else ();
	binddconslist(sdcl,sign)
    end

fun elabDec (coreEnv,error,errorMatch,region) =

let val completeMatch = ElabUtil.completeMatch(coreEnv,"Match")
    val completeBind = ElabUtil.completeMatch(coreEnv,"Bind")
 
 (**** TYPE DECLARATION ****)

 fun elabTB (region:region) 
	    (ctx as (env:Modules.env,rpath:IP.path))
	    (notwith:bool) (tb : Ast.tb) : (Absyn.tb list * Modules.env) =
     case tb
       of Tb{tyc,def,tyvars} =>
	    let val tvs = elabTyvList error region tyvars
		val ty = elabType error region (env,NONE) def
	     in makeTB(tvs,tyc,ty,error region) notwith ctx
	    end
	| MarkTb (tb,region) =>
	    elabTB region ctx notwith tb

 fun elabTBlist region (ctx as (env:Modules.env,rpath:IP.path)) 
       (notwith:bool) (tbl:Ast.tb list) : (Absyn.tb list * Modules.env) =
     foldl
       (fn (typb2,(tbs2,env2)) =>
	  let val (tb3,env3) =
	          elabTB region (Env.atop (env2,env),rpath) notwith typb2
	   in (tb3 @ tbs2, Env.atop(env3,env2))
	  end)
       ([],Env.empty) tbl

 fun elabTYPEdec
       region
       (ctx as (env:Modules.env,rpath:IP.path,st:Stamps.scope))
       (tbl : Ast.tb list): (Absyn.dec * Modules.env * tyvarset * tyvUpdate) =
     let val tbenv1 = elabTBlist region (env,rpath) true tbl
	 val (dec4,env4) = makeTYPEdec (tbenv1,error region)
      in (dec4,env4,no_tyvars,no_updt)
     end

 fun elabDATATYPEdec region (env,rpath:IP.path,scope:Stamps.scope) (db,tb) =
     let (* predefine datatypes *)
	 fun predefine region (Db{tyc=id,def,tyvars}) = 
	       let val r = ref(DATAtyc nil)
		   val tyc = 
		     GENtyc{path=IP.extend(rpath,id), arity=length tyvars,
			    eq=ref DATA, stamp=Stamps.newStamp scope (), kind=r}
		   val tvs = elabTyvList error region tyvars
	        in (tvs,id,tyc,def,r,region)
	       end
	   | predefine _ (MarkDb(db,region)) = predefine region db

	 val (env1,db1,datatycs,datatyc_names,datacon_names) = 
	   foldl
	     (fn (db2,(env2,dbs2,dts2,dtns2,dcns2)) => 
		let val (tvs,id3,tyc3,def3,r3,region3) = predefine region db2
		 in (Env.bind(id3,TYCbind tyc3,env2),
		     ((tvs,id3,def3,region3),r3)::dbs2, tyc3::dts2, id3::dtns2,
		      (map #1 def3) @ dcns2)
		end)
	     (Env.empty,[],[],[],[]) db

	 (* define types associated *)
	 val (withtycs,env4) = 
	   elabTBlist region (Env.atop(env1,env),rpath) false tb

	 (* builds the resulting environment *)
	 val env5 = Env.atop(env4, env1)
	 val fullEnv = (Env.atop(env5,env),NONE)

	 (* look at the definition of datatypes *)
	 fun redefine ((db,r),env6) = 
	     let val (dcons,env7) = elabDB (env6,fullEnv,rpath,error) db 
	      in r := DATAtyc(dcons); env7
	     end
	 val env8 = foldl redefine env5 db1
	 val withtyc_names =
	     map (fn TB{tyc=DEFtyc{path,...},...} => IP.last path
		   | _ => ErrorMsg.impossible "CoreLang.elabDATATYPEdec")
		 withtycs
	 val dtdec = DATATYPEdec{datatycs=rev datatycs,withtycs=rev withtycs}
      in checkUniq (error region, "duplicate type names in type declaration")
		   (datatyc_names @ withtyc_names);
	 checkUniq (error region, "duplicate datacon names in datatype declaration")
		   datacon_names;
	 app (defineEqTycon (fn x => x)) datatycs;
	 (dtdec,env8,no_tyvars,no_updt)
     end

 (**** EXCEPTION DECLARATION ****)

 fun elabEb (region:region) (env:Modules.env) (eb:Ast.eb) =
     case eb
       of EbGen{exn=id,etype=NONE} =>
	    let val exn = DATACON{name=id,const=true,typ=exnTy,orig=NONE,
				  rep=VARIABLEc(LVAR(namedLvar id)), sign=[]}
	     in ([EBgen{exn=exn,etype=NONE,ident=STRINGexp(Symbol.name id)}], 
		 Env.bind(id, CONbind exn, Env.empty),no_tyvars)
	    end
	| EbGen{exn=id,etype=SOME typ} =>
	    let val (ty,vt) = elabType error region (env,NONE) typ
		val exn = DATACON{name=id,const=false,typ=(ty --> exnTy),
                          orig=NONE,rep=VARIABLE(LVAR(namedLvar id)),sign=[]}
	     in ([EBgen{exn=exn,etype=SOME ty,
                        ident=STRINGexp(Symbol.name id)}],
	         Env.bind(id,CONbind exn, Env.empty),vt) 
	    end
	| EbDef{exn=id,edef=qid} =>
	    let val edef as DATACON{const,typ,rep,sign,orig,...} =
		  lookEXN(env,SP.SPATH qid,error region)
		val exn = DATACON{name=id,const=const,typ=typ,
                                  sign=sign,orig=orig,
				  rep=if const 
                                      then VARIABLEc(LVAR(namedLvar id))
				      else VARIABLE(LVAR(namedLvar id))}
	     in ([EBdef{exn=exn,edef=edef}],
		 Env.bind(id,CONbind exn,Env.empty),no_tyvars)
	    end
	| MarkEb(eb,region) => elabEb region env eb

 fun elabEXCEPTIONdec region (env: Modules.env) (excbinds:Ast.eb list) =
     let val (ebs,env,vt) = 
	   foldl
	     (fn (exc1,(ebs1,env1,vt1)) =>
		let val (eb2,env2,vt2) = elabEb region env exc1
		in (eb2@ebs1,Env.atop(env2,env1),union(vt1,vt2,error region)) end)
	      ([],Env.empty,no_tyvars) excbinds
	 fun getname(EBgen{exn=DATACON{name,...},...}) = name
	   | getname(EBdef{exn=DATACON{name,...},...}) = name
      in checkUniq (error region, "duplicate exception declaration")
	   (map getname ebs);
	 (EXCEPTIONdec (rev ebs),env,vt,no_updt)
     end

 (**** PATTERNS ****)

 fun apply_pat (MarkPat(c,(l1,r1)),MarkPat(p,(l2,r2))) = 
       MarkPat(AppPat{constr=c, argument=p},(min(l1,l2),max(r1,r2)))
   | apply_pat (c ,p) = AppPat{constr=c, argument=p}

 fun tuple_pat (MarkPat(a,(l,_)),MarkPat(b,(_,r))) = MarkPat(TuplePat[a,b],(l,r))
   | tuple_pat (a,b) = TuplePat[a,b]

 val patParse = Precedence.parse{apply=apply_pat, pair=tuple_pat}

 exception FreeOrVars
 fun elabPat (region:region) (env:Modules.env) 
	     (pat:Ast.pat) : Absyn.pat * tyvarset =
   case pat
   of WildPat => (WILDpat,no_tyvars)
    | VarPat  path => 
	(clean_pat (error region) (pat_id(SP.SPATH path,env,error region)),
	 no_tyvars)
    | IntPat s => (INTpat(s,TypesUtil.mkRefMETAty 0),no_tyvars)
    | WordPat s => (WORDpat(s,TypesUtil.mkRefMETAty 0),no_tyvars)
    | RealPat r => (REALpat r,no_tyvars)
    | StringPat s => (STRINGpat s,no_tyvars)
    | CharPat s => (CHARpat s,no_tyvars)
    | RecordPat {def,flexibility} =>
	 let val (lps,tyv) = elabPLabel region env def
	 in (makeRECORDpat (lps,flexibility,error region),tyv) end
    | ListPat nil =>
	   (NILpat, no_tyvars)
    | ListPat (a::rest) =>
         let val (p, tyv) = elabPat region env (TuplePat[a,ListPat rest])
	  in (CONSpat p, tyv)
	 end
    | TuplePat pats =>
	 let val (ps,tyv) = elabPatList region env pats
	  in (TUPLEpat ps,tyv)
	 end
    | VectorPat pats =>
	 let val (ps,tyv) = elabPatList region env pats
	 in (VECTORpat(ps,UNDEFty),tyv) end
    | OrPat pats =>
	(* Check that the sub-patterns of an or-pattern have exactly the same
	 * free variables, and rewrite the sub-pattersn so that all instances
	 * of a given free variable have the same type ref and the same access.
	 *)
	let val (ps, tyv) = elabPatList region env pats
	    fun freeOrVars (pat::pats) =
		let val tbl : (access * ty ref * int) IntStrMap.intstrmap =
			IntStrMap.new(16, FreeOrVars)
		    fun symbToIntStr f symb =
			(f tbl (Symbol.number symb, Symbol.name symb))
		    val ins =
			let val ins' = IntStrMap.add tbl
			 in fn (symb, x) =>
			      ins' (Symbol.number symb, Symbol.name symb, x)
			end
		    val look =
			let val look' = IntStrMap.map tbl
			 in fn symb => look' (Symbol.number symb, Symbol.name symb)
			end
		    fun errorMsg x = 
			  error region COMPLAIN
			    ("variable "^x^
			     " does not occur in all branches of or-pattern")
			    nullErrorBody
		    fun insFn (id, access, ty) =
			(ins(id, (access, ty, 1)); (access,ty))
		    fun bumpFn (id, access0, ty0) =
			(let val (access, ty, n) = look id
			  in ins (id, (access, ty, n+1)); (access,ty)
			 end
			 handle FreeOrVars => 
				 (errorMsg(Symbol.name id); (access0,ty0)))
		    fun checkFn (id, access0, ty0) = (let val (access, ty, _) = 
			look id in (access, ty) end
			handle FreeOrVars => 
				(errorMsg(Symbol.name id); (access0, ty0)))
		    fun doPat(insFn: (Symbol.symbol*access*ty ref)->access*ty ref) =
			let fun doPat' (VARpat(VALvar{access, path, typ})) =
				  let val (access,typ) = 
				      insFn(SymPath.first path,access,typ)
				   in VARpat(VALvar{access=access, path=path,
						    typ=typ})
				  end
			      | doPat' (RECORDpat{fields, flex, typ}) =
				  RECORDpat
				    {fields = map (fn (l, p) => (l, doPat' p))
				                  fields,
				     flex = flex, typ = typ}
			      | doPat' (APPpat(dc, ty, pat)) =
				  APPpat(dc, ty, doPat' pat)
			      | doPat' (CONSTRAINTpat(pat, ty)) =
				  CONSTRAINTpat(doPat' pat, ty)
			      | doPat' (LAYEREDpat(p1, p2)) =
				  LAYEREDpat(doPat' p1, doPat' p2)
			      | doPat' (ORpat(p1, p2)) =
				  ORpat(doPat' p1, doPat checkFn p2)
			      | doPat' (VECTORpat(pats, ty)) =
				  VECTORpat(map doPat' pats, ty)
			      | doPat' pat = pat
			   in doPat'
			  end
		  (* check that each variable occurs in each sub-pattern *)
		    fun checkComplete m (_, id, (_, _, n)) =
			if (n = m) then () else (errorMsg id)
		    val pats = (doPat insFn pat) :: (map (doPat bumpFn) pats)
		 in IntStrMap.app (checkComplete (length pats)) tbl;
		    pats
		end (* freeOrVars *)
	    val (pat::pats) = freeOrVars ps
	    fun foldOr (p, []) = p
	      | foldOr (p, p'::r) = ORpat(p, foldOr(p', r))
	 in (foldOr(pat, pats), tyv)
        end
    | AppPat {constr, argument} =>
	let fun getVar (MarkPat(p,region),region') = getVar(p,region)
	      | getVar (VarPat path, region') = 
		   let val dcb = pat_id (SP.SPATH path,env,error region')
		       val (p,tv) = elabPat region env argument
		   in (makeAPPpat (error region) (dcb,p),tv) end
	      | getVar (_, region') = 
		(error region' COMPLAIN 
		      "non-constructor applied to argument in pattern"
		      nullErrorBody;
		 (WILDpat,no_tyvars))
	 in getVar(constr,region)
	end
    | ConstraintPat {pattern=pat,constraint=ty} =>
	let val (p1,tv1) = elabPat region env pat
	    val (t2,tv2) = elabType error region (env,NONE) ty
	 in (CONSTRAINTpat(p1,t2), union(tv1,tv2,error region))
	end
    | LayeredPat {varPat,expPat} =>
	let val (p1,tv1) = elabPat region env varPat
	    val (p2,tv2) = elabPat region env expPat
	 in (makeLAYEREDpat(p1,p2,error region),union(tv1,tv2,error region))
	end
    | MarkPat (pat,region) =>
	let val (p,tv) = elabPat region env pat
	 in (p,tv)
	end
    | FlatAppPat pats => elabPat region env (patParse(pats,env,error))

 and elabPLabel (region:region) (env:Modules.env) labs =
     foldl
       (fn ((lb1,p1),(lps1,lvt1)) => 
	   let val (p2,lvt2) = elabPat region env p1
	   in ((lb1,p2) :: lps1, union(lvt2,lvt1,error region)) end)
       ([],no_tyvars) labs

 and elabPatList (region:region) (env:Modules.env) ps =
     foldr
       (fn (p1,(lps1,lvt1)) => 
	   let val (p2,lvt2) = elabPat region env p1
	   in (p2 :: lps1, union(lvt2,lvt1,error region)) end)
       ([],no_tyvars) ps

 (**** EXPRESSIONS ****)

 val expParse = Precedence.parse
                  {apply=fn(f,a) => AppExp{function=f,argument=a},
		   pair=fn (a,b) => TupleExp[a,b]}

 fun elabExp (region: region) 
	     (ctx as (env: Modules.env, st: Stamps.scope))
	     (exp: Ast.exp) : (Absyn.exp * tyvarset * tyvUpdate) =
     case exp
       of VarExp path =>
	    (varcon(lookVARCON(env,SP.SPATH path,error region)),no_tyvars,no_updt)
        | IntExp s => (INTexp(s,TypesUtil.mkRefMETAty 0),no_tyvars,no_updt)
        | WordExp s => (WORDexp(s,TypesUtil.mkRefMETAty 0),no_tyvars,no_updt)
	| RealExp r => (REALexp r,no_tyvars,no_updt)
	| StringExp s => (STRINGexp s,no_tyvars,no_updt)
	| CharExp s => (CHARexp s,no_tyvars,no_updt)
	| RecordExp cells => 
	    let val (les,tyv,updt) = elabELabel region ctx cells
	     in (makeRECORDexp (les,error region),tyv,updt)
	    end
	| SeqExp exps =>
	    let val (es,tyv,updt) = elabExpList region ctx exps
	     in (SEQexp es,tyv,updt)
	    end
	| ListExp nil => (NILexp, no_tyvars, no_updt)
	| ListExp (a::rest) =>
	    let val (e,tyv,updt) = elabExp region ctx (TupleExp[a,ListExp rest])
	     in (APPexp(CONSexp,e), tyv, updt)
	    end
	| TupleExp exps =>
	    let val (es,tyv,updt) = elabExpList region ctx exps
	     in (TUPLEexp es,tyv,updt)
	    end
	| VectorExp exps =>
	    let val (es,tyv,updt) = elabExpList region ctx exps
	     in (VECTORexp(es,UNDEFty),tyv,updt)
	    end
	| AppExp {function,argument} =>
	    let val (e1,tv1,updt1) = elabExp region ctx function and
		    (e2,tv2,updt2) = elabExp region ctx argument
		fun updt tv = (updt1 tv;updt2 tv)
	     in (APPexp (e1,e2),union(tv1,tv2,error region),updt)
	    end
	| ConstraintExp {expr=exp,constraint=ty} =>
	    let val (e1,tv1,updt) = elabExp region ctx exp
		val (t2,tv2) = elabType error region (env,NONE) ty
	     in (CONSTRAINTexp(e1,t2), union(tv1,tv2,error region),updt)
	    end
	| HandleExp {expr,rules} =>
	    let val (e1,tv1,updt1) = elabExp region ctx expr
		val (rls2,tv2,updt2) = elabMatch region ctx rules
		fun updt tv = (updt1 tv;updt2 tv)
	     in (makeHANDLEexp (e1, rls2), union(tv1,tv2,error region),updt)
	    end
	| RaiseExp exp =>
	    let val (e,tyv,updt) = elabExp region ctx exp
	     in (RAISEexp(e,UNDEFty),tyv,updt)
	    end
	| LetExp {dec,expr} => 
	    let val (d1,e1,tv1,updt1) =
		       elabDec' region (env,IP.IPATH[],st) dec
		val (e2,tv2,updt2) = elabExp region (Env.atop(e1,env),st) expr
		fun updt tv = (updt1 tv;updt2 tv)
	     in (LETexp(d1,e2), union(tv1,tv2,error region),updt)
	    end
	| CaseExp {expr,rules} =>
	    let val (e1,tv1,updt1) = elabExp region ctx expr
		val (rls2,tv2,updt2) = elabMatch region ctx rules
		fun updt tv = (updt1 tv;updt2 tv)
	     in (CASEexp (e1,completeMatch rls2),
		 union(tv1,tv2,error region),updt)
	    end
	| IfExp {test,thenCase,elseCase} =>
	    let val (e1,tv1,updt1) = elabExp region ctx test and
		    (e2,tv2,updt2) = elabExp region ctx thenCase and
		    (e3,tv3,updt3) = elabExp region ctx elseCase
		fun updt tv = (updt1 tv;updt2 tv;updt3 tv)
	     in (IFexp(e1,e2,e3),
		 union(tv1,union(tv2,tv3,error region),error region),
		 updt)
	    end
	| AndalsoExp (exp1,exp2) =>
	    let val (e1,tv1,updt1) = elabExp region ctx exp1 and 
		    (e2,tv2,updt2) = elabExp region ctx exp2
		fun updt tv = (updt1 tv;updt2 tv)
	     in (IFexp(e1, e2, FALSEexp), union(tv1,tv2,error region),updt)
	    end
	| OrelseExp (exp1,exp2) =>
	    let val (e1,tv1,updt1) = elabExp region ctx exp1 and 
		    (e2,tv2,updt2) = elabExp region ctx exp2
		fun updt tv = (updt1 tv;updt2 tv)
	     in (IFexp(e1 ,TRUEexp, e2), union(tv1,tv2,error region),updt)
	    end
	| WhileExp {test,expr} =>
	    let val (e1,tv1,updt1) = elabExp region ctx test and 
		    (e2,tv2,updt2) = elabExp region ctx expr
		fun updt tv = (updt1 tv;updt2 tv)
	     in (WHILEexp(e1,e2), union(tv1,tv2,error region), updt)
	    end
	| FnExp rules => 
	    let val (rls,tyv,updt) = elabMatch region ctx rules
	     in (FNexp (completeMatch rls,UNDEFty),tyv,updt)
	    end
	| MarkExp (exp,region) => 
	    let val (e,tyv,updt) = elabExp region ctx exp
	     in (if !Control.markabsyn then MARKexp(e,region) else e,tyv,updt)
	    end
	| SelectorExp s => 
	    (let val v = mkVALvar s
	      in FNexp(completeMatch
		       [RULE(RECORDpat{fields=[(s,VARpat v)], flex=true,
				       typ= ref UNDEFty},
			     MARKexp(VARexp(ref v,NONE),region))],UNDEFty)
	     end,
	     no_tyvars, no_updt)
        | FlatAppExp items => elabExp region ctx (expParse(items,env,error))

 and elabELabel region ctx labs =
     let val (les1,lvt1,updt1) =
	   foldr 
	     (fn ((lb2,e2),(les2,lvt2,updts2)) => 
		 let val (e3,lvt3,updt3) = elabExp region ctx e2
		  in ((lb2,e3) :: les2, union(lvt3,lvt2,error region),
		      updt3 :: updts2)
		 end)
	     ([],no_tyvars,[]) labs
	 fun updt tv : unit = app (fn f => f tv) updt1
      in (les1, lvt1, updt)
     end

 and elabExpList region ctx es =
     let val (les1,lvt1,updt1) =
	   foldr 
	     (fn (e2,(es2,lvt2,updts2)) => 
		 let val (e3,lvt3,updt3) = elabExp region ctx e2
		  in (e3 :: es2, union(lvt3,lvt2,error region), updt3 :: updts2)
		 end)
	     ([],no_tyvars,[]) es
	 fun updt tv: unit = app (fn f => f tv) updt1
      in (les1, lvt1, updt)
     end

 and elabMatch region ctx rs = 
     let val (rs,lvt,updt1) =
	   foldr 
	     (fn (r1,(rs1,lvt1,updt1)) => 
		 let val (r2,lvt2,updt2) = elabRule region ctx r1
		 in (r2 :: rs1, union(lvt2,lvt1,error region), updt2::updt1) end)
	     ([],no_tyvars,[]) rs
	 fun updt tv: unit = app (fn f => f tv) updt1
      in (rs, lvt, updt)
     end

 and elabRule region (env,st) (Rule{pat,exp}) =
     let val region' = case pat of MarkPat (p,reg) => reg | _ => region
	 val (p,tv1) = elabPat region env pat
	 val env' = Env.atop(bindVARp ([p],error region'), env)
	 val (e,tv2,updt) = elabExp region (env',st) exp
      in (RULE(p,e),union(tv1,tv2,error region),updt)
     end

 (**** ELABORATE SIMPLE DECLARATIONS ****)

 and elabDec' region 
	     (ctx as (env,rpath:IP.path,scope:Stamps.scope)) (dec:Ast.dec) : 
	 (Absyn.dec * Modules.env * tyvarset * tyvUpdate) =
     case dec 
       of TypeDec tbs => 
	    elabTYPEdec region ctx tbs
	| DatatypeDec{datatycs,withtycs} => 
	    elabDATATYPEdec region ctx (datatycs,withtycs)
	| AbstypeDec{abstycs,withtycs,body} =>
	    elabABSTYPEdec region ctx (abstycs,withtycs,body)
	| ExceptionDec ebs =>
	    elabEXCEPTIONdec region env ebs
	| ValDec vbs =>
	    elabVALdec region ctx vbs
	| FunDec fbs =>
	    elabFUNdec region ctx fbs
	| ValrecDec rvbs =>
	    elabVALRECdec region ctx rvbs
	| SeqDec ds =>
	    elabSEQdec region ctx ds
	| LocalDec ld => elabLOCALdec region ctx ld
	| OpenDec ds => elabOPENdec region ctx ds
	| FixDec (ds as {fixity,ops}) => 
	    let val env = 
		foldr (fn (id,env) =>
			Env.bind(id,FIXbind(FIXvar{name=id,binding=fixity}),env))
		     Env.empty ops
	     in (FIXdec ds,env,no_tyvars,no_updt)
	    end
	| OvldDec dec  => elabOVERLOADdec region ctx dec
	| MarkDec(dec,region) =>
	    let val (d,env,tv,updt)= elabDec' region ctx dec
	     in (if !Control.markabsyn then MARKdec(d,region) else d,
		 env,tv,updt)
	    end
	| StrDec _ => impossible "strdec"
	| AbsDec _ => impossible "absdec"
	| ImportDec _ => impossible "importdec"
	| FctDec _ => impossible "fctdec"
	| SigDec _ => impossible "sigdec"
	| FsigDec _ => impossible "fsigdec"

 (**** ABSTRACT TYPES DECLARATION ****)

 and elabABSTYPEdec region (ctx as (env,rpath:IP.path,scope:Stamps.scope))
		    (db,tb,ldecs) =
     let val (dtv,env0,_,_) =
	   elabDATATYPEdec region ctx (db,tb)
	 val (DATATYPEdec{datatycs,withtycs}) = dtv
	 val withtycons = map (fn TB{tyc,...} => tyc) withtycs 
	 val (body,env'',tv,updt) = 
	   elabDec' region (Env.atop(env0,env),rpath,scope) ldecs
	 fun bind (tyc,env') = Env.bind(tycName tyc,TYCbind tyc,env')
	 (* will become abstycs during type checking *)
	 val envdt = foldr bind Env.empty datatycs
	 val envwt = foldr bind envdt withtycons (* withtycs *)
      in (ABSTYPEdec{abstycs=datatycs,withtycs=withtycs,body=body},
	  Env.atop(env'',envwt),tv,updt)
     end

 (**** OVERLOADING ****)

 and elabOVERLOADdec region (env,rpath:IP.path,stamps) (id,typ,exps)=
     let val (body,tyvars) = elabType error region (env,NONE) typ 
	 val tvs = get_tyvars tyvars
	 val scheme = (TypesUtil.bindTyvars tvs; TypesUtil.compressTy body;
		       TYFUN{arity=length tvs, body=body})
	 fun option (MARKexp(e,_)) = option e
	   | option (VARexp(ref (v as VALvar{typ,...}),_)) =
	       {indicator = TypesUtil.matchScheme(scheme,!typ), variant = v}
	   | option _ = ErrorMsg.impossible "CoreLang.makeOVERLOADdec.option"
	 val exps = map (elabExp region (env,stamps)) exps
	 val exps1 = map #1 exps and exps3 = map #3 exps
	 fun updt tv: unit = app (fn f => f tv) exps3
	 val ovldvar = OVLDvar{name=id,scheme=scheme,
			       options=ref(map option exps1)}
      in (OVLDdec ovldvar, Env.bind(id,VARbind ovldvar,Env.empty),no_tyvars,updt)
     end

 and elabLOCALdec region (env,rpath:IP.path,st) (ldecs1,ldecs2)=
     let val (ld1,env1,tv1,updt1) = elabDec' region (env,IP.IPATH[],st) ldecs1
	 val (ld2,env2,tv2,updt2) =
	       elabDec' region (Env.atop(env1,env),rpath,st) ldecs2
	 fun updt tv = (updt1 tv;updt2 tv)
      in (LOCALdec(ld1,ld2), env2,union(tv1,tv2,error region),updt)
     end

 and elabOPENdec region (env,_,_) qid_p = 
     let val strs = map (fn s => lookSTR(env,SP.SPATH s,error region)) qid_p
	 fun openit (str,env) = openStructureVar (env,str)
      in (OPENdec strs, foldl openit Env.empty strs, no_tyvars,no_updt)
     end

 (****  VALUE DECLARATION ****)

 and elabVB _  ctx (MarkVb(vb,region)) = 
       elabVB region ctx vb
   | elabVB region (env,st) (Vb{pat,exp}) =
       let val (pat,pv) = elabPat region env pat
	   val (exp,ev,updtexp) = elabExp region (env,st) exp
	   fun stripped (MARKexp(e,_)) = stripped e
	     | stripped (CONSTRAINTexp(e,_)) = stripped e
	     | stripped e = e
	   val tvref = ref []
	   fun updt tv: unit =
	       let val localtyvars = 
			 diff_tyvars(union_tyvars(pv,ev,error region),tv,
				     error region)
		   val localtyvarlist = get_tyvars localtyvars
		   val downtyvars = union_tyvars(localtyvars,tv,error region)
		in tvref := localtyvarlist; updtexp downtyvars
	       end
	   val pat = 
	       case stripped exp
		 of VARexp(ref(VALvar{access as INLINE _,...}),_) =>
		      (case pat
		       of CONSTRAINTpat(VARpat(VALvar{path,typ,...}), ty) =>
			    CONSTRAINTpat(VARpat(VALvar{path=path,typ=typ,
							access=access}),ty)
			| VARpat(VALvar{path, typ,...}) =>
			    VARpat(VALvar{path=path,typ=typ,access=access})
			| CONSTRAINTpat(WILDpat, ty) => CONSTRAINTpat(WILDpat, ty)
			| pat => pat)
		  | _ => pat

	in if bindpat(pat) then 
	      let val (newpat,oldvar,newvar) = patproc pat
		  val newexp = map (fn v => VARexp(ref v,NONE)) newvar
		  val r = RULE(newpat,TUPLEexp(newexp))
		  val bpat = case oldvar of [] => WILDpat
					  | _ => TUPLEpat(oldvar)
	       in (VB{exp=CASEexp(exp,completeBind [r]),
		      tyvars=tvref,pat=bpat},
		   updt)
	      end
	   else (VB{exp=exp, tyvars=tvref, pat=pat},updt)
       end


 and elabVALdec region (env,rpath:IP.path,st) vb =
     let val (ds,updt1) = 
	   foldr 
	     (fn (vdec,(ds1,updt1)) => 
		let val (d2,updt2) = elabVB region (env,st) vdec
		in (d2::ds1,updt2::updt1) end)
	     ([],[]) vb
	 fun updt tv : unit = app (fn f => f tv) updt1
     in (VALdec ds, bindVARp (map (fn VB{pat,...}=>pat) ds,error region),
	 no_tyvars,updt)
    end

 and elabRVB _ ctx (MarkRvb(rvb,region)) =
       elabRVB region ctx rvb
   | elabRVB region (env,st) (Rvb{var,fixity,exp,resultty}) =
       let val (e,ev,updt) = elabExp region (env,st) exp
	   val (t,tv) = 
	       case resultty 
		 of SOME t1 => 
		      let val (t2,tv2) = elabType error region (env,NONE) t1
		       in (SOME t2,tv2)
		      end
		  | NONE => (NONE,no_tyvars)
        in case fixity of NONE => ()
	    | SOME(f,region) => case lookFIX(env,f) of Fixity.NONfix => ()
		  | _ => error region COMPLAIN ("infix symbol \""^ 
						Symbol.name f ^
			    "\" used where a nonfix identifier was expected")
			nullErrorBody;
	    ({match = e , ty = t, name=var},union(ev,tv,error region),updt)
       end

 and elabVALRECdec  region (ctx as (env,rpath:IP.path,st)) rvb =
     let val env' = ref(Env.empty: Modules.env)
	 fun makevar region (p as Rvb{var,...}) =
	       let val v = mkVALvar var
	        in checkBoundConstructor(env,var,error region);
		   env' := Env.bind(var,VARbind v,!env');
		   (v,p)
	       end
	   | makevar _ (p as MarkRvb(rvb,region)) = 
	       let val (v,_) = makevar region rvb in (v,p) end
	 val rvbs' = map (makevar region) rvb
	 val env'' = Env.atop(!env', env)
	 val new_ctx = (env'',st)
	 val (rvbs,tyvars,updt)=
	     foldl (fn((v,rvb1),(rvbs1,tvs1,updt1))=>
			let val (rvb2,tv2,updt2) =
			      elabRVB region new_ctx rvb1
			 in ((v,rvb2)::rvbs1, 
			     union_tyvars(tv2,tvs1,error region),
			     updt2::updt1)
			end) 
		     ([],no_tyvars,[]) rvbs' 
	 fun updtexp tv : unit = app (fn f => f tv) updt
	 val tvref = ref []
	 fun updt tv : unit =
	     let val downtyvars = union(tyvars,tv,error region)
		 val localtyvarlist =
		       get_tyvars(diff_tyvars(tyvars,tv,error region))
	      in tvref := localtyvarlist; updtexp downtyvars
	     end
      in checkUniq (error region,"duplicate function name in val rec dec")
	   (map (fn (v,{name,...}) => name) rvbs);
	 (VALRECdec (map (fn (v,{ty,match,name}) =>
			    RVB{var=v,resultty=ty,tyvars=tvref, exp=match})
		         rvbs),
          !env', no_tyvars, updt)
     end

 and elabFUNdec region (env,_,st) fb =
     let fun makevar _ (MarkFb(fb,region),ctx) = makevar region (fb,ctx)
	   | makevar region (Fb clauses,(lcl,env')) =
	      let 
		  fun getfix(SOME f) = lookFIX(env,f)
		    | getfix NONE = Fixity.NONfix
		  fun ensureInfix{item,fixity,region} =
		      (case getfix fixity
		        of Fixity.NONfix => error region COMPLAIN
			    "infix operator required, or delete parentheses" 
			    nullErrorBody
			 | _ => ();
		       item)
		  fun ensureNonfix{item,fixity,region} =
		      (case (getfix fixity, fixity)
		        of (Fixity.NONfix,_) => ()
			 | (_,SOME sym) => error region COMPLAIN
			       ("infix operator \"" ^ Symbol.name sym ^
				"\" used without \"op\" in fun dec")
			       nullErrorBody;
		       item)

		  fun getname(MarkPat(p,region),_) = getname(p,region)
		    | getname(VarPat[v], _) = v
		    | getname(_, region) = (error region COMPLAIN
					    "illegal function symbol in clause"
					    nullErrorBody; 
					    bogusID)

		  fun parse'({item=FlatAppPat[a,b as {region,...},c],...}::rest) = 
		         (getname(ensureInfix b, region),
			   tuple_pat(ensureNonfix a, ensureNonfix c)
			   :: map ensureNonfix rest)
		    | parse' [{item,region,...}] = 
			     (error region COMPLAIN
			      "can't find function arguments in clause"
			      nullErrorBody;
			      (getname(item,region), [WildPat]))
		    | parse' ((a as {region,...}) :: rest) =
		          (getname(ensureNonfix a, region), 
			   map ensureNonfix rest)

		  fun parse({item=MarkPat(p,_),region,fixity}::rest) = 
		          parse({item=p,region=region,fixity=fixity}::rest)
		    | parse (pats as [a as {region=ra,...},
			              b as {item,fixity,region},c]) =
		        (case getfix fixity
			  of Fixity.NONfix => parse' pats
			   | _ => (getname(item,region),
				   [tuple_pat(ensureNonfix a, ensureNonfix c)]))
		    | parse pats = parse' pats

		  fun parseClause(Clause{pats,resultty,exp}) =
		      let val (funsym,argpats) = parse pats
		       in {funsym=funsym,argpats=argpats,resultty=resultty,
			   exp=exp}
		      end

		  val clauses as {funsym=var,...}::_ = map parseClause clauses

		  val _ = if List.exists (fn {funsym,...} => 
				     not(Symbol.eq(var,funsym))) clauses
			  then  error region COMPLAIN 
				  "clauses don't all have function name"
				  nullErrorBody
			  else ()

		  val _ = checkBoundConstructor(env,var,error region)

		  val v = mkVALvar var
		  val _ = 
		    case clauses
		    of ({argpats,...})::rest => 
			 let val len = length argpats
			  in if List.exists (fn ({argpats,...}) => len <> length argpats)
				  rest
			     then error region COMPLAIN 
				   "clauses don't all have same number of patterns"
				   nullErrorBody
			     else ()
			 end
		     | [] => impossible "elabFUNdec 1"
	       in ((v,clauses,region)::lcl,Env.bind(var,VARbind v,env'))
	      end
	 val (clauses,env') = foldl (makevar region) ([],Env.empty) fb
	 val env'' = Env.atop(env',env)
	 fun makeclause region ({argpats=pats,resultty,exp,funsym}) =
	     let val (pats,tv1) = elabPatList region env pats
		 val (exp,tv2,updt) = 
		   elabExp region (Env.atop(bindVARp(pats,error region),env''),st)
		     exp
		 val (ty,tv3) =
		     case resultty
		       of NONE => (NONE,no_tyvars)
			| SOME t => 
			    let val (t4,tv4) = elabType error region (env,NONE) t
			     in (SOME t4,tv4)
			    end
	      in (CLAUSE{pats=pats,resultty=ty,exp=exp},
		  union(tv1,union(tv2,tv3,error region),error region),updt)
	     end
	 fun evalclauses ((var,clauses,region),(fs,tvs,updt)) = 
	     let val (cs1,tvs1,updt1) =
	           foldl (fn (c2,(cs2,tvs2,updt2)) =>
			  let val (c3,tvs3,updt3) = makeclause region c2
			   in (c3::cs2,union(tvs3,tvs2,error region),updt3::updt2)
			  end) 
		       ([],no_tyvars,[]) clauses
	      in ((var,rev cs1)::fs,union(tvs1,tvs,error region),updt1 @ updt)
	     end
	 val (fbs1,fv,updt1) = foldl evalclauses ([],no_tyvars,[]) clauses
	 val tvref = ref []
	 fun updt tv : unit =  
	     let val localtyvars = diff_tyvars(fv,tv,error region)
		 val downtyvars = union_tyvars(localtyvars,tv,error region)
		 val localtyvarlist = get_tyvars localtyvars
	      in tvref := localtyvarlist; app (fn f => f downtyvars) updt1
	     end
	 fun makefb (v as VALvar{path=SymPath.SPATH[_],...},cs) =
	       (FB{var=v,clauses=cs, tyvars=tvref})
	   | makefb _ = ErrorMsg.impossible "makeFUNdec.makefb"
      in checkUniq (error region,"duplicate function names in fun dec") 
		   (map (fn (VALvar{path=SymPath.SPATH[x],...},_) => x
			  | _ => impossible "makeFUNdec:checkuniq")
			fbs1);
	 (FUNdec(completeMatch,map makefb fbs1,errorMatch,region),
	  env', no_tyvars, updt)
     end

 and elabSEQdec region (ctx as (env,rpath:IP.path,st)) ds =
     let val (ds1,env1,tv1,updt1) = 
	   foldl 
	    (fn (decl2,(ds2,env2,tvs2,updt2)) =>
	       let val (d3,env3,tvs3,updt3) =
			elabDec' region (Env.atop(env2,env),rpath,st) decl2
		in (d3::ds2, Env.atop(env3,env2), union(tvs3,tvs2,error region),
		    updt3::updt2)
	       end)
	    ([],Env.empty,no_tyvars,[]) ds 
	 fun updt tv : unit = app (fn f => f tv) updt1
      in (SEQdec(rev ds1),env1,tv1,updt)
     end

 in elabDec' region
end (* elabDec *)

end (* structure ElabCore *)
