(* dbgenv.new.sml
 *
 * COPYRIGHT (c) 1996 Bell Laboratories.
 *)

(* DebugEnv
 
   Support for synthetic environments when halted under debugger.
   Intended to be layered into appropriate spot in interactive debugger
   environment.

   The static environment special lookup
   function searches for a given symbol in the environment 
   current at envTime.  If the symbol is found,
   a synthetic binding for it is returned.  For variables and constructors,
   type reconstruction is performed and the resulting type is placed 
   in the synthetic binding. For type constructors and structures, 
   all component types are reconstructed.  For symbols with dynamic 
   representations, the synthetic binding contains the original pid.

   The dynamic component of the special environment is a ref holding 
   a normal dynamic environment mapping pids to appropriate value objects 
   for all identifiers ever looked up.

    Only pids that appear in bindings returned by lookups in
   the static synthetic environment should ever be looked up in the dynamic
   synthetic environment.

*)


signature DEBUG_ENV =
sig
  type time
  val debugEnvironment: time -> Environment.environment
      (* synthetic debugger environment *)
end

structure DebugEnv: DEBUG_ENV =
struct
  open DebugUtil DebugRun DebugBindings DebugMotions Types Variables
       Access Absyn Modules PrintUtil

  structure SP = SymPath

  exception Unbound = Env.Unbound

 fun debugEnvironment(envTime : time) =
 let 

  val dynEnvR = ref DynamicEnv.empty
  fun dynbind (pid,binding) = dynEnvR := DynamicEnv.bind(pid,binding,!dynEnvR)

  fun pidOf v =  PersStamps.stringToStamp
		       (substring(makestring v ^ "                ",0,16))

  (* look up symbol in a given namespace, and return new binding *)
  fun lookVARCON (symbol:Symbol.symbol) : binding  =
	let val n = Symbol.name symbol
	    val _ = dbgprint ("lookVARCON " ^ n ^ "\n")
	    val (t,c,(i,binding)) = findVARCONBind ([n],envTime,0) 
	in case binding
           of VARbind(VALvar{typ=ref ty,access=LVAR v,...}) =>
	       let val pid = pidOf v
		   val (evn,args) = evnArgsAt t
		   val bv = nth (nthArgs(evn,c,args),i)
		   val ty = dynTypeAt (t,c) ty
		   val _ = dbgprint "gotVALVAR\n"
	       in dynbind(pid,bv);
(*		  print "binding ";print (PersStamps.stampToString pid);
		  print"\n";
*)		  VARbind(VALvar{access=EXTERN pid,
				 path=SP.SPATH [symbol],
				 typ=ref ty})
	       end
	   | VARbind(OVLDvar _) =>
	       (dbgprint "gotOVLDVAR\n";
		binding)
	   | CONbind(dc as (DATACON{const,rep,sign,typ,orig,...}))  =>
	       let val typ = dynTypeAt (t,c) typ
		   val rep = case rep of
			       (VARIABLE(LVAR v)) => 
				  let val pid = pidOf v
				      val (evn,args) = evnArgsAt t
				      val ev = nth(nthArgs(evn,c,args),i)
				  in dynbind(pid,ev);
				     VARIABLE(EXTERN pid)
				  end
			     | (VARIABLEc(LVAR v)) => 
				  let val pid = pidOf v
				      val (evn,args) = evnArgsAt t
				      val ev = nth(nthArgs(evn,c,args),i)
				  in dynbind(pid,ev);
				     VARIABLEc(EXTERN pid)
				  end
			     | _ => rep
		   val _ = dbgprint "gotCON\n"
	       in CONbind(DATACON{name=symbol,const=const,orig=orig,
				  typ=typ,rep=rep,sign=sign})
	       end
	end
  
  fun lookSTR  (symbol:Symbol.symbol) : binding =
	  let val n = Symbol.name symbol
	      val _ = dbgprint ("lookSTR " ^ n ^ "\n")
	      val (t,c,(i,STRbind(STRvar{binding=s,access=LVAR v,...}))) = 
			     findSTRBind ([n],envTime,0)
	      val pid = pidOf v
	      val s = dynStrAt t s
	      val sobj =
		  if i >= 0 then 
		    let val (evn,args) = evnArgsAt t
		    in nth(nthArgs(evn,c,args),i+1)
		    end
		  else let val args = argsAt (pred t)
		       in hd args
		       end
	      val _ = dbgprint "gotSTR\n"
	  in dynbind(pid,sobj);
	     STRbind(STRvar{name=symbol,access=EXTERN pid,binding=s})
	  end
  
  fun lookFCT (symbol:Symbol.symbol) : binding =
	  let val n = Symbol.name symbol
	      val _ = dbgprint ("lookFCT " ^ n ^ "\n")
	      val (t,c,(i,FCTbind(FCTvar{binding=f,access=LVAR v,...}))) = 
		  findFCTBind ([n], envTime,0)
	      val pid = pidOf v
	      val (evn,args) = evnArgsAt t
	      val fobj = nth(nthArgs(evn,c,args),i)
	      val _ = dbgprint "gotFCT\n"
	  in dynbind(pid,fobj);
	     FCTbind(FCTvar{name=symbol,access=EXTERN pid,binding=f})
	  end
  
  fun lookSIG (symbol:Symbol.symbol) : binding =
	  let val n = Symbol.name symbol
	      val _ = dbgprint ("lookSIG " ^ n ^ "\n")
	      val (t,c,(i,SIGbind sv)) = findSIGBind ([n],envTime,0)
	      val _ = dbgprint "gotSIG\n"
	  in SIGbind(sv)
	  end
  
  fun lookFSIG (symbol:Symbol.symbol) : binding =
	  let val n = Symbol.name symbol
	      val _ = dbgprint ("lookFSIG " ^ n ^ "\n")
	      val (t,c,(i,FSIGbind fsv)) = findFSIGBind ([n],envTime,0)
	      val _ = dbgprint "gotFSIG\n"
	  in FSIGbind(fsv)
	  end
  
  fun lookTYC (symbol:Symbol.symbol) : binding = 
	  let val n = Symbol.name symbol
	      val _ = dbgprint ("lookTYC " ^ n ^ "\n")
	      val (t,c,(i,TYCbind tycon)) = findTYCBind ([n],envTime,0)
	      val tycon = dynTyconAt t tycon
	      val _ = dbgprint "gotTYC\n"
	  in TYCbind(tycon)
	  end
  
  fun lookFIX (symbol:Symbol.symbol) : binding =
	  let val n = Symbol.name symbol
	      val _ = dbgprint ("lookFIX " ^ n ^ "\n")
	      val (t,c,(i,FIXbind fv)) = findFIXBind ([n],envTime,0)
	      val _ = dbgprint "gotFIX\n"
	  in FIXbind(fv)
	  end
  

  fun statLooker' (symbol:Symbol.symbol) :binding =
	let val look = case Symbol.nameSpace symbol of
			 Symbol.VALspace => lookVARCON
		       | Symbol.TYCspace => lookTYC
		       | Symbol.SIGspace => lookSIG
		       | Symbol.FSIGspace => lookFSIG
		       | Symbol.STRspace => lookSTR
		       | Symbol.FCTspace => lookFCT
		       | Symbol.FIXspace => lookFIX
		       | _ => raise Unbound
	in case (withEstablishedTime (fn _ => 
		   SOME (look symbol)
		      handle Unbound => NONE
		           | QueryInterrupted => NONE)) of
	     NOTRUNNING => raise Unbound
           | COMPLETED(SOME binding) => binding
           | COMPLETED NONE => raise Unbound
	   | INTERRUPTED _ => raise Unbound  (* somewhat superfluously *)
	end

   fun dynLooker (pid: PersStamps.persstamp)  = 
       DynamicEnv.look (!dynEnvR) pid

  in  {static=StaticEnv.special (statLooker,NONE),
       dynamic=DynamicEnv.special (dynLooker,DynamicEnv.empty)}
 end

end
