(* md.sig
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *
 * Specifies the instructions available on the machine and their properties.
 *
 * Todo: Data representation for instructions.
 *
 *)

signature INSTRUCTIONS = sig
    structure C : CELLS
    
    type instruction
end




(* INSN_PROPERTIES - defines properties of the machine instruction 
 *	that can typically be inferred from a more concise 
 *	machine description.
 *)

signature INSN_PROPERTIES = sig
   structure I : INSTRUCTIONS
   structure C : CELLS

   sharing I.C = C

   structure FR : GETREG
   structure GR : GETREG
	    

   (** common **)
   datatype kind = IK_JUMP | IK_NOP | IK_INSTR
   datatype target = LABELLED of Label.label | FALLTHROUGH | ESCAPES

   val instrKind     : I.instruction -> kind
   val defUse	     : I.instruction -> C.cellset * C.cellset
   val branchTargets : I.instruction -> target list
   val nop 	     : unit -> I.instruction


   (** register allocation functions **)

   val spillInit : unit -> unit
   val spill1    : int -> I.instruction list * I.instruction list
   val spill2    : int -> I.instruction list * I.instruction list
   val spill3    : int -> I.instruction list * I.instruction list

   val getreg1 : int list * int list -> int
   val getreg2 : int list * int list -> int
   val getreg3 : int list * int list -> int

   (* Implicit is an assumption that there is only one source and 
    * destination for the move instruction - see register allocator
    *)
   val moveInstr : I.instruction -> bool

  (** basic block scheduling **)

   val branchDelayedArch : bool
   val latency 	: I.instruction -> int
   val needsNop : I.instruction * I.instruction list -> int

   (* minSize and maxSize are not restricted to SDIs but instructions
    *  that may require NOPs after them, etc. 
    *)
   val minSize	: I.instruction -> int
   val maxSize	: I.instruction -> int
   val isSdi	: I.instruction -> bool

   (* sdiSize(instr,labMap,loc) - size of sdi instruction instr, at 
    * 	location loc, using the labMap assignment to labels
    *)
   val sdiSize	: I.instruction * (Label.label -> int) * int -> int

   (* expand(instr,size,labMap) - expands sdi instruction instr,
    *  into size bytes, using the labMap assignment of labels
    *)
   val expand	: I.instruction * int * (int->int) -> I.instruction list
       
   val numResources : int
   val mayNeedNops : I.instruction -> int
   val bdefUse  : int Array.array list -> I.instruction -> int list * int list
end

signature MACHINE_PROPERTIES = sig
    val startgcOffset 	   : int
    val constBaseRegOffset : int

    val varptr_indexable   : bool
end
