(* flowgraph.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *
 * Defines the flowgraph data structure used as the intermediate program
 * representation.
 *)

signature FLOWGRAPH = sig

  structure C : CELLS
  structure I : INSTRUCTIONS
  sharing I.C = C

  datatype block =
      MARK
    | LABEL of Label.label
    | REALconst of Label.label * string
    | STRINGconst of Label.label * int * string
    | JMPtable of Label.label * Label.label list
    | BBLOCK of { blknum  : int,
		  liveIn  : C.cellset ref,
		  liveOut : C.cellset ref,
		  succ 	  : int list ref,
		  pred 	  : int list ref,
		  insns	  : I.instruction list ref
	        }
    | CODE of I.instruction Vector.vector ref 

  datatype cluster = 
      CLUSTER of {blocks : block list, 
		  regmaps : int Array.array list}

  val prBlock : block -> unit
end


(*  Create the flowgraph data structure specialized towards	
 *  a specific type of cells and instructions.
 *)
functor FlowGraph(structure Instr : INSTRUCTIONS) : FLOWGRAPH = 
struct
  structure I = Instr
  structure C = Instr.C

  datatype block =
       MARK
     | LABEL of Label.label
     | REALconst of Label.label * string
     | STRINGconst of Label.label * int * string
     | JMPtable of Label.label * Label.label list
     | BBLOCK of { blknum  : int,
		   liveIn  : C.cellset ref,
		   liveOut : C.cellset ref,
		   succ    : int list ref,
		   pred    : int list ref,
		   insns   : I.instruction list ref
		 }
     | CODE of I.instruction Vector.vector ref

  datatype cluster = 
      CLUSTER of {blocks : block list, 
		  regmaps : int Array.array list}

  fun prBlock MARK               = print "MARK\n"
    | prBlock(LABEL lab)         = print ("LABEL " ^ Label.nameOf lab ^ "\n")
    | prBlock(REALconst(lab,r))  = print ("REAL " ^ r ^ "\n")
    | prBlock(STRINGconst(_,_,s))= print ("STRING " ^ s ^ "\n")
    | prBlock(CODE _)            = print "CODE\n"
    | prBlock(JMPtable(lab,labs))= let
	  fun prLabs [] = print "\n"
	    | prLabs(lab::labs) = (print ("\t\t" ^ Label.nameOf lab ^ "\n");
				  prLabs labs)
	in
	    print ("JTABLE \t" ^ Label.nameOf lab ^ "\n");
	    prLabs labs
	end
    | prBlock(BBLOCK{blknum,succ,pred,liveOut,liveIn,...}) = let

	fun prBlkList [] = print "\n"
	  | prBlkList ((blknum:int)::blknums) = 
	      (print (makestring blknum ^ ","); prBlkList blknums)

	fun prCells cells = 
	  (print (C.cellset2string cells); 
	   print "\n")
      in
	  print("BLOCK" ^ makestring blknum ^ "\n");
	  print("\t liveIn: ");  prCells(!liveIn);
	  print("\t succ: ");    prBlkList(!succ);
	  print("\t pred: ");    prBlkList(!pred);
	  print("\t liveOut: "); prCells(!liveOut)
      end
end


