(* ctreeComp.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *
 * Top level loop that compiles clusters to machine code.
 *)

signature MLTREECOMP = sig
    structure Flowgraph : FLOWGRAPH

    val mltreeComp : MLTree.mltree -> unit		  
end


functor CTreeComp(structure MLTreeGen : MLTREE_GEN
		  structure MLTreeComp : MLTREECOMP
		  structure CPSRegs : CELLS
		  structure MachSpec: MACH_SPEC ) : CPSGEN = 
struct

  structure MachSpec = MachSpec

  val _ =if MachSpec.numRegs <> length(CPSRegs.miscregs) + 3
           then ErrorMsg.impossible "ctreeComp: wrong number of miscregs"
	   else ()

  val cps_spill = false

  fun error msg = ErrorMsg.impossible ("ctreeComp." ^ msg)

  structure M = MLTree
  val mltreeComp = MLTreeComp.mltreeComp

  fun codegen (funcs0,limits,err) = let
      exception Labbind

      val allocLimits =  (fn (a,_) => a) o limits

      (* correspondence between CPS.lvar and Label.label *)
      val labTable : Label.label Intmap.intmap = Intmap.new (32,Labbind)
      val addlabbinding = Intmap.add labTable
      val labmap = Intmap.map labTable

      exception Strings and Reals
      val stringTable : Label.label IntStrMap.intstrmap	= IntStrMap.new (32,Strings)
      val realTable : Label.label IntStrMap.intstrmap	= IntStrMap.new (32,Reals)

      exception Typbind
      val typTable : CPS.cty Intmap.intmap = Intmap.new(32, Typbind)
      val addtypbinding = Intmap.add typTable
      fun iscont v = case (Intmap.map typTable v) of CPS.CNTt => true | _ => false

      (* list of all gc invocation subroutines that need to 
       * be generated for this compilation unit.
       *)
      val gcInvocationList = ref ([] : MLTreeGen.gcInfo list)

      (* Generate a label for each function in the clusters.
       * Also install entries for function names and parameters.	
       *)
      fun makeLabTable cluster =
	app (fn (fk,f,vl,tl,_) =>
	         (addlabbinding (f, Label.newLabel (makestring (f:int)));
		  case fk of CPS.CONT => addtypbinding(f,CPS.CNTt)
			   | _ => addtypbinding(f,CPS.PTRt (NONE))
	         (*esac*);
	         List2.app2 addtypbinding (vl,tl)))
	    cluster

      fun emitMLRiscUnit f =
	(mltreeComp (M.BEGINCLUSTER);
	 CPSRegs.resetRegs();
	 f ();
	 mltreeComp (M.ENDCLUSTER))
	 
      fun doCluster cluster = let
	(* given a CPS.lvar, give me the knowledge of the function *)
	exception Know
	val knowtable : Frag.frag Intmap.intmap   = Intmap.new (8, Know)
	val addknow                               = Intmap.add knowtable
	val know                                  = Intmap.map knowtable

	val fragList = ref ([] : (Label.label * Frag.frag) list)
	fun addfrags frags = fragList := frags @ (!fragList)

	fun makefrags ((cpsfun as (fk,f,vl,tl,e))::funcs) = let
	      val lab = labmap f
	      val (funseq,varfmls,frags) =
		CTreeify.ctreeify (cpsfun, stringTable, realTable, typTable)
	      val knowledge =
		(case (fk,allocLimits f)
		 of (CPS.ESCAPE,alloc) =>
			 Frag.STANDARD(ref(SOME(f,varfmls,tl,funseq)),alloc)
		  | (CPS.CONT,alloc)   =>
			 Frag.STANDARD(ref(SOME(f,varfmls,tl,funseq)),alloc)
		  | (CPS.KNOWN,alloc)  =>
			 Frag.KNOWNFUN(ref(Frag.UNGEN(varfmls,tl,funseq)))
		  | (CPS.KNOWN_CHECK,alloc) =>
			 Frag.KNOWNCHK(ref(Frag.UNGEN(varfmls,tl,funseq)),alloc)
		  | _	 => error "makefrag: knowledge"
		 (*esac*))
	    in
	      addknow(f,knowledge);
	      (lab,knowledge)::(frags @ makefrags funcs)
	    end (*makefrag*)
	  | makefrags [] = []

	(* The bulk of makefrags is ctreeification *)
	val makefrags = 
	  Stats.doPhase (Stats.makePhase "Compiler 122 ctreeify") makefrags

	  (* returns a list of gc invocations subroutines 
	   * that are required for this cluster.
	   *)
	  fun genfrag([], gc) = gc
	    | genfrag((_, Frag.KNOWNFUN _)::rest, gc) = genfrag(rest, gc)
	    | genfrag((_, Frag.KNOWNCHK _)::rest, gc) = genfrag(rest, gc)
	    | genfrag((_, Frag.STANDARD (ref NONE, _))::rest, gc) = 
		genfrag(rest, gc)
	    | genfrag((lab, Frag.STANDARD
			      (r as ref(SOME(f,varfmls,tl,funseq)), 
			       alloc))::rest, 
		       gc) =
		(r := NONE;
		 genfrag(rest, MLTreeGen.mltreeGen
				      (f, lab, varfmls, funseq, tl, typTable,
				       labmap, mltreeComp, know, alloc) 
			       @ gc))
	    | genfrag((lab, Frag.REALfrag r)::rest, gc) =
		((mltreeComp (M.REALCONST (lab, IEEEReal.realconst r)))
		     handle IEEEReal.BadReal _ =>
			 error ("real constant out of range: "^r);
		 genfrag(rest, gc))
	    | genfrag((lab, Frag.STRINGfrag s)::rest, gc) =
		(mltreeComp (M.STRINGCONST (lab, s)); 
		 genfrag(rest, gc))

	  (* The bulk of genfrags in mltreeGen *)
	  val genfrag = 
	    Stats.doPhase(
	       Stats.makePhase "Compiler 124 machine codegen") genfrag

	  (* collapse multiple jumps to the same gc subroutine into one. *)
	  fun longJumpsToGC(gcCalls) = let
	    (* Finds if a gc invocation subroutine already exists. *)
	    fun findGcLabel(MLTreeGen.GCINFO{lab=label, maskRegs, 
			    fRegs, i32Regs, ret=M.JMP return}) = let
		 (* gcInvocationList is likely to be small, so a linear
		  * search is used
		  *)
		  fun search [] = let
			val lab = Label.newLabel""
		      in
			gcInvocationList := 
			  MLTreeGen.GCINFO{
			    lab=lab, maskRegs=maskRegs, i32Regs=i32Regs, 
			    fRegs=fRegs, ret=M.JMP return} :: (!gcInvocationList);
			(lab, label)
		      end
		    | search(MLTreeGen.GCINFO{
			lab, maskRegs=mr, i32Regs=i32, 
			fRegs=fr, ret=M.JMP return'} :: rest) = let
			fun eqR([], []) = true
			  | eqR(M.REG(x)::xs, M.REG(y)::ys) = 
			      x = y andalso eqR(xs, ys)
			  | eqR _ = false
			fun eqF([], []) = true
			  | eqF(M.FREG(x)::xs, M.FREG(y)::ys) = 
			      x = y andalso eqF(xs, ys)
			  | eqF _ = false
		      in
			if eqR(mr, maskRegs) andalso eqR(i32, i32Regs) andalso 
			  eqF(fr, fRegs) andalso return' = return
			then (lab, label)
			else search rest
		      end
		in
		  search (!gcInvocationList)
		end
	      | findGcLabel _ = error "longJumpToGC:findGcLabel"

	    (* group target labels in this cluster that call the same
	     * GC subroutine 
	     *)
	    fun collapse(gcCall::rest, ans) = let
		  val (subroutine, label) = findGcLabel(gcCall)
		  fun add [] = [(subroutine, [label])]
		    | add((p as (lab, labs))::rest) = 
		      if Label.id lab = Label.id subroutine then 
			  (lab, label::labs)::rest
		      else p::add(rest)
		in
		  collapse(rest, add(ans))
		end
	      | collapse([], ans) = ans

	    (* emit the target labels for each of the CHECKLIMITs and 
	     * the (potential) long jump to the GC subroutine. 
	     *)
	    fun emitLongJumps (lab, labs) = 
	      (app (fn lab => mltreeComp (M.LABEL lab)) labs;
	       mltreeComp (M.CODE(M.BR lab)))
	  in
	    app emitLongJumps (collapse(gcCalls, []))
	  end

	in
	  (* Generate all the code fragments for this cluster. *)
	  addfrags (makefrags cluster);

	  (*
	   * Now generate code for each fragment
	   * The known functions will only be generated, by fall-through,
	   * if they are referenced in an application (APP_LABEL).
	   *)
	  if !Control.debugging then app PPCps.printcps0 cluster else ();
	  emitMLRiscUnit (fn () => longJumpsToGC (genfrag (!fragList, [])))
	end (* doCluster *)

      val callGc = MLTreeGen.callGc mltreeComp

      val clusters = Cluster.cluster funcs0
    in
      app makeLabTable clusters;
      app doCluster clusters;
      emitMLRiscUnit(fn () => app callGc  (!gcInvocationList))
    end	(* codegen *)
end (* CtreeComp *)
