(* ctree.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *
 * Describes the ctree data structure.
 *)

structure CTree = 
struct
  type var = int

  val itoa = Int.toString
    
  datatype br_type = B_EQ | B_LT | B_LE | B_LTU | B_LEU

  datatype exprbase
    = VAR of var
    | LABEL of CPS.lvar
    | INT of int
    | INT32 of Word32.word
    | REAL of Label.label
    | STRING of Label.label
    (* arith *)
    | ADD of {signed:bool}
    | SUB of {signed:bool}
    | MUL of {signed:bool}
    | DIV of {signed:bool}
    | NEG
    | FADD
    | FSUB
    | FMUL
    | FDIV
    (* pure *)
    | LENGTH
    | OBJLENGTH
    | RSHIFT
    | RSHIFTL
    | LSHIFT
    | ORB
    | ANDB
    | XORB
    | NOTB
    | W32ADD
    | W32SUB
    | W32MUL
    | W32DIV
    | W32RSHIFT
    | W32RSHIFTL
    | W32LSHIFT
    | W32ANDB
    | W32ORB
    | W32XORB
    | W32NOTB
    | W32TOINT
    | W32FROMINT
    | W31TOINT
    | W31FROMINT
    | W31TOW32
    | W31FROMW32
    | FNEG
    | FABS
    | REALOFINT
    | SUBSCRIPTV
    | GETTAG
    | MKSPECIAL
    (* looker *)
    | DEREF
    | GETHDLR
    | GETVAR
    | SUBSCRIPT
    | SUBSCRIPTF
    | ORDOF
    | GETSPECIAL
    (* record *)
    | RSTORE      of int * CPS.accesspath		  (* offset and path *)
    | RSTOREF     of int * CPS.accesspath		  (* offset and path *)
    | RSTOREF0    of int		            (* store float at offset *)
    | RSTOREDESC  of bool * int * int	      (* alignment * offset and desc *)
    | RSTORESTORE of int					   (* offset *)
    | RSET        of int					   (* offset *)
    | RSETSTORE   of int					   (* offset *)
    | RINC        of int				    (* offset to add *)
    | SELECT      of int
    | SELECTF     of int
    | OFFSET      of int
    (* side effect operations, return NOTHING *)
    | STORE
    | UNBOXEDUPDATE
    | CALCADDR
    | UPDATEF
    | SETHDLR
    | SETVAR
    | SETPSEUDO
    | SETSPECIAL
    (* flow control primitives, return NEVER *)
    | APP_VAR   of CPS.lvar * CPS.cty list
    | APP_LABEL of CPS.lvar * CPS.cty list
    | ACONS
    | ACONSF 
    | ANIL
    | SWITCH    of seq list
    | BR_BOXED  of seq * seq
    | BR_INT    of br_type * seq * seq
    | BR_WORD32  of br_type * seq * seq
    | BR_FLOAT  of br_type * seq * seq
    | BR_STRING of seq * seq

    | GETPSEUDO
    | WRAP
    | UNWRAP
    | CAST

  and expr = EX of exprbase * expr list * int
  (* int: the order of evaluation of the kids (expr list) *)
  withtype seq = (int * int)		   (* first and last var (inclusive) *)
    
  type ctree = expr

  type ctreeInfo = {ctree:ctree,deptype:int} Array.array

  type funCtree = (int * int) * ctreeInfo

  fun printExpr(blk,info) (EX(base, children, order)) = let
      fun brtype B_EQ  = "eq"
	| brtype B_LT  = "lt"
	| brtype B_LE  = "le"
	| brtype B_LTU = "ltu"
	| brtype B_LEU = "leu"

      fun signedOp(oper, true) = oper
	| signedOp(oper, false) = oper ^ "U"

      val name =
	case base of
	  VAR v 		=> "CT.VAR."^(itoa v)
	| LABEL l 		=> "LABEL."^(itoa l)
	| INT i 		=> "CT.INT."^(itoa i)
	| INT32 w 		=> "INT32." ^ (Word32.toString w)
	| REAL _ 		=> "REAL.?"
	| STRING _ 		=> "STRING.?"
	(* arith *)	
	| ADD{signed}		=> signedOp("ADD", signed)
	| SUB{signed}		=> signedOp("SUB", signed)
	| MUL{signed}		=> signedOp("MUL", signed)
	| DIV{signed}		=> signedOp("DIV", signed)
	| NEG 			=> "NEG"
	| FADD 			=> "FADD"
	| FSUB 			=> "FSUB"
	| FMUL 			=> "FMUL"
	| FDIV 			=> "FDIV"
	(* pure *)
	| LENGTH 		=> "LENGTH"
	| OBJLENGTH 		=> "OBJLENGTH"
	| RSHIFT 		=> "RSHIFT"
	| RSHIFTL 		=> "RSHIFT_L"
	| LSHIFT 		=> "LSHIFT"
	| ORB 			=> "ORB"
	| ANDB 			=> "ANDB"
	| XORB 			=> "XORB"
	| NOTB 			=> "NOTB"
	| W32ADD 		=> "W32ADD"
	| W32SUB 		=> "W32SUB"
	| W32MUL 		=> "W32MUL"
	| W32DIV 		=> "W32DIV"
	| W32RSHIFT 		=> "W32RSHIFT"
	| W32RSHIFTL 		=> "W32RSHIFTL"
	| W32LSHIFT 		=> "W32LSHIFT"
	| W32ANDB 		=> "W3WANDB"
	| W32ORB 		=> "W32ORB"
	| W32XORB 		=> "W32XORB"
	| W32NOTB 		=> "W32NOTB"
	| W32TOINT 		=> "W32TOINT"
	| W32FROMINT 		=> "W32FROMINT"
	| W31TOINT 		=> "W31TOINT"
	| W31FROMINT 		=> "W31FROMINT"
	| W31TOW32		=> "W31TOW32"
	| W31FROMW32		=> "W31FROMW32"
	| FNEG 			=> "FNEG"
	| FABS 			=> "FABS"
	| REALOFINT 		=> "REALOFINT"
	| SUBSCRIPTV 		=> "SUBSCRIPTV"
	| GETTAG 		=> "GETTAG"
	| MKSPECIAL 		=> "MKSPECIAL"
	(* looker *)
	| DEREF 		=> "DEREF"
	| GETHDLR 		=> "GETHDLR"
	| GETVAR 		=> "GETVAR"
	| SUBSCRIPT 		=> "SUBSCRIPT"
	| SUBSCRIPTF 		=> "SUBSCRIPTF"
	| ORDOF 		=> "ORDOF"
	| GETSPECIAL 		=> "GETSPECIAL"
	(* record *)
	| RSTORE(i,path) 	=> "RSTORE."^(itoa i)
	| RSTOREF(i,path) 	=> "RSTOREF."^(itoa i)
	| RSTOREF0(i) 		=> "RSTOREF0."^(itoa i)
	| RSTOREDESC(b,i,d)	=> "RSTOREDESC."^(itoa i)^":"^(itoa d)
	| RSTORESTORE i  	=> "RSTORESTORE."^(itoa i)
	| RSET i 		=> "RSET."^(itoa i)
	| RSETSTORE i 		=> "RSETSTORE."^(itoa i)
	| RINC i 		=> "RINC."^(itoa i)
	| SELECT i 		=> "SELECT."^(itoa i)
	| SELECTF i 		=> "SELECTF."^(itoa i)
	| OFFSET i 		=> "OFFSET."^(itoa i)
	(* side effect operations *)
	| STORE 		=> "STORE"
	| UNBOXEDUPDATE 	=> "UNBOXEDUPDATE"
	| CALCADDR 		=> "CALCADDR"
	| UPDATEF 		=> "UPDATEF"
	| SETHDLR 		=> "SETHDLR"
	| SETVAR 		=> "SETVAR"
	| SETPSEUDO 		=> "SETPSEUDO"
	| SETSPECIAL 		=> "SETSPECIAL"
	(* flow control primitives *)
	| APP_VAR (v,_) 	=> "APP_VAR."^(itoa v)
	| APP_LABEL(l,_) 	=> "APP_LABEL."^(itoa l)
	| ACONS 		=> "ACONS"
	| ACONSF 		=> "ACONSF"
	| ANIL 			=> "ANIL"
	| SWITCH _ 		=> "SWITCH"
	| BR_BOXED _ 		=> "BR_BOXED"
	| BR_INT (t,_,_) 	=> "BR_INT."^(brtype t)
	| BR_WORD32 (t,_,_) 	=> "RB_WORD32."^(brtype t)
	| BR_FLOAT (t,_,_) 	=> "BR_FLOAT."^(brtype t)
	| BR_STRING _  		=> "BR_STRING"
	| GETPSEUDO  		=> "GETPSEUDO"
	| WRAP 			=> "WRAP"
	| UNWRAP 		=> "UNWRAP"
	| CAST	 		=> "CAST"

      val seql =
	case base of
	  SWITCH seql =>		seql
	| BR_BOXED (s1,s2) =>	[s1,s2]
	| BR_INT (_,s1,s2) =>	[s1,s2]
	| BR_FLOAT (_,s1,s2) =>	[s1,s2]
	| BR_STRING(s1,s2) =>   [s1,s2]
	| _ => []
  in
      (print (blk^name^" (order "^(itoa order)^")\n");
       app (printExpr((blk^"  "),info)) children ;
       app (fn (s as (b,e)) => (print (blk^"[ ("
				       ^(itoa b)^","
				       ^(itoa e)^")\n");
				printSeq (blk^"  ",s, info);
				print (blk^"]\n"))) seql)
  end

  and printSeq (blk, (fst,lst), info:{ctree:expr,deptype:int}Array.array) = let
      fun loop v =
	  if v>lst then () 
	  else let 
	      val {ctree,deptype}=Array.sub (info, v)
		  handle zap => (print ("handle.printSeq v="
					^(itoa v)^"\n"); raise zap)
	    in
	      if deptype = ~1 orelse deptype = 6 then () else
		(print (blk^(itoa v)^"=\n");
		 printExpr ((blk^" "),info) ctree);
		loop (v+1)
	    end
  in
    loop fst
  end
end (* CTree *)
