(* Copyright 1989 by AT&T Bell Laboratories *)
(* cps.sml *)

structure CPS = struct

structure P =  struct

    (* numkind includes kind and size *)
    datatype numkind = INT of int | UINT of int | FLOAT of int

    datatype arithop = + | - | * | / | ~ | abs
	             | lshift | rshift | rshiftl | andb | orb | xorb | notb

    datatype cmpop = > | >= | < | <= | gtu | geu | ltu | leu | eql | neq

  (* These are two-way branches dependent on pure inputs *)
    datatype branch
      = cmp of {oper: cmpop, kind: numkind}
      | boxed | unboxed | peql | pneq
      | streq | strneq (* streq(n,a,b) is defined only if strings a and b have
	         exactly the same length n>1 *)

  (* These all update the store *)
    datatype setter
      = numupdate of {kind: numkind}
      | unboxedupdate | boxedupdate | update
      | sethdlr | setvar | uselvar | setspecial
      | free | acclink | setpseudo | setmark

  (* These fetch from the store, never have functions as arguments. *)
    datatype looker
      = ! | subscript | numsubscript of {kind: numkind} | getspecial | deflvar
      | getrunvec | gethdlr | getvar | getpseudo

  (* These might raise exceptions, never have functions as arguments.*)
    datatype arith
      = arith of {oper: arithop, kind: numkind}
      | round of {floor: bool, fromkind: numkind, tokind: numkind}

  (* These don't raise exceptions and don't access the store. *)
    datatype pure
      = pure_arith of {oper: arithop, kind: numkind}
      | pure_numsubscript of {kind: numkind}
      | length | objlength | makeref
      | real of {fromkind: numkind, tokind: numkind}
      | subscriptv
      | gettag | mkspecial | wrap | unwrap | cast | getcon | getexn
      | fwrap | funwrap | iwrap | iunwrap | i32wrap | i32unwrap

    local fun f op > = op <= | f op <= = op >
	    | f op < = op >= | f op >= = op <
	    | f gtu = leu | f geu = ltu
	    | f ltu = geu | f leu = gtu
	    | f eql = neq | f neq = eql
    in fun opp boxed = unboxed | opp unboxed = boxed
	 | opp strneq = streq | opp streq = strneq
	 | opp peql = pneq | opp pneq = peql
	 | opp (cmp{oper,kind}) = cmp{oper=f oper,kind=kind}
    end

    val iadd = arith{oper=op +,kind=INT 31}
    val isub = arith{oper=op -,kind=INT 31}
    val imul = arith{oper=op *,kind=INT 31}
    val idiv = arith{oper=op /,kind=INT 31}
    val ineg = arith{oper=op ~,kind=INT 31}

    val fadd = arith{oper=op +,kind=FLOAT 64}
    val fsub = arith{oper=op -,kind=FLOAT 64}
    val fmul = arith{oper=op *,kind=FLOAT 64}
    val fdiv = arith{oper=op /,kind=FLOAT 64}
    val fneg = arith{oper=op ~,kind=FLOAT 64}

    val ieql = cmp{oper=eql,kind=INT 31}
    val ineq = cmp{oper=neq,kind=INT 31}
    val igt = cmp{oper=op >,kind=INT 31}
    val ige = cmp{oper=op >=,kind=INT 31}
    val ile = cmp{oper=op <=,kind=INT 31}
    val ilt = cmp{oper=op <,kind=INT 31}
    val iltu = cmp{oper=ltu,kind=INT 31}
    val igeu = cmp{oper=geu,kind=INT 31}

    val feql = cmp{oper=eql,kind=FLOAT 64}
    val fneq = cmp{oper=neq,kind=FLOAT 64}
    val fgt = cmp{oper=op >,kind=FLOAT 64}
    val fge = cmp{oper=op >=,kind=FLOAT 64}
    val fle = cmp{oper=op <=,kind=FLOAT 64}
    val flt = cmp{oper=op <,kind=FLOAT 64}

    fun arity op ~ = 1
      | arity _ = 2

end (* P *)

type lvar = Access.lvar

datatype value 
  = VAR of lvar
  | LABEL of lvar
  | INT of int
  | INT32 of Word32.word
  | REAL of string
  | STRING of string
  | OBJECT of System.Unsafe.object
  | VOID

datatype accesspath 
  = OFFp of int 
  | SELp of int * accesspath

datatype cty = INTt | INT32t | PTRt of int option 
             | FUNt | FLTt | CNTt | DSPt

val BOGt = PTRt(NONE)  (* bogus pointer type whose length is unknown *)

datatype fun_kind
  = CONT           (* continuation functions *)
  | KNOWN          (* general known functions *)
  | KNOWN_REC      (* known recursive functions *)
  | KNOWN_CHECK    (* known functions that need a heap limit check *)
  | KNOWN_TAIL     (* tail-recursive kernal *)
  | KNOWN_CONT     (* known continuation functions *)
  | ESCAPE         (* before the closure phase, any user function;
	              after the closure phase, escaping user function *)
  | NO_INLINE_INTO (* before the closure phase,
		      a user function inside of which no in-line expansions
		      should be performed; 
		      does not occur after the closure phase *)


datatype cexp
  = RECORD of Access.record_kind * (value * accesspath) list * lvar * cexp
  | SELECT of int * value * lvar * cty * cexp
  | OFFSET of int * value * lvar * cexp
  | APP of value * value list
  | FIX of function list * cexp
  | SWITCH of value * lvar * cexp list
  | BRANCH of P.branch * value list * lvar * cexp * cexp
  | SETTER of P.setter * value list * cexp
  | LOOKER of P.looker * value list * lvar * cty * cexp
  | ARITH of P.arith * value list * lvar * cty * cexp
  | PURE of P.pure * value list * lvar * cty * cexp
withtype function = fun_kind * lvar * lvar list * cty list * cexp

fun combinepaths(p,OFFp 0) = p
  | combinepaths(p,q) = 
    let val rec comb =
	fn (OFFp 0) => q
	 | (OFFp i) => (case q of
		          (OFFp j) => OFFp(i+j)
		        | (SELp(j,p)) => SELp(i+j,p))
	 | (SELp(i,p)) => SELp(i,comb p)
    in comb p
    end

fun lenp(OFFp _) = 0
  | lenp(SELp(_,p)) = 1 + lenp p

local structure LT = LambdaType in

fun ctype t = case LT.out t
 of LT.INT => INTt
  | LT.INT32 => INT32t
  | LT.BOOL => INTt
  | LT.REAL => FLTt
  | LT.BOXED => BOGt
  | LT.RBOXED => BOGt
  | LT.SRCONT => FUNt
  | (LT.CONT _) => CNTt
  | (LT.ARROW _) => FUNt
  | (LT.RECORD []) => INTt
  | (LT.SRECORD []) => INTt
  | (LT.RECORD l) => PTRt (SOME (length l))
  | (LT.SRECORD l) => PTRt (SOME (length l))
  | _ => BOGt

end

end (* CPS *)
