(* Copyright 1989 by AT&T Bell Laboratories *)
(* convert.sml *)

(***************************************************************************
 *                         IMPORTANT NOTES                                 *
 *                                                                         *
 *          OFFSET and RECORD accesspath SELp should not be                *
 *                  generated by this module.                              *
 ***************************************************************************)

functor Convert(MachSpec : MACH_SPEC) : CONVERT = struct

  open Access CPS 
  structure LT = LambdaType
 
val rep_flag = MachSpec.representations
fun which (a,b) = if rep_flag then a else fn x => b

val mergeLty = which(LT.mergeLty, LT.BOGUS)
val arrowLty = which(LT.arrowLty, (LT.BOGUS,LT.BOGUS))
val contLty = which(LT.contLty,  LT.BOGUS)
val recordLty = which(LT.recordLty, [LT.BOGUS])
val selectLty = which(LT.selectLty, LT.BOGUS)

structure P = CPS.P (* to avoid confusing SourceGroup *) 
structure AP = Access.P

(***************************************************************************
 *              CONSTANTS AND UTILITY FUNCTIONS                            *
 ***************************************************************************)
val OFFp0 = OFFp 0
val id = fn x => x

fun err s = ErrorMsg.impossible s


val IntOpTy = LT.injARROW(LT.injRECORD[LT.injINT,LT.injINT],LT.injINT)
val seqTy = LT.injARROW(LT.injRECORD[LT.injBOXED,LT.injBOXED],LT.inj LT.BOOL)

fun getConTy t = case LT.out t
  of LT.ARROW(t1,t2) => t2    (* non-constant data constructor *)
   | _ => t                   (* constant data constructor *)

fun getDeconTy t = case LT.out t
 of LT.ARROW(t1,t2) => t1    
  | _ => t


fun numkind (AP.INT bits) = P.INT bits
  | numkind (AP.UINT bits) = P.UINT bits
  | numkind (AP.FLOAT bits) = P.FLOAT bits

fun cmpop(stuff,argt) = case stuff
      of {oper=AP.EQL,kind=AP.INT 31} => 
	     if LT.eq(argt,LT.injRECORD[LT.injBOXED,LT.injBOXED])
		 then (Control.Print.say "int-equality used for ptr-equality\n";
		       P.peql)
                 else P.ieql
       | {oper=AP.NEQ,kind=AP.INT 31} => 
	     if LT.eq(argt,LT.injRECORD[LT.injBOXED,LT.injBOXED])
		 then (Control.Print.say "int-equality used for ptr-equality\n";
		       P.pneq)
                 else P.ineq
       | {oper,kind} =>
	 let fun c AP.> = P.> | c AP.>= = P.>= | c AP.< = P.< | c AP.<= = P.<=
	       | c AP.LEU=P.leu|c AP.LTU=P.ltu|c AP.GEU=P.geu|c AP.GTU=P.gtu
	       | c AP.EQL=P.eql| c AP.NEQ=P.neq
	 in P.cmp{oper=c oper, kind=numkind kind} end

fun arity AP.~ = 1
  | arity AP.ABS = 1
  | arity AP.NOTB = 1
  | arity AP.+ = 2
  | arity AP.- = 2
  | arity AP.* = 2
  | arity AP./ = 2
  | arity AP.LSHIFT = 2
  | arity AP.RSHIFT = 2
  | arity AP.RSHIFTL = 2
  | arity AP.ANDB = 2
  | arity AP.ORB = 2
  | arity AP.XORB = 2
    
fun arithop AP.~ = P.~
  | arithop AP.ABS = P.abs
  | arithop AP.NOTB = P.notb
  | arithop AP.+ = P.+
  | arithop AP.- = P.-
  | arithop AP.* = P.*
  | arithop AP./ = P./
  | arithop AP.LSHIFT = P.lshift
  | arithop AP.RSHIFT = P.rshift
  | arithop AP.RSHIFTL = P.rshiftl
  | arithop AP.ANDB = P.andb
  | arithop AP.ORB = P.orb
  | arithop AP.XORB = P.xorb

(***************************************************************************
 *                        THE MAIN FUNCTION                                *
 *     convert : Lambda.lexp -> CPS.cexp * CPS.lty Intmap.intmap           *
 ***************************************************************************)
fun convert lexp = let 
 
val cvtrfty = if (MachSpec.newListRep) then TransList.cvtrfty 
              else (fn x => x)
val selectLty = if (MachSpec.newListRep) then TransList.selectLty
                else selectLty

(* the following should be reconfigured in the future *)
(**  (* replaced with below to avoid infinite loop in spill when #fpregs=7 *)
val maxrepregs1 = if not rep_flag then 0
  else (let val k = MachSpec.numRegs - MachSpec.numCalleeSaves
         in min(k-2,MachSpec.numArgRegs)
        end)

val maxrepregs2 = if not rep_flag then 0
  else (let val k = MachSpec.numRegs - MachSpec.numCalleeSaves
         in min(k-2,MachSpec.maxRepRegs)
        end)
**)

 val maxrepregs1 = if not rep_flag then 0
  else (let val k = MachSpec.numRegs - MachSpec.numCalleeSaves
         in min(min(k-2,MachSpec.numFloatRegs-1),MachSpec.numArgRegs)
        end)
 
 val maxrepregs2 = if not rep_flag then 0
   else (let val k = MachSpec.numRegs - MachSpec.numCalleeSaves
          in min(min(k-2,MachSpec.numFloatRegs-1),MachSpec.maxRepRegs)
         end)

local open Intmap
      exception Rename
      val m : value intmap = new(32, Rename)
      val rename = map m

   in fun ren v = rename v handle Rename => VAR v
      val newname = add m
  end

local open Intmap
   in exception TypeInfo
      val typtable : LT.lty intmap = new(32, TypeInfo)
      val mapty = 
           if rep_flag then 
             (fn v => (map typtable v) 
                   handle TypeInfo => 
       	              (List.app Control.Print.say
		       ["The lvar ", Access.lvarName v,
                        " is not in the current hashtable!\n"];
                     err "TypeInfo hash table in convert.sml"))
           else (fn v => LT.injBOXED)
      val addty = if rep_flag then (add typtable) else (fn v => ())
      val rmvty = if rep_flag then (rmv typtable) else (fn v => ())
      val nthty = if rep_flag then List.nth else (fn _ => LT.injBOXED)
      fun grabty(VAR v) = mapty v
        | grabty(LABEL v) = mapty v
        | grabty(INT _) = LT.injINT
	| grabty(INT32 _) = LT.inj LT.INT32
        | grabty(REAL _) = LT.injREAL
        | grabty _ = LT.injBOXED
  end

fun mkfn(f,t) = let val v = mkLvar()
                in  addty(v,t);
		    f v
                end

fun mkv(t) = let val v = mkLvar()
	     in  addty(v,t);
		 v
             end

val bogus_cont = mkv(LT.inj(LT.CONT LT.injBOXED))

val unboxedfloat = MachSpec.unboxedFloats
val untaggedint = MachSpec.untaggedInt
val flatfblock = (!Control.CG.flatfblock) andalso unboxedfloat

fun unwrapfloat(u,x,ce) = PURE(P.funwrap,[u],x,FLTt,ce)
fun wrapfloat(u,x,ce) = PURE(P.fwrap,[u],x,BOGt,ce)
fun unwrapint(u,x,ce) = PURE(P.iunwrap,[u],x,INTt,ce)
fun wrapint(u,x,ce) = PURE(P.iwrap,[u],x,BOGt,ce)
fun unwrapi32(u,x,ce) = PURE(P.i32unwrap,[u],x,INT32t,ce)
fun wrapi32(u,x,ce) = PURE(P.i32wrap,[u],x,BOGt,ce)

fun primwrap(INTt) = P.iwrap
  | primwrap(INT32t) = P.i32wrap
  | primwrap(FLTt) = P.fwrap
  | primwrap _ = P.wrap

fun primunwrap(INTt) = P.iunwrap
  | primunwrap(INT32t) = P.i32unwrap
  | primunwrap(FLTt) = P.funwrap
  | primunwrap _ = P.unwrap

(* check if a record contains only reals *)
fun isFloatRec t = case LT.out t
 of LT.RECORD (l as _::_) =>
       let fun h [] = flatfblock
             | h (LT.REAL::r) = h r
             | h _ = false
        in h(map LT.out l)
       end
  |  _ => false

fun selectFL(i,u,x,ct,ce) = SELECT(i,u,x,ct,ce)
fun selectNM(i,u,x,ct,ce) =
  (case (ct,unboxedfloat,untaggedint)
    of (FLTt,true,_) => let val v = Access.mkLvar()
                         in SELECT(i,u,v,BOGt,unwrapfloat(VAR v,x,ce))
                        end
     | (INTt,_,true) => let val v = Access.mkLvar()
                         in SELECT(i,u,v,BOGt,unwrapint(VAR v,x,ce))
                        end
     | (INT32t,_,_) => let val v = Access.mkLvar()
                        in SELECT(i,u,v,BOGt,unwrapi32(VAR v,x,ce))
                       end
     | _ => SELECT(i,u,x,ct,ce))

fun recordFL(ul,_,w,ce) = 
  let val nul = map (fn u => (u,OFFp 0)) ul
   in RECORD(RK_FBLOCK,nul,w,ce)
  end

fun recordNM(ul,tyl,w,ce) =
  let fun g(FLTt::r,u::z,l,h) = 
             if unboxedfloat then 
               (let val v = Access.mkLvar()
                 in g(r, z, (VAR v,OFFp 0)::l, fn ce => h(wrapfloat(u,v,ce)))
                end)
             else g(r, z, (u,OFFp 0)::l, h)
        | g(INTt::r,u::z,l,h) = 
             if untaggedint then 
               (let val v = Access.mkLvar()
                 in g(r, z, (VAR v,OFFp 0)::l, fn ce => h(wrapint(u,v,ce)))
                end)
             else g(r, z, (u,OFFp 0)::l, h)
        | g(INT32t::r,u::z,l,h) = 
             let val v = Access.mkLvar()
              in g(r, z, (VAR v,OFFp 0)::l, fn ce => h(wrapi32(u,v,ce)))
             end
        | g(_::r,u::z,l,h) = g(r, z, (u,OFFp0)::l, h)
        | g([],[],l,h) = (rev l, h)
        | g _ = err "unexpected in recordNM in convert"

      val (nul,header) = 
        if rep_flag then g(map ctype tyl,ul,[],fn x => x)
        else (map (fn u => (u,OFFp 0)) ul, fn x => x)
   in header(RECORD(RK_RECORD,nul,w,ce))
  end

fun convpath(Access.LVAR v, k) = k(ren v)
  | convpath(Access.PATH(i,p), k) =
       let fun kont(v) =
            let val t = selectLty(grabty(v),i)
                val w = mkv(t)
             in SELECT(i, v, w, ctype t, k(VAR w))
            end
        in convpath(p,kont)
       end

(* BUG: The defintion of E_word is clearly incorrect since it can raise
 *	an overflow at code generation time. A clean solution would be 
 *	to add a WORD constructor into the CPS language -- daunting! The
 *	revolting hack solution would be to put the right int constant 
 *	that gets converted to the right set of bits for the word constant.
 *)
val do_switch = Switch.switch {
   E_int = fn i => if i < ~0x20000000 orelse i >= 0x20000000
                 then raise Switch.TooBig else CPS.INT i, 
   E_real = fn s => REAL s,
   E_switchlimit = 4,
   E_neq = P.ineq,
   E_w32neq = P.cmp{oper=P.neq,kind=P.UINT 32},
   E_word32 = INT32,
   E_wneq = P.cmp{oper=P.neq, kind=P.UINT 31},
   E_word = fn w => INT (Word.toInt w),
   E_pneq = P.pneq,
   E_fneq = P.fneq,
   E_less = P.ilt,
   E_branch= fn(cmp,x,y,a,b) => BRANCH(cmp,[x,y],mkv(LT.injINT),a,b),
   E_strneq= fn(w,str,a,b) => BRANCH(P.strneq, [INT(size str),w,STRING str],
				     mkv(LT.injINT), a, b),
   E_switch= fn(v,list) => SWITCH(v, mkv(LT.injINT), list),
   E_add= fn(x,y,c) => let val v = mkv(LT.injINT) in ARITH(P.iadd,[x,y],v,INTt,c(VAR v))
		     end,
   E_gettag= fn(x,c) => let val v = mkv(LT.injINT)
                     in PURE(P.getcon,[x],v,INTt,c(VAR v))
		    end,
   E_unwrap= fn(x,c) => let val v = mkv(LT.injINT)
                     in PURE(P.unwrap,[x],v,INTt,c(VAR v))
		    end,
   E_getexn= fn(x,c) => let val v = mkv(LT.injBOXED)
                     in PURE(P.getexn,[x],v,BOGt,c(VAR v))
		    end,
   E_length= fn(x,c) => let val v = mkv(LT.injINT)
                     in PURE(P.length,[x],v,INTt,c(VAR v))
		    end,
   E_boxed= fn(x,a,b) => BRANCH(P.boxed,[x],mkv(LT.injINT),a,b),
   E_path= convpath}

		     
(***************************************************************************
 *        mkArgIn : lty * lvar -> lvar list * cty list * (cexp -> cexp)    *
 *       mkArgOut : lty * value -> value list * (cexp -> cexp)             *
 *                                                                         *
 * When the type of the argument x of a function f(x) is an "small"        *
 * unboxed record, f will be transformed to a multi-argument function      *
 * with #1(mkArgIn(...,x)) as its list of arguments.                       *
 *                                                                         *
 * When a function f is applied to a argument x, and x is of a "small"     *
 * unboxed record type, x will be flattened. #1(mkArgOut(...,x)) will      *
 * become the actual arguments of the function call f.                     *
 *                                                                         *
 * When the Control.CG.representations flag is turned off, all             *
 * these effects are gone.  (l >> 0)                                       *
 ***************************************************************************)

fun mkArgIn0(t,v) = 
  let val l = LT.sizeLty(t)
      fun megl((vl1,cl1,f1),(vl2,cl2,f2)) = (vl1 @ vl2, cl1 @ cl2, f1 o f2)

      (* recFlat: recursive flatten *)
      fun recFlat(tt,p) = case LT.out tt
       of LT.RECORD nil =>  ([p],[INTt],id)
        | LT.RECORD args => 
          let val ul = map (fn t => mkv(t)) args
              val recordCE = if isFloatRec tt then recordFL else recordNM  
              val header = fn ce => recordCE(map VAR ul,args,p,ce)
           in foldr megl ([], [], header) (List2.map2 recFlat (args,ul))
          end
        | _ => ([p],[ctype tt],id)

      (* oneFlat: flatten only one level *)
      fun oneFlat (tt,p) = case LT.out tt
        of LT.RECORD args => 
	  let val wl = map (fn t => mkv(t)) args
              val cl = map ctype args
              val recordCE = if isFloatRec tt then recordFL else recordNM  
              val header = fn ce => recordCE(map VAR wl,args,p,ce)
           in (wl,cl,header) 
          end
        | _ => ([p],[ctype(tt)],id)

   in if l < maxrepregs1 then recFlat(t,v)
      else (let val s = LT.lengthLty(t)
             in if s < maxrepregs2 then oneFlat(t,v)
                else ([v],[ctype(t)],id)
            end)
  end

fun mkArgIn(t,v) = mkArgIn0(cvtrfty t,v)

fun mkArgOut0(t,z as VAR v) = 
  let val l = LT.sizeLty(t)
      fun megr((vl1,f1),(vl2,f2)) = ((vl1 @ vl2), f2 o f1)
  
      fun recFlat (tt,p) = case LT.out tt
       of LT.RECORD nil =>  ([VAR p],id)
        | LT.RECORD args =>
	  let val wl = map (fn t => (t,mkv(t))) args
              val selectCE = if isFloatRec tt then selectFL else selectNM

              fun sel((t,x)::tl,i) = 
                   let val header = sel(tl,i+1)
                    in fn ce => selectCE(i, VAR p, x, ctype(t), header(ce))
                   end
                | sel(nil,i) = id

              val header = sel(wl,0)
                     
           in foldr megr ([], header) (map recFlat wl)
          end
        | _ => ([VAR p],id)

      fun oneFlat (tt,p) = case LT.out tt 
       of LT.RECORD args =>
	  let val wl = map (fn t => (mkv(t),ctype(t))) args
              val selectCE = if isFloatRec tt then selectFL else selectNM
              fun sel((x,ct)::tl,i) = 
                   let val header = sel(tl,i+1)
                    in fn ce => selectCE(i, VAR p, x, ct, header(ce))
                   end
                | sel(nil,i) = id
              val header = sel(wl,0)
           in (map (VAR o #1)  wl,header) 
          end
        | _ => ([VAR p],id)
                            
   in if l < maxrepregs1 then recFlat(t,v)
      else (let val s = LT.lengthLty(t)
             in if s < maxrepregs2 then oneFlat(t,v)
                else ([z],id)
            end)
  end
  | mkArgOut0(t,z) = ([z],id) 

fun mkArgOut(t,v) = mkArgOut0(cvtrfty t,v)


(***************************************************************************
 *           preventEta : cexp * lty -> cexp * value                       *
 ***************************************************************************)
fun preventEta(c,argt) =
  let val f = mkv((LT.inj o LT.CONT) argt) and v = mkv(argt)
      val (vl,cl,header) = mkArgIn(argt,v)
      val b = header(c(VAR v))
   in case b
       of APP(w as VAR w', [VAR v']) => 
            if v=v' andalso v<>w'
		(* The case v=w' never turns up in practice,
		   but v<>v' does turn up. *)
	    then (id,w)
	    else (fn x => FIX([(CONT,f,vl,cl,b)],x),VAR f)
	| _ => (fn x => FIX([(CONT,f,vl,cl,b)],x),VAR f)
  end

(***************************************************************************
 *   convlist : Lambda.lexp list * (value list -> cexp) -> cexp            *
 ***************************************************************************)
fun convlist (el,c) =
  let fun f(le::r, vl) = conv(le, fn v => f(r,v::vl))
	| f(nil, vl) = c (rev vl)
   in f (el,nil)
  end

(***************************************************************************
 *   getargs : int * Lambda.lexp * (value list -> cexp) -> cexp            *
 ***************************************************************************)
and getargs(1,a,g) = conv(a, fn z => g[z])
  | getargs(n,Lambda.RECORD l,g) = convlist(l,g)
  | getargs(n,Lambda.VECTOR l,g) = convlist(l,g)
  | getargs(0,a,g) = g(nil)
  | getargs(n,a,g) =
     let fun kont(v) = 
           let val lt = grabty(v)
               val selectCE = if (isFloatRec lt) then selectFL else selectNM
               fun f(j,wl) = 
                 if j = n then g(rev wl)
                 else (let val tt = selectLty(lt,j)
                           fun h(w) = 
                             selectCE(j,v,w,ctype(tt),f(j+1,VAR w :: wl))
                        in mkfn(h,tt)
                       end)
            in f(0,nil)
           end
      in conv(a,kont)
     end

(***************************************************************************
 *   conv : Lambda.lexp * (value list -> cexp) -> cexp                     *
 ***************************************************************************)
and conv (le, c : value -> cexp) = 

 case le 
  of Lambda.APP(Lambda.PRIM(AP.CALLCC,_), f) =>
       let fun kont(vf) =
             let val (t1,t2) = arrowLty(grabty(vf))
                 val h = mkv((LT.inj LT.SRCONT)) 
                 (* t1 must be SRCONTty here *)
                 val k' = mkv(t1) and x' = mkv(t2) 
                 val (header,F) = preventEta(c,t2)
                 val (vl,cl,_) = mkArgIn(t2,x')
                 val z = mkv((LT.inj o LT.CONT) LT.injBOXED) (* bogus cont *)
              in header(LOOKER(P.gethdlr, [], h, FUNt,
                         FIX([(ESCAPE, k', z::vl, CNTt::cl, 
                                   SETTER(P.sethdlr, [VAR h],
                                                APP(F, map VAR vl)))],
                             APP(vf,[F, VAR k']))))
             end
        in conv(f,kont)
       end
   | Lambda.APP(Lambda.PRIM(AP.CAPTURE,_), f) =>
       let fun kont(vf) = 
             let val (t1,t2) = arrowLty(grabty(vf))
                 val k' = mkv(t1) and x' = mkv(t2)
                 val (header,F) = preventEta(c,t2)
                 val (vl,cl,_) = mkArgIn(t2,x')     
                 val z = mkv((LT.inj o LT.CONT) LT.injBOXED) (* bogus cont *)
                 (* this k' is one kind of eta redexes that optimizer
                  * should not reduce! The type of k' and F is different.
                  *)
              in header(FIX([(ESCAPE, k', z::vl, CNTt::cl, 
                              APP(F, map VAR vl))],
                            APP(vf,[F, VAR k'])))
             end
        in conv(f,kont)
       end
(* We can't do this because the of representation type problems:
   | Lambda.APP(Lambda.PRIM(AP.THROW,_), v) => conv(v,c)
*)
   | Lambda.APP(Lambda.PRIM(AP.THROW,_), v) => 
        let fun kont(kv) =
              let val t = LT.injARROW(LT.injBOXED,LT.injBOXED)
                  val f = mkv(t)
               in PURE(P.cast,[kv],f,ctype(t),c(VAR f))
              end
         in conv(v,kont)
        end
   | Lambda.APP(Lambda.PRIM(AP.CAST,lt), x) => 
       let fun kont(vx) =
             let val (_,t) = arrowLty(lt)
              in mkfn(fn u => PURE(P.cast,[vx],u,ctype(t),c(VAR u)), t)
             end
        in conv(x,kont)
       end
   | Lambda.APP(Lambda.PRIM(i,lt), a) => 
       let val (argt,t) = arrowLty(lt)
           val ct = ctype t

           fun arith(n,i) = 
             let fun kont(vl) = mkfn(fn w => ARITH(i,vl,w,ct,c(VAR w)),t)
              in getargs(n,a,kont)
             end

           fun setter(n,i) = 
             let fun kont(vl) = SETTER(i,vl,c(INT 0))
              in getargs(n,a,kont)
             end

           fun looker(n,i) =
             let fun kont(vl) = mkfn(fn w => LOOKER(i,vl,w,ct,c(VAR w)),t)
              in getargs(n,a,kont)
             end

           fun pure(n,i) =
             let fun kont(vl) = mkfn(fn w => PURE(i,vl,w,ct,c(VAR w)),t)
              in getargs(n,a,kont)
             end

  	   fun branch(n,i)= 
             let val (header,F) = preventEta(c,t) 
                 fun kont(vl) = header(BRANCH(i,vl,mkv(LT.injINT),
                                              APP(F,[INT 1]),APP(F,[INT 0])))
              in getargs(n,a,kont)
             end

        in case i
	    of AP.BOXED => branch(1,P.boxed)
	     | AP.UNBOXED => branch(1,P.boxed)
	     | AP.CMP stuff => branch(2,cmpop(stuff,argt))
	     | AP.PTREQL => branch(2,P.peql)
	     | AP.PTRNEQ => branch(2,P.pneq)

	     | AP.ARITH{oper,kind,overflow=true} =>
		arith(arity oper,
		      P.arith{oper=arithop oper,kind=numkind kind})
	     | AP.ARITH{oper,kind,overflow=false} =>
		pure(arity oper,
		     P.pure_arith{oper=arithop oper,kind=numkind kind})

	     | AP.ROUND{floor,fromkind,tokind} =>
		arith(1,P.round{floor=floor,
				fromkind=numkind fromkind,
				tokind=numkind tokind})

             | AP.REAL{fromkind,tokind} =>
		pure(1,P.real{tokind=numkind tokind,
			      fromkind=numkind fromkind})

	     | AP.SUBSCRIPTV => pure(2,P.subscriptv)
	     | AP.MAKEREF => pure(1,P.makeref)
	     | AP.LENGTH => pure(1,P.length)
	     | AP.OBJLENGTH => pure(1,P.objlength)
	     | AP.GETTAG => pure(1, P.gettag)
	     | AP.MKSPECIAL => pure(2, P.mkspecial)
		
	     | AP.SUBSCRIPT => looker(2,P.subscript)
	     | AP.NUMSUBSCRIPT{kind,immutable=false,checked=false} => 
		   looker(2,P.numsubscript{kind=numkind kind})
	     | AP.NUMSUBSCRIPT{kind,immutable=true,checked=false} => 
		   pure(2,P.pure_numsubscript{kind=numkind kind})
	     | AP.DEREF => looker(1,P.!)
	     | AP.GETRUNVEC => looker(0, P.getrunvec)
	     | AP.GETHDLR => looker(0,P.gethdlr)
	     | AP.GETVAR  => looker(0,P.getvar)
             | AP.GETPSEUDO => looker(1,P.getpseudo)
	     | AP.GETSPECIAL => looker(1, P.getspecial)
	     | AP.DEFLVAR  => looker(0,P.deflvar)
		
	     | AP.SETHDLR => setter(1,P.sethdlr)
	     | AP.NUMUPDATE{kind,checked=false} =>
		   setter(3,P.numupdate{kind=numkind kind})
	     | AP.UNBOXEDUPDATE => setter(3,P.unboxedupdate)
	     | AP.BOXEDUPDATE => setter(3,P.boxedupdate)
	     | AP.UPDATE => setter(3,P.update)
	     | AP.SETVAR => setter(1,P.setvar)
             | AP.SETPSEUDO => setter(2,P.setpseudo)
             | AP.SETMARK => setter(1,P.setmark)
             | AP.DISPOSE => setter(1,P.free)
	     | AP.SETSPECIAL => setter(2, P.setspecial)
	     | AP.USELVAR => setter(1,P.uselvar)
	     | AP.MARKEXN => getargs(2,a,fn[x,m']=>
	  	  let val bty = LT.injBOXED
                      val ety = LT.injRECORD[bty,bty,bty]

                      val xx = mkv ety
		      val x0 = mkv bty
		      val x1 = mkv bty
		      val x2 = mkv bty

                      val y = mkv ety
                      val y' = mkv bty

		      val z = mkv(LT.injRECORD[bty,bty])
                      val z' = mkv bty

                   in PURE(P.unwrap,[x],xx,ctype(ety),
                        SELECT(0,VAR xx,x0,BOGt,
   		        SELECT(1,VAR xx,x1,BOGt,
		        SELECT(2,VAR xx,x2,BOGt,
		          RECORD(RK_RECORD,[(m',OFFp0),(VAR x2,OFFp0)],z,
                          PURE(P.wrap,[VAR z],z',BOGt,
  		          RECORD(RK_RECORD,[(VAR x0,OFFp0),
				            (VAR x1,OFFp0),
					    (VAR z', OFFp0)], y,
                          PURE(P.wrap,[VAR y], y', BOGt,c(VAR y')))))))))
                  end)

	     | _ => err ("calling with bad primop \"" 
                                         ^ (AP.pr_primop i) ^ "\"")
       end
   | Lambda.PRIM(i,lt) => 
       let (* val _ = print ("prim chkarrow "^(AP.pr_primop i)^"\n") *)
           val (t,_) = arrowLty(lt)
           val v = mkLvar()
           val e = Lambda.FN(v,t,Lambda.APP(le,Lambda.VAR v))
        in conv(e,c)
       end
   | Lambda.EXNF(e,lt) =>
       let fun kont(u) = 
             let val x = mkv(LT.injBOXED) (* and y = mkv(lt) *)
              in PURE(P.makeref,[u],x,BOGt,c(VAR x))
                 (****>> PURE(P.cast,[VAR x],y,FUNt,c(VAR y)) <<****)
             end
        in conv(e,kont)
       end
   | Lambda.EXNC e => 
       let fun kont(u) =
             let val x = mkv(LT.injBOXED) and z = mkv(LT.injBOXED)
                 and y = mkv(LT.injRECORD[LT.injBOXED,LT.injINT,LT.injBOXED])
              in PURE(P.makeref,[u],x,BOGt,
                   RECORD(RK_RECORD,[(VAR x,OFFp0),(INT 0,OFFp0),
				     (INT 0, OFFp0)],y,
                     PURE(P.wrap,[VAR y],z,BOGt,c(VAR z))))
             end
        in conv(e,kont)
       end
   | Lambda.VAR v => c(ren v)
   | Lambda.APP(Lambda.FN(v,_,e),a) =>
       let fun kont(w) = 
             let val _ = newname(v,w)
                 val _ = addty(v,grabty(w))
                 val _ = case w of VAR w' => Access.sameName(v,w')
                                 | _ => ()
              in conv(e,c)
             end
        in conv(a,kont)
       end
   | Lambda.FN(v,t,e) =>   (* using "save" the reference cell is 
                              dirty, but i can't find better way *)
       let val _ = addty(v,t)
           val save = ref LT.injBOXED and k = mkLvar()
           fun kont(vb) =
             let val t = grabty(vb)
                 val _ = (save := t)
                 val (ul,header) = mkArgOut(t,vb)
              in header(APP(VAR k,ul))
             end
           val ce = conv(e,kont)
           val t1 = !save
           val f = mkv(LT.injARROW(t,t1)) and _ = (addty(k,(LT.inj o LT.CONT) t1))
           val (vl,cl,header) = mkArgIn(t,v)
        in FIX([(ESCAPE,f,k::vl,CNTt::cl,header(ce))],c(VAR f))
       end
   | Lambda.APP(f,a) =>   (* different from the old version in 
                             that header is now put in the middle
                             of evaluations between f and a, a bit odd *)
       let fun kont(vf) =
             let val (t1,t2) = arrowLty(grabty(vf))
                 val (header,F) = preventEta(c,t2)
                 fun kont'(va) = 
                   let val (ul,header') = mkArgOut(t1,va)
                    in header'(APP(vf,F::ul))
                   end
              in header(conv(a,kont'))
             end
        in conv(f,kont)
       end
   | Lambda.FIX(fl, tl,  el, body) =>
       let fun g(f::fl, t::tl, Lambda.FN(v,_,b)::el) =
                let val (t1,t2) = arrowLty(t)
                    val _ = addty(v,t1) and k = mkv((LT.inj o LT.CONT)(t2))
                    val (vl,cl,header) = mkArgIn(t1,v)
                    fun kont(vb) = 
                       let val (ul,header') = mkArgOut(t2,vb)
                        in header'(APP(VAR k,ul))
                       end
                    val be = conv(b,kont)
                 in (ESCAPE,f,k::vl,CNTt::cl,header(be))::g(fl,tl,el)
                end
             | g(nil, nil, nil) = nil
             | g _ = err "convert.conv.FIX1"

           fun h(f::fl,t::tl) = (addty(f,t);h(fl,tl))
             | h(nil,nil) = ()
             | h _ = err "convert.conv.FIX2"

           val _ = h(fl,tl)
        in FIX(g(fl,tl,el),conv(body,c))
       end
   | Lambda.INT i => ((i+i+2; c(INT i))
      handle Overflow =>
	 let open Lambda
 	  in conv(APP(PRIM(AP.IADD,IntOpTy),
                      RECORD([INT(i div 2),INT(i - i div 2)])),c)
	 end)
   | Lambda.WORD w => let
       val maxWord = Word.<<(0wx2000, 0w16)
     in
       if Word.<(w, maxWord) then c(INT (Word.toInt w)) 
       else let
	   open Lambda
	   val addu = AP.ARITH{oper=AP.+, overflow=false, kind=AP.UINT 31}
	   val half = Word.div(w, 0w2)
	   val x1 = Word.toInt half
	   val x2 = Word.toInt(Word.-(w, half))
 	 in 
	   conv(APP(PRIM(addu, IntOpTy), RECORD([INT x1, INT x2])),c)
	 end
     end
   | Lambda.WORD32 w => c(INT32 w)
   | Lambda.REAL i => c(REAL i)
   | Lambda.STRING s =>  c(STRING s)
   | Lambda.RECORD [] => c(INT 0) 
                         (* err "zero length records in convert" *)
   | Lambda.SRECORD [] => c(INT 0) 
                         (* err "zero length records in convert" *)
   | Lambda.VECTOR [] => err "zero length vectors in convert"
   | Lambda.RECORD l => 
       let fun kont(vl) =
             let val tyl = map grabty vl
                 val lt = LT.injRECORD tyl
                 val recordCE = 
                   if (isFloatRec lt) then recordFL else recordNM
                 val w = mkv(lt)
              in recordCE(vl,tyl,w,c(VAR w))
             end
        in convlist(l,kont)
       end                
   | Lambda.SRECORD l => 
       let fun kont(vl) =
             let val tyl = map grabty vl
                 val w = mkv(LT.inj(LT.SRECORD(tyl)))
              in recordNM(vl,tyl,w,c(VAR w))
             end
        in convlist(l,kont)
       end                
   | Lambda.VECTOR l => 
       let fun kont(vl) =
             let val w = mkv(LT.injBOXED)
              in RECORD(RK_VECTOR, map (fn v => (v, OFFp0)) vl, w, c(VAR w))
             end
        in convlist(l,kont)
       end
   | Lambda.SELECT(i, e) => 
       let fun kont(v) =
            let val lt = grabty(v)
                val t = selectLty(lt,i)
                val w = mkv(t)
                val selectCE = if (isFloatRec lt) then selectFL else selectNM
             in selectCE(i, v, w, ctype t, c(VAR w))
            end
        in conv(e,kont)
       end
   | Lambda.CON((_,CONSTANT i,t),e) =>            (* coercion needed *)
       let val t1 = getConTy t
        in case LT.out t1 
            of LT.BOXED => 
                  mkfn(fn x => PURE(P.wrap,[INT i],x,BOGt,c(VAR x)),LT.injBOXED)
             | _ => c(INT i)
       end             
   | Lambda.CON((x,LISTNIL,t),e) => conv(Lambda.CON((x,CONSTANT 0,t),e),c)
   | Lambda.CON((x,LISTCONS,t),e) => conv(Lambda.CON((x,UNTAGGEDREC 2,t),e),c)
   | Lambda.CON((_,TAGGED i,t),e) => 
       let fun kont(v) =
             let val tyl = [LT.injINT,grabty v]
                 val ul = [INT i, v]
                 val x = mkv(LT.injRECORD tyl)
                 val y = mkv(LT.injBOXED)
                 val be = PURE(P.wrap,[VAR x],y,BOGt,c(VAR y))
              in recordNM(ul,tyl,x,be)
             end
        in conv(e,kont)
       end
   | Lambda.CON((_,TRANSPARENT,t),e) => 
       let val t1 = getConTy t
           fun kont(w) = mkfn(fn x => PURE(P.cast,[w],x,ctype t1,c(VAR x)),t1)
        in conv(e,kont)
       end
   | Lambda.CON((_,UNTAGGEDREC _,t),e) => 
       let val t1 = getConTy t
           fun kont(w) = mkfn(fn x => PURE(P.wrap,[w],x,BOGt,c(VAR x)),t1)
        in conv(e,kont)
       end
   | Lambda.CON((_,UNTAGGED,t),e) => 
       let val t1 = getConTy t
           fun kont(w) = mkfn(fn x => PURE(P.wrap,[w],x,BOGt,c(VAR x)),t1)
        in conv(Lambda.RECORD[e],kont)
       end
   | Lambda.CON((_,TAGGEDREC(tag,n),t),e) => 
       let val t1 = getConTy t
           fun kont(v) =
             let val lt = grabty v
                 val argty = recordLty(lt)
                 val selectCE = if isFloatRec lt then selectFL else selectNM

                 fun f(i) =
                   if i=n then (nil,id)
                   else (let val (vl,header) = f(i+1)
                             val tt = nthty(argty,i)
                             val z = mkv(tt)
                          in ((VAR z)::vl,
                                 fn ce => selectCE(i,v,z,ctype tt,header(ce)))
                         end)
                 val (wl,header) = f(0)
                 val ul = (INT tag)::wl
                 val tyl = LT.injINT::argty
                 val u = mkv(LT.injRECORD(tyl))
                 val x = mkv(t1)
                 val be = PURE(P.wrap,[VAR u],x,BOGt,c(VAR x))
              in header(recordNM(ul,tyl,u,be))
             end
        in conv(e,kont)
       end
   | Lambda.CON((_,VARIABLE p,t),e) =>
       let fun kont(w) = 
             let fun g(LVAR v) = (id, ren v, mapty v)
                   | g(PATH(i,r)) = 
                       let val (header,nv,t) = g(r)
                           val t' = selectLty(t,i)
                           val z = mkv(t')
                        in (fn ce => header(SELECT(i,nv,z,ctype t',ce)), 
                            VAR z, t')
                       end
                   | g _ = err "convert.278"
                 val (header,nv,t1) = g(p)
                 val ul = [nv,w,INT 0]
                 val tyl = [t1,grabty(w),LT.injBOXED]
                 val u = mkv(LT.injRECORD tyl)
                 val x = mkv(LT.injBOXED)
                 val be = PURE(P.wrap,[VAR u],x,BOGt,c(VAR x))
              in header(recordNM(ul,tyl,u,be))
             end
        in conv(e,kont)
       end
   | Lambda.CON((_,VARIABLEc p,_),e) => 
       let fun g (LVAR v) = Lambda.VAR v
	     | g (PATH(i,r)) = Lambda.SELECT(i, g r)
             | g _ = err "convert.2137"
        in conv(g p, c)
       end
   | Lambda.DECON((_,CONSTANT _,_),e) => 
       mkfn(fn x => PURE(P.wrap,[INT 0],x,BOGt,c(VAR x)),LT.injBOXED)
   | Lambda.DECON((x,LISTNIL,t),e) => conv(Lambda.DECON((x,CONSTANT 0,t),e),c)
   | Lambda.DECON((x,LISTCONS,t),e) => conv(Lambda.DECON((x,UNTAGGEDREC 2,t),e),c)
   | Lambda.DECON((_,TAGGED i,t),e) => 
       let val t1 = getDeconTy(t)
           fun kont(v) = 
            let val x = mkv(LT.injRECORD[LT.injINT,t1])
                val y = mkv(t1) 
             in PURE(P.unwrap,[v],x,BOGt,                 
                     selectNM(1,VAR x,y,ctype t1,c(VAR y)))
            end
        in conv(e,kont)
       end
   | Lambda.DECON((_,TRANSPARENT,t),e) => 
       let val t1 = getDeconTy t
           fun kont(w) =mkfn(fn x => PURE(P.cast,[w],x,ctype t1,c(VAR x)),t1)
        in conv(e,kont)
       end
   | Lambda.DECON((_,UNTAGGEDREC _,t),e) => 
       let val t1 = getDeconTy t
           fun kont(w) = 
             mkfn(fn x => PURE(P.unwrap,[w],x,ctype t1,c(VAR x)),t1)
        in conv(e,kont)
       end
   | Lambda.DECON((_,UNTAGGED,t),e) => 
       let val t1 = getDeconTy(t)
           fun kont(v) = 
             let val x = mkv(LT.injRECORD[t1])
              in PURE(P.unwrap,[v],x,BOGt,
                      conv(Lambda.SELECT(0,Lambda.VAR x),c))
             end
        in conv(e,kont)
       end
   | Lambda.DECON((_,VARIABLE p,t),e) => 
       let val t1 = getDeconTy(t)
           fun kont(v) = 
             let val u = mkv(LT.injRECORD[t,t1,LT.injBOXED])
                 val header = fn ce => PURE(P.unwrap,[v],u,BOGt,ce) 
                 val x = mkv(t1)
              in header(selectNM(1,VAR u,x,ctype t1,c(VAR x)))
             end
        in conv(e,kont)
       end
   | Lambda.DECON((_,TAGGEDREC(tag,n),t),e) => 
       let val lt = getDeconTy(t)
           val argty = recordLty(lt)
           fun kont(v) = 
             let val x = mkv(LT.injRECORD(LT.injINT::argty))
                 fun f(i) = 
                  if i = n then (id,nil)
                  else (let val (header,vl) = f(i+1)
                            val tt = nthty(argty,i)
                            val ct = ctype tt
                            val w = mkv(tt)
                         in (fn ce => selectNM(i+1,VAR x,w,ct,header(ce)),
                             (VAR w)::vl)
                        end)
                 val (header,vl) = f(0)
                 val recordCE = if isFloatRec(lt) then recordFL else recordNM
                 val w = mkv(lt)
              in PURE(P.unwrap,[v],x,BOGt,
                      header(recordCE(vl,argty,w,c(VAR w))))
             end
        in conv(e,kont)
       end
   | Lambda.SWITCH(e,l,[a as (Lambda.DATAcon(_,Access.CONSTANT 0,_),_),
		        b as (Lambda.DATAcon(_,Access.CONSTANT 1,_),_)], 
                   NONE) =>
       conv(Lambda.SWITCH(e,l,[b,a],NONE),c)
   | Lambda.SWITCH(x as (Lambda.APP(oper, args),_,
                   [(Lambda.DATAcon(_,Access.CONSTANT 1,_),e1),
	           (Lambda.DATAcon(_,Access.CONSTANT 0,_),e2)],NONE)) =>
       let fun g i' =
	     let val k = mkLvar() and save = ref LT.injBOXED
                 fun kont(w) = 
                   let val t = grabty(w) 
                       val _ = (save := t)
                       val (ul,header1) = mkArgOut(t,w)
                    in header1(APP(VAR k,ul))
                   end
                 val ce1 = conv(e1,kont) and ce2 = conv(e2,kont)
                 val t = !save
                 val _ = addty(k,(LT.inj o LT.CONT) t) and v = mkv(t)
                 val (vl,cl,header) = mkArgIn(t,v)
	      in FIX([(CONT,k,vl,cl,header(c(VAR v)))],
                  getargs(2,args,fn vl => BRANCH(i',vl,mkv(LT.injINT),ce1,ce2)))
	     end
        in case oper
	    of Lambda.PRIM(AP.CMP stuff,lt) => g(cmpop(stuff,#1(arrowLty lt)))
	     | Lambda.PRIM(AP.PTREQL,_) => g(P.peql)
	     | Lambda.PRIM(AP.PTRNEQ,_) => g(P.pneq)
	     | _ => genswitch(x,c)
       end
   | Lambda.SWITCH x => genswitch(x,c)
   | Lambda.RAISE(e,t) =>
       let fun kont(w) = 
             let val h = mkv((LT.inj LT.SRCONT))
                 val _ = mkfn(fn u => c(VAR u), t)
              in LOOKER(P.gethdlr,[],h,FUNt,APP(VAR h,[VAR bogus_cont,w]))
             end
        in conv(e,kont)
       end
   | Lambda.HANDLE(a,b) =>
       let fun kont(vb) =
             let val (_,t) = arrowLty(grabty(vb))
                 val h = mkv((LT.inj LT.SRCONT)) and v = mkv(LT.injBOXED)
                 val k = mkv((LT.inj LT.SRCONT))
                 val (header,F) = preventEta(c,t)
                 fun kont1(va) = 
                   let val (ul,header1) = mkArgOut(t,va)
                    in SETTER(P.sethdlr,[VAR h],
                              header1(APP(F,ul)))
                   end
              in LOOKER(P.gethdlr,[],h,FUNt,
                   header(FIX([(ESCAPE,k,[mkv((LT.inj o LT.CONT) LT.injBOXED),v],[CNTt,BOGt],
                               SETTER(P.sethdlr,[VAR h],APP(vb,[F,VAR v])))],
                              SETTER(P.sethdlr,[VAR k],conv(a,kont1)))))
             end
        in conv(b,kont)
       end
   | Lambda.WRAP(t,le) => 
       let fun kont(w) = 
             let val t = grabty(w)
                 val ct = ctype t
                 val x = mkv(LT.injBOXED)
              in PURE(primwrap ct,[w],x,BOGt,c(VAR x))
             end
        in conv(le,kont)
       end
   | Lambda.UNWRAP(t,le) => 
       let val ct = ctype t
           fun kont(w) = 
             let val x = mkv(t)
              in PURE(primunwrap ct,[w],x,ct,c(VAR x))
             end
        in conv(le,kont)
       end
   | _ => err "convert.sml 7432894"


(***************************************************************************
 * genswitch : (Lambda.lexp * Access.conrep list * (Lambda.con *           *
 *                 Lambda.lexp) list * Lambda.lexp option) *               *
 *              (value -> cexp) -> cexp                                    *
 ***************************************************************************)
and genswitch ((e, sign, l: (Lambda.con * Lambda.lexp) list, d),c) =
  let val df = mkv((LT.inj o LT.CONT) LT.injINT) and save = ref LT.injBOXED and k = mkLvar()
      fun kont1(z) = 
        let val t = grabty z
            val _ = (save := t)
            val (ul,header) = mkArgOut(t,z)
         in header(APP(VAR k,ul))
        end

      val l' = map (fn(c,e)=>(c,conv(e,kont1))) l

      fun kont(w) = do_switch{sign=sign,exp=w,cases=l',default=APP(VAR df,[INT 0])}
      val body = conv(e,kont)

      val body' = case d 
         of NONE => body
	  | SOME d' => FIX([(CONT,df,[mkv(LT.injINT)],[INTt],conv(d',kont1))],
			   body)

      val t = !save
      val v = mkv(t) and _ = (addty(k,(LT.inj o LT.CONT) t))
      val (vl,cl,header) = mkArgIn(t,v)
   in FIX([(CONT,k,vl,cl,header(c(VAR v)))],body')
  end

val save = ref LT.injBOXED and k = mkLvar() and f = mkLvar() and v = mkLvar()
fun kont(w) = 
  let val t = grabty(w)
      val (t1,t2) = arrowLty(t)
      val _ = (addty(k,(LT.inj o LT.CONT) t2); addty(f,t); addty(v,t1); save := t1)
      val (ul,header) = mkArgOut(t1,VAR v)
   in header(APP(w,(VAR k)::ul))
  end
val lexp = if (MachSpec.newListRep) 
           then (TransList.translist(MachSpec.listCellSz,lexp)) else lexp
(* val _ = MCprint.printLexp lexp *)
val body = conv(lexp,kont)
val (vl,cl,header) = mkArgIn(!save,v)

val bogus_knownf = mkv((LT.inj o LT.CONT)(LT.injBOXED))
val bogushead = 
     fn ce => FIX([(KNOWN,bogus_knownf,[mkv(LT.injBOXED)],[BOGt],
                    APP(VAR bogus_knownf,[STRING "bogus"]))],
                  FIX([(CONT,bogus_cont,[mkv(LT.injBOXED)],[BOGt],
                        APP(VAR bogus_knownf,[STRING "bogus"]))],ce))

in ((ESCAPE,f,k::vl,CNTt::cl,header(bogushead(body))),typtable)
end

end

