(* profast.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *)

structure ProfileInternals =
struct
val update = System.Unsafe.update
val subscript = System.Unsafe.subscript
val op + : int * int -> int = op +
val profiling = ref true

    val times = System.Unsafe.Assembly.times
    val _ = times := array(3,0)


    datatype compunit = UNIT of {base: int,
				 size: int,
				 counts: int Array.array,
				 names: string
				}
			   
    val units = ref [ UNIT{base=0,size=3,counts=Array.array(3,0),
			   names="Other\nCompilation\nGarbage Collection\n"} ];
local
    fun newlines s = 
	let val n = size s
	    fun f(i,count) = if i=n then count 
			     else if substring(s,i,1)="\n"
				      then f(i+1,count+1)
				      else f(i+1,count)
         in f(0,0)
	end
					     
in
    fun register names =
	let val ref (list as UNIT{base,size,...}::_) = units
	    val count = newlines names
	    val a = array(count,0)
	    val b = base+size
	    val n = b+count
	 in if n <= Array.length(!times) then ()
	    else let open Array
		     val t = !times
		     val oldlen = Array.length t
		     val newlen = n+n
		     val new = array(newlen,0)
		     fun copy(i) = if i<oldlen then (update(new,i,t sub i); 
						     copy(i+1))
				   else ()
		 in copy(0);
		     times := new
		 end;
	     units := UNIT{base=b,size=count,counts=a,names=names}::list;
	    (b,a,System.Unsafe.Assembly.profCurrent)
        end
end

local 

open Compiler.Ast

infix -->

(* Profiling globals *)
val profileList = ref([]: (Access.lvar * string) list)

fun App(f,a) = AppExp{function=f,argument=a}

		
val anonSym = varSymbol "anon"
val ProfileInternals = strSymbol "ProfileInternals"
val updateOp =  VarExp[ProfileInternals,varSymbol "update"]
fun updateExp (a,k,b) = App(updateOp, TupleExp[a, k, b])

val subop =  VarExp[ProfileInternals,varSymbol "subscript"]

val addop = VarExp[ProfileInternals,"+"]

val registerop = VarExp[ProfileInternals,"register"]

fun tmpvar str = [varSymbol str]

fun clean (path as name::names) = 
           if Symbol.eq(name,anonSym) then names else path
  | clean x = x

in 

fun instrumDec ast =
 let val countarrayvar = tmpvar "countarray"
     val countarray = VarExp countarrayvar
     val basevar = tmpvar "base"
     val baseexp = VarExp basevar
     val currentvar = tmpvar "profCurrent"
     val currentexp = VarExp currentvar
     
     val entries = ref (nil: string list)
     val entrycount = ref 0
     fun makeEntry(name) = let val i = !entrycount
	                    in entries := "\n" :: name :: !entries;
			       entrycount := i+1;
			       i
			   end

     fun BUMPCCexp ccvara = 
	 let val lvar = tmpvar "indexvar"
	  in updateExp (countarray,
			IntExp ccvara,
			App(addop,
			    TupleExp[
			      App(subop,
				     TupleExp[countarray, IntExp ccvara]),
				     IntExp 1]))
	 end
     fun SETCURRENTexp ccvara =
	 let val lvar = tmpvar "indexvar"
	  in LetExp{dec=ValDec[Vb{pat=VarPat lvar,
				  exp=App(addop,
					  TupleExp[IntExp ccvara, baseexp])}],
		    exp=updateExp (currentexp, IntExp 0, VarExp lvar)}
	 end

   fun instrdec(sp as (names,ccvara), ValDec vbl) = 
    let fun instrvb (Vb{pat as VarPat[n],exp}) =
	      Vb{pat=pat,exp=instrexp (n::clean names,ccvara) false exp}
	  | instrvb (Vb{pat as ConstraintPat(VarPat[n]),exp}) =
	      Vb{pat=pat,exp=instrexp (n::clean names,ccvara) false exp}
	  | instrvb (Vb{pat,exp}) = Vb{pat=pat, exp=instrexp sp false exp}
          | instrvb (MarkVb(v,a,b)) = MarkVb(instrvb v, a, b)
    in VALdec (map instrvb vbl)
    end
  | instrdec(sp as (names,ccvara), ValrecDec rvbl) = 
    let fun instrrvb (Rvb{var,exp,resultty}) =
               RVB{var=var, 
		   exp=instrexp (var::clean names, ccvara) false exp, 
		   resultty=resultty}
	  | instrrvb (MarkRvb(v,a,b)) = MarkRvb(instrrvb v, a, b)
    in ValrecDec(map instrrvb rvbl)
    end
  | instrdec(sp, AbstypeDec{abstycs,withtycs, body}) =
	AbstypeDec {abstycs=abstycs,withtycs=withtycs, body=instrdec(sp,body)}
  | instrdec(sp, StrDec strbl) = StrDec (map (fn strb => instrstrb(sp,strb)) strbl)
  | instrdec(sp, AbsDec strbl) = AbsDec (map (fn strb => instrstrb(sp,strb)) strbl)
  | instrdec(sp, FctDec fctbl) = FctDec (map (fn fctb => instrfctb(sp,fctb)) fctbl)
  | instrdec(sp, LocalDec(localdec,visibledec)) =
	LocalDec(instrdec (sp,localdec), instrdec (sp,visibledec))
  | instrdec(sp, SeqDec decl) = SeqDec (map (fn dec => instrdec(sp,dec)) decl)
  | instrdec(sp, MarkDec(dec,a,b)) = MarkDec(instrdec (sp,dec), a,b)
  | instrdec(sp, other) = other

and instrstrexp(names, StructStr dec) = StructStr(instrdec((names,0),dec))
  | instrstrexp(names, AppStr (path,args) =
		AppStr(path,map(fn(a,b)=>(instrstrexp(names,a),b)) args)
  | instrstrexp(names, VarStr x) = VarStr x
  | instrstrexp(names, LetStr(d,body)) = 
		LetStr(instrdec((names,0),d), instrstrexp(names,body))
  | instrstrexp(names,MarkStr(body,a,b)) = MarkStr(instrstrexp(names,body),a,b)

and instrstrb ((names,ccvara), Strb{name,def,constraint}) =
    Strb{name=name,def=instrstrexp(name::names,def),constraint=constraint}
  | instrstrb (z,MarkStrb(strb,a,b)) = MarkStrb(instrstrb(z,strb),a,b)

and instrfctb ((names,ccvara), Fctb{name,def} =
	       Fctb{name=name,def=instrfctexp(name::names,def)}
  | instrfctb (z,MarkFctb(f,a,b)) = MarkFctb(instrfctb(z,f),a,b)

and instrfctexp(names,v as VarFct _) = v
  | instrfctexp(names,FctFct{params,body,constraint}) =
    FctFct{params=params,body=instrstrexp(names,body),constraint=constraint}
  | instrfctexp(names,LetFct(dec,fctexp)) =
    LetFct(instrdec((names,0),dec), instrfctexp(names, fctexp))
  | instrfctexp(names,AppFct(f,args,constraint)) =
    AppFct(f,map(fn(s,b)=>(instrstrexp(names,s),b)) args, constraint)
  | instrfctexp(names,MarkFct(f,a,b)) = MarkFct(instrfctexp(names,f),a,b)

and instrexp(sp as (names,ccvara)) =
 let fun istail tail =
     let fun iinstr exp = istail false exp
	 fun oinstr exp = istail true exp
	 fun instrrules tr = 
	     map (fn (Rule{pat,exp}) => Rule{pat=pat, exp=tr exp))
	 val rec instr:(exp->exp) =
	     fn RecordExp l => RecordExp(map (fn (lab,exp) => (lab,iinstr exp)) l)
	      | VectorExp l => VectorExp(map iinstr l)
	      | SeqExp l =>
		let fun seq [e] = [instr e]
		      | seq (e::r) = (iinstr e)::(seq r)
		      | seq nil = nil
		in SeqExp (seq l)
		end
	      | AppExp{function=f,argument=a} =>
	           if tail
		    then App (iinstr f, iinstr a)
		    else let val lvar = tmpvar "appvar"
			  in LetExp{dec=ValDec[Vb{pat=VarPat lvar,
						  exp=App(iinstr f,iinstr a)}],
				    exp=SeqExp[SETCURRENTexp ccvara, 
					       VarExp lvar]}
			 end
	      | ConstraintExp{expr,constraint} => 
		    ConstraintExp{expr=instr e, constraint=constraint}
	      | HandleExp{expr,rules} =>
		let fun tr exp = SeqExp[SETCURRENTexp ccvara, instr e])
		in HandleExp{expr=instr expr, rules=instrrules tr rules}
		end
	      | RaiseExp e => RaiseExp (oinstr e)
	      | LetExp{dec,expr} => 
		           LetExp{dec=instrdec(sp,dec), expr= instr expr}
	      | CaseExp{expr,rules} =>
		    CaseExp{expr=iinstr e, rules=instrrules instr rules}
	      | FnExp l =>
		let fun dot (a,[z]) = Symbol.name z :: a
		      | dot (a,x::rest) = dot("." :: Symbol.name x :: a, rest)
		      | dot _ = ErrorMsg.impossible "no path in instrexp"
		    val name =  concat (dot ([], names))
		    val ccvara' = makeEntry(name)
		    val lvar = tmpvar "fnvar";
		in FnExp [Rule{pat=VarPat lvar, 
			       expr=SeqExp [BUMPCCexp ccvara',
				   SETCURRENTexp ccvara',
			           CaseExp(VarExp lvar,
			    	       instrrules (instrexp (anonSym::names,
			    				     ccvara') true) rules)]}]
		end
	      | MarkExp(e,a,b) => MarkExp(instr e, a, b)
	      | e => e 
     in instr
     end
 in istail
 end

   val ast' = instrdec(([],0),ast)

in LocalDec(ValDec[Vb{pat=TuplePat[VarPat basevar,
				   VarPat countarrayvar,
				   VarPat currentvar],
		      exp=App(registerop, StringExp(concat(rev(!entries))))}],
	    ast')
end




end (* local *)

end


signature PROFILE =
sig
  val use_prof : string -> unit     (* compile a file with profiling *)
  val profileOn : unit -> unit      (* turn interrupt timer on *)
  val profileOff : unit -> unit     (* turn interrupt timer off *)
  val reset : unit -> unit          (* reset profiling counts to zero *)
  val report : outstream -> unit    (* print profiling report to stream *)
end


structure Profile : PROFILE = struct

    val current : int ref = System.Unsafe.Assembly.current

    open ProfileInternals

    open Array List infix 9 sub

      local
	open System.Timer
	val t0 = TIME{sec=0, usec=0} and t10 = TIME{sec=0, usec=10000}
      in
      val timerval = (1 (* ITIMER_VIRTUAL *), t10, t10)
      val timerval0 = (1 (* ITIMER_VIRTUAL *), t0, t0)
      end

   local open System.Signals
    in 
	fun profileOn () = 
	    ((*setHandler(SIGVTALRM,
			SOME(
			fn (n,c) => let val t = !times and i = !current
			             in update(t,i, t sub i + n);
					c
				    end)); *)
	     System.Unsafe.CInterface.setitimer timerval; 
	     ())
        fun profileOff () = 
	    (System.Unsafe.CInterface.setitimer timerval0; 
	     (* setHandler(SIGVTALRM,NONE); *)
	     ())
    end

    fun zero a = let val len = Array.length a
	             fun f i = if i=len then () else (update(a,i,0); f(i+1))
		  in f 0
		 end

    fun reset() = (zero (!times);
		   app (fn UNIT{counts,...}=> zero counts) (!units))

       

    
    datatype entry = ENTRY of {name: string, count: int, time: int}

    fun splitlines "" = nil
      | splitlines s =
	let fun f(l,i,j) = if i=0 then substring(s,0,j-1)::l
	                   else if substring(s,i-1,1)="\n"
	                   then f(substring(s,i,j-i-1)::l,i-1,i)
			       else f(l,i-1,j)
	 in f(nil,size s - 1, size s)
        end

    fun join(entries, base, _, counts, times, nil) = entries
      | join(entries, base, n, counts, times, line::lines) =
          join(ENTRY{name=line,count=counts sub n,
		     time = times sub (base+n)}::entries,
	       base, n+1, counts, times, lines)

    fun batch(UNIT{base,size,counts,names}) =
	join(nil, base, 0, counts, !times, splitlines names)

    fun log10 0 = 0
      | log10 i = 1 + log10(i div 10)

    fun field (st,w) = 
	  let val s = "                    " ^ st
	   in substring(s,String.size s - w, w)
	  end

    fun decimal(st,w) =
	(substring(st,0,size st - w) handle Substring => "") ^ "." ^
	      let val st' = "00000000000" ^ st
	       in substring(st',size st' - w,w)
	      end


    fun muldiv(i,j,k) =
	  (i*j div k) 
	     handle Overflow => muldiv(i,j div 2, k div 2)

    fun decfield(n,j,k,w1,w2) = 
	  field(decimal(makestring (muldiv(n,j,k)),w1)
		  handle Div => "",w2)

    fun report outstream =
	let val biglist = fold (op @) (map batch (!units)) nil

	    val biglist' = Sort.sort 
		             (fn (ENTRY{time=a,count=ca,name=na,...},
				  ENTRY{time=b,count=cb,name=nb,...})=>
			       a<b orelse 
			       a=b andalso (ca<cb orelse 
					    ca=cb andalso na>nb)
			       )
			     biglist

	    val tot = fold(fn (ENTRY{time=a,...},b)=> a+b) biglist' 0
	    val maxc = fold (fn (ENTRY{count=a,...},b)=> max(a,b)) biglist' 0

	    val digits_cum = log10 tot
            val digits_cnt = max(6,1+log10 maxc)

	    val pr = outputc outstream

	    fun printlines (ENTRY{time,name,count}::rest,cum) =
		(pr(decfield(time,10000,tot,2,6));
		 if (digits_cum > 4)
		     then pr(field(makestring(cum+time+50 div 100),7))
		     else pr(decfield(cum+time,1,1,2,7));
		 pr(field(makestring count,digits_cnt));
(*		 pr(decfield(time,50000,count,4,10)); *)
		 pr "  "; pr name; pr "\n";
		 printlines(rest,cum+time))
	      | printlines (nil,_) = ()

	 in pr(field("%time",6));
	    pr(field("cumsec",7));
	    pr(field("#call",digits_cnt));
(*	    pr(field("ms/call",10)); *)
	    pr("  name\n");
	    printlines(biglist',0);
	    flush_out outstream
        end

end

