(* prof.sml
 *
 * Copyright 1989 by AT&T Bell Laboratories
 *)

signature PROF =
  sig
    val instrumDec : StaticEnv.staticEnv (* coreEnv*) -> Absyn.dec -> Absyn.dec
  end

abstraction Prof : PROF =
struct

open Access Absyn ElabUtil Variables Modules Types BasicTypes ErrorMsg
structure P = Access.P (* to avoid confusing SourceGroup *)
structure SP = SymPath

infix -->
val varSymbol = Symbol.varSymbol

(* Profiling globals *)
val profiling = System.Unsafe.profiling
val profileList = ref([]: (Access.lvar * string) list)
		
val anonSym = varSymbol "anon"

val intreftype = CONty(refTycon,[intTy])

fun poly1 ty = POLYty{sign=[{weakness=infinity,eq=false}],abs=0,
				  tyfun=TYFUN{arity=1,body=ty}}
val updateop = VALvar{path=SP.SPATH[Symbol.varSymbol "unboxedupdate"],
		      access=INLINE(P.UNBOXEDUPDATE),
	 	      typ = ref(poly1(tupleTy[CONty(arrayTycon,[IBOUND 0]),
					 intTy, IBOUND 0] --> unitTy))}

(* val assignop = VALvar{path=SP.SPATH[Symbol.varSymbol "unboxedupdate"],
		      access=INLINE(P.UNBOXEDUPDATE),
	 	      typ = ref(poly1(tupleTy[CONty(refTycon,[IBOUND 0]),
					 intTy, IBOUND 0] --> unitTy))} *)

val assignop = VALvar{path=SP.SPATH[Symbol.varSymbol ":="],
		      access=INLINE(P.ASSIGN),
	 	      typ = ref(poly1(tupleTy[CONty(refTycon,[IBOUND 0]),
					 intTy, IBOUND 0] --> unitTy))}

val subop = VALvar{path=SP.SPATH[Symbol.varSymbol "subscript"],
		   access=INLINE(P.SUBSCRIPT),
		   typ=ref(poly1(tupleTy[CONty(arrayTycon,[IBOUND 0]),
					 intTy] --> IBOUND 0))}
val derefop = VALvar{path=SP.SPATH [Symbol.varSymbol "!"],
		    access=INLINE(P.DEREF),
		    typ= ref(poly1(CONty(refTycon,[IBOUND 0]) --> IBOUND 0))}

val addop = VALvar{path = SP.SPATH[varSymbol "iadd"],
		   access = INLINE P.IADD,
		   typ = ref(tupleTy[intTy,intTy] --> intTy)}

fun tmpvar(str,ty) = 
    let val sym = Symbol.varSymbol str
     in VALvar{access=LVAR(namedLvar sym), path=SP.SPATH[sym], typ=ref ty}
    end

fun varexp(v as VALvar{typ=ref ty,path,...}) =
    case TypesUtil.headReduceType ty
      of POLYty _ =>
	   ErrorMsg.impossible ("poly["^SP.makestring path^"] in Prof")
       | ty' => VARexp(ref v, NONE) (* VARexp(ref v, SOME ty') *)

fun clean (path as name::names) = if Symbol.eq(name,anonSym) then names else path
  | clean x = x

fun instrumDec' coreEnv absyn =
 let val countarrayvar = tmpvar("countarray",CONty(arrayTycon,[intTy]))
     val countarray = varexp countarrayvar
     val basevar = tmpvar("base",intTy)
     val baseexp = varexp basevar
     val currentvar = tmpvar("profCurrent",CONty(refTycon,[intTy]))
     val currentexp = varexp currentvar
     
     val VARbind register = ModuleUtil.lookVARCON 
	 (coreEnv,
	  SP.SPATH [Symbol.strSymbol "Core",
			 Symbol.varSymbol "profile_register"],
	  fn _ => fn s => fn _ => ErrorMsg.impossible "222 in prof")

     local val VALvar{typ=ref ty,...} = register
           val CONty(reff,[ty']) = TypesUtil.headReduceType ty
      in val profDerefTy = ty --> ty'
     end

     val entries = ref (nil: string list)
     val entrycount = ref 0
     fun makeEntry(name) = let val i = !entrycount
	                    in entries := "\n" :: name :: !entries;
			       entrycount := i+1;
			       i
			   end

     val intUpdTy = tupleTy[CONty(arrayTycon,[intTy]),intTy,intTy] --> unitTy
     val intSubTy = tupleTy[CONty(arrayTycon,[intTy]),intTy] --> intTy

     fun BUMPCCexp (ccvara : int) = 
	 let val lvar = tmpvar("indexvar",intTy)
	  in 
             APPexp(VARexp(ref updateop,SOME intUpdTy),
		    TUPLEexp[countarray,
			     INTexp (makestring ccvara, intTy),
			     APPexp(varexp addop,
				    TUPLEexp[APPexp(VARexp(ref subop,SOME(intSubTy)),
						    TUPLEexp[countarray, 
							     INTexp (makestring ccvara, intTy)]),
					     INTexp ("1",intTy)])])
	 end

     val intAssTy = tupleTy[CONty(refTycon,[intTy]),intTy] --> unitTy

     fun SETCURRENTexp (ccvara : int) =
	 let val lvar = tmpvar("indexvar",intTy)
	  in LETexp(VALdec[VB{pat=VARpat(lvar),
			    exp=APPexp(varexp addop,
				       TUPLEexp[INTexp (makestring ccvara,
							intTy),
						baseexp]),
			    tyvars=ref nil}],
		    APPexp(VARexp(ref assignop,SOME intAssTy),
			   TUPLEexp[currentexp, varexp lvar]))
	 end

   fun instrdec(sp as (names,ccvara), VALdec vbl) = 
       let fun getvar(VARpat v) = SOME v
	     | getvar(CONSTRAINTpat(p,_)) = getvar p
	     | getvar _ = NONE

	   fun instrvb(vb as VB{pat,exp,tyvars}) =
	       case getvar pat
		of SOME(VALvar{access=INLINE _,...}) => vb
		 | SOME(VALvar{path=SP.SPATH[n],...}) =>
		     VB{pat=pat, tyvars=tyvars,
			exp=instrexp (n::clean names,ccvara) false exp}
		 | _ => VB{pat=pat, exp=instrexp sp false exp, tyvars=tyvars}

	in VALdec (map instrvb vbl)
       end
  | instrdec(sp as (names,ccvara), VALRECdec rvbl) = 
    let fun instrrvb (RVB{var as VALvar{path=SP.SPATH[n],...},
			  exp,resultty,tyvars}) =
               RVB{var=var,
		   exp=instrexp(n::clean names, ccvara) false exp,
		   resultty=resultty, tyvars=tyvars}
	  | instrrvb _ = impossible "VALRECdec in instrdec"
     in VALRECdec(map instrrvb rvbl)
    end
  | instrdec(sp, ABSTYPEdec {abstycs,withtycs,body}) = 
	ABSTYPEdec {abstycs=abstycs,withtycs=withtycs, body=instrdec(sp,body)}
  | instrdec(sp, STRdec strbl) = STRdec (map (fn strb => instrstrb(sp,strb)) strbl)
  | instrdec(sp, ABSdec strbl) = ABSdec (map (fn strb => instrstrb(sp,strb)) strbl)
  | instrdec(sp, FCTdec fctbl) = FCTdec (map (fn fctb => instrfctb(sp,fctb)) fctbl)
  | instrdec(sp, LOCALdec(localdec,visibledec)) =
	LOCALdec(instrdec (sp,localdec), instrdec (sp,visibledec))
  | instrdec(sp, SEQdec decl) = SEQdec (map (fn dec => instrdec(sp,dec)) decl)
  | instrdec(sp, MARKdec(dec,region)) = MARKdec(instrdec (sp,dec), region)
  | instrdec(sp, other) = other

and instrstrexp(names, STRUCTstr {body,locations,str}) =
      STRUCTstr{body = (map (fn dec => instrdec((names,0),dec)) body),
				    locations=locations,str=str}
  | instrstrexp(names, APPstr {oper,instancelty,argexp,argthin,str}) = 
      APPstr{oper=oper, argexp=instrstrexp(names,argexp), argthin=argthin,
             instancelty=instancelty, str=str}
  | instrstrexp(names, VARstr x) = VARstr x
  | instrstrexp(names, LETstr(d,body)) = 
		LETstr(instrdec((names,0),d), instrstrexp(names,body))
  | instrstrexp(names,MARKstr(body,region)) =
      MARKstr(instrstrexp(names,body),region)

and instrstrb ((names,ccvara), STRB{strvar=strvar as STRvar{name,...},
				    abslty,def,thin,constraint}) = 
        STRB{strvar=strvar, def=instrstrexp(name::names,def),
	     abslty=abslty, thin=thin, constraint=constraint}

and instrfctb ((names,ccvara), FCTB{fctvar=fctvar as FCTvar{name,...},
				    def=FCTfct{param,def,thin,constraint}}) =
        FCTB{fctvar=fctvar,
             def=FCTfct{param=param,def=instrstrexp(name::names,def),
                      thin=thin, constraint=constraint}}
  | instrfctb ((names,ccvara),fctb) = fctb

and instrexp(sp as (names,ccvara)) =
 let fun istail tail =
     let fun iinstr exp = istail false exp
	 fun oinstr exp = istail true exp
	 fun instrrules tr = map (fn (RULE(p,e)) => RULE(p, tr e))
	 val rec instr:(exp->exp) =
	     fn RECORDexp l => RECORDexp(map (fn (lab,exp) => (lab,iinstr exp)) l)
	      | VECTORexp(l,t) => VECTORexp((map iinstr l),t)
	      | SEQexp l =>
		let fun seq [e] = [instr e]
		      | seq (e::r) = (iinstr e)::(seq r)
		      | seq nil = nil
		in SEQexp (seq l)
		end
	      | exp as APPexp (f,a) =>
	        let fun safe(VARexp(ref(VALvar{access=INLINE P.CALLCC,
                                               ...}),_)) = false
		      | safe(VARexp(ref(VALvar{access=INLINE _,
                                               ...}),_)) = true
		      | safe(MARKexp(e,_)) = safe e
		      | safe(CONSTRAINTexp(e,_)) = safe e
		      | safe(SEQexp[e]) = safe e
		      | safe _ = false
		    fun rator_instr a = case a
			of APPexp(randf,_) =>
			         if safe randf then iinstr else oinstr
			 | VARexp _ => oinstr
			 | MARKexp(e,_) => rator_instr e
			 | CONSTRAINTexp(e,_) => rator_instr e
			 | SEQexp[e] => rator_instr e
			 | _ => iinstr

		    val f' = rator_instr a f

 		 in if tail orelse (safe f)
		    then APPexp (f', oinstr a)
		    else let val ty = Reconstruct.expType exp
			     val lvar = tmpvar("appvar",ty)
			  in LETexp (VALdec[VB{pat=VARpat(lvar),
					       exp=APPexp(f', oinstr a),
					       tyvars=ref nil}],
			     SEQexp([SETCURRENTexp(ccvara), varexp lvar]))
			 end
		end
	      | CONSTRAINTexp(e,t) => CONSTRAINTexp(instr e, t)
	      | HANDLEexp (e, HANDLER(FNexp(l,t)))=> 
		let fun rule(RULE(p,e)) = 
				RULE(p,SEQexp[SETCURRENTexp ccvara, instr e])
		in HANDLEexp (instr e, HANDLER(FNexp(map rule l,t)))
		end
	      | RAISEexp(e,t) => RAISEexp(oinstr e,t)
	      | LETexp (d,e) => LETexp (instrdec(sp,d), instr e)
	      | CASEexp (e,l) => CASEexp(iinstr e, instrrules instr l)
	      | FNexp(l,t) =>
		let fun dot (a,[z]) = Symbol.name z :: a
		      | dot (a,x::rest) = dot("." :: Symbol.name x :: a, rest)
		      | dot _ = impossible "no path in instrexp"
		    val name =  concat (dot ([], names))
		    val ccvara' = makeEntry(name)
		    val lvar = tmpvar("fnvar",t);

                    val CONbind exnMatch = ModuleUtil.lookVARCON 
                       (coreEnv,SP.SPATH [Symbol.strSymbol "Core", 
                                          Symbol.varSymbol "Match"],
  	                       fn _ => fn s => fn _ => 
                                    ErrorMsg.impossible "250 in prof")
                    val RULE(_,special) = List.last l
		 in FNexp ([RULE(VARpat(lvar), 
		                SEQexp ([BUMPCCexp(ccvara'),
					 SETCURRENTexp(ccvara'),
					 CASEexp(varexp lvar,
					 instrrules (instrexp (anonSym::names,
					                 ccvara') true) l)])),
                            RULE(WILDpat,RAISEexp(CONexp(exnMatch,NONE),
                                                 Reconstruct.expType special))
		           ],
		    t)
		end
	      | MARKexp(e,region) => MARKexp(instr e, region)
	      | e => e 
     in instr
     end
 in istail
 end

   val absyn1 = instrdec(([],0),absyn)

   val absyn2 = LOCALdec(VALdec[VB{pat=TUPLEpat[VARpat basevar,
				   VARpat countarrayvar,
				   VARpat currentvar],
		      exp=APPexp(APPexp(VARexp(ref derefop, SOME profDerefTy),
					varexp register),
				 STRINGexp(concat(rev(!entries)))),
		      tyvars=ref nil}],
                  absyn1)

in absyn2

end

fun instrumDec coreEnv absyn = 
    if !profiling then instrumDec' coreEnv absyn else absyn

end (* structure Instrum *)
