(* compile.sml
 *
 * COPYRIGHT (c) 1994 AT&T Bell Laboratories.
 *)

local
  exception Compile of string
in

functor CompileF(Machm : CODEGENERATOR) : COMPILE =
struct

  structure SE = StaticEnv and EM = ErrorMsg

  type lvar = Access.lvar
  type absyn = Absyn.dec
  type lambda = Lambda.lexp
  type clambda = lambda
  type plambda = lambda
  type pid = PersStamps.persstamp
  type obj = System.Unsafe.object
  type lsegments = { l0: lambda, ln: lambda list }
  type csegments = { c0: string, cn: string list }

  val architecture = Machm.architecture

  exception Compile = Compile

  fun pickle cl = cl
  val unpickle = Lambda.rehashcons

  fun fail s = raise (Compile s)

  val parsePhase = Stats.makePhase "Compiler 010 Parse"

  fun parse source =
      let val parser = Parse.parse source
          val _ = (CheckLty.fname_ref := #fileName(source))

	  fun loop asts = 
	      case parser()
               of Parse.EOF => Ast.SeqDec(rev asts)
		| Parse.ABORT => fail "syntax error"
		| Parse.ERROR => fail "syntax error"
		| Parse.PARSE ast => loop(ast::asts)

       in loop nil
      end

  val parse = Stats.doPhase parsePhase parse

  fun parseOne (source: Source.inputSource) =
      let val parser = Parse.parse source
	  val parser = Stats.doPhase parsePhase parser (* for correct timing *)
       in fn () =>
	     case parser ()
	       of Parse.EOF => NONE
		| Parse.ABORT => fail "syntax error"
		| Parse.ERROR => fail "syntax error"
		| Parse.PARSE ast => SOME ast
     end

  fun showPid pid =
      let val >> = Word.>> infix 3 >>
	  val fsay = Control.Print.say
	  fun hexdig i = fsay(substring("0123456789abcdef",Word.toInt i,1))
	  fun printhex i = (hexdig(i >> 0w4); hexdig(Word.andb(i, 0w15)))
       in app (printhex o Word.fromInt o Char.ord) 
	          (explode (PersStamps.stampToString pid));
	  fsay "\n"
      end 

  fun elaborate{errors as {error,errorMatch,anyErrors}, corenv,compenv, ast,
		transform} =
     let val freeStampScope = Stamps.freeScope
	 val (absyn,newenv) =
	   ElabStr.elaborateTop(ast,compenv,corenv,error,errorMatch,transform)
         val (absyn,newenv) = 
           if !anyErrors then (Absyn.SEQdec nil, StaticEnv.empty)
	   else (absyn, newenv)
         val foo = (!Control.CG.mtderiv) andalso true (* (!Control.CG.representations) *)
         val absyn = if foo then (MTDeriv.mtderivDec(newenv,absyn))
                     else absyn
         val basePid = PersStamps.stringToStamp (HashEnv.hashEnv newenv)
	 val (newenv', exportLexp, exportPid) =
	     Environment.exports (newenv, basePid)
     in(* showPid basePid;*)
        { absyn=absyn,newenv=newenv', exportPid = exportPid,
	  exportLexp = exportLexp, staticPid = basePid }
    end

  val elaborate = Stats.doPhase(Stats.makePhase "Compiler 030 Elaborate")
                      elaborate


  fun makePid se = PersStamps.stringToStamp (HashEnv.hashEnv se)

  fun instrument{source,compenv,corenv} = 
      SProf.instrumDec corenv source 
      o Prof.instrumDec corenv

  fun translate args = let
      val { errors = { error, errorMatch, anyErrors },
	    absyn, exportLexp, corenv, exportPid,
	    statenv (* for printing Ast in messages *)} = args
      val { genLambda, importPids } =
	  Translate.transDec corenv statenv error errorMatch absyn exportLexp
  in
      { genLambda = genLambda, imports = importPids }
  end

  fun symDelta (NONE, _) = SymbolicEnv.empty
    | symDelta (_, NONE) = SymbolicEnv.empty
    | symDelta (SOME pid, SOME l) = SymbolicEnv.singleton (pid, l)
 
  val translate = Stats.doPhase (Stats.makePhase "Compiler 040 Translate")
			        translate

  fun codeopt lambda = let
	fun prLexp (s,le) = let
	      val outS = IO.open_append ((!CheckLty.fname_ref)^s);
	      val saveOut = !Control.Print.out
	      in
		Control.Print.out := {
		    say = IO.outputc outS,
		    flush = fn () => IO.flush_out outS
		  };
		MCprint.printLexp (le);
		close_out outS;
		Control.Print.out := saveOut
	      end

        val _ = if !Control.CG.printLambda then MCprint.printLexp lambda
                else ()
        val _ = if !Control.CG.checklty1 then
                 (if CheckLty.check(lambda) then prLexp(".log1",lambda)
                  else ())
                else ()

        val lambda = LambdaOpt.lambdaopt lambda
        val _ = if !Control.CG.checklty2 then
                 (if CheckLty.check(lambda) then prLexp(".log2",lambda)
                  else ())
                else ()

        val lambda = Reorder.reorder lambda
        val _ = if !Control.CG.checklty3 then
                 (if CheckLty.check(lambda) then prLexp(".log3",lambda)
                     else ())
                else ()

     in lambda
    end

  val codeopt = Stats.doPhase (Stats.makePhase "Compiler 050 CodeOpt") codeopt

  fun inline { genLambda, imports, symenv } =
      genLambda (map (SymbolicEnv.look symenv) imports)

  (* `conservative' splitting (i.e., none) *)
  fun split { lambda, enable } = let
      val (lambda_e, lambda_i) =
	  (* act as if it were always disabled *)
	  ({ l0 = lambda, ln = [] }, NONE)
      (* canonicalize lambda_i, so it can be hash'd *)
      val lambda_i = case lambda_i of
	  NONE => NONE
	| SOME l => let
	      val genlvar = Access.newCanonicalLvars ()
	      val canonicalize = Lambda.copy genlvar
	  in
	      SOME (canonicalize l)
	  end
      (* calculate lambda_i's pid *)
      val pid = PersStamps.stringToStamp (Lambda.hashlambda lambda_i)
  in
      { lambda_e = lambda_e, lambda_i = lambda_i, pid = pid }
  end

  fun codegen { errors, lambda = { l0, ln } } = let
      fun cg l = (Machm.codegen (errors, codeopt l); Machm.collect ())
  in
      { c0 = cg l0, cn = map cg ln }
  end

  val codegen = Stats.doPhase (Stats.makePhase "Compiler 140 CodeGen") codegen

  fun csegsize { c0, cn } =
      foldl (fn (x: string, y) => (size x) + y) (size c0) cn

  val addCode = Stats.addStat (Stats.makeStat "Code Size")

  val codegen = fn x => let val c = codegen x
                         in addCode(csegsize c); c
                        end

  type ovec = obj vector
  val mkCode0 = System.Unsafe.CInterface.c_function "SMLNJ-RunT" "mkCode"
  val mkCodeV : string -> (string * (ovec -> obj)) = mkCode0
  val mkCodeO : string -> (string * (obj -> obj)) = mkCode0

  fun applyCode { c0, cn } =
      foldl (fn (c, r) => (#2 (mkCodeO c)) o r) (#2 (mkCodeV c0)) cn

  fun execute { executable, imports, exportPid, dynenv } = let
      val result =
	  executable (Vector.fromList (map (DynamicEnv.look dynenv) imports))
	  handle DynamicEnv.Unbound =>
	      (app (fn p => (print "lookup ";
			     print(PersStamps.stampToString p);
			     print "\n")) imports;
	       fail "imported objects not found or inconsistent")
  in
      case exportPid of
	  NONE => DynamicEnv.empty
	| SOME p => DynamicEnv.singleton (p, result)
  end

  val execute = Stats.doPhase (Stats.makePhase "Execute") execute

end (* functor CompileF *)

end (* local *)
