(* word31.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *)

structure Word31 : WORD =
  struct
    infix 7 * div mod
    infix 6 + -
    infix 4 > < >= <=

    structure W31 = InlineT.Word31

    type word = word

    val wordSize = 31

    val toLargeWord   : word -> LargestWord.word = W31.toword32
(** NOTE: the following should be a primop **)
    fun toLargeWordX (w : word) =
	  InlineT.Word32.rshift(InlineT.Word32.lshift(W31.toword32 w, 0w1), 0w1)
    val fromLargeWord : LargestWord.word -> word = W31.fromword32

    val toLargeInt   : word -> LargestInt.int = W31.toint
    val toLargeIntX  : word -> LargestInt.int = W31.toint
    val fromLargeInt : LargestInt.int -> word = W31.fromint

    val toInt   : word -> int = W31.toint
    val toIntX  : word -> int = W31.toint
    val fromInt : int -> word = W31.fromint

    val orb  : word * word -> word = W31.orb
    val xorb : word * word -> word = W31.xorb
    val andb : word * word -> word = W31.andb
    val notb : word -> word = W31.notb

  (** These should be inline functions **)
    fun << (w : word, k : word) = if (W31.<=(0w31, k))
	  then 0w0
	  else W31.lshift(w, k)
    fun >> (w : word, k : word) = if (W31.<=(0w31, k))
	  then 0w0
	  else W31.rshiftl(w, k)
    fun ~>> (w : word, k : word) = if (W31.<=(0w31, k))
	  then W31.rshift(w, 0w31)
	  else W31.rshift(w, k)

    val op * : word * word -> word = W31.*
    val op + : word * word -> word = W31.+
    val op - : word * word -> word = W31.-
    val op div : word * word -> word = W31.div
    fun op mod(a:word,b:word):word = a-(a div b)*b

    val op > : word * word -> bool = W31.>
    val op >= : word * word -> bool = W31.>=
    val op < : word * word -> bool = W31.<
    val op <= : word * word -> bool = W31.<=

    val w32_to_w31 : word32 -> word = W31.fromword32

    fun fmt radix = (NumFormat.fmtWord radix) o  W31.toword32
    val toString = fmt StringCvt.HEX

    fun scan radix = let
	  val scanLargest = NumScan.scanWord radix
	  fun scan getc cs = (case (scanLargest getc cs)
		 of NONE => NONE
		  | (SOME(w, cs')) => if InlineT.Word32.>(w, 0wx7FFFFFFF)
		      then raise Overflow
		      else SOME(w32_to_w31 w, cs')
		(* end case *))
	  in
	    scan
	  end
    val fromString = PreBasis.scanString (scan StringCvt.HEX)

  end  (* structure Word31 *)

(* to avoid losing inlining *)
structure Word31 =
  struct
    open Word31
    val toLargeWord   : word -> LargestWord.word = InlineT.Word31.toword32
(** NOTE: the following should be a primop **
    fun toLargeWordX (w : word) = InlineT.Word32.rshift(InlineT.cast w, 0w1)
**)
    val fromLargeWord : LargestWord.word -> word = InlineT.Word31.fromword32
    val toLargeInt   : word -> LargestInt.int = InlineT.Word31.toint
    val toLargeIntX  : word -> LargestInt.int = InlineT.Word31.toint
    val fromLargeInt : LargestInt.int -> word = InlineT.Word31.fromint
    val toInt   : word -> int = InlineT.Word31.toint
    val toIntX  : word -> int = InlineT.Word31.toint
    val fromInt : int -> word = InlineT.Word31.fromint
    val notb : word -> word = InlineT.Word31.notb
    val orb  : word * word -> word = InlineT.Word31.orb
    val xorb : word * word -> word = InlineT.Word31.xorb
    val andb : word * word -> word = InlineT.Word31.andb
    val op * : word * word -> word = InlineT.Word31.*
    val op + : word * word -> word = InlineT.Word31.+
    val op - : word * word -> word = InlineT.Word31.-
    val op div : word * word -> word = InlineT.Word31.div
    val op > : word * word -> bool = InlineT.Word31.>
    val op >= : word * word -> bool = InlineT.Word31.>=
    val op < : word * word -> bool = InlineT.Word31.<
    val op <= : word * word -> bool = InlineT.Word31.<=
  end;

