(* Copyright 1989 by AT&T Bell Laboratories *)
(* boot/system.sig *)

signature RUNTIMECONTROL =
  sig
    val gc          : int -> unit
  end

signature POLL = 
  sig
    exception BadPollFreq
    val pollEvent : bool ref
    val setHandler : (unit cont -> unit cont) option -> unit
    val inqHandler : unit -> ((unit cont -> unit cont) option)
    val setFreq : int option -> unit
    val inqFreq : unit -> (int option)
  end

signature TAGS =
  sig
    type tag
    val width_tags : int (* number of bits to hold a tag *)
    val power_tags : int (* 2 ^ width_tags *)
    val max_length : int (* one greater than max length value *)
  (* tag values *)
    val tag_record : tag
    val tag_pair : tag
    val tag_array : tag
    val tag_string : tag
    val tag_bytearray : tag
    val tag_reald : tag
    val tag_realdarray : tag
    val tag_cont : tag
    val tag_block : tag
    val tag_variant : tag (* currently not used *)
    val tag_special : tag
    val tag_backptr : tag
  (* build a descriptor from a tag and length *)
    val make_desc : (int * tag) -> int
  (* fixed descriptors *)
    val desc_pair : int
    val desc_reald : int
  (* special descriptors *)
    val desc_special : int
    val special_evaled_susp : int
    val special_unevaled_susp : int
    val special_weak : int
    val special_nulled_weak : int
  end


signature WEAK = 
sig
  type 'a weak
  val weak : 'a -> 'a weak
  val strong : 'a weak -> 'a option
  type weak'
  val weak' : 'a -> weak'
  val strong' : weak' -> bool
end (* WEAK *)

signature SUSP =
  sig
    type 'a susp
    val delay : (unit -> 'a) -> 'a susp
    val force : 'a susp -> 'a
  end (* SUSP *)

(* fully polymorphic (and thus not typesafe) versions of the continuation
 * operations
 *)
signature POLY_CONT =
sig
  type 'a cont
  val callcc : ('a cont -> 'a) -> 'a
  val throw : 'a cont -> 'a -> 'b
  type 'a control_cont
  val capture : ('a control_cont -> 'a) -> 'a
  val escape : 'a control_cont -> 'a -> 'b
end

signature UNSAFE =
sig
  structure Assembly : ASSEMBLY
  structure CInterface : CINTERFACE
  structure Susp : SUSP
  structure PolyCont : POLY_CONT
  type object
  type instream and outstream
  val boxed : 'a -> bool
  val ordof : 'a * int -> char
  val slength : 'a -> int		(* string length *)
  val objLength : 'a -> int		(* object length *)
  val getObjTag : 'a -> int		(* get width_tag bits of object descriptor *)
  val special : (int * 'a) -> 'b
  val setSpecial : ('a * int) -> unit
  val getSpecial : 'a -> int
  val store : string * int * char -> unit
  val subscript : 'a array * int -> 'a
  val update : 'a array * int * 'a -> unit
  val subscriptv : 'a vector * int -> 'a
  val subscriptf : Assembly.A.realarray * int -> real
  val updatef : Assembly.A.realarray * int * real -> unit
  val getvar : unit -> 'a
  val setvar : 'a -> unit
  val getpseudo : int -> 'a
  val setpseudo : 'a * int -> unit
  val setmark : 'a -> unit
  val dispose : 'a -> unit
  val gethdlr : unit -> 'a cont
  val sethdlr : 'a cont -> unit
  val cast : 'a -> 'b
  val blastWrite : 'a -> Word8Vector.vector
  val blastRead  : Word8Vector.vector -> 'a
  val create_s : int -> string
  val create_b : int -> Assembly.A.bytearray
  val store_s : string * int * char -> unit
  val topLevelCont : unit cont ref
  exception Boxity
  val tuple : object -> object vector
  val string : object -> string
  val real : object -> real
  val int : object -> int
  val word : object -> Word.word
  val word8 : object -> Word8.word
  val word32 : object -> Word32.word

  val profiling : bool ref
  val sprofiling : bool ref
  val profile_register : (string -> int * int array * int ref) ref
  val profile_sregister : (Assembly.object * string -> Assembly.object) ref
end

signature SYSTEM =
sig
  structure Runtime : RUNTIMECONTROL
  structure Tags : TAGS
  structure Unsafe : UNSAFE
  structure Weak : WEAK
  structure CleanUp : CLEAN_UP
  val runtimeStamp : string ref
  val errorMatch : string ref
  val interactive : bool ref
end

