(* num-scan.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *
 * The string conversion for the largest int and word types.
 * All of the other scan functions can be implemented in terms of them.
 *)

structure NumScan : sig

    val scanWord : StringCvt.radix
	  -> (char, 'a) StringCvt.reader -> (word32, 'a) StringCvt.reader
    val scanInt : StringCvt.radix
	  -> (char, 'a) StringCvt.reader -> (int, 'a) StringCvt.reader
	(** should be to int32 **)
    val scanFloat : (char, 'a) StringCvt.reader -> (real, 'a) StringCvt.reader
	(** should be to LargestFloat.real **)

  end = struct

    structure W = InlineT.Word32
    structure I = InlineT.Int31
    structure R = InlineT.Float64
    type word = word32

    val op <  = W.<
    val op >= = W.>=
    val op +  = W.+
    val op -  = W.-
    val op *  = W.*

    val largestWordDiv10 : word = 0w429496729	(* 2^32-1 divided by 10 *)
    val largestWordMod10 : word = 0w5		(* remainder *)
    val largestNegInt : word = 0w1073741824	(* absolute value of ~2^30 *)
    val largestPosInt : word = 0w1073741823	(* 2^30-1 *)
    val minInt = ~1073741824

  (* A table for mapping digits to values.  Whitespace characters map to
   * 128, "+" maps to 129, "-","~" map to 130, "." maps to 131, and the
   * characters 0-9,A-Z,a-z map to their * base-36 value.  All other
   * characters map to 255.
   *)
    local
      val cvtTable = "\
	    \\255\255\255\255\255\255\255\255\255\128\128\255\255\255\255\255\
	    \\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
	    \\128\255\255\255\255\255\255\255\255\255\255\129\255\130\131\255\
	    \\000\001\002\003\004\005\006\007\008\009\255\255\255\255\255\255\
	    \\255\010\011\012\013\014\015\016\017\018\019\020\021\022\023\024\
	    \\025\026\027\028\029\030\031\032\033\034\035\255\255\255\255\255\
	    \\255\010\011\012\013\014\015\016\017\018\019\020\021\022\023\024\
	    \\025\026\027\028\029\030\031\032\033\034\035\255\255\255\130\255\
	    \\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
	    \\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
	    \\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
	    \\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
	    \\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
	    \\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
	    \\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
	    \\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
	  \"
    val ord = InlineT.Char.ord
    in
    fun code (c : char) =
	  W.fromint(ord(InlineT.CharVector.sub(cvtTable, ord c)))
    val wsCode : word = 0w128
    val plusCode : word = 0w129
    val minusCode : word = 0w130
    val ptCode : word = 0w131
    val eCode : word = 0w14	(* code for #"e" and #"E" *)
    end (* local *)

  (* skip leading whitespace and any sign (+, -, or ~) *)
    fun scanPrefix getc cs = let
	  fun skipWS cs = (case (getc cs)
		 of NONE => NONE
		  | (SOME(c, cs')) => let val c' = code c
		      in
			if (c' = wsCode) then skipWS cs' else SOME(c', cs')
		      end
		(* end case *))
	  fun getNext (neg, cs) = (case (getc cs)
		 of NONE => NONE
		  | (SOME(c, cs)) => SOME{neg=neg, next=code c, rest=cs}
		(* end case *))
	  in
	    case (skipWS cs)
	     of NONE => NONE
	      | (SOME(c, cs')) =>
		  if (c = plusCode) then getNext(false, cs')
		  else if (c = minusCode) then getNext(true, cs')
		  else SOME{neg=false, next=c, rest=cs'}
	    (* end case *)
	  end

  (* for power of 2 bases (2, 8 & 16), we can check for overflow by looking
   * at the hi (1, 3 or 4) bits.
   *)
    fun chkOverflow mask w =
	  if (W.andb(mask, w) = 0w0) then () else raise Overflow

    fun scanBin getc cs = (case (scanPrefix getc cs)
	   of NONE => NONE
	    | (SOME{neg, next, rest}) => let
		fun isDigit (d : word) = (d < 0w2)
		val chkOverflow = chkOverflow 0wx80000000
		fun cvt (w, rest) = (case (getc rest)
		       of NONE => SOME{neg=neg, word=w, rest=rest}
			| SOME(c, rest') => let val d = code c
			    in
			      if (isDigit d)
				then (
				  chkOverflow w;
				  cvt(W.+(W.lshift(w, 0w1), d), rest'))
				else SOME{neg=neg, word=w, rest=rest}
			    end
		      (* end case *))
		in
		  if (isDigit next)
		    then cvt(next, rest)
		    else NONE
		end
	  (* end case *))

    fun scanOct getc cs = (case (scanPrefix getc cs)
	   of NONE => NONE
	    | (SOME{neg, next, rest}) => let
		fun isDigit (d : word) = (d < 0w8)
		val chkOverflow = chkOverflow 0wxE0000000
		fun cvt (w, rest) = (case (getc rest)
		       of NONE => SOME{neg=neg, word=w, rest=rest}
			| SOME(c, rest') => let val d = code c
			    in
			      if (isDigit d)
				then (
				  chkOverflow w;
				  cvt(W.+(W.lshift(w, 0w3), d), rest'))
				else SOME{neg=neg, word=w, rest=rest}
			    end
		      (* end case *))
		in
		  if (isDigit next)
		    then cvt(next, rest)
		    else NONE
		end
	  (* end case *))

    fun isDigit10 (d : word) = (d < 0w10)

    fun scanDec getc cs = (case (scanPrefix getc cs)
	   of NONE => NONE
	    | (SOME{neg, next, rest}) => let
		fun cvt (w, rest) = (case (getc rest)
		       of NONE => SOME{neg=neg, word=w, rest=rest}
			| SOME(c, rest') => let val d = code c
			    in
			      if (isDigit10 d)
				then (
				  if ((w >= largestWordDiv10)
				  andalso ((largestWordDiv10 < w)
				    orelse (largestWordMod10 < d)))
				    then raise Overflow
				    else ();
				  cvt (0w10*w+d, rest'))
				else SOME{neg=neg, word=w, rest=rest}
			    end
		      (* end case *))
		in
		  if (isDigit10 next)
		    then cvt(next, rest)
		    else NONE
		end
	  (* end case *))

    fun scanHex getc cs = (case (scanPrefix getc cs)
	   of NONE => NONE
	    | (SOME{neg, next, rest}) => let
		fun isDigit (d : word) = (d < 0w16)
		val chkOverflow = chkOverflow 0wxF0000000
		fun cvt (w, rest) = (case (getc rest)
		       of NONE => SOME{neg=neg, word=w, rest=rest}
			| SOME(c, rest') => let val d = code c
			    in
			      if (isDigit d)
				then (
				  chkOverflow w;
				  cvt(W.+(W.lshift(w, 0w4), d), rest'))
				else SOME{neg=neg, word=w, rest=rest}
			    end
		      (* end case *))
		in
		  if (isDigit next)
		    then cvt(next, rest)
		    else NONE
		end
	  (* end case *))

    fun finalWord scanFn getc cs = (case (scanFn getc cs)
	   of NONE => NONE
	    | (SOME{neg=true, ...}) => NONE
	    | (SOME{neg=false, word, rest}) => SOME(word, rest)
	  (* end case *))

    fun scanWord StringCvt.BIN = finalWord scanBin
      | scanWord StringCvt.OCT = finalWord scanOct
      | scanWord StringCvt.DEC = finalWord scanDec
      | scanWord StringCvt.HEX = finalWord scanHex

    fun finalInt scanFn getc cs = (case (scanFn getc cs)
	   of NONE => NONE
	    | (SOME{neg=true, word, rest}) =>
		if (word < largestNegInt)
		  then SOME(I.~(W.toint word), rest)
		else if (largestNegInt < word)
		  then raise Overflow
		  else SOME(minInt, rest)
	    | (SOME{word, rest, ...}) =>
		if (largestPosInt < word)
		  then raise Overflow
		  else SOME(W.toint word, rest)
	  (* end case *))

    fun scanInt StringCvt.BIN = finalInt scanBin
      | scanInt StringCvt.OCT = finalInt scanOct
      | scanInt StringCvt.DEC = finalInt scanDec
      | scanInt StringCvt.HEX = finalInt scanHex

  (* scan a string of decimal digits (starting with d), and return their
   * value as a real number.  Also return the number of digits, and the
   * rest of the stream.
   *)
    fun fscan10 getc (d, cs) = let
	  fun wordToReal w = InlineT.real(W.toint w)
	  fun scan (accum, n, cs) = (case (getc cs)
		 of (SOME(c, cs')) => let val d = code c
		      in
			if (isDigit10 d)
			  then scan(R.+(R.*(10.0, accum), wordToReal d), I.+(n, 1), cs')
			  else SOME(accum, n, cs)
		      end
		  | NONE => SOME(accum, n, cs)
		(* end case *))
	  in
	    if (isDigit10 d) then scan(wordToReal d, 1, cs) else NONE
	  end

    local
      val negTbl = #[
	      1.0E~0, 1.0E~1, 1.0E~2, 1.0E~3, 1.0E~4,
	      1.0E~5, 1.0E~6, 1.0E~7, 1.0E~8, 1.0E~9
	    ]
      val posTbl = #[
	      1.0E0, 1.0E1, 1.0E2, 1.0E3, 1.0E4,
	      1.0E5, 1.0E6, 1.0E7, 1.0E8, 1.0E9
	    ]
      fun scale (tbl, step10 : real) = let
	    fun f (r, 0) = r
	      | f (r, exp) = if (I.<(exp, 10))
		  then (R.*(r, InlineT.PolyVector.sub(tbl, exp)))
		  else f (R.*(step10, r), I.-(exp, 10))
	    in
	      f
	    end
    in
    val scaleUp = scale (posTbl, 1.0E10)
    val scaleDown = scale (negTbl, 1.0E~10)
    end

    fun scanFloat getc cs = let
	  fun scan10 cs = (case (getc cs)
		 of (SOME(c, cs)) => fscan10 getc (code c, cs)
		  | NONE => NONE
		(* end case *))
	  fun getFrac rest = (case (scan10 rest)
		 of SOME(frac, n, rest) => (SOME(scaleDown(frac, n)), rest)
		  | NONE => (NONE, rest)
		(* end case *))
	  fun combine (SOME whole, SOME frac) = R.+(whole, frac)
	    | combine (SOME whole, NONE) = whole
	    | combine (NONE, SOME frac) = frac
	    | combine _ = raise Option
	  fun negate (true, num) = R.~ num
	    | negate (false, num) = num
	  fun scanExp cs = (case (getc cs)
		 of SOME(c, cs) => let
		      val d = code c
		      fun scan (accum, cs) = (case (getc cs)
			     of SOME(c, cs') => let val d = code c
				  in
				    if (isDigit10 d)
				      then scan (I.+(I.*(accum, 10), W.toint d), cs')
				      else (accum, cs)
				  end
			      | NONE => (accum, cs)
			    (* end case *))
		      in
			if (isDigit10 d)
			  then SOME (scan (W.toint d, cs))
			  else NONE
		      end
		  | NONE => NONE
		(* end case *))
	  fun getExp cs = (case (getc cs)
		 of (SOME(c, cs)) => if (code c = eCode)
		      then (case (getc cs)
			 of SOME(c, cs') => let
			      val (isNeg, cs) = if (code c = minusCode)
				    then (true, cs')
				    else (false, cs)
			      in
			        case scanExp cs
				 of SOME(exp, cs) => SOME(isNeg, exp, cs)
				  | NONE => NONE
				(* end case *)
			      end
			  | NONE => NONE
			(* end case *))
		      else NONE
		  | NONE => NONE
		(* end case *))
	  in
	    case (scanPrefix getc cs)
	     of NONE => NONE
	      | (SOME{neg, next, rest}) => let
		  val (whole, hasPt, rest) = if (next = ptCode)
			then (NONE, true, rest)
			else let
			  val (whole, rest) = (case fscan10 getc (next, rest)
				 of SOME(whole, _, rest) => (SOME whole, rest)
				  | NONE => (NONE, rest)
				(* end case *))
			  in
			    case (getc rest)
			     of SOME(#".", rest) => (whole, true, rest)
			      | _ => (whole, false, rest)
			    (* end case *)
			  end
		  val (frac, rest) = if hasPt then getFrac rest else (NONE, rest)
		  val num = negate (neg, combine (whole, frac))
		  in
		    case (getExp rest)
		     of (SOME(isNeg, exp, rest)) =>
			  if isNeg
			    then SOME(scaleDown(num, exp), rest)
			    else SOME(scaleUp(num, exp), rest)
		      | NONE => SOME(num, rest)
		    (* end case *)
		  end
	    (* end case *)
	  end
	    handle Option => NONE

  end;

