(* int31.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *
 * The following structures must be without signatures so that inlining 
 * can take place: Bits, Vector, Array, RealArray, Int, Real
 *)

structure Int31 : INTEGER =
  struct
    structure I31 = InlineT.Int31

    exception Div = Assembly.Div
    exception Overflow = Assembly.Overflow

    type int = int

    val precision = SOME 31
    val minIntVal = ~1073741824
    val minInt = SOME minIntVal
    val maxInt = SOME 1073741823

    val toLarge = InlineT.identity
    val fromLarge = InlineT.identity
    val toInt = InlineT.identity
    val fromInt = InlineT.identity

    val ~ 	: int -> int = I31.~
    val op * 	: int * int -> int  = I31.*
    val op + 	: int * int -> int  = I31.+
    val op - 	: int * int -> int  = I31.-
    val op div 	: int * int -> int  = I31.div
    val op mod 	: int * int -> int  = I31.mod
    val op quot : int * int -> int  = I31.quot
    val op rem 	: int * int -> int  = I31.rem
    val min 	: int * int -> int  = I31.min
    val max 	: int * int -> int  = I31.max
    val abs 	: int -> int = I31.abs

    fun sign 0 = 0
      | sign i = if I31.<(i, 0) then ~1 else 1
    fun sameSign (i, j) = (I31.andb(I31.xorb(i, j), minIntVal) = 0)

    fun compare (i, j) =
	  if (I31.<(i, j)) then General.LESS
	  else if (I31.>(i, j)) then General.GREATER
	  else General.EQUAL
    val op > 	: int * int -> bool = I31.>
    val op >= 	: int * int -> bool = I31.>=
    val op < 	: int * int -> bool = I31.<
    val op <= 	: int * int -> bool = I31.<=

    val scan = NumScan.scanInt
    val fmt = NumFormat.fmtInt

    val toString = fmt StringCvt.DEC
    val fromString = PreBasis.scanString (scan StringCvt.DEC)

  end  (* structure Int31 *)

(* rebind the structure to avoid loss of inlining *)
structure Int31 =
  struct
    open Int31
    val ~ 	: int -> int = InlineT.Int31.~
    val op * 	: int * int -> int  = InlineT.Int31.*
    val op + 	: int * int -> int  = InlineT.Int31.+
    val op - 	: int * int -> int  = InlineT.Int31.-
    val op > 	: int * int -> bool = InlineT.Int31.>
    val op >= 	: int * int -> bool = InlineT.Int31.>=
    val op < 	: int * int -> bool = InlineT.Int31.<
    val op <= 	: int * int -> bool = InlineT.Int31.<=
    val op div 	: int * int -> int  = InlineT.Int31.div
    val op mod 	: int * int -> int  = InlineT.Int31.mod
    val op quot : int * int -> int  = InlineT.Int31.quot
    val op rem 	: int * int -> int  = InlineT.Int31.rem
    val min 	: int * int -> int  = InlineT.Int31.min
    val max 	: int * int -> int  = InlineT.Int31.max
    val abs 	: int -> int = InlineT.Int31.abs
    val toLarge : int -> LargestInt.int = InlineT.identity
    val fromLarge : LargestInt.int -> LargestInt.int = InlineT.identity
    val toInt : int -> Int.int = InlineT.identity
    val fromInt : int -> Int.int = InlineT.identity
  end;

