(* float64.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *)

structure Float64 : FLOAT = 
  struct
    structure I = InlineT.DfltInt

    structure Math = Math64

    type real = real
    val ~ = InlineT.Float64.~
    val op +  = InlineT.Float64.+
    val op -  = InlineT.Float64.-
    val op *  = InlineT.Float64.*
    val op /  = InlineT.Float64./
    val op >  = InlineT.Float64.>
    val op <  = InlineT.Float64.<
    val op >= = InlineT.Float64.>=
    val op <= = InlineT.Float64.<=
    val floor = Assembly.A.floor
    fun trunc n = if n < 0.0 then I.~(floor(~n)) else floor n
    fun ceil n = I.~(floor(~n))
    fun round x = floor(x+0.5)
    val abs : real -> real = InlineT.Float64.abs
    val real : int -> real = InlineT.real

    fun sign x = if (x < 0.0) then ~1 else if (x > 0.0) then 1 else 0
    fun sameSign (x, y) = (case (x < 0.0, y < 0.0)
	   of (true, true) => true
	    | (false, false) => true
	    | _ => false
	  (* end case *))

    val toDefault   : real -> Real.real = InlineT.identity
    val fromDefault : Real.real -> real = InlineT.identity

    val radix = 2
    val precision = 52

    val two_to_the_54 = 18014398509481984.0

(** NOTE logb and scalb are also defined in math64.sml; do we need both??? **)
    fun logb x = (case Assembly.A.logb x
	   of ~1023 => (* denormalized number *)
		I.-(Assembly.A.logb(x * two_to_the_54), 54)
	    | i => i
	  (* end case *))

    fun scalb (x, k) = if I.ltu(I.+(k,1022),2046)
	  then Assembly.A.scalb(x,k)
          else let val k1 = I.div(k, 2)
	    in
	      scalb(scalb(x, k1), I.-(k, k1))
	    end
  
    fun nextAfter _ = raise Fail "Float.nextAfter unimplemented"

  (* The next three values are computed laboriously, partly to
   * avoid problems with inaccurate string->float conversions
   * in the compiler itself.
   *)

  (* This will infinite-loop on the SPARC and RS/6000 until the logb
   * bug is fixed.
   *)
    val maxFinite = let
	  fun f(x,i) = if i=1023 then x else f(x*2.0, I.+(i, 1))
	  val y = f(1.0,0)
	  fun g(z, y, 0) = z
	    | g(z, y, i) = g(z+y, y*0.5, I.-(i, 1))
	  in
	    g(0.0,y,53)
	  end

    val minNormalPos = let
	  fun f(x) = let
		val y = x * 0.5
		in
		  if Assembly.A.logb y = ~1023 then x else f y
		end
	  in
	    f 1.0
	  end

    val minPos = let
	  fun f(x) = let
		val y = x * 0.5
		in
		  if y=0.0 then x else f y
                end
	  in
	    f minNormalPos
	  end

    val fmt = RealFormat.fmtReal
    val toString = fmt (StringCvt.GEN NONE)
    val scan = NumScan.scanFloat
    val fromString = StringCvt.scanString scan

  end (* Float64 *)

structure Float64 =
  struct
    open Float64
    val ~ = InlineT.Float64.~
    val op +  = InlineT.Float64.+
    val op -  = InlineT.Float64.-
    val op *  = InlineT.Float64.*
    val op /  = InlineT.Float64./
    val op >  = InlineT.Float64.>
    val op <  = InlineT.Float64.<
    val op >= = InlineT.Float64.>=
    val op <= = InlineT.Float64.<=
  end;

structure Float = Float64

