(* char.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *)

structure Char : CHAR =
  struct

    structure C = InlineT.Char

    val op + = InlineT.DfltInt.+
    val op - = InlineT.DfltInt.-
    val op * = InlineT.DfltInt.*

    val itoc : int -> char = InlineT.cast
    val ctoi : char -> int = InlineT.cast

    type char = PrimTypes.char

    exception Chr = C.Chr

    val minChar : char	= C.chr 0
    val maxChar	: char	= C.chr C.maxOrd
    val maxOrd		= C.maxOrd

    fun pred (c : char) : char = let
	  val c' = (ctoi c - 1)
	  in
	    if InlineT.DfltInt.< (c', 0) then raise Chr else (itoc c')
	  end
    fun succ (c : char) : char = let
	  val c' = (ctoi c + 1)
	  in
	    if InlineT.DfltInt.< (maxOrd, c') then raise Chr else (itoc c')
	  end

    val chr = C.chr
    val ord = C.ord

    val (op <)  = C.<
    val (op <=) = C.<=
    val (op >)  = C.>
    val (op >=) = C.>=

    fun compare (c1 : char, c2 : char) =
	  if (c1 = c2) then EQUAL
	  else if (c1 < c2) then LESS
	  else GREATER

  (* testing character membership *)
    local
      fun mkArray (s, sLen) = let
	    val cv = Assembly.A.create_s(maxOrd+1)
	    fun init i = if InlineT.DfltInt.<= (i, maxOrd)
		  then (InlineT.CharVector.update(cv, i, #"\000"); init(i+1))
		  else ()
	    fun ins i = if InlineT.DfltInt.< (i, sLen)
		  then (
		    InlineT.CharVector.update (
		      cv, ord(InlineT.CharVector.sub(s, i)), #"\001");
		    ins(i+1))
		  else ()
	    in
	      init 0; ins 0; cv
	    end
    in
    fun contains "" = (fn c => false)
      | contains s = let val sLen = InlineT.CharVector.length s
	  in
	    if (sLen = 1)
	      then let val c' = InlineT.CharVector.sub(s, 0)
		in fn c => (c = c') end
	      else let val cv = mkArray (s, sLen)
		in fn c => (InlineT.CharVector.sub(cv, ord c) <> #"\000") end
	  end
    fun notContains "" = (fn c => true)
      | notContains s = let val sLen = InlineT.CharVector.length s
	  in
	    if (sLen = 1)
	      then let val c' = InlineT.CharVector.sub(s, 0)
		in fn c => (c <> c') end
	      else let val cv = mkArray (s, sLen)
		in fn c => (InlineT.CharVector.sub(cv, ord c) = #"\000") end
	  end
    end (* local *)

  (* For each character code we have an 8-bit vector, which is interpreted
   * as follows:
   *   0x01  ==  set for upper-case letters
   *   0x02  ==  set for lower-case letters
   *   0x04  ==  set for digits
   *   0x08  ==  set for white space characters
   *   0x10  ==  set for punctuation characters
   *   0x20  ==  set for control characters
   *   0x40  ==  set for hexadecimal characters
   *   0x80  ==  set for SPACE
   *)
    val ctypeTbl = "\
	    \\032\032\032\032\032\032\032\032\032\040\040\040\040\040\032\032\
	    \\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\032\
	    \\136\016\016\016\016\016\016\016\016\016\016\016\016\016\016\016\
	    \\068\068\068\068\068\068\068\068\068\068\016\016\016\016\016\016\
	    \\016\065\065\065\065\065\065\001\001\001\001\001\001\001\001\001\
	    \\001\001\001\001\001\001\001\001\001\001\001\016\016\016\016\016\
	    \\016\066\066\066\066\066\066\002\002\002\002\002\002\002\002\002\
	    \\002\002\002\002\002\002\002\002\002\002\002\016\016\016\016\032\
	    \\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
	    \\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
	    \\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
	    \\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
	    \\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
	    \\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
	    \\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
	    \\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
	  \"
    fun inSet (c, s) = let
	  val m = ord(InlineT.CharVector.sub(ctypeTbl, ord c))
	  in
	    (InlineT.DfltInt.andb(m, s) <> 0)
	  end

  (* predicates on integer coding of Ascii values *)
    fun isAlpha c	= inSet(c, 0x03)
    fun isUpper c	= inSet(c, 0x01)
    fun isLower c	= inSet(c, 0x02)
    fun isDigit c	= inSet(c, 0x04)
    fun isHexDigit c	= inSet(c, 0x40)
    fun isAlphaNum c	= inSet(c, 0x07)
    fun isSpace c	= inSet(c, 0x08)
    fun isPunct c	= inSet(c, 0x10)
    fun isGraph c	= inSet(c, 0x17)
    fun isPrint c	= inSet(c, 0x97)
    fun isCntrl c	= inSet(c, 0x20)
    fun isAscii c    	= InlineT.DfltInt.< (ord c, 128)

    val offset = ctoi #"a" - ctoi #"A"
    fun toUpper c = if (isLower c) then itoc(ctoi c - offset) else c
    fun toLower c = if (isUpper c) then itoc(ctoi c + offset) else c

  (* conversions between characters and printable representations *)
    fun scan getc rep = let
	  fun get2 rep = (case (getc rep)
		 of (SOME(c1, rep')) => (case (getc rep')
		       of (SOME(c2, rep'')) => SOME(c1, c2, rep'')
			| _ => NONE
		      (* end case *))
		  | _ => NONE
		(* end case *))
	  in
	    case (getc rep)
	     of NONE => NONE
	      | (SOME(#"\\", rep')) => (case (getc rep')
		   of NONE => NONE
		    | (SOME(#"\\", rep'')) => (SOME(#"\\", rep''))
		    | (SOME(#"\"", rep'')) => (SOME(#"\"", rep''))
		    | (SOME(#"n", rep'')) => (SOME(#"\n", rep''))
		    | (SOME(#"t", rep'')) => (SOME(#"\t", rep''))
		    | (SOME(#"^", rep'')) => (case (getc rep'')
			 of NONE => NONE
	    		  | (SOME(c, rep''')) =>
			      if ((#"@" <= c) andalso (c <= #"_"))
			        then SOME(chr(ord c - ord #"@"), rep''')
			        else NONE
			(* end case *))
		    | (SOME(d1, rep'')) => if (isDigit d1)
			then (case (get2 rep'')
			   of SOME(d2, d3, rep''') => let
				fun cvt d = (ord d - ord #"0")
				in
				  if (isDigit d2 andalso isDigit d3)
				    then SOME(
				      chr(100*(cvt d1)+10*(cvt d2)+(cvt d3)),
				      rep''')
				    else NONE
			        end
			    | NONE => NONE
			  (* end case *))
			else NONE
		  (* end case *))
	      | (SOME(#"\"", rep')) => NONE
	      | (SOME(c, rep')) => if (isPrint c) then (SOME(c, rep')) else NONE
	    (* end case *)
	  end

    val fromString = StringCvt.scanString scan

    val itoa = NumFormat.fmtInt StringCvt.DEC

    fun toString #"\t" = "\\t"
      | toString #"\n" = "\\n"
      | toString #"\"" = "\\\""
      | toString c =
	  if (isPrint c)
	    then InlineT.PolyVector.sub (PreString.chars, ord c)
(** NOTE: we should probably recognize the control characters **)
	    else let
	      val c' = ord c
	      in
		if InlineT.DfltInt.>=(c', 100)
		  then PreString.concat2("\\", itoa c')
		else if InlineT.DfltInt.>=(c', 10)
		  then PreString.concat2("\\0", itoa c')
		  else PreString.concat2("\\00", itoa c')
	      end

  end (* Char *)


(* can't be signature constrained because of inline components *)
structure Char =
  struct
    open Char
    open InlineT.Char
  end
