(* built-in.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *
 * Interfaces to the compiler built-ins, infixes, etc.
 *)

infix 7  * / quot mod rem div
infix 6 ^ + -
infix 3 := o
infix 4 > < >= <= = <>
infixr 5 :: @
infix 0 before

structure PrimTypes1 = struct open PrimTypes end
   (* this silliness is to prevent elabstr.sml from sticking a NO_ACCESS
       in the wrong place *)

open PrimTypes1

structure Assembly = Core.Assembly

(* create a type-safe version of the InLine structure while preserving
 * the inline property of the functions.
 *)
structure InlineT =
  struct
    local
      abstraction CC : sig type 'a control_cont end =
        struct type 'a control_cont = 'a cont end
    in open CC end

    val capture 	: ('1a cont -> '1a) -> '1a = InLine.capture
    val callcc		: ('1a cont -> '1a) -> '1a = InLine.callcc
    val unsafecallcc	: ('a cont -> 'a) -> 'a = InLine.callcc
    val unsafecapture	: ('a control_cont -> 'a) -> 'a = InLine.capture
    val escape		: 'a control_cont -> 'a -> 'b = InLine.throw   
    val throw	 	: 'a cont -> 'a -> 'b = InLine.throw
    val !	 	: 'a ref -> 'a = InLine.!
    val op := 		: 'a ref * 'a -> unit = InLine.:=
    val makeref 	: 'a -> 'a ref = InLine.makeref
    val op = 		: ''a * ''a -> bool  = InLine.=
    val op <> 		: ''a * ''a -> bool = InLine.<>
    val boxed 		: 'a -> bool = InLine.boxed
    val unboxed 	: 'a -> bool = InLine.unboxed
    val cast 		: 'a -> 'b = InLine.cast
    val identity	: 'a -> 'a = InLine.cast
    val objlength	: 'a -> int = InLine.objlength
    val mkspecial	: int * 'a -> 'b = InLine.mkspecial
    val getspecial	: 'a -> int = InLine.getspecial
    val setspecial	: ('a * int) -> unit = InLine.setspecial
    val getpseudo	: int -> 'a = InLine.getpseudo 
    val setpseudo	: 'a * int -> unit = InLine.setpseudo 
    val gethdlr 	: unit -> 'a cont = InLine.gethdlr
    val sethdlr 	: 'a cont -> unit = InLine.sethdlr
    val getvar		: unit -> 'a = InLine.getvar
    val setvar		: 'a -> unit = InLine.setvar
    val compose 	: ('b -> 'c) * ('a -> 'b) -> ('a -> 'c) = InLine.compose
    val op before	: ('a * 'b) -> 'a = InLine.before
    val gettag		: 'a -> int = InLine.gettag
    val setmark 	: 'a -> unit = InLine.setmark 
    val dispose 	: 'a -> unit = InLine.dispose 
    val inlnot		: bool -> bool = InLine.inlnot
    val real		: int -> real = InLine.real   

    structure Int31 =
      struct
        val op *    : int * int -> int  = InLine.i31mul
        val op quot : int * int -> int  = InLine.i31quot
        val op rem  : int * int -> int  = InLine.i31rem
        val op +    : int * int -> int  = InLine.i31add
        val op -    : int * int -> int  = InLine.i31sub 
        val ~       : int -> int = InLine.i31neg
        val andb    : int * int -> int  = InLine.i31andb
        val orb     : int * int -> int  = InLine.i31orb
        val xorb    : int * int -> int  = InLine.i31xorb
        val rshift  : int * int -> int  = InLine.i31rshift
        val lshift  : int * int -> int  = InLine.i31lshift
        val notb    : int -> int = InLine.i31notb
        val op <    : int * int -> bool = InLine.i31lt
        val op <=   : int * int -> bool = InLine.i31le
        val op >    : int * int -> bool = InLine.i31gt
        val op >=   : int * int -> bool = InLine.i31ge
        val op =    : int * int -> bool = InLine.i31eq
        val op <>   : int * int -> bool = InLine.i31ne
        val ltu     : int * int -> bool = InLine.i31ltu
        val geu     : int * int -> bool = InLine.i31geu
  
        val op mod  : int * int -> int  = InLine.i31mod
        val op div  : int * int -> int  = InLine.i31div
        val min     : int * int -> int  = InLine.i31min
        val max     : int * int -> int  = InLine.i31max
        val abs     : int -> int = InLine.i31abs
      end

    structure Float64 =
      struct
        val op +   : real * real -> real = InLine.f64add
        val op -   : real * real -> real = InLine.f64sub
        val op /   : real * real -> real = InLine.f64div
        val op *   : real * real -> real = InLine.f64mul
        val op =   : real * real -> bool = InLine.f64eq
        val op <>  : real * real -> bool = InLine.f64ne
        val op >=  : real * real -> bool = InLine.f64ge
        val op >   : real * real -> bool = InLine.f64gt
        val op <=  : real * real -> bool = InLine.f64le
        val op <   : real * real -> bool = InLine.f64lt
        val ~      : real -> real = InLine.f64neg
        val abs    : real -> real = InLine.f64abs
      end

    structure Word32 =
      struct
        val toint   : word32 -> int = InLine.w32toint
        val fromint : int -> word32 = InLine.w32fromint
        val orb     : word32 * word32 -> word32 = InLine.w32orb
        val xorb    : word32 * word32 -> word32 = InLine.w32xorb
        val andb    : word32 * word32 -> word32 = InLine.w32andb
        val op *    : word32 * word32 -> word32 = InLine.w32mul
        val op +    : word32 * word32 -> word32 = InLine.w32add
        val op -    : word32 * word32 -> word32 = InLine.w32sub
        val op div  : word32 * word32 -> word32 = InLine.w32div
        val op >    : word32 * word32 -> bool   = InLine.w32gt
        val op >=   : word32 * word32 -> bool   = InLine.w32ge
        val op <    : word32 * word32 -> bool   = InLine.w32lt
        val op <=   : word32 * word32 -> bool   = InLine.w32le
        val rshift  : word32 * word -> word32 = InLine.w32rshift
        val rshiftl : word32 * word -> word32 = InLine.w32rshiftl
        val lshift  : word32 * word -> word32 = InLine.w32lshift
        val notb    : word32 -> word32 = InLine.w32notb
      end

    structure Word31 =
      struct
        val toint      : word -> int = InLine.w31toint
        val fromint    : int -> word = InLine.w31fromint
	val toword32   : word -> word32 = InLine.w31tow32 
	val fromword32 : word32 -> word = InLine.w31fromw32
        val orb     : word * word -> word = InLine.w31orb
        val xorb    : word * word -> word = InLine.w31xorb
        val andb    : word * word -> word = InLine.w31andb
        val op *    : word * word -> word = InLine.w31mul
        val op +    : word * word -> word = InLine.w31add
        val op -    : word * word -> word = InLine.w31sub
        val op div  : word * word -> word = InLine.w31div
        val op >    : word * word -> bool   = InLine.w31gt
        val op >=   : word * word -> bool   = InLine.w31ge
        val op <    : word * word -> bool   = InLine.w31lt
        val op <=   : word * word -> bool   = InLine.w31le
        val rshift  : word * word -> word = InLine.w31rshift
        val rshiftl : word * word -> word = InLine.w31rshiftl
        val lshift  : word * word -> word = InLine.w31lshift
        val notb    : word -> word = InLine.w31notb
      end

    structure Word8 =
      struct
      (* temporary framework, because the actual word8 operators are not implemented*)
        val toint    : word8 -> int = InLine.cast
	val toword32 : word8 -> word32 = InLine.w31tow32
	val fromword32 : word32 -> word8 = InLine.w31fromw32
        val orb     : word8 * word8 -> word8 = InLine.i31orb
        val xorb    : word8 * word8 -> word8 = InLine.i31xorb
        val op div  : word8 * word8 -> word8 = InLine.i31div
        val andb    : word8 * word8 -> word8 = InLine.i31andb
        val op >    : word8 * word8 -> bool  = InLine.i31gt
        val op >=   : word8 * word8 -> bool  = InLine.i31ge
        val op <    : word8 * word8 -> bool  = InLine.i31lt
        val op <=   : word8 * word8 -> bool  = InLine.i31le
        val rshift  : word8 * word -> word8 = InLine.i31rshift
        val rshiftl : word8 * word -> word8 = InLine.i31rshift (* high bits always 0 *)
        val lshift  : word8 * word -> word8 = InLine.i31lshift
(* WARNING! the following operators don't get the high-order bits right *)
        val notb    : word8 -> word8 = InLine.i31notb  
        val op *    : word8 * word8 -> word8 = InLine.i31mul
        val op +    : word8 * word8 -> word8 = InLine.i31add
        val op -    : word8 * word8 -> word8 = InLine.i31sub
        val fromint : int -> word8 = InLine.cast
      end

    structure Char =
      struct

        val maxOrd = 255
        exception Chr

      (* the following should be an inline operator *)
        fun chr i = if (Int31.geu(i, Int31.+(maxOrd,1)))
	    then raise Chr
	    else ((InLine.cast i) : char)

        val ord : char -> int = InLine.cast

        val (op <)  : (char * char) -> bool = InLine.i31lt
        val (op <=) : (char * char) -> bool = InLine.i31le
        val (op >)  : (char * char) -> bool = InLine.i31gt
        val (op >=) : (char * char) -> bool = InLine.i31ge
      end

    structure PolyArray =
      struct
        val length    : 'a array -> int = InLine.length
        val sub       : 'a array * int -> 'a = InLine.arrSub
        val chkSub    : 'a array * int -> 'a = InLine.arrChkSub
        val update    : 'a array * int * 'a -> unit = InLine.arrUpdate
        val chkUpdate : 'a array * int * 'a -> unit = InLine.arrChkUpdate
      end

    structure PolyVector =
      struct
        val length    : 'a vector -> int = InLine.length 
        val sub       : 'a vector * int -> 'a = InLine.vecSub
        val chkSub    : 'a vector * int -> 'a = InLine.vecChkSub
      end

  (* The type of this ought to be float64array *)
    structure Float64Array =
      struct
        val length    : Assembly.A.realarray -> int = InLine.length
        val sub       : Assembly.A.realarray * int -> real = InLine.f64Sub
        val chkSub    : Assembly.A.realarray * int -> real = InLine.f64chkSub
        val update    : Assembly.A.realarray * int * real -> unit = InLine.f64Update
        val chkUpdate : Assembly.A.realarray * int * real -> unit = InLine.f64chkUpdate
      end

    structure Word8Array =
      struct
        val length    : Assembly.A.bytearray -> int = InLine.length
    (* BUG: using "ordof" for W8A.sub is dangerous, because ordof is
     (technically) fetching from immutable things.  A fancy optimizer might
     someday be confused. *)
        val sub       : Assembly.A.bytearray * int -> word8 = InLine.ordof
        val chkSub    : Assembly.A.bytearray * int -> word8 = InLine.inlbyteof
        val update    : Assembly.A.bytearray * int * word8 -> unit = InLine.store
        val chkUpdate : Assembly.A.bytearray * int * word8 -> unit = InLine.inlstore
      end

    structure Word8Vector =
      struct
        val length    : string -> int = InLine.length
        val sub       : string * int -> word8 = InLine.ordof
        val chkSub    : string * int -> word8 = InLine.inlordof
        val update	  : string * int * word8 -> unit = InLine.store
      end

    structure CharArray =
      struct
	val length    : Assembly.A.bytearray -> int
		= InLine.length
	val chkSub    : (Assembly.A.bytearray * int) -> char
		= InLine.inlordof
	val chkUpdate : (Assembly.A.bytearray * int * char) -> unit
		= InLine.inlstore
	val sub       : (Assembly.A.bytearray * int) -> char
		= InLine.ordof
	val update    : (Assembly.A.bytearray * int * char) -> unit
		= InLine.store
      end

    structure CharVector =
      struct
	val length    : string -> int			= InLine.length
	val chkSub    : (string * int) -> char		= InLine.inlordof
	val sub       : (string * int) -> char		= InLine.ordof
	val update    : (string * int * char) -> unit	= InLine.store
      end

  (* This is only here to temporarily support the ByteArray module.
   * Both will be deleted soon.
   *)
    val blength : Assembly.A.bytearray -> int = InLine.length
    val byteof : Assembly.A.bytearray * int -> int = InLine.ordof
    val inlbyteof : Assembly.A.bytearray * int -> int = InLine.inlbyteof
    val store : string * int * char -> unit = InLine.store
    val bstore : Assembly.A.bytearray * int * int -> unit = InLine.store

    structure DfltInt  = Int31
    structure DfltWord  = Word31
    structure DfltReal = Float64

  end  (* structure InlineT *)
