(* posix-bin-prim-io.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *
 * This implements the UNIX version of the OS specific binary primitive
 * IO structure.  The Text IO version is implemented by a trivial translation
 * of these operations (see posix-text-prim-io.sml).
 *)

structure PosixBinPrimIO : OS_PRIM_IO = 
  struct

    structure PrimIO = BinPrimIO

    structure Vec = Word8Vector
    structure PF = Posix.FileSys
    structure PIO = Posix.IO

    type file_desc = PF.file_desc

    val toFPI = Position.fromInt  (* temporary *)

    fun announce s x y = (
	  (*print "Posix: "; print (s:string); print "\n"; *)
	  x y)

  (* The buffer size is chosen as the largest string allocated in the
   * nursery.
   *)
    val bufferSzB = 2048

    fun mkReader {fd, name, initPos, initBlkMode} = let
	  val closed = ref false
          val pos = ref initPos
          val blocking = ref initBlkMode
          fun updateStatus () =
		announce "setfl"
		  PIO.setfl(fd, if !blocking then PIO.O.flags[] else PIO.O.nonblock)
	  fun incPos k = pos := Position.+(!pos, toFPI k)
	  fun readVec n = let
		val v = announce "read" PIO.readVec(fd, n)
		in
		  incPos (Vec.length v); v
		end
	  fun readArr {data, first, nelems} = let
		val k = announce "readBuf" 
			PIO.readArr(fd, {buf=data, i=first, sz=SOME nelems})
		in
		  incPos k; k
		end
	  fun blockWrap f x = f x
	  fun noBlockWrap f x = SOME(f x)
		handle (e as OS.SysErr(_, SOME cause)) =>
                     if cause = Posix.Error.again then NONE else raise e

	  fun ensureOpen () = if !closed then raise ClosedStream else ()
	  fun ensureBlock (x) =
		if !blocking = x then () else (blocking := x; updateStatus())
	  fun readv block wrap n = (
		ensureOpen(); ensureBlock block; wrap readVec n)
	  fun reada block wrap arg = (
		ensureOpen(); ensureBlock block; wrap readArr arg)
	  fun close () = if !closed
		then ()
		else (closed:=true; announce "close" PIO.close fd)
	  fun getPos () = !pos
	  fun findPos ({data,first,nelems}, p) = 
		Position.+(p, Position.fromInt nelems)
	  fun setPos p = (
		ensureOpen(); pos := announce "lseek" PIO.lseek(fd,p,PIO.SEEK_SET))
	  fun endPos () = (
		ensureOpen();
		case PF.ST.size(announce "fstat" PF.fstat fd)
		 of SOME p => p
		  | _ => raise OS.SysErr("cannot fstat", SOME Posix.Error.spipe)
		(* end case *))
	  in
	    BinPrimIO.Rd{
		readBlock	= SOME(readv true blockWrap),
		readNoBlock	= SOME(readv false noBlockWrap),
		readaBlock	= SOME(reada true blockWrap),
		readaNoBlock	= SOME(reada false noBlockWrap),
		block		= NONE,
		canInput	= NONE,
		name		= name,
		chunkSize	= bufferSzB,
		close		= close,
		getPos		= getPos,
		findPos		= findPos,
		setPos		= setPos,
		endPos		= endPos,
		desc		= SOME(PF.fdToIOD fd)
	      }
	  end

	     
    fun openRd name = mkReader{
	    fd = announce "openf" PF.openf(name,PIO.O_RDONLY,PF.O.flags[]),
	    name = name,
	    initPos = Position.fromInt 0, 
	    initBlkMode = true
	  }


    fun mkWriter {fd, name, initBlkMode, appendMode, chunkSize, lineBuf} = let
	  val closed = ref false
	  val blocking = ref initBlkMode
	  val appendFS = PIO.O.flags(if appendMode then [PIO.O.append] else nil)
	  fun updateStatus() = let
		val flgs = if !blocking
		      then appendFS
		      else PIO.O.flags[PIO.O.nonblock, appendFS]
		in
		  announce "setfl" PIO.setfl(fd, flgs)
		end
	  fun ensureOpen () = if !closed then raise ClosedStream else ()
	  fun ensureBlock (x) =
		if !blocking = x then () else (blocking := x; updateStatus())
	  fun putV x = announce "writeVec" PIO.writeVec x
	  fun putA x = announce "writeArr" PIO.writeArr x
	  fun write (put, block) {data,first,nelems} = (
		ensureOpen(); ensureBlock block; 
		put(fd, {buf=data, i=first, sz=SOME nelems}))
	  fun handleBlock writer arg = SOME(writer arg)
		handle (e as OS.SysErr(_, SOME cause)) => 
 		  if cause = Posix.Error.again then NONE else raise e
	  fun close () = if !closed
		then ()
		else (closed:=true; announce "close" PIO.close fd)
	  fun getPos p = (
		ensureOpen(); announce "lseek" PIO.lseek(fd, toFPI 0, PIO.SEEK_CUR))
	  fun setPos p = (
		ensureOpen(); announce "lseek" PIO.lseek(fd, p, PIO.SEEK_SET); ())
	  fun endPos () = (
		ensureOpen();
		case (PF.ST.size(announce "fstat" PF.fstat fd))
		 of SOME p => p
		  | _ => raise OS.SysErr("can't stat file", SOME(Posix.Error.spipe))
		(* end case *))
	  in
	    BinPrimIO.Wr{
		writeBlock	= SOME(write(putV,true)),
		writeNoBlock	= SOME(handleBlock(write(putV,false))),
		writeaBlock	= SOME(write(putA,true)),
		writeaNoBlock	= SOME(handleBlock(write(putA,false))),
		block		= NONE,
		canOutput	= NONE,
		name		= name,
		chunkSize	= chunkSize,
		lineBuf		= lineBuf,
		close		= close,
		getPos		= getPos,
		setPos		= setPos,
		endPos		= endPos,
		desc		= SOME(PF.fdToIOD fd)
	      }
	  end

    val standardMode = PF.S.flags[	(* mode 0666 *)
	    PF.S.irusr, PF.S.iwusr,
	    PF.S.irgrp, PF.S.iwgrp,
	    PF.S.iroth, PF.S.iwoth
	  ]
    fun createFile (name, mode, flags) =
	  announce "createf" PF.createf(name, mode, flags, standardMode)

    fun openWr name = mkWriter{
	    fd=createFile(name, PIO.O_WRONLY, PF.O.trunc),
	    name=name,
	    initBlkMode=true,
	    appendMode=false,
	    lineBuf=NONE,
	    chunkSize=bufferSzB
	  }

    fun openApp name = mkWriter{
	    fd		= createFile(name, PIO.O_WRONLY, PF.O.append),
	    name	= name,
	    initBlkMode	= true,
	    appendMode	= true,
	    lineBuf	= NONE,
	    chunkSize	= 1	(* don't want to buffer these streams!! *)
	  }


  end; (* PosixPrimIO *)

