(* net-db.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *)

structure NetDB : NET_DB =
  struct

    structure SysW = SysWord

    val netdbFun = CInterface.c_function "SMLNJ-Sockets"

    datatype net_addr = NETADDR of SysW.word

    type addr_family = PreSock.addr_family

    datatype entry = NETENT of {
	name : string,
	aliases : string list,
	addrType : addr_family,
	addr : net_addr
      }

    local
      fun conc field (NETENT a) = field a
    in
    val name = conc #name
    val aliases = conc #aliases
    val addrType = conc #addrType
    val addr = conc #addr
    end (* local *)

  (* Network DB query functions *)
    local
      type netent
	= (string * string list * PreSock.af * SysWord.word)
      fun getNetEnt NONE = NONE
	| getNetEnt (SOME(name, aliases, addrType, addr)) = SOME(NETENT{
	      name = name, aliases = aliases,
	      addrType = PreSock.AF addrType, addr = NETADDR addr
	    })
      val getNetByName' : string -> netent option
	    = netdbFun "getNetByName"
      val getNetByAddr' : (SysWord.word * PreSock.af) -> netent option
	    = netdbFun "getNetByAddr"
    in
    val getByName = getNetEnt o getNetByName'
    fun getByAddr (NETADDR addr, PreSock.AF af) =
	  getNetEnt(getNetByAddr'(addr, af))
    end (* local *)

    fun stringToNetAddr s = let
	  val (op +) = SysW.+
	  in
	    case (PreSock.toWords s)
	     of [a, b, c, d] =>
		SOME(NETADDR(SysW.<<(a, 0w24)+SysW.<<(b, 0w16)+SysW.<<(c, 0w8)+d))
	      | [a, b, c] => SOME(NETADDR(SysW.<<(a, 0w24)+SysW.<<(b, 0w16)+c))
	      | [a, b] => SOME(NETADDR(SysW.<<(a, 0w24)+b))
	      | [a] => SOME(NETADDR a)
	      | _ => NONE
	    (* end case *)
	  end

    fun netAddrToString (NETADDR addr) = let
	  fun get n = Word8.fromLargeWord(SysW.toLargeWord((SysW.>>(addr, n))))
	  in
	    PreSock.fromBytes (get 0w24, get 0w16, get 0w8, get 0w0)
	  end

  end
