(* bin-io-fn.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *)

functor BinIOFn (

    structure OSPrimIO : OS_PRIM_IO
      sharing type OSPrimIO.PrimIO.elem = Word8.word
      sharing type OSPrimIO.PrimIO.vector = Word8Vector.vector
      sharing type OSPrimIO.PrimIO.array = Word8Array.array
      sharing type OSPrimIO.PrimIO.pos = Position.int

  ) : BIN_IO = struct

(***** INCLUDE "stream-io-fn.sml" *****)
functor StreamIO(structure PrimIO : PRIM_IO
                 structure Vec: MONO_VECTOR
                 structure Arr: MONO_ARRAY
                 val someElem : PrimIO.elem
		 val posLess : PrimIO.pos * PrimIO.pos -> bool
		 val posDiff : ({lo: PrimIO.pos, hi: PrimIO.pos} -> 
				                    Position.int) option
		 type subvector
                 val base: subvector -> Vec.vector * int * int
               sharing type PrimIO.elem = Arr.elem = Vec.elem 
               sharing type PrimIO.vector=Arr.vector=Vec.vector
               sharing type PrimIO.array=Arr.array
               ) : STREAM_IO =
let 
 structure P = PrimIO
 abstraction Body : sig include STREAM_IO sharing PrimIO=P end =
 struct
  structure PrimIO=PrimIO
  type elem = PrimIO.elem
  type array = PrimIO.array
  type vector = PrimIO.vector
  type subvector = subvector
  type pos = PrimIO.pos

    type finder = ({data: vector, first:int, nelems: int}*pos) ->pos 

    datatype buffer = Buf of {
	more     : more ref,
        data     : vector,
	basePos  : pos,
	findPos  : finder,
	emptyMeansEof: bool,
	name : string
      }				

    and more = GETmore of PrimIO.reader
             | ISmore of buffer
             | NOmore

    and instream = In of {
        pos      : int,
        buffer   : buffer
      }


 (* NOTE: I haven't implemented check to ensure blocking reads are possible.*)

    val empty = Vec.fromList[]

    fun mkInstream r = 
	let val r' as PrimIO.Rd{name,getPos,findPos,...} = PrimIO.augmentIn r
	 in In{pos=0, 
	       buffer=Buf{name=name,
			  emptyMeansEof=false,
			  data=empty,
			  more = ref(GETmore r'),
			  basePos=getPos(),
			  findPos=findPos}}
	end

    fun handler(Buf{name,...},function,e) = 
	     raise Io{function=function,name=name,cause=e}

    fun closeBuf (Buf{more=ref NOmore,...}) = ()
      | closeBuf (Buf{more=ref(ISmore buf),...}) = closeBuf buf
      | closeBuf (b as Buf{more as ref(GETmore(PrimIO.Rd{close,...})),...}) =
	    (close() handle e => handler(b,"closeIn",e); 
	     more := NOmore)

    fun closeIn (In{buffer,...}) = closeBuf buffer


    exception WouldBlock  (* not to be raised out of this module *)

    fun filbuf (doRead: PrimIO.reader -> vector, mlOp: string) 
	       (buffer as Buf{data,more,emptyMeansEof,name,basePos,findPos})=
         let val len = Vec.length data
          in if len = 0 andalso emptyMeansEof
              then {eof=true, 
		    rest=Buf{data=empty,more=more,name=name,
			     emptyMeansEof=false,basePos=basePos,
			     findPos=findPos}}
             else case !more
		  of ISmore(Buf{data=d,more=m,emptyMeansEof=e,
				name=n,basePos=b,findPos=f}) =>
			     {eof=e andalso Vec.length d = 0,
			      rest=Buf{data=d,more=m,
				       emptyMeansEof=false,
				       name=n,basePos=b,findPos=f}}
                   | NOmore => {eof=true,
				rest=Buf{data=empty,more=more,
					 emptyMeansEof=true,
					 name=name,basePos=basePos,
					 findPos=findPos}}
		   | m as GETmore (gm as PrimIO.Rd{getPos,...}) => 
			   (let val basePos' = getPos()
				val v = doRead gm
(*val _ = (print "got "; print (Vec.length v); print "\n")*)
				val buf' = Buf{data=v, more=ref m,
					       name=name, basePos=basePos',
					       findPos=findPos,
					       emptyMeansEof=true}
			     in more := ISmore buf'
			    end handle e => handler(buffer,mlOp,e);
			    filbuf (doRead,mlOp) buffer)
          end

    fun generalizedInput (filbuf': buffer -> {eof: bool, rest: buffer}) 
	              : instream -> vector * instream =
    let fun get(In{pos,buffer as Buf{data,...}}) =
       let val len = Vec.length data
        in (*print "pos=";print pos; print " len="; print len; print"\n";*)
	    if pos < len
	    then (Vec.extract(data, pos, SOME(len - pos)),
		  In{pos=len,buffer=buffer})
	    else case filbuf' buffer
		  of {eof=true,rest} => (empty,In{pos=0,buffer=rest})
		   | {eof=false,rest} => get (In{pos=0,buffer=rest})
       end
     in get
    end

   fun chunk1 (PrimIO.Rd{chunkSize,readBlock=SOME read,...}) = read chunkSize
     | chunk1 _ = raise BlockingNotSupported


   fun chunkN n (PrimIO.Rd{chunkSize=1,readBlock=SOME read,...}) = read n
     | chunkN n (PrimIO.Rd{chunkSize=k,readBlock=SOME read,...}) =
                      (* round n up to the next multiple of k *)
	               read (((n-1+k) div k) * k)
     | chunkN _ _ = raise BlockingNotSupported

   val eofFilbuf = filbuf (chunk1, "endOfStream")

   fun endOfStream (In{pos,buffer as Buf{data,...}}) =
       if pos < Vec.length data then false
	   else #eof(eofFilbuf buffer)

   val input = generalizedInput(filbuf(chunk1, "input"))

 local

   fun bigchunk (arg as PrimIO.Rd{getPos,endPos,...}) =
       case posDiff
        of SOME f => (chunkN (Position.toInt(Position.max(f{hi=endPos(),lo=getPos()}, Position.fromInt 1)))
		      handle _ => chunk1) 
	             arg
         | NONE => chunk1 arg

   val biginput = generalizedInput(filbuf(bigchunk,"inputAll"))
		      
  in
    fun inputAll f =
	(* Try "biginput" the first time.  If that doesn't
	    get everything, treat it as unreliable and unnecessarily
	    expensive, i.e. don't waste time with "endPos" in the
	    rest of the input operations. *)
	let fun loop f = 
	       let val (s,rest) = input f
		in if Vec.length s = 0 then nil else s :: loop rest
               end
	    val (s,rest) = biginput f
         in if Vec.length s = 0 then s else Vec.concat(s :: loop rest)
        end
  end
	     
   local 
       fun nonBlockChunk (PrimIO.Rd{chunkSize,readNoBlock=SOME read,...})=
		      (case read chunkSize
			of NONE => raise WouldBlock
		         | SOME stuff => stuff)
         | nonBlockChunk _ = raise NonBlockingNotSupported

       val inpNob = generalizedInput (filbuf (nonBlockChunk, "inputNoBlock"))
    in fun inputNoBlock x = SOME(inpNob x) handle WouldBlock => NONE
  end

    val input1Filbuf = filbuf(chunk1, "input1")

    fun input1(In{pos,buffer as Buf{data,...}}) =
       let val len = Vec.length data
        in if pos < len
	        then (SOME(Vec.sub(data,pos)), In{pos=pos+1,buffer=buffer})
	    else case input1Filbuf buffer
		  of {eof=true,rest} => (NONE,In{pos=0,buffer=rest})
		   | {eof=false,rest} => input1(In{pos=0,buffer=rest})

       end

    fun listInputN(In{pos,buffer as Buf{data,...}}, n) =
       let val len = Vec.length data
        in if pos + n <= len
	        then ([Vec.extract(data,pos,SOME n)], In{pos=pos+n,buffer=buffer})
	    else if pos < len
		then let val hd = Vec.extract(data,pos,SOME(len-pos))
		         val (tl,f') = listInputN(In{pos=len,buffer=buffer},
						  n-(len-pos))
		      in (hd::tl,f')
		     end
	    else case filbuf (chunkN n, "inputN") buffer
		  of {eof=true,rest} => (nil,In{pos=0,buffer=rest})
		   | {eof=false,rest} => listInputN(In{pos=0,buffer=rest},n)
       end

    fun inputN(f,n) = 
	let val (vl,f') = listInputN(f,n)
         in (Vec.concat vl, f')
        end

    fun getPosIn(In{pos,buffer as Buf{data,basePos,findPos,...}}) =
            findPos({data=data,first=0,nelems=pos},basePos)
	     handle e => handler(buffer,"getPosIn",e)

    fun posLessEq(a,b) = Bool.not(posLess(b,a))

    fun setPosIn(In{pos,buffer as Buf{data,basePos,more,name,findPos,...}}, n) =
     let fun binarySearch(first,0) = In{pos=first,buffer=buffer}
	   | binarySearch(first,len) =
	      let val half = len div 2
	       in if posLessEq(n,findPos({data=data,first=first,nelems=half},basePos))
		    then binarySearch(first,half)
		    else binarySearch(first+half+1,len-(half+1))
	      end

         val datalen = Vec.length data
	 
      in if posLessEq(basePos,n) 
	  andalso posLessEq (n,
                          findPos({data=data,first=0,nelems=datalen},basePos))
	     then binarySearch(0,datalen)
	 else case !more
	    of ISmore f' => setPosIn(In{pos=0,buffer=f'}, n)
             | NOmore => handler(buffer,"setPosIn",TerminatedStream)
	     | m as GETmore(PrimIO.Rd{setPos,...}) =>
		(more := NOmore;
		 setPos n handle e => handler(buffer,"setPosIn",e);
		 In{pos=0,buffer=Buf{data=empty,more=ref m,name=name,
				     emptyMeansEof=false,
				     basePos=n,findPos=findPos}})

     end
         
    fun getReader' (Buf{more=ref(NOmore),...}) = raise TerminatedStream
      | getReader' (Buf{more=ref(ISmore b),...}) = getReader' b
      | getReader' (Buf{more=ref(GETmore m),...}) = m

    fun getReader(In{pos,buffer}) = 
	     getReader' buffer handle e => handler(buffer,"getReader",e)

    fun endPosIn(In{pos,buffer}) =
	let val PrimIO.Rd{endPos,...} = getReader' buffer
         in endPos()
        end handle e => handler(buffer,"endPosIn",e)


    (* OUTPUT: *)

    datatype outstream = Out of {
        data     : array,
        pos      : int ref,
        writer   : PrimIO.writer
      }

    fun handler(Out{writer=PrimIO.Wr{name,...},...}, function, cause) =
                     raise Io{name=name,function=function,cause=cause}

    fun mkOutstream w =
	case PrimIO.augmentOut w
         of w' as PrimIO.Wr{chunkSize=1,...} =>
	     Out{data=Arr.array(0,someElem), pos=ref 0, writer=w'}
          | w' as PrimIO.Wr{chunkSize,...} =>
		Out{data=Arr.array(chunkSize,someElem), pos=ref 0, writer=w'}

    fun flushOut' (Out{data,pos,
			writer=PrimIO.Wr{writeaBlock=SOME write,...},...}) =
       let val p = !pos
	   fun loop i = if i<p 
	        then loop(i+write{data=data,first=i,nelems=p-i}
			  handle e => (Arr.copy{src=data,si=i,len=SOME(p-i),
						dst=data,di=0};
				       pos := p-i;
				       raise e))

		else ()
	in pos := 0; (* do this first, in case of interrupt *)
	   loop 0
       end      
      | flushOut' _ = raise BlockingNotSupported

    fun closeOut(f as Out{writer=PrimIO.Wr{close,name,...},...})= 
			(flushOut' f;  close()) 
			handle e => handler(f,"closeOut",e)
				 

    fun flushOut f = flushOut' f handle e => handler(f,"flushOut",e)

    fun bigoutput(f as Out{writer=PrimIO.Wr{writeBlock=SOME write,...},...},
		  buf as {data,first,nelems}) =
	   if nelems=0 then ()
            else let val written = write buf (* may raise exception! *)
		  in bigoutput(f, {data=data,first=first+written,
				   nelems=nelems-written})
		 end
     | bigoutput (Out{writer=PrimIO.Wr{writeBlock=NONE,...},...}, _) =
			  raise BlockingNotSupported

    fun output'(f as Out{data,pos,writer=PrimIO.Wr{lineBuf,...},...}, s, 
		spos,nelems,opname) =
      let val blen = Arr.length data
          val p = !pos
	  fun copy(i,offset) = if i<nelems 
                                 then (Arr.update(data,i+offset,
						  Vec.sub(s,i+spos));
				       copy(i+1,offset))
                                 else pos := offset + nelems;
	  fun copyLB(i,offset,nl) = if i<nelems 
                                 then let val x = Vec.sub(s,i+spos)
				       in Arr.update(data,i+offset,x);
					   if nl x 
					       then (copy(i,offset);
						     flushOut' f)
					       else copyLB(i+1,offset,nl)
				      end
                                 else pos := offset + nelems
          fun copyAt offset =
	      case lineBuf
               of SOME nl => copyLB(0,offset,nl)
                | NONE => copy(0,offset)
       in if p+nelems < blen
           then copyAt p
           else ((flushOut' f;
		  if nelems < blen
		     then copyAt 0
                     else bigoutput(f,{data=s,first=spos,nelems=nelems}))
		 handle e => handler(f,opname,e))
      end

    fun output(f,s) = output'(f,s,0,Vec.length s,"output")
          
    fun outputS(f as Out{data,pos,...}, sv) =
      let val (s,first,nelems) = base sv
       in output'(f,s,first,nelems,"outputS")
      end


    fun output1(f as Out{data,pos,writer=PrimIO.Wr{lineBuf,...},...}, e) =
      let val blen = Arr.length data
          val p = !pos
       in if p < blen
           then (Arr.update(data,p,e); pos := p+1;
		 case lineBuf
                  of SOME nl => 
		      if nl e then flushOut' f 
			            handle e=> handler(f,"output1",e)
			      else ()
		   | NONE => ())
           else if p=0
	         then bigoutput(f,{data=Vec.fromList[e],first=0,nelems=1})
		 else (flushOut' f handle e=> handler(f,"output1",e);
		       output1(f,e))
      end

    fun getPosOut(f as Out{writer=PrimIO.Wr{getPos,...},...}) =
	(flushOut' f; getPos()) handle e => handler(f,"getPosOut",e)

    fun setPosOut(f as Out{writer=PrimIO.Wr{setPos,...},...},p) =
	(flushOut' f; setPos p) handle e => handler(f,"setPosOut",e)

    fun endPosOut(f as Out{writer=PrimIO.Wr{endPos,...},...}) =
	(flushOut' f; endPos()) handle e => handler(f,"endPosOut",e)

    fun getWriter (Out{writer,...}) = writer

 end
 in Body
end
(***** END "stream-io-fn.sml" *****)

    structure BinStreamIO = StreamIO (
	structure PrimIO = OSPrimIO.PrimIO
	structure Vec = Word8Vector
	structure Arr = Word8Array
	val someElem: Word8.word = 0w0
	val posLess = Position.<
	val posDiff = SOME(fn {hi,lo} => Position.-(hi,lo))
	type subvector = Vec.vector
	fun base v = (v,0,Vec.length v))
               

    structure BinIO' = ImperativeIO (structure S = BinStreamIO)

    open BinIO'

    val openIn     = mkInstream o BinStreamIO.mkInstream o OSPrimIO.openRd
    val openOut    = mkOutstream o BinStreamIO.mkOutstream o OSPrimIO.openWr
    val openAppend = mkOutstream o BinStreamIO.mkOutstream o OSPrimIO.openApp

  end

