(* Copyright 1990, 1991 by AT&T Bell Laboratories *)
(* stamps.sml *)

abstraction Stamps : STAMPS =
struct

  type counter = int ref

  fun inc r = r := !r+1

  (* A fresh FREESCOPE scope is created for each compilation unit.
     The string ref part of a FREESCOPE initially contains "", but this
     gets replaced by the 16 byte intrinsic stamp calculated by
     hashing the static environment of the compilation unit. *)

  val freecounter = ref 0

  datatype scope
    = FREESCOPE
    | BOUNDSCOPE of int * counter
  
  datatype stamp
    = BOUND of {scope: int, count: int}
    | FREE of {scope: string, count: int} ref
    | NULL
    | ERROR

  fun isExternal(FREE(ref{scope= "",...})) = false
    | isExternal(FREE _) = true
    | isExternal _ = false


  (* nextBoundScope is used to generate BOUNDSCOPE numbers used to
     distinguish bound scopes.  This should be reset for each compilation
     unit, though the hashing function for environments currently ignores
     the scope number of BOUNDSCOPE. *)
  val nextBoundScope = ref 1

  fun newBoundScope () =
      BOUNDSCOPE(!nextBoundScope,ref 0) before inc nextBoundScope

  val freeScope = FREESCOPE

  fun setFreeStamp(FREE r, scope, count) = r:={scope=scope,count=count}
    | setFreeStamp _ = ErrorMsg.impossible "Stamps.setFreeStamp"

  fun newStamp FREESCOPE = 
        (fn () => FREE(ref{scope="",count= !freecounter before inc freecounter}))
    | newStamp (BOUNDSCOPE(s,c)) =
        (fn () => BOUND{scope=s,count= !c before inc c})

  (* one predefined freescope used for newfree *)
  val newFree = newStamp FREESCOPE

  fun specialFree s = FREE(ref{scope= s, count=0})   (* a bit of a hack *)

  val null = NULL
  val error = ERROR

  fun isFree (FREE _) = true
    | isFree _ = false

  fun isBound FREESCOPE = (fn FREE(ref{scope="",...}) => true | _ => false)
      (* we assume that the FREESCOPE is always the 
           "current" one.  Hope it's true! *)
    | isBound (BOUNDSCOPE(s,_)) = (fn BOUND{scope=s',...} => s=s' | _ => false)

  fun eq (BOUND{scope=s1,count=n1}, BOUND{scope=s2,count=n2}) =
             s1=s2 andalso n1=n2
    | eq (FREE(r1 as ref{scope=s1,count=n1}), 
	  FREE(r2 as ref{scope=s2,count=n2})) =
                   r1=r2 orelse s1=s2 andalso n1=n2
    | eq _ =  false

  fun less (BOUND{scope=s1,count=n1}, BOUND{scope=s2,count=n2}) =
        if s1 <> s2 then ErrorMsg.impossible "Stamps.less -- bscopes"
	else n1 < n2
    | less (FREE(ref{scope=s1,count=n1}),FREE(ref{scope=s2,count=n2})) =
        if s1 <> s2 then ErrorMsg.impossible "Stamps.less -- fscopes"
	else n1 < n2
    | less _ =  ErrorMsg.impossible "Stamps.less - bad arg"

  fun greater (BOUND{scope=s1,count=n1}, BOUND{scope=s2,count=n2}) =
        if s1 <> s2 then ErrorMsg.impossible "Stamps.greater -- bscopes"
	else n1 > n2
    | greater (FREE(ref{scope=s1,count=n1}),FREE(ref{scope=s2,count=n2})) =
        if s1 <> s2 then ErrorMsg.impossible "Stamps.greater -- fscopes"
	else n1 > n2
    | greater _ = ErrorMsg.impossible "Stamps.greater - bad arg"

  fun hex pid =
      let val >> = Word.>> infix 3 >>
	  fun hexdig i = substring("0123456789abcdef",Word.toInt i,1)
	  fun printhex i = (hexdig(i >> 0w4) ^ hexdig(Word.andb(i, 0w15)))
       in concat(map (printhex o Word.fromInt o Char.ord) (explode pid))
      end 

  fun stampToString(BOUND{scope=i,count=j}) =
        "BOUND(scope="^makestring i^ ",stamp="^makestring j^")"
    | stampToString(FREE(ref{scope=s,count=i})) =
        "FREE("^ hex s ^ "," ^ makestring i^")"
    | stampToString NULL = "NULL"
    | stampToString ERROR = "ERROR"

  (* this is a temporary fix until we do alpha conversion of
     "exported" stamps in static environments during factoring *)
  (* c is a crc hasing function *)
  fun hashStamp(c,_,_,_,BOUND _) = c 0
       (* note bound, ignore scope and count *)
    | hashStamp(c,int,_,enter,s as FREE(ref{scope="",count})) =
		(c 4; int(enter(s,count)))
    | hashStamp(c,int,string,_,FREE(ref{scope,count})) =
	(c 1; string scope; int count)
    | hashStamp(c,_,_,_,NULL) = c 2
    | hashStamp(c,_,_,_,ERROR) = c 3

  (* stamp maps *)
  (* finite mappings over stamps.  Works for domains consisting
     of mixed free and bound stamps from multiple scopes. *)

  type 'a stampMap = (stamp * 'a) list Intmap.intmap * exn

  fun newMap ex = (Intmap.new(20,ex), ex)

  fun num(BOUND{count,...}) = count
    | num(FREE(ref{count,...})) = count
    | num _ = ErrorMsg.impossible "Stamps.num"

  fun applyMap((m,ex),st) =
   let fun f((a,b)::r) = if eq(a,st) then b else f r
         | f nil = raise ex
    in case st
	of BOUND{count,...} => f(Intmap.map m count)
         | FREE(ref{count,...}) => f(Intmap.map m count)
	 | _ => raise ex
   end

  fun updateMap (m,ex) (st,v) =
   let val n = num st
       val old = Intmap.map m n handle _ => nil
    in Intmap.add m (n, (st,v)::old)
   end
   
end (* structure Stamps *)
