(* Copyright 1989 by AT&T Bell Laboratories *)
(* basictypes.sml *)

structure BasicTypes : BASICTYPES = struct

open Types Symbol Access
structure IP = InvPath

val trueSym	 = varSymbol "true"
val falseSym	 = varSymbol "false"
val listSym	 = tycSymbol "list"
val consSym	 = varSymbol "::"
val nilSym	 = varSymbol "nil"
val refVarSym	 = varSymbol "ref"
val refTycSym	 = tycSymbol "ref"
val contSym	 = tycSymbol "cont"
val arraySym	 = tycSymbol "array"
val vectorSym	 = tycSymbol "vector"

(* primitive type constructors and types *)

infix -->
val arrowStamp = Stamps.specialFree "->"
val arrowTycon =
    GENtyc{stamp = arrowStamp, path = IP.IPATH [tycSymbol "->"],
	   arity = 2, eq = ref NO, kind = ref(PRIMtyc)}
fun t1 --> t2 = CONty(arrowTycon,[t1,t2])
fun isArrowType(CONty(GENtyc{stamp,...},_)) = Stamps.eq(stamp,arrowStamp)
  | isArrowType(VARty(ref(INSTANTIATED ty))) = isArrowType ty
  | isArrowType _ = false
fun domain(CONty(_,[ty,_])) = ty
  | domain _ = ErrorMsg.impossible "domain"
fun range(CONty(_,[_,ty])) = ty
  | range _ = ErrorMsg.impossible "range"

fun primtyc sym = GENtyc{stamp = Stamps.specialFree sym,
		      path = IP.IPATH[tycSymbol sym], arity = 0,
		      eq = ref YES, kind = ref(PRIMtyc)}

val intTycon = primtyc "int"
val intTy = CONty(intTycon,nil)

val realTycon = primtyc "real"
val realTy = CONty(realTycon,nil)

val wordTycon = primtyc "word"
val wordTy = CONty(wordTycon,nil)

val word8Tycon = primtyc "word8"
val word8Ty = CONty(word8Tycon,nil)

val word32Tycon = primtyc "word32"
val word32Ty = CONty(word32Tycon,nil)

val stringTycon = primtyc "string"
val stringTy = CONty(stringTycon,nil)

val charTycon = primtyc "char"
val charTy = CONty(charTycon,nil)

val exnTycon = GENtyc{stamp = Stamps.specialFree "exn",
		      path = IP.IPATH[tycSymbol "exn"], arity = 0,
		      eq = ref NO, kind = ref(PRIMtyc)}
val exnTy = CONty(exnTycon,nil)

val contTycon = GENtyc{stamp = Stamps.specialFree "cont",
		       path = IP.IPATH[contSym], arity = 1,
 		       eq = ref NO, kind = ref(PRIMtyc)}

val arrayTycon = GENtyc{stamp = Stamps.specialFree "array",
		        path = IP.IPATH[arraySym], arity = 1,
		        eq = ref OBJ, kind = ref(PRIMtyc)}

(* vectors *)
val vectorTycon =
    GENtyc{stamp = Stamps.specialFree "vector",
	   path = IP.IPATH[vectorSym], arity = 1,
	   eq = ref YES, kind = ref(PRIMtyc)}

(* building record and product types *)

fun recordTy(fields: (label * ty) list) : ty = 
    CONty(Tuples.mkRECORDtyc(map (fn (a,b) => a) fields),
	  (map (fn(a,b)=>b) fields))

fun tupleTy(tys: ty list) : ty =
    CONty(Tuples.mkTUPLEtyc(length tys), tys)

val unitTycon = Tuples.mkTUPLEtyc 0
val unitTy = CONty(unitTycon, nil)

(* predefined datatypes *)

val alpha = IBOUND 0

(* bool *)

val kind = ref (DATAtyc nil)
val boolTycon =
    GENtyc{stamp = Stamps.specialFree "bool",
	   path = IP.IPATH[tycSymbol "bool"], arity = 0,
	   eq = ref YES, kind = kind}
val boolTy = CONty(boolTycon,nil)
val boolsign = [CONSTANT 0, CONSTANT 1]
val falseDcon = 
    DATACON
      {name = falseSym,
       const = true,
       rep = CONSTANT 0,
       typ = boolTy,
       orig = NONE,
       sign = boolsign}
val trueDcon =
    DATACON
      {name = trueSym,
       const = true,
       rep = CONSTANT 1,
       typ = boolTy,
       orig = NONE,
       sign = boolsign}
val _ = kind := DATAtyc [falseDcon,trueDcon]

(* option;  unnecessary; removed by appel

val kind = ref (DATAtyc nil)

val optionTycon =
    GENtyc{stamp = Stamps.specialFree "option",
	   path = IP.IPATH[optionSym], arity = 1,
	   eq = ref YES, kind = kind}
val optionsign = [CONSTANT 0, UNTAGGED]
val NONEDcon = 
    DATACON
      {name = NONESym,
       const = true,
       rep = CONSTANT 0,
       typ = POLYty {sign = [{weakness=infinity,eq=false}], abs=0,
			   tyfun = TYFUN{arity=1,
					 body=CONty(optionTycon,[alpha])}},
       orig = NONE,
       sign = optionsign}
val SOMEDcon =
    DATACON
      {name = SOMESym,
       const = false,
       rep = UNTAGGED,
       typ = POLYty {sign = [{weakness=infinity,eq=false}], abs=0,
		   tyfun = TYFUN
			    {arity = 1,
			     body = CONty(arrowTycon,
				      [alpha, CONty(optionTycon,[alpha])])}},
       orig = NONE,
       sign = optionsign}
val _ = kind := DATAtyc [NONEDcon,SOMEDcon]
*)

(* references *)

val kind = ref (DATAtyc nil)
val refTycon =
    GENtyc{stamp = Stamps.specialFree "ref",
	   path = IP.IPATH[refTycSym], arity = 1,
	   eq = ref OBJ, kind = kind}
val refTyfun = TYFUN {arity = 1, body = alpha --> CONty(refTycon, [alpha])}
val refDcon = 
    DATACON
      {name = refVarSym,
       const = false,
       rep = REF,
       typ = POLYty {sign = [{weakness=1,eq=false}], tyfun = refTyfun, abs=0},
       orig = NONE,
       sign = [REF]}
val refPatType = POLYty {sign = [{weakness=infinity,eq=false}], tyfun = refTyfun, abs=0}
val _ = kind := DATAtyc [refDcon]

(* lists *)
val kind = ref (DATAtyc nil)
val listsign = [UNTAGGEDREC 2,CONSTANT 0] (* [LISTCONS,LISTNIL] *) 
val listTycon =
    GENtyc{stamp = Stamps.specialFree "list",
	   path = IP.IPATH[listSym], arity = 1,
	   eq = ref YES, kind = kind}
val consDcon =
    DATACON 
      {name = consSym,
       const = false,
       rep = UNTAGGEDREC 2,   (* LISTCONS, *)
       typ = POLYty {sign = [{weakness=infinity,eq=false}], abs=0,
		   tyfun = TYFUN
			    {arity = 1,
			     body = CONty(arrowTycon,
				      [tupleTy[alpha, CONty(listTycon,[alpha])],
				       CONty(listTycon,[alpha])])}},
       orig = NONE,
       sign = listsign}
val nilDcon = 
    DATACON
      {name = nilSym,
       const = true,
       rep = CONSTANT 0, (* LISTNIL, *)
       typ = POLYty {sign = [{weakness=infinity,eq=false}], abs=0,
		   tyfun = TYFUN{arity=1,body=CONty(listTycon,[alpha])}},
       orig = NONE,
       sign = listsign}
val _ = kind := DATAtyc [consDcon,nilDcon]


(* unrolled lists *)
val kind = ref (DATAtyc nil)
val ulistsign = [LISTCONS,LISTNIL]  
val ulistTycon =
    GENtyc{stamp = Stamps.specialFree "ulist",
	   path = IP.IPATH[listSym], arity = 1,
	   eq = ref YES, kind = kind}

val uconsDcon =
   DATACON 
      {name = consSym,
       const = false,
       rep = LISTCONS, 
       typ = POLYty {sign = [{weakness=infinity,eq=false}], abs=0,
	 	     tyfun = TYFUN
			    {arity = 1,
			     body = CONty(arrowTycon,
			      [tupleTy[alpha, CONty(ulistTycon,[alpha])],
			       CONty(ulistTycon,[alpha])])}},
       orig = NONE,
       sign = ulistsign}

val unilDcon = 
   DATACON
      {name = nilSym,
       const = true,
       rep = LISTNIL, (* CONSTANT 0 *)
       typ = POLYty {sign = [{weakness=infinity,eq=false}], abs=0,
  		     tyfun = TYFUN{arity=1,body=CONty(ulistTycon,[alpha])}},
       orig = NONE,
       sign = ulistsign}

val _ = kind := DATAtyc [uconsDcon,unilDcon]

(* frags *)
val kind = ref (DATAtyc nil)

val fragTycon =
    GENtyc{stamp = Stamps.specialFree "frag",
	   path = IP.IPATH[tycSymbol "frag"], arity = 1,
	   eq = ref YES, kind = kind}
val fragsign = [TAGGED 0, TAGGED 1] 
val ANTIQUOTEDcon =
    DATACON
      {name = varSymbol "ANTIQUOTE",
       const = false,
       rep = TAGGED 0,
       typ = POLYty {sign = [{weakness=infinity,eq=false}], abs=0,
		     tyfun = TYFUN
			    {arity = 1,
			     body = CONty(arrowTycon,
				      [alpha, CONty(fragTycon,[alpha])])}},
       orig = NONE,
       sign = fragsign}
val QUOTEDcon = 
    DATACON
      {name = varSymbol "QUOTE",
       const = false,
       rep = TAGGED 1,
       typ = POLYty {sign = [{weakness=infinity,eq=false}], abs=0,
		     tyfun = TYFUN
			    {arity = 1,
			     body = CONty(arrowTycon,
				      [stringTy, CONty(fragTycon,[alpha])])}},
       orig = NONE,
       sign = fragsign}
val _ = kind := DATAtyc [ANTIQUOTEDcon,QUOTEDcon]

end (* structure BasicTypes *)

