(* alpha32MC.sml
 *
 * COPYRIGHT (c) 1996 Bell Laboratories.
 *)

structure Alpha32MCEmitter : EMITTER_NEW =
  struct
    structure I = Alpha32Instr

    structure T = System.Tags
    structure AS = Alpha32AsmEmitter

    val << = Word.<<
    val >> = Word.>>
    val ~>> = Word.~>>
    val ++ = Word.orb
    val & = Word.andb
    infix << >> ~>> ++ &

    val itow  = Word.fromInt

    fun error msg = ErrorMsg.impossible ("Alpha32MCEmitter." ^ msg)

    val loc = ref 0

    fun emitByte n = let
      val i = !loc
      val wtob  = Word8.fromLargeWord o Word.toLargeWord
    in
      loc:= i+1; CodeString.update(i, wtob n)
    end

    (* Alpha32 is low endian *)
    fun emitWord n = (emitByte(n & 0w255); emitByte((n >> 0w8) & 0w255))

    fun emitLong n = let 
      val w = itow n
    in
      emitWord(w & 0w65535); 
      emitWord(w >> 0w16)
    end

    fun emitLongX n = let 
      val w = itow n
    in
      emitWord(w & 0w65535); 
      emitWord(w ~>> 0w16)
    end

    fun emitQuad n = error "emitQuad: not yet"

    fun align () = 
      case Word.andb(itow(!loc), 0w7) 
       of 0w0 => ()
        | 0w4 => emitLong 0
	| _ => error "align"

    fun mark () = emitLong (T.make_desc((!loc + 4) quot 4, T.tag_backptr))
      
    fun defineLabel  lab = ()

    fun emitstring s = let
      val emit = emitByte o itow o Char.ord o String.sub
      fun copy i = (emit (s, i); copy(i+1))
    in
      (copy 0) handle _ => ()
    end

    fun emitString (lab,size,s) =		(* padd strings GGG *)
      (mark();
       emitLong(T.make_desc(size,T.tag_string));
       defineLabel lab;
       emitstring s)

    fun emitReal (lab, f) = 
      (align();
       mark();
       emitLong(T.make_desc(size f,T.tag_reald));
       defineLabel lab;
       emitstring (implode(rev(explode f))))

    fun emitJmpTable(base,targets) = let
      val baseOff = Label.addrOf base
      fun emitOffset lab = emitLongX(Label.addrOf lab - baseOff)
    in
      app emitOffset targets
    end
  
    fun comment msg = ()
      
    fun init n = (CodeString.init n;  loc:=0)

    open Label
    
    fun emitMCInstr(instr,regmaps) = let
      datatype register = REG | FREG

      val regMap = hd regmaps
      val fregMap = hd(tl regmaps)
      fun rNum r = itow(Array.sub(regMap,r))
      fun fNum f = itow(Array.sub(fregMap,f))

      fun valOf(I.POSLAB(lab,k)) = Label.addrOf lab + k
	| valOf(I.NEGLAB(lab,k)) = k - Label.addrOf lab

      fun regmask regs = 
	Alpha32Mask.regmask (map (fn r => Array.sub(regMap, r)) regs)
	

      local 
	fun split i = let
	  val w = Word.fromInt i
	  val hi = Word.~>>(w, 0w16)
	  val lo = Word.andb(w, 0w65535)
	in
	  if lo <  0w32768 then (hi, lo) else (hi+0w1, lo-0w65536)
	end
      in
	fun high n = #1 (split n)
	fun low n  = #2 (split n)
      end

      fun immed16(I.IMMop i) =
	(if i < ~32768 orelse i > 32767 then
	   error ("immed16 - " ^ makestring i) else ();
	   (itow i & 0w65535))                     
	| immed16(I.LOLABop labexp) = (low (valOf labexp))
	| immed16(I.HILABop labexp) = (high (valOf labexp))
	| immed16(I.LOMASKop regs)  = (low (regmask regs)) 
	| immed16(I.HIMASKop regs)  = (high (regmask regs))
	| immed16 _ = error "immed16"

      fun immed8(I.IMMop i) =
	(if i < 0 orelse i > 255 then
	   error ("immed8 - " ^ makestring i) else ();
	   (itow i & 0w255))                     
	| immed8 _ = error "immed8"
	
      fun immed21 i =
	(if i < ~1048576 orelse i > 1048575 then
	   error ("immed21 - " ^ makestring i) else ();
	   itow i & 0w2097151)                      

	  
      fun Branch typ opcode (ra', lab) = let
	  val ra = (case typ of REG => rNum ra' | FREG => fNum ra')
	  val disp = (((Label.addrOf lab) - !loc - 4) quot 4)
	  val testdisp = immed21 disp
	  val lowdisp = testdisp & 0w65535
	in
	  emitWord lowdisp;
	  emitWord((itow opcode << 0w10) ++ (ra << 0w5) ++ (testdisp >> 0w16))
	end

      fun Memory typ opcode (ra', (rb, disp)) = let
	  val ra = (case typ of REG => rNum ra' | FREG => fNum ra')
	in
	  emitWord (itow disp);
	  emitWord((itow opcode << 0w10) ++ (ra << 0w5) ++ rNum rb)
	end

      fun MemoryW typ opcode (ra', (rb, disp)) = let
	  val ra = (case typ of REG => rNum ra' | FREG => fNum ra')
	in
	  emitWord disp;
	  emitWord((itow opcode << 0w10) ++ (ra << 0w5) ++ rNum rb)
	end

      fun Jump opcode bits (ra, (rb, disp)) =
	let
        (* disp is just a hint to the alpha - it does not affect destination *)
	  val _ = if disp <> 0 then error "Jump" else ()
	  val d = (itow disp & 0w16383) ++ (itow bits << 0w14)
	in
	  MemoryW REG opcode (ra, (rb, d))
	end
      
      fun Operate opcode func (ra, I.REGop rb, rc) =
 	   (emitWord((itow func << 0w5) ++ rNum rc);
	    emitWord((itow opcode << 0w10) ++ (rNum ra << 0w5) ++ rNum rb))
	| Operate opcode func (ra, opnd, rc) =
	  let
	    val testi = immed8 opnd
	  in
	    emitWord((testi << 0w13) ++ 0w4096 ++ 
					(itow func << 0w5) ++ rNum rc);
	    emitWord((itow opcode << 0w10) ++ (rNum ra << 0w5) ++ 
					      (testi >> 0w3))
	  end
	
      fun FOperate opcode func (ra, rb, rc) =
	 (emitWord((itow func << 0w5) ++ fNum rc);
	  emitWord((itow opcode << 0w10) ++ (fNum ra << 0w5) ++ fNum rb))

    in
	case instr of
	  I.DEFFREG _           => error "DEFFREG"
	| I.LDA (ra, rb, disp)	=> MemoryW REG 8   (ra, (rb, immed16 disp)) 
	| I.LDAH (ra, rb, disp) => MemoryW REG 9   (ra, (rb, immed16 disp))
	| I.LDL arg   		=> Memory REG 40  arg
	| I.LDQ arg  		=> Memory REG 41  arg
	| I.LDQ_U arg  		=> Memory REG 11  arg
	| I.STL arg   		=> Memory REG 44  arg
	| I.STQ arg  		=> Memory REG 45  arg
	| I.STQ_U arg  		=> Memory REG 15  arg

	| I.BR arg 		=> Branch REG 48 arg
	 
	| I.JMPL (arg, _) 	=> Jump 26 0 arg
	| I.JSR(arg,_,_)	=> Jump 26 1 arg 
	| I.BEQ arg 		=> Branch REG 57 arg
	| I.BGE arg 		=> Branch REG 62 arg
	| I.BGT arg 		=> Branch REG 63 arg
	| I.BLE arg 		=> Branch REG 59 arg
	| I.BLT arg 		=> Branch REG 58 arg
	| I.BNE arg 		=> Branch REG 61 arg
   	| I.BLBS arg 		=> Branch REG 60 arg
   	| I.BLBC arg 		=> Branch REG 56 arg

	| I.ZAP arg 		=> Operate 18 48 arg
	| I.ADDL arg 		=> Operate 16 0 arg
	| I.ADDLV arg 		=> Operate 16 64 arg
	| I.ADDQ arg 		=> Operate 16 32 arg
	| I.SUBL arg 		=> Operate 16 9 arg
	| I.SUBLV arg 		=> Operate 16 73 arg
	| I.SUBQ arg 		=> Operate 16 41 arg
	| I.MULL arg 		=> Operate 19 0 arg
	| I.MULLV arg 		=> Operate 19 64 arg
	| I.CMPULE arg 		=> Operate 16 61 arg
	| I.CMPULT arg 		=> Operate 16 29 arg
	| I.CMPEQ arg 		=> Operate 16 45 arg
	| I.CMPLE arg 		=> Operate 16 109 arg
	| I.CMPLT arg 		=> Operate 16 77 arg
 	| I.SGNXL (s,d)         => emitMCInstr((I.ADDL (s,I.REGop 31,d)),regmaps)
	    
	| I.AND arg 		=> Operate 17 0 arg
	| I.BIS arg 		=> Operate 17 32 arg
	| I.XOR arg 		=> Operate 17 64 arg
	| I.SRA arg 		=> Operate 18 60 arg
	| I.SRL arg 		=> Operate 18 52 arg
	| I.SLL arg 		=> Operate 18 57 arg
	  
	| I.INSBL arg 		=> Operate 18 11 arg
	| I.EXTBL arg 		=> Operate 18 6 arg
	| I.EXTQH arg 		=> Operate 18 122 arg
	| I.MSKBL arg 		=> Operate 18 2 arg
	| I.MSKLH arg 		=> Operate 18 98 arg
	  
	| I.LDT arg   		=> Memory FREG 35 arg 
	| I.STT arg   		=> Memory FREG 39 arg
	
	| I.FBEQ arg    	=> Branch FREG 49 arg
	| I.FBGE arg    	=> Branch FREG 54 arg
	| I.FBGT arg    	=> Branch FREG 55 arg
	| I.FBLE arg    	=> Branch FREG 51 arg
	| I.FBLT arg    	=> Branch FREG 50 arg
	| I.FBNE arg    	=> Branch FREG 53 arg
	  
	| I.CPYS arg		=> FOperate 23 32 arg
	| I.CPYSN arg		=> FOperate 23 33 arg
	  
	| I.CMPTEQ arg 		=> FOperate 22 0x5a5 arg
	| I.CMPTLT arg 		=> FOperate 22 0x5a6 arg
	| I.CMPTLE arg 		=> FOperate 22 0x5a7 arg
	  
	| I.ADDT arg		=> FOperate 22 0x5a0 arg
	| I.SUBT arg		=> FOperate 22 0x5a1 arg
	| I.MULT arg		=> FOperate 22 0x5a2 arg
	| I.DIVT arg		=> FOperate 22 0x5a3 arg  (* chopped rounding *)
	  
	| I.CVTQT arg		=> FOperate 22 0x0be arg
	| I.CVTTQ arg		=> FOperate 22 0x02f arg (*rounds towards zero*)

	| I.TRAPB      		=> Memory REG 24 (I.C.zeroR, (I.C.zeroR, 0))
	  
	| I.LADDR _ 		=> error "emitInstr: LADDR"
	| I.BRANCH _   		=> error "emitinstr: BRANCH"
    end

    fun emitInstr(i,regmaps) = 	 emitMCInstr(i,regmaps)
  end			      






  
