(* alpha32Asm.sml
 *
 * COPYRIGHT (c) 1996 Bell Laboratories.
 *)

structure Alpha32AsmEmitter : EMITTER_NEW = struct
  structure I = Alpha32Instr
  structure T = System.Tags

  fun ms n = if n<0 then ("-"^Int.toString (~n)) else Int.toString n

  fun error msg = ErrorMsg.impossible ("Alpha32AsEmitter." ^ msg)

  fun emit s = output(!AsmStream.asmOutStream,s)

  val loc = ref 0
 
  fun incLoc n = loc := (!loc) + n

  fun emitLong n = (emit (".long\t" ^ ms n ^ "\n"); incLoc 4)

  fun align () = case Word.andb(Word.fromInt(!loc), 0w7) of
		   0w0 => ()
		 | 0w4 => emitLong 0
		 | _ => error "align"

  fun mark () =
    emitLong (T.make_desc((!loc + 4) quot 4,T.tag_backptr))

  fun emitString (lab, sz, s) = (
	mark();
	emitLong(T.make_desc(sz,T.tag_string));
	emit (concat[
	  Label.nameOf lab, ":\t.ascii ", AsmUtil.printableString s, "\n"]);
	incLoc (size s))

  fun emitReal (lab,f) = (
	align();
	mark ();
	emitLong(T.make_desc(size f,T.tag_reald));
	emit(Label.nameOf lab ^ ":" ^ "\t.double " ^ f ^ "\n");
	incLoc 8)

  fun emitJmpTable(base,targets) = 
    (emit(".jumpTable\n" ^ Label.nameOf base ^ ":\n");
     app (fn lab => emit("\t\t" ^ Label.nameOf lab ^ "\n")) targets;
     incLoc (4*length targets))

  fun defineLabel(lab as Label.Label{comment,...}) = 
        (emit(Label.nameOf lab ^ ":\t\t/* at " ^ AsmUtil.itoa(!loc));
	 emit(if comment = "" then "*/\n"
	      else  " = lvar: " ^ comment ^ "*/\n"))

  fun comment msg = emit ("/*" ^ msg ^ "*/\n")

  fun init size = (comment ("Code Size = " ^ ms size);
		   emit ".set\tnoat\n";
		   loc := 0)

    
  fun emitInstr(instr,regmaps) = let
    datatype register = REG | FREG

    val regmap = hd regmaps
    val fregmap = hd(tl regmaps)
      
    fun comma() 	    	= emit ", "
    fun tab()               	= emit "\t"
    fun eReg (i) 		= emit ("$" ^ ms(Array.sub(regmap, i))) 
    fun eFreg(f)         	= emit("$f" ^ ms(Array.sub(fregmap, f)))
    fun eLabel lab       	= emit (Label.nameOf lab)
    fun newline ()          	= emit "\n"
    fun eDisp(rd, 0) 		= (emit "("; eReg rd; emit ")")
      | eDisp(rd, disp) 	= (emit (ms disp);
				   emit "(";
				   eReg rd;
				   emit ")")
    fun emitLExp(lab,k:int) = 
	  (emit "("; eLabel lab; emit "+"; emit(ms k); emit ")")
    fun emitLExp(I.POSLAB(lab,k)) =
	  (emit"("; emit(ms k); emit"+"; eLabel lab; emit")" )
      | emitLExp(I.NEGLAB(lab,k)) =
	  (emit"("; emit(ms k); emit"-"; eLabel lab; emit")" )

    fun valOf(I.POSLAB(lab,k)) = Label.addrOf lab + k
      | valOf(I.NEGLAB(lab,k)) = k - Label.addrOf lab

    fun eOperand (I.REGop r) = eReg r
      | eOperand (I.IMMop n) = emit (ms n)
      | eOperand (I.HIMASKop l) = (emit "HI("; app eReg l; emit ")")
      | eOperand (I.LOMASKop l) = (emit "LO("; app eReg l; emit ")")
      | eOperand (I.LOLABop l)  = (emit "LO("; emitLExp l;emit ")")
      | eOperand (I.HILABop l) =  (emit "HI("; emitLExp l; emit ")")

    fun eMemFormat REG (reg, disp) =
          (eReg reg; comma(); eDisp disp)
      | eMemFormat FREG (freg, disp) =
          (eFreg freg; comma(); eDisp disp)

    fun eBrFormat REG (reg, lab) =
          (eReg reg; comma(); eLabel lab)
      | eBrFormat FREG (freg, lab) =
	  (eFreg freg; comma(); eLabel lab)

    fun eOpFormat (rs, opnd, rd) =
      (eReg rs; comma(); eOperand opnd; comma(); eReg rd)

    fun eFOpFormat (f1, f2, fd) = 
      (eFreg f1; comma(); eFreg f2; comma(); eFreg fd)

    fun eFCVTOpFormat (f1, f2, fd) =
      (eFreg f1; comma(); eFreg f2)
 
    fun emitLDA (r1, r2, opnd) =
      (eReg r1; comma(); eOperand opnd; emit "("; eReg r2; emit ")")
      
  in
      ((case instr of
	  I.DEFFREG f   => (emit "\tdeffreg\t"; eFreg f)
	| I.LDA  arg	=> (emit "\tlda\t"; emitLDA arg)
	| I.LDAH arg 	=> (emit "\tldah\t"; emitLDA arg)
	| I.LDL arg 	=> (emit "\tldl\t"; eMemFormat REG arg)
	| I.LDQ  arg 	=> (emit "\tldq\t"; eMemFormat REG arg)
	| I.LDQ_U arg 	=> (emit "\tldq_u\t"; eMemFormat REG arg)
	| I.STL arg 	=> (emit "\tstl\t"; eMemFormat REG arg)
	| I.STQ arg	=> (emit "\tstq\t"; eMemFormat REG arg)
	| I.STQ_U arg 	=> (emit "\tstq_u\t"; eMemFormat REG arg)
	    
	| I.BR arg 	=> (emit "\tbr\t"; eBrFormat REG arg)
	| I.JMPL(arg,_)	=> (emit "\tjmp\t"; eMemFormat REG arg)

	| I.JSR(arg,_,_)=> (emit "\tjsr\t"; eMemFormat REG arg)
	| I.BEQ arg 	=> (emit "\tbeq\t"; eBrFormat REG arg)
	| I.BGE arg 	=> (emit "\tbge\t"; eBrFormat REG arg)
	| I.BGT arg 	=> (emit "\tbgt\t"; eBrFormat REG arg)
	| I.BLE arg 	=> (emit "\tble\t"; eBrFormat REG arg)
	| I.BLT arg 	=> (emit "\tblt\t"; eBrFormat REG arg)
	| I.BNE arg 	=> (emit "\tbne\t"; eBrFormat REG arg)
   	| I.BLBC arg 	=> (emit "\tblbc\t"; eBrFormat REG arg)
   	| I.BLBS arg 	=> (emit "\tblbs\t"; eBrFormat REG arg)
	| I.FBEQ arg 	=> (emit "\tfbeq\t"; eBrFormat FREG arg)
	| I.FBGE arg 	=> (emit "\tfbge\t"; eBrFormat FREG arg)
	| I.FBGT arg 	=> (emit "\tfbgt\t"; eBrFormat FREG arg)
	| I.FBLE arg 	=> (emit "\tfble\t"; eBrFormat FREG arg)
	| I.FBLT arg 	=> (emit "\tfblt\t"; eBrFormat FREG arg)
	| I.FBNE arg 	=> (emit "\tfbne\t"; eBrFormat FREG arg)

	| I.ZAP arg   	=> (emit"\tzap\t";  eOpFormat arg)
	| I.ADDL arg  	=> (emit"\taddl\t";  eOpFormat arg)
	| I.ADDLV arg  	=> (emit"\taddlv\t";  eOpFormat arg)
	| I.ADDQ arg  	=> (emit"\taddq\t";  eOpFormat arg)
	| I.SUBL arg  	=> (emit"\tsubl\t";  eOpFormat arg)
	| I.SUBLV arg  	=> (emit"\tsublv\t";  eOpFormat arg)
	| I.SUBQ arg  	=> (emit"\tsubq\t";  eOpFormat arg)
	| I.MULL arg  	=> (emit"\tmull\t";  eOpFormat arg)
	| I.MULLV arg  	=> (emit"\tmullv\t";  eOpFormat arg)
	| I.CMPULE arg 	=> (emit"\tcmpule\t";  eOpFormat arg)
	| I.CMPULT arg 	=> (emit"\tcmpult\t";  eOpFormat arg)
	| I.CMPEQ arg  	=> (emit"\tcmpeq\t";  eOpFormat arg)
	| I.CMPLE arg  	=> (emit"\tcmple\t";  eOpFormat arg)
	| I.CMPLT arg  	=> (emit"\tcmplt\t";  eOpFormat arg)
 	| I.SGNXL (s,d) => (emit"\taddl\t"; eReg s; comma();
 			                    emit "$31"; comma(); eReg d;
 					    emit "\t/* sign extend */")
	    
	| I.AND arg  	=> (emit"\tand\t";  eOpFormat arg)
	| I.BIS arg  	=> (emit"\tbis\t";  eOpFormat arg)
	| I.XOR arg  	=> (emit"\txor\t";  eOpFormat arg)
	| I.SRA arg  	=> (emit"\tsra\t";  eOpFormat arg)
	| I.SRL arg  	=> (emit"\tsrl\t";  eOpFormat arg)
	| I.SLL arg  	=> (emit"\tsll\t";  eOpFormat arg)
	    
	| I.INSBL arg  	=> (emit"\tinsbl\t";  eOpFormat arg)
	| I.EXTBL arg  	=> (emit"\textbl\t";  eOpFormat arg)
	| I.EXTQH arg  	=> (emit"\textqh\t";  eOpFormat arg)
	| I.MSKBL arg  	=> (emit"\tmskbl\t";  eOpFormat arg)
	| I.MSKLH arg  	=> (emit"\tmsklh\t";  eOpFormat arg)

	| I.LDT arg 	=> (emit "\tldt\t"; eMemFormat FREG arg)
	| I.STT arg 	=> (emit "\tstt\t"; eMemFormat FREG arg)
	    
	| I.CPYS arg  	=> (emit"\tcpys\t";  eFOpFormat arg)
	| I.CPYSN arg  	=> (emit"\tcpysn\t";  eFOpFormat arg)
	| I.CVTQT arg  	=> (emit"\tcvtqt\t";  eFCVTOpFormat arg)
	| I.CVTTQ arg  	=> (emit"\tcvttqc\t";  eFCVTOpFormat arg)
	| I.CMPTEQ arg 	=> (emit"\tcmpteq/su\t"; eFOpFormat arg)    
	| I.CMPTLT arg 	=> (emit"\tcmptlt/su\t"; eFOpFormat arg)    
	| I.CMPTLE arg 	=> (emit"\tcmptle/su\t"; eFOpFormat arg)    
	| I.ADDT arg  	=> (emit"\taddt/su\t";  eFOpFormat arg)
	| I.SUBT arg  	=> (emit"\tsubt/su\t";  eFOpFormat arg)
	| I.MULT arg  	=> (emit"\tmult/su\t";  eFOpFormat arg)
	| I.DIVT arg  	=> (emit"\tdivt/su\t";  eFOpFormat arg)
	    
	| I.TRAPB 	=> emit"\ttrapb\t"

	| I.LADDR(rs,opnd,rd) => 
	      		   (emit"\tladdr\t"; eReg rd; comma(); 
			    emitLExp opnd; eReg rs)
			    
	| I.BRANCH _ 	=> error "emitInstr: BRANCH" 
      (* endcase *));
      emit "\n";
      incLoc 4)
  end

end



