/* run-ml.c
 *
 * COPYRIGHT (c) 1993 by AT&T Bell Laboratories.
 */

#include "ml-base.h"
#include "ml-limits.h"
#include "ml-values.h"
#include "vproc-state.h"
#include "reg-mask.h"
#include "ml-state.h"
#include "tags.h"
#include "ml-request.h"
#include "ml-objects.h"
#include "ml-globals.h"
#include "ml-signals.h"
#include "c-library.h"

/* local functions */
PVT void UncaughtExn (ml_val_t e);


/* ApplyMLFn:
 *
 * Apply the ML closure f to arg and return the result.  If the flag useCont
 * is set, then the ML state has already been initialized with a return
 * continuation (by SaveCState).
 */
ml_val_t ApplyMLFn (ml_state_t *msp, ml_val_t f, ml_val_t arg, bool_t useCont)
{
    int		i;

    InitMLState (msp);

  /* initialize the calling context */
    msp->ml_exnCont	= PTR_CtoML(handle_v+1);
    msp->ml_varReg      = ML_unit;
    msp->ml_arg		= arg;
    if (! useCont)
	msp->ml_cont	= PTR_CtoML(return_c);
    msp->ml_closure	= f;
    msp->ml_pc		=
    msp->ml_linkReg	= GET_CODE_ADDR(f);

    RunML (msp);

    return msp->ml_arg;

} /* end of ApplyMLFn */


/* RaiseMLExn:
 *
 * Modify the ML state, so that the given exception will be raised
 * when ML is resumed.
 */
void RaiseMLExn (ml_state_t *msp, ml_val_t exn)
{
    ml_val_t	kont = msp->ml_exnCont;

/** NOTE: we should have a macro defined in ml-state.h for this **/
    msp->ml_arg		= exn;
    msp->ml_closure	= kont;
    msp->ml_cont	= ML_unit;
    msp->ml_pc		=
    msp->ml_linkReg	= GET_CODE_ADDR(kont);

} /* end of RaiseMLExn. */


/* RunML:
 */
void RunML (ml_state_t *msp)
{
    int		request;
    vproc_state_t *vsp = msp->ml_vproc;

    while (TRUE) {

	request = restoreregs(msp);

	if (request == REQ_GC) {
	    if (vsp->vp_handlerPending) { /* this is really a signal */
	      /* check for GC */
		if (NeedGC (msp, 4*ONE_K))
		    InvokeGC (msp, 0);
	      /* invoke the ML signal handler */
		ChooseSignal (vsp);
		msp->ml_arg		= MakeHandlerArg (msp, sigh_resume);
		msp->ml_cont		= PTR_CtoML(sigh_return_c);
		msp->ml_exnCont		= PTR_CtoML(handle_v+1);
		msp->ml_closure		= DEREF(MLSignalHandler);
		msp->ml_pc		=
		msp->ml_linkReg		= GET_CODE_ADDR(msp->ml_closure);
		vsp->vp_inSigHandler	= TRUE;
		vsp->vp_handlerPending	= FALSE;
	    }
#ifdef SOFT_POLL
	    else if (msp->ml_pollPending && !msp->ml_inPollHandler) { 
	      /* this is a poll event */
#if defined(MP_SUPPORT) && defined(MP_GCPOLL)
	      /* Note: under MP, polling is used for GC only */
#ifdef POLL_DEBUG
SayDebug ("run-ml: poll event\n");
#endif
	        msp->ml_pollPending = FALSE;
	        InvokeGC (msp,0);
#else
	      /* check for GC */
		if (NeedGC (msp, 4*ONE_K))
		    InvokeGC (msp, 0);
		msp->ml_arg		= MakeResumeCont(msp, pollh_resume);
		msp->ml_cont		= PTR_CtoML(pollh_return_c);
		msp->ml_exnCont		= PTR_CtoML(handle_v+1);
		msp->ml_closure		= DEREF(MLPollHandler);
		msp->ml_pc		=
		msp->ml_linkReg		= GET_CODE_ADDR(msp->ml_closure);
		msp->ml_inPollHandler	= TRUE;
		msp->ml_pollPending	= FALSE;
#endif /* MP_SUPPORT */
	    } 
#endif /* SOFT_POLL */
	    else
	        InvokeGC (msp, 0);
	}
	else {
#ifdef BASE_INDX
	    msp->ml_baseReg = ML_unit;  /* not a live root */
#endif
	    switch (request) {
	      case REQ_RETURN:
	      /* do a minor collection to clear the store list; we set the PC to
	       * a non-root value, incase the minor collection triggers a major
	       * collection.
	       */
		msp->ml_pc = ML_unit;
		InvokeGC (msp, 0);
		return;

	      case REQ_EXN: /* an UncaughtExn exception */
		UncaughtExn (msp->ml_arg);
		return;

	      case REQ_FAULT: /* a hardware fault */
		RaiseMLExn (msp, msp->ml_faultExn);
		break;

	      case REQ_BIND_CFUN:
		msp->ml_arg = BindCFun (
		    REC_SELPTR(char, msp->ml_arg, 0),
		    REC_SELPTR(char, msp->ml_arg, 1));
		SETUP_RETURN(msp);
		break;

	      case REQ_CALLC: {
		    ml_val_t    (*f)(), arg;

		    SETUP_RETURN(msp);
		    msp->ml_liveRegMask = RET_MASK;
		    if (NeedGC (msp, 8*ONE_K))
			InvokeGC (msp, 0);

#ifdef INDIRECT_CFUNC
		    f = ((cfunc_binding_t *)REC_SELPTR(Word_t, msp->ml_arg, 0))->cfunc;
#else
		    f = (cfunc_t) REC_SELPTR(Word_t, msp->ml_arg, 0);
/*SayDebug("CALLC: %#x (%#x)\n", f, arg);*/
#endif
		    arg = REC_SEL(msp->ml_arg, 1);
		    msp->ml_arg = (*f)(msp, arg);
		} break;

	      case REQ_ALLOC_STRING:
		msp->ml_arg = ML_AllocString (msp, INT_MLtoC(msp->ml_arg));
		SETUP_RETURN(msp);
		break;

	      case REQ_ALLOC_BYTEARRAY:
		msp->ml_arg = ML_AllocBytearray (msp, INT_MLtoC(msp->ml_arg));
		SETUP_RETURN(msp);
		break;

	      case REQ_ALLOC_REALDARRAY:
		msp->ml_arg = ML_AllocRealdarray (msp, INT_MLtoC(msp->ml_arg));
		SETUP_RETURN(msp);
		break;

	      case REQ_ALLOC_ARRAY:
		msp->ml_arg = ML_AllocArray (msp,
		    REC_SELINT(msp->ml_arg, 0), REC_SEL(msp->ml_arg, 1));
		SETUP_RETURN(msp);
		break;

	      case REQ_ALLOC_VECTOR:
		msp->ml_arg = ML_AllocVector (msp,
		    REC_SELINT(msp->ml_arg, 0), REC_SEL(msp->ml_arg, 1));
		SETUP_RETURN(msp);
		break;

	      case REQ_SIG_RETURN:
#ifdef SIGNAL_DEBUG
SayDebug("REQ_SIG_RETURN: arg = %#x, pending = %d, inHandler = %d\n",
msp->ml_arg, vsp->vp_handlerPending, vsp->vp_inSigHandler);
#endif
	      /* throw to the continuation */
		SETUP_THROW(msp, msp->ml_arg, ML_unit);
	      /* note that we are exiting the handler */
		vsp->vp_inSigHandler = FALSE;
		break;

#ifdef SOFT_POLL
	      case REQ_POLL_RETURN:
	      /* throw to the continuation */
		SETUP_THROW(msp, msp->ml_arg, ML_unit);
	      /* note that we are exiting the handler */
		msp->ml_inPollHandler = FALSE;
		ResetPollLimit (msp);
		break;
#endif

#ifdef SOFT_POLL
	      case REQ_POLL_RESUME:
#endif
	      case REQ_SIG_RESUME:
#ifdef SIGNAL_DEBUG
SayDebug("REQ_SIG_RESUME: arg = %#x\n", msp->ml_arg);
#endif
		LoadResumeState (msp);
		break;

	      case REQ_BUILD_LITERALS:
		SETUP_RETURN(msp);
		msp->ml_liveRegMask = RET_MASK;
		if (NeedGC (msp, 32*ONE_K))
		    InvokeGC (msp, 0);
		msp->ml_arg = BuildLiterals (msp, msp->ml_arg);
		break;

	      default:
		Die ("unknown request code = %d", request);
		break;
	    } /* end switch */
	}
    } /* end of while */

} /* end of RunML */


/* UncaughtExn:
 * Handle an uncaught exception.
 */
PVT void UncaughtExn (ml_val_t e)
{
    ml_val_t	val = REC_SEL(e, 1);
    ml_val_t	name = REC_SEL(REC_SEL(e, 0), 0);

    Error ("Uncaught exception %.*s with ", OBJ_LEN(name), PTR_MLtoC(char, name));

    if (isUNBOXED(val))
	Error ("%d\n", INT_MLtoC(val));
    else {
	Word_t	tag = OBJ_TAG(val);
	if (tag == DTAG_string)
	    Error ("\"%.*s\"\n", OBJ_LEN(val), PTR_MLtoC(char, val));
	else
	    Error ("<unknown>\n");
    }

    Exit (1);

} /* end of UncaughtExn */
