(*
 * sys/compiler.sig: signature matching SML/NJ's visual compiler interface
 *
 *   Copyright (c) 1992, 1995 by AT&T Bell Laboratories
 *
 * contact: Matthias Blume (blume@cs.princeton.edu)
 *)
signature AST = sig

    type fixity
    type symbol
    val infixleft:  int -> fixity
    val infixright: int -> fixity
    type 'a fixitem

    type srcpos (* to mark positions in files *)
    sharing type srcpos = int

    (* EXPRESSIONS *)

    datatype exp =
	VarExp of symbol list		(* variable *)
      | FnExp of rule list		(* abstraction *)
      | FlatAppExp of { item: exp, region: srcpos * srcpos,
		        fixity: symbol option } list
      | AppExp of { function: exp, argument: exp }
					(* application *)
      | CaseExp of { expr: exp, rules: rule list }
					(* case expression *)
      | LetExp of { dec: dec, expr: exp }
					(* let expression *)
      | SeqExp of exp list		(* sequence of expressions *)
      | IntExp of string		(* integer *)
      | WordExp of string		(* word literal *)
      | RealExp of string		(* floating point coded as a string *)
      | StringExp of string		(* string *)
      | CharExp of string		(* char *)
      | RecordExp of (symbol * exp) list
					(* record *)
      | ListExp of exp list		(* list (derived form) *)
      | TupleExp of exp list		(* tuple (derived form) *)
      | SelectorExp of symbol		(* selector of a record field *)
      | ConstraintExp of { expr: exp, constraint: ty }
					(* type constraint *)
      | HandleExp of { expr: exp, rules: rule list }
					(* exception handler *)
      | RaiseExp of exp			(* raise an exception *)
      | IfExp of { test: exp, thenCase: exp, elseCase: exp }
					(* if expression (derived form) *)
      | AndalsoExp of exp * exp		(* andalso (derived form) *)
      | OrelseExp of exp * exp		(* orelse (derived form) *)
      | VectorExp of exp list		(* vector *)
      | WhileExp of { test: exp, expr: exp }
					(* while (derived form) *)
      | MarkExp of exp * (srcpos * srcpos)

    (* RULE for case functions and exception handler *)
    and rule = Rule of { pat: pat, exp: exp }

    (* PATTERN *)
    and pat =
	WildPat				(* empty pattern *)
      | VarPat of symbol list		(* variable pattern *)
      | IntPat of string		(* integer *)
      | WordPat of string		(* word literal *)
      | RealPat of string		(* floating point number *)
      | StringPat of string		(* string *)
      | CharPat of string		(* char *)
      | RecordPat of { def: (symbol * pat) list, flexibility: bool }
					(* record *)
      | ListPat of pat list		(* list *)
      | TuplePat of pat list		(* tuple *)
      | FlatAppPat of { item: pat, region: srcpos * srcpos,
		        fixity:symbol option } list
      | AppPat of { constr: pat, argument: pat }
					(* application *)
      | ConstraintPat of { pattern: pat, constraint: ty }
					(* constraint *)
      | LayeredPat of { varPat: pat, expPat: pat }
					(* as expressions *)
      | VectorPat of pat list		(* vector pattern *)
      | MarkPat of pat * (srcpos * srcpos)
      | OrPat of pat list		(* or-pattern *)

    (* STRUCTURE EXPRESSION *)
    and strexp =
	VarStr of symbol list		(* variable structure *)
      | StructStr of dec		(* defined structure *)
      | AppStr of symbol list * (strexp * bool) list
					(* application *)
      | LetStr of dec * strexp		(* let in structure *)
      | MarkStr of strexp * (srcpos * srcpos)

    (* FUNCTOR EXPRESSION *)
    and fctexp =
	VarFct of symbol list * fsigexp option
					(* functor variable *)
      | FctFct of  {			(* definition of a functor *)
		    params:     (symbol option * sigexp) list,
		    body:       strexp,
		    constraint: sigexp option
		   }
      | LetFct of dec * fctexp
      | AppFct of symbol list * (strexp * bool) list * fsigexp option
					(* application *)
      | MarkFct of fctexp * (srcpos * srcpos)

    (* SIGNATURE EXPRESSION *)
    and sigexp =
	VarSig of symbol		(* signature variable *)
      | SigSig of spec list		(* defined signature *)
      | MarkSig of sigexp * (srcpos * srcpos)

    (* FUNCTOR SIGNATURE EXPRESSION *)
    and fsigexp =
	VarFsig of symbol		(* funsig variable *)
      | FsigFsig of {param: (symbol option * sigexp) list, def:sigexp}
					(* defined funsig *)
      | MarkFsig of fsigexp * (srcpos * srcpos)

    (* SPECIFICATION FOR SIGNATURE DEFINITIONS *)
    and spec =
	StrSpec of (symbol * sigexp) list (* structure *)
      | TycSpec of ((symbol * tyvar list * ty option) list * bool)
      | FctSpec of (symbol * fsigexp) list
					(* functor *)
      | ValSpec of (symbol * ty) list	(* value *)
      | DataSpec of { datatycs: db list, withtycs: tb list }
					(* datatype *)
      | ExceSpec of (symbol * ty option) list
					(* exception *)
      | FixSpec of  { fixity: fixity, ops: symbol list }
					(* fixity *)
      | ShareSpec of symbol list list	(* structure sharing *)
      | ShatycSpec of symbol list list	(* type sharing *)
      | LocalSpec of spec list * spec list
					(* local specif *)
      | IncludeSpec of symbol		(* include specif *)
      | OpenSpec of symbol list list	(* open structures *)
      | MarkSpec of spec * (srcpos * srcpos)

    (* DECLARATIONS (let and structure) *)
    and dec =
	ValDec of vb list		(* values *)
      | ValrecDec of rvb list		(* recursive values *)
      | FunDec of fb list		(* recurs functions *)
      | TypeDec of tb list		(* type dec *)
      | DatatypeDec of { datatycs: db list, withtycs: tb list }
					(* datatype dec *)
      | AbstypeDec of { abstycs: db list, withtycs: tb list, body: dec }
					(* abstract type *)
      | ExceptionDec of eb list		(* exception *)
      | StrDec of strb list		(* structure *)
      | AbsDec of strb list		(* abstract struct *)
      | FctDec of fctb list		(* functor *)
      | SigDec of sigb list		(* signature *)
      | FsigDec of fsigb list		(* funsig *)
      | LocalDec of dec * dec		(* local dec *)
      | SeqDec of dec list		(* sequence of dec *)
      | OpenDec of symbol list list	(* open structures *)
      | OvldDec of symbol * ty * exp list
					(* overloading (internal) *)
      | FixDec of { fixity: fixity, ops: symbol list }
					(* fixity *)
      | ImportDec of string list	(* import (unused) *)
      | MarkDec of dec * (srcpos * srcpos)

  (* VALUE BINDINGS *)
    and vb =
	Vb of {pat:pat, exp:exp}
      | MarkVb of vb * (srcpos * srcpos)

    (* RECURSIVE VALUE BINDINGS *)
    and rvb =
	Rvb of { var:symbol, exp:exp, resultty: ty option,
		 fixity: (symbol * (srcpos*srcpos)) option}
      | MarkRvb of rvb * (srcpos * srcpos)

    (* RECURSIVE FUNCTIONS BINDINGS *)
    and fb =
	Fb of clause list
      | MarkFb of fb * (srcpos * srcpos)

    (* CLAUSE: a definition for a single pattern in a function binding *)
    and clause = Clause of { pats: { item: pat, region: srcpos * srcpos,
				     fixity: symbol option } list, 
			    resultty: ty option, exp:exp }

    (* TYPE BINDING *)
    and tb =
	Tb of { tyc: symbol, def: ty, tyvars: tyvar list }
      | MarkTb of tb * (srcpos * srcpos)

    (* DATATYPE BINDING *)
    and db =
	Db of { tyc: symbol, tyvars: tyvar list,
	       def: (symbol * ty option) list }
      | MarkDb of db * (srcpos * srcpos)

    (* EXCEPTION BINDING *)
    and eb =
	EbGen of { exn: symbol, etype: ty option }  (* Exception definition *)
      | EbDef of { exn: symbol, edef: symbol list } (* defined by equality *)
      | MarkEb of eb * (srcpos * srcpos)

    (* STRUCTURE BINDING *)
    and strb =
	Strb of { name: symbol, def: strexp, constraint: sigexp option }
      | MarkStrb of strb * (srcpos * srcpos)

    (* FUNCTOR BINDING *)
    and fctb =
	Fctb of { name: symbol, def: fctexp }
      | MarkFctb of fctb * (srcpos * srcpos)

    (* SIGNATURE BINDING *)
    and sigb =
	Sigb of { name: symbol, def: sigexp }
      | MarkSigb of sigb * (srcpos * srcpos)

    (* FUNSIG BINDING *)
    and fsigb =
	Fsigb of { name: symbol, def: fsigexp }
      | MarkFsigb of fsigb * (srcpos * srcpos)

    (* TYPE VARIABLE *)
    and tyvar =
	Tyv of symbol
      | MarkTyv of tyvar * (srcpos * srcpos)

    (* TYPES *)
    and ty = 
	VarTy of tyvar			(* type variable *)
      | ConTy of symbol list * ty list	(* type constructor *)
      | RecordTy of (symbol * ty) list 	(* record *)
      | TupleTy of ty list		(* tuple *)
      | MarkTy of ty * (srcpos * srcpos)
end (* structure Ast *)

signature ENVIRONMENT = sig
    type environment
    type staticEnv
    type dynenv
    type symenv
    type symbol

    val primEnv: staticEnv

    val emptyEnv:         environment
    val staticPart:       environment -> staticEnv
    val dynamicPart:      environment -> dynenv
    val symbolicPart:     environment -> symenv
    val mkenv:            { static: staticEnv,
			    dynamic: dynenv,
			    symbolic: symenv } -> environment
    val layerStatic:      staticEnv * staticEnv -> staticEnv
    val layerSymbolic:    symenv * symenv -> symenv
    val layerEnv:         environment * environment -> environment
    val concatEnv:        environment * environment -> environment
    val filterStaticEnv:  staticEnv * symbol list -> staticEnv
    val filterEnv:        environment * symbol list -> environment
    val catalogEnv:       staticEnv -> symbol list
    val consolidateEnv:   environment -> environment
    val consolidateStatic: staticEnv -> staticEnv
    val consolidateSymbolic: symenv -> symenv

    datatype cmEnv =
	CM_NONE
      | CM_ENV of symbol -> cmEnv

    val cmEnvOfModule: staticEnv -> symbol -> cmEnv

    (* Core env *)
    val coreEnvRef:      { get: unit->staticEnv, set: staticEnv->unit } 

    (* interactive top level env *)
    val topLevelEnvRef:  { get: unit->environment, set: environment->unit }

    (* pervasive environment *)
    val pervasiveEnvRef: { get: unit->environment, set: environment->unit }
end


signature COMPILER = sig

    structure Control: sig
	structure Print: sig
	    val out:       { say: string -> unit, flush: unit -> unit } ref
	    val say:       string -> unit
	    val flush:     unit -> unit
	    val linewidth: int ref
	end
    end

  structure Ast: AST

  structure Symbol: sig
      type symbol
      datatype namespace =
	  VALspace | TYCspace | SIGspace 
	| STRspace | FCTspace | FIXspace 
	| LABspace | TYVspace | FSIGspace

      val eq: symbol * symbol -> bool
      val symbolCMLt: symbol * symbol -> bool
      val nameSpace: symbol -> namespace
      val name: symbol -> string
      val symbolToString: symbol -> string
      val strSymbol: string -> symbol
      val sigSymbol: string -> symbol
      val fctSymbol: string -> symbol
      val fsigSymbol: string -> symbol
  end

  structure Index: sig
      val openIndexFile: string -> outstream option
      val indexing: bool ref
  end

  structure Source: sig
      type inputSource
      type charpos

      val newSource:
	  string * int * instream * bool *
	  {
	   consumer: string -> unit,
	   linewidth: int,
	   flush: unit -> unit
	  } *
	  outstream option 
	  -> inputSource

      val closeSource: inputSource -> unit

      val filepos: inputSource -> charpos -> string * int * int
  end

  sharing type Source.charpos = Ast.srcpos

  structure Environment: ENVIRONMENT
  structure BareEnvironment: ENVIRONMENT 

  sharing
  type Symbol.symbol = Ast.symbol = Environment.symbol = BareEnvironment.symbol

  structure CoerceEnv: sig
      val b2e: BareEnvironment.environment -> Environment.environment
      val e2b: Environment.environment -> BareEnvironment.environment
  (* should perhaps have coercers to work on static parts too *)
  end

  structure PersStamps: sig
      type persstamp
      val stringToStamp: string -> persstamp
      val stampToString: persstamp -> string
      val less : persstamp * persstamp -> bool
  end

  structure DynamicEnv: sig
      type object
      type dynenv
      exception Unbound
      val look: dynenv -> PersStamps.persstamp -> object
  end

  sharing
  type DynamicEnv.dynenv = Environment.dynenv =BareEnvironment.dynenv

  structure ErrorMsg: sig
      type errors
      val errors: Source.inputSource -> errors
      val anyErrors: errors -> bool
  end

  structure Interact: sig
      val installCompManager:
	  (Ast.dec *
	   { get: unit -> BareEnvironment.environment,
	     set: BareEnvironment.environment -> unit } *
	   { get: unit -> BareEnvironment.environment,
	     set: BareEnvironment.environment -> unit }
	   -> unit) option
	  -> unit
  end

  structure Compile: sig
      type lvar
      type absyn
      type pid = PersStamps.persstamp
      type lambda			(* normal lambda *)
      type clambda			(* canonical lambda *)
      type plambda			(* pickled lambda *)
      type obj = System.Unsafe.object
      type lsegments = { l0: lambda, ln: lambda list }
      type csegments = { c0: string, cn: string list }
	
      exception Compile of string

      val pickle: clambda -> plambda
      val unpickle: plambda -> clambda

      val parse: Source.inputSource -> Ast.dec

      val parseOne:
	  Source.inputSource -> unit -> Ast.dec option

      val elaborate:
	  {
	   errors: ErrorMsg.errors,
	   corenv: BareEnvironment.staticEnv,
	   compenv: BareEnvironment.staticEnv,
	   transform: absyn -> absyn,	(* TEMPORARY *)
	   ast: Ast.dec
	  } ->
	  {
	   absyn: absyn, 
	   newenv: BareEnvironment.staticEnv,
	   exportLexp: lambda,
	   exportPid: pid option,
	   staticPid: pid
	  }

      val makePid: BareEnvironment.staticEnv -> pid

      val instrument:
	  {
	   source: Source.inputSource,
	   corenv: BareEnvironment.staticEnv,
	   compenv: BareEnvironment.staticEnv
	  } ->
	  absyn -> absyn
		
      val translate:
	  {
	   errors: ErrorMsg.errors,
	   corenv: BareEnvironment.staticEnv,
	   absyn: absyn,
	   exportLexp: lambda,
	   exportPid: pid option,
	   statenv: BareEnvironment.staticEnv
	  } ->
	  {
	   genLambda: lambda option list -> lambda,
	   imports: pid list
	  }

      val symDelta: pid option * clambda option
	  -> Environment.symenv

      val inline: { genLambda: lambda option list -> lambda,
		    imports: pid list,
		    symenv: Environment.symenv }
	  -> lambda

      val split: { lambda: lambda, enable: bool }
	  -> { lambda_e: lsegments, lambda_i: clambda option, pid: pid }

      val architecture : string
      val codegen: { errors: ErrorMsg.errors, lambda: lsegments } -> csegments
		
      val applyCode: csegments -> obj vector -> obj

      (* the functions above raise ONLY the exception Compile;
       execute can raise other exceptions *)

      val execute:
	  {
	   executable: obj vector -> obj,
	   imports: pid list,
	   exportPid: pid option,
	   dynenv: Environment.dynenv
	  } ->
	  Environment.dynenv		(* new "delta" dynEnv *)
  end

  structure PickleEnv: sig
      type pickledEnv
      val pickleEnv:
	  { env:BareEnvironment.staticEnv, context: Environment.staticEnv }
	  -> pickledEnv
      val unPickleEnv:
	  { env:pickledEnv, context: Environment.staticEnv }
	  -> BareEnvironment.staticEnv
      val debugging: bool ref
  end

  val architecture: string
  val version: { system: string, version_id: int list, date: string }
  val banner: string

end
