(*
 * srctypes/smlsource.sml:
 *   Data structure which associates SML source files with everything
 *   you ever wanted to know about them.
 *
 *   Copyright (c) 1995 by AT&T Bell Laboratories
 *
 * author: Matthias Blume (blume@cs.princeton.edu)
 *)
functor SmlSourceFun (structure Decl: DECL
		      structure FnameRules: FNAME_RULES
		      structure CUnit: CUNIT
		      structure Control: CONTROL
		      sharing
			  Decl.Compiler =
			  CUnit.Compiler): SML_SOURCE =
struct

    structure MD = Decl.MD
    structure Compiler = CUnit.Compiler
    structure CUnit = CUnit
    structure Iid = CUnit.Iid
    structure Env = Compiler.Environment

    exception SourceFileDoesNotExist of string
    and UserCodeExn of string * exn
    and SmlSourceInternalError

    type dec = Compiler.Ast.dec
    type source = Compiler.Source.inputSource
    type senv = Env.staticEnv
    type symenv = Env.symenv
    type env = Env.environment

    datatype cu =
	SUCCESS of CUnit.t
      | FAILURE of exn

    datatype t = S of {
		       name: AbsPath.t,
		       describer: (unit -> string) option,
		       timestamp: Time.time option ref,
		       declfile: FnameRules.textfile,
		       binfile: FnameRules.binfile,
		       decl: MD.decl option ref,
		       (*
			* meaning of `cu' field ({} indicates a future):
			* NONE: no compilation was started
			* SOME {...}: compilation was started
			* SOME {FAILURE exn}: compilation failed with exn
			* SOME {SUCCESS cu}: compilation succeeded
			*)
		       cu: cu Futures.future option ref,
		       stable: bool,
		       parsed: { ast: dec,  source: source } option ref
		      }

    val cache: (AbsPath.t, t) Table.table = Table.create AbsPath.sameFile

    fun describe (pn, NONE) = AbsPath.elab pn
      | describe (pn, SOME f) = concat [AbsPath.elab pn, " (", f (), ")"]

    fun nonexistent (S { name, describer, ... }) =
	SourceFileDoesNotExist (describe (name, describer))

    (* the timestamp is set to NONE if the file doesn't exist *)
    fun modtime pn =
	if AbsPath.exists pn then
	    SOME (AbsPath.modTime pn)
	else
	    NONE

    (*
     * We make zeroTime special -- it's the time stamp for stable
     * sources in stable entities.
     *)
    fun earlier (NONE, NONE) = false	(* file still doesn't exist *)
      | earlier (NONE, SOME _) = true	(* a file has appeared *)
      | earlier (SOME _, NONE) = false	(* a file has disappeared *)
      | earlier (SOME t1, SOME t2) =
	Time.<= (t1, Time.zeroTime) orelse Time.< (t1, t2)

    fun get (name, desc) = let
	fun update (s as S { timestamp, decl, cu, ... },
		    modtime_opt) =
	    (timestamp := modtime_opt; decl := NONE; cu := NONE)
    in
	case Table.find (cache, name) of
	    NONE => let
		val s = S {
			   name = name,
			   describer = desc,
			   timestamp = ref (modtime name),
			   declfile = FnameRules.declFileFor name,
			   binfile = FnameRules.binFileFor name,
			   decl = ref NONE,
			   cu = ref NONE,
			   stable = false,
			   parsed = ref NONE
			  }
	    in
		Table.enter (cache, name, s); s
	    end
	  | SOME (s as S { timestamp, stable = false, ... }) => let
		val modtime = modtime name
	    in
		if earlier (!timestamp, modtime) then
		    (update (s, modtime); s)
		else
		    s
	    end
	  | SOME s => s
    end

    fun mkstable (name, decl) =
	case Table.find (cache, name) of
	    SOME (S { stable = true, ... }) => ()
	  | _ => let
		val desc = (AbsPath.elab name) ^ " (stable)"
		val s = S {
			   name = name,
			   describer = SOME (fn () => desc),
			   timestamp = ref (SOME Time.zeroTime),
			   declfile =
			     FnameRules.errorTextFile SmlSourceInternalError,
			   binfile = FnameRules.binFileFor name,
			   decl = ref (SOME decl),
			   cu = ref NONE,
			   stable = true,
			   parsed = ref NONE
			  }
	    in
		Table.enter (cache, name, s)
	    end

    fun name (S { name, ... }) = name
    fun binfile (S { binfile, ... }) = binfile

    fun makestring (S { name, describer, ... }) = describe (name, describer)

    fun parse (S { parsed = ref (SOME p), ... }, true) = p
      | parse (s as S { name, parsed, ... }, quiet) = let
	    val desc = makestring s
	    val _ = if quiet then ()
		    else Control.vsay (concat ["[parsing ", desc, "]\n"])
	    val p = CUnit.parse { file = name, desc = desc }
	in
	    parsed := SOME p;
	    p
	end

    fun forget_parse (S { parsed, ... }) = parsed := NONE

    fun warn_stable (t, name, action) =
	if Time.<= (t, Time.zeroTime) then
	    Control.say
	      (concat ["!% CM Warning: need to ", action, " ",
		       AbsPath.elab name,
		       " (even though it is `stable')\n"])
	else
	    ()

    fun decl (s as S { name, timestamp, declfile, decl, ... }) =
	case !decl of
	    SOME d => d
	  | NONE => let
		val df = declfile
	    in
		case Decl.recover (#open_in df, #name df, !timestamp) of
		    SOME d => (decl := SOME d; d)
		  | NONE =>
			(case !timestamp of
			     NONE => raise (nonexistent s)
			   | SOME t => let
				 val _ = warn_stable (t, name, "reconsult")
				 val { ast, source } = parse (s, false)
				 val d =
				     Decl.create (ast, #open_out df, #name df,
						  source)
				     handle exn =>
					 (Compiler.Source.closeSource source;
					  raise exn)
				 val _ = Compiler.Source.closeSource source
			     in
				 decl := SOME d; d
			     end)
	    end

    fun cunit { smlsource, mkcompenv, senvOf, symenvOf, runEnvOf } = let
	val S { timestamp, name, binfile, cu, ... } = smlsource
	val run_code = case runEnvOf of NONE => false | _ => true
	fun changed desc = Control.say
	    (concat
	     ["!* WARNING: ", desc,
	      " was modified after being analyzed.\n",
	      "!* -------  (It might be necessary to re-run the analysis.)\n"])

	fun userCodeExn exn = UserCodeExn (AbsPath.elab name, exn)

	fun work old_cu () = let

	    val (envf, provided) = mkcompenv ()

	    fun mayrun u =
		case runEnvOf of
		    NONE => SUCCESS u
		  | SOME cvt => let
			val e = cvt (envf ())
			val de = Env.dynamicPart e
		    in
			(CUnit.execute (u, de);
			 SUCCESS u)
			handle exn => FAILURE (userCodeExn exn)
		    end

	    fun fromfile () = let
		val e = envf ()
		val senv = senvOf e
		val symenv = symenvOf e
	    in
		case CUnit.recover {
				    opener = #open_in binfile,
				    binfile = #name binfile,
				    se = senv,
				    sourcetime = !timestamp,
				    provided = provided,
				    keep_code = run_code
				   } of
		    SOME { u, ... } => (forget_parse smlsource; mayrun u)
		  | NONE =>
			(case !timestamp of
			     NONE => raise (nonexistent smlsource)
			   | t as SOME ts => let
				 val _ = warn_stable (ts, name, "recompile")
				 val desc = makestring smlsource
				 val _ =
				     Control.vsay
				       (concat ["[compiling ", desc,
						" -> ",
						AbsPath.elab (#name binfile),
						"]\n"])
				 val S { name, ... } = smlsource
				 val _ =
				     if earlier (t, modtime name) then
					 changed desc
				     else ()
				 val { ast, source } = parse (smlsource, true)
			     in
				 forget_parse smlsource;
				 mayrun
				   (CUnit.create {
						  ast = ast,
						  source = source,
						  name = name,
						  opener = #open_out binfile,
						  binfile = #name binfile,
						  senv = senv,
						  symenv = symenv,
						  provided = provided,
						  keep_code = run_code
						 })
				 before Compiler.Source.closeSource source
			     end)
	    end handle exn => FAILURE exn
	      
	in
	    case old_cu of
		NONE => fromfile ()
	      | SOME uf =>
		    (case Futures.get uf of
			 FAILURE _ => fromfile ()
		       | SUCCESS u =>	(* so let's check it *)
			     if CUnit.isValid (u, provided, run_code) then
				 mayrun u
			     else
				 fromfile ())
	end

	val new_cu = Futures.future (work (!cu))

    in
	cu := SOME new_cu; new_cu
    end

    fun cunit_again (S { cu = ref (SOME cuf), ... }) = cuf
      | cunit_again _ = raise SmlSourceInternalError

    fun clearcache () = Table.clear cache

    fun sweepcache () = let
	fun is_bad (S { name, describer, timestamp, stable, ... }) =
	    (not stable) andalso earlier (!timestamp, modtime name)
	fun keep_good (name, s, l) =
	    if is_bad s then l else (name, s) :: l
	val good_list = Table.fold keep_good cache []
    in
	Table.clear cache;
	app (fn (n, s) => Table.enter (cache, n, s)) good_list
    end

end
