(*
 * cm/cm.sml: `CM' Compilation Manager (constructing the main structure)
 *
 *   Copyright (c) 1995 by AT&T Bell Laboratories
 *
 * author: Matthias Blume (blume@cs.princeton.edu)
 *)
functor CompilationManagerFun
    (structure Compiler: COMPILER
     val version: string
     val singlebindir: AbsPath.t option
     val hostconf: Arch.conf
     val targetos: Arch.os): FULL_CM =

struct

    structure Compiler = Compiler

    val targetconf = { cpu = Arch.cpu Compiler.architecture,
		       os = targetos }

    structure SymVal = SymValFun
	(val conf = targetconf
	 val version = #version_id Compiler.version)

    structure Control = ControlFun (Compiler)

    structure SysEnv = SysEnvFun (Compiler)

    structure FnameRules = FnameRulesFun
	(structure Compiler = Compiler
	 val singlebindir = singlebindir
	 val namelength_limited = CmConfig.namelength_limited)

    structure ModuleName = ModuleNameFun (Compiler)

    structure ModDecl = ModDeclFun (ModuleName)

    structure Convert = ConvertFun (ModDecl)

    structure Decl = DeclFun
	(structure Convert = Convert
	 structure Control = Control)

    structure ImpExp = ImpExpFun
	(structure MD = ModDecl
	 structure Control = Control)

    structure Iid = IIDFun
	(structure Compiler = Compiler)

    structure CUnit = CUnitFun
	(structure Compiler = Compiler
	 structure Control = Control
	 structure Iid = Iid)

    structure LazyEnv = LazyEnvFun
	(val threshold = CmConfig.consolidate_threshold)

    structure SmlSource = SmlSourceFun
	(structure Decl = Decl
	 structure FnameRules = FnameRules
	 structure CUnit = CUnit
	 structure Control = Control)

    structure YaccSource = YaccSourceFun
	(structure Control = Control
	 val processor = CmConfig.yacc)

    structure LexSource = LexSourceFun
	(structure Control = Control
	 val processor = CmConfig.lex)

    structure BurgSource = BurgSourceFun
	(structure Control = Control
	 val processor = CmConfig.burg)

    structure RcsSource = RcsSourceFun
	(structure Control = Control
	 val processor = CmConfig.rcsco)

    structure GroupDag = GroupDagFun
	(structure IE = ImpExp
	 structure SmlSource = SmlSource)

    structure EntityDesc = EntityDescFun
	(structure Decl = Decl
	 structure FnameRules = FnameRules
	 structure SmlSource = SmlSource
	 structure Control = Control)

    structure Source = SourceFun
	(structure SmlSource = SmlSource
	 structure YaccSource = YaccSource
	 structure LexSource = LexSource
	 structure BurgSource = BurgSource
	 structure RcsSource = RcsSource)

    structure SysDag = SysDagFun
	(structure GroupDag = GroupDag
	 structure Source = Source
	 structure EntityDesc = EntityDesc
	 structure Decl = Decl
	 structure FnameRules = FnameRules
	 structure Control = Control)

    structure DagTraversal = DagTraversalFun (SysDag)

    fun clear () = (SmlSource.clearcache (); EntityDesc.clear ())

    structure Driver = DriverFun
	(structure SysDag = SysDag
	 structure SysEnv = SysEnv
	 structure Control = Control
	 val path = CmConfig.path
	 val symval = SymVal.lookup)

    structure Dot = DotFun (DagTraversal)

    structure GenDot = GenDotFun
	(structure Dot = Dot
	 structure Driver = Driver)

    structure LinearSched = LinearSchedFun (DagTraversal)
    structure Lists = ListsFun (LinearSched)

    structure GenLists = GenListsFun
	(structure Lists = Lists
	 structure Driver = Driver)

    structure Prune = PruneFun (DagTraversal)

    structure Recompile = RecompileFun
	(structure DagTraversal = DagTraversal
	 structure Driver = Driver
	 structure SysEnv = SysEnv
	 structure Control = Control
	 structure LE = LazyEnv
	 structure Prune = Prune)

    structure AutoLoad = AutoLoadFun
	(structure Control = Control
	 structure Convert = Convert
	 structure Recompile = Recompile)

    val rootfile = ref "sources.cm"
    val verbose = Control.verbose
    val debug = Control.debug
    val keep_going = Control.keep_going
    val show_exports = Control.show_exports
    val path = Driver.path

    fun set_path p = path := map AbsPath.current p

    structure Export = ExportFun
	(structure Compiler = Compiler
	 val slists = [("CM_PATH", #":", set_path)]
	 val bools = [("CM_VERBOSE", verbose),
		      ("CM_DEBUG", debug),
		      ("CM_KEEP_GOING", keep_going),
		      ("CM_SHOW_EXPORTS", show_exports)]
	 val strings = [("CM_ROOT", rootfile)])

    structure Complain = ComplainFun
	(structure GroupDag = GroupDag
	 structure ImpExp = ImpExp
	 structure SysDag = SysDag
	 structure CUnit = CUnit
	 structure Decl = Decl
	 structure Driver = Driver
	 structure ED = EntityDesc
	 structure Recompile = Recompile
	 structure BurgSource = BurgSource
	 structure LexSource = LexSource
	 structure YaccSource = YaccSource
	 structure SmlSource = SmlSource
	 structure Source = Source
	 structure Control = Control)

    type desc = SysDag.desc

    val version = version

    val canon = AbsPath.current

    val cmfile = SysDag.CMFILE o canon
    val scfile = SysDag.SCGROUP o canon

    fun unprime1 f = fn () => f (cmfile (!rootfile))
    fun unprime2 f = fn arg => f (cmfile (!rootfile), arg)

    fun set_root r = rootfile := r

    (*
     * '-versions (root description file given explicitly):
     *)
    val dot' = Complain.complaining GenDot.genDot
    val names' = Complain.complaining GenLists.names
    val binfiles' = Complain.complaining GenLists.binfiles
    val strings' = Complain.complaining GenLists.strings
    val recompile' = Complain.complaining Recompile.only
    val make' =
	Complain.complaining (Recompile.and'run (hostconf, targetconf, NONE))
    val mkusefile' = Complain.complaining GenLists.mkusefile
    val autoload' = Complain.complaining (Recompile.withAe AutoLoad.register)

    fun testbed' (desc, files) =
	Complain.complaining (Recompile.and'run (hostconf, targetconf, SOME files))
	desc

    fun stabilize' (desc, recursive) =
	Complain.complaining (Recompile.and'stabilize recursive) desc
	before clear ()

    fun destabilize' desc = let

	fun deleteFile name = OS.FileSys.remove name handle _ => ()

	fun c _ = "bogus"
	fun d _ = false
	fun v _ = NONE
	    
	val ep = EntityDesc.EP { path = !path, classify = c,
				 lparm = { strdef = d, sigdef = d, fctdef = d,
					   fsigdef = d, symval = v } }

	fun get_entity (SysDag.CMFILE f) = EntityDesc.read ep f
	  | get_entity (SysDag.SCGROUP f) = EntityDesc.readSCGroup ep f
	  | get_entity (SysDag.SCLIBRARY f) = EntityDesc.readSCLibrary ep f

	val EntityDesc.ENTITY { location, stable, ... } = get_entity desc

    in
	(if stable then
	    deleteFile (AbsPath.elab (#name (FnameRules.stableFileFor location)))
	 else ());
	clear ()
    end

    (*
     * convenient versions: root description file taken from !rootfile:
     *)
    val dot = unprime2 dot'
    val names = unprime1 names'
    val binfiles = unprime1 binfiles'
    val strings = unprime1 strings'
    val recompile = unprime1 recompile'
    val make = unprime1 make'
    val mkusefile = unprime2 mkusefile'
    val stabilize = unprime2 stabilize'
    val destabilize = unprime1 destabilize'
    val autoload = unprime1 autoload'

    fun testbed files = testbed' (cmfile "all-files.cm", files)

    fun autoloading true =
	Compiler.Interact.installCompManager (SOME AutoLoad.manager)
      | autoloading false =
	(Compiler.Interact.installCompManager NONE; AutoLoad.clear ())

    type env = Compiler.Environment.environment

    val export = let
	fun export (imagefile, env0) =
	    Export.export ("CM&CMB " ^ version, imagefile, env0)
    in
	Complain.complaining export
    end

    val sweep = SmlSource.sweepcache

end
