(* crc-util.sml
 *
 * COPYRIGHT (c) 1995 by AT&T Bell Laboratories
 *)

structure CRCUtil = struct

  local
    open Access

    val say = Control.Print.say

    structure LW = LargestWord
    val low8bits = LW.fromInt 0xff
    val sh8bits = 0wx8
  in
    val debugging = ref false
    fun debugmsg msg = if !debugging then say msg else ()

    (* Passing in dont_ignore_lvar is a hack, because for some reason
     * there _are_ LVAR entries in static environment, and we must ignore
     * the actual number there.  However, when hashing Lambda.lexp we cannot
     * ignore it. *)
    fun mkUtils (crc, dont_ignore_lvar) = let

	val c = CRC.append crc

	val c =
	    if !debugging then
		fn i => (Control.Print.say (makestring i ^ ","); c i)
	    else c

	fun for(i, j) f = if i <= j then (f i; for (i + 1, j) f) else ()

	local
	    val l0 = LW.fromInt 0
	in
	    fun largeWord w = let
		val low = LW.toInt (LW.andb (w, low8bits))
		val hig = LW.>> (w, sh8bits)
	    in
		c low;
		if hig > l0 then largeWord hig else ()
	    end
	end

	(* val int = largeWord o LW.fromInt *)
	fun int i =
	    if i < 0 then (c 250; int (~i))
	    else if i < 250 then c i
	    else if i < 65536 then (c 251; c (i mod 256); c (i div 256))
	    else (c 252; c (i mod 256); c(i div 256 mod 256);
		  c (i div (256 * 256) mod 256); c (i div (256 * 256 * 256)))

	val word = largeWord o Word.toLargeWord
	val word32 = largeWord o Word32.toLargeWord

	fun bool true = c 1 | bool false = c 0

	fun string s = (int (size s); app (c o Char.ord) (explode s))

	fun list f l = (int (length l); app f l)

	fun vector f v = (int (Vector.length v); Vector.app f v)

	fun option f NONE = c 0 | option f (SOME s) = (c 1; f s)

	fun pid p = string (PersStamps.stampToString p)

	fun primop p = string (P.pr_primop p)

	fun access (SLOT i) = (c 0; int i)
	  | access (LVAR i) = (c 1; if dont_ignore_lvar then int i else ())
	  | access (EXTERN p) = (c 2; pid p)
	  | access (INLINE p) = (c 3; primop p)
	  | access (PATH (i, a)) = (c 4; int i; access a)
	  | access NO_ACCESS = c 5

	fun conrep UNTAGGED = c 0
	  | conrep (TAGGED i) = (c 1; int i)
	  | conrep (TAGGEDREC (i, j)) = (c 2; int i; int j)
	  | conrep (UNTAGGEDREC i) = (c 3; int i)
	  | conrep (CONSTANT i) = (c 4; int i)
	  | conrep TRANSPARENT = c 5
	  | conrep REF = c 6
	  | conrep LISTCONS = c 7  (* skip LISTNIL, as it will go away *)
	  | conrep (VARIABLE a) = (c 8; access a)
	  | conrep (VARIABLEc a) = (c 9; access a)
	  | conrep LISTNIL = c 10

    in
	{
	 app = c,
	 for = for,
	 int = int,
	 word = word,
	 word32 = word32,
	 largeWord = largeWord,
	 bool = bool,
	 string = string,
	 list = list,
	 option = option,
	 vector = vector,
	 access = access,
	 conrep = conrep,
	 primop = primop
	}
    end
  end
end
