(* Copyright 1989 by AT&T Bell Laboratories *)
(* checklty.sml *)

signature CHECKLTY = sig 
    val check : Lambda.lexp -> bool
    val newlam_ref : Lambda.lexp ref
    val fname_ref : string ref
end

structure CheckLty : CHECKLTY = struct

  open  Lambda 
 structure LT = LambdaType

val newlam_ref : Lambda.lexp ref = ref (RECORD[])
val fname_ref : string ref = ref "yyy"
val anyerror = ref false
val clickerror = fn () => (anyerror := true)

val err = ErrorMsg.impossible
val say = Control.Print.say

fun app2(f,nil,nil) = ()
  | app2(f,a::al,b::bl) = (f(a,b);app2(f,al,bl))
  | app2(f,_,_) = err "CheckLty app2 783"

fun complain le (t1,t2,s) = 
       (clickerror();
        say (s^"  **** Type conflicting in lexp =====> \n    ");
        LT.printLty t1; say "\n and   \n    "; LT.printLty t2;
        say "\n \n";  MCprint.printLexp le;
        say "***************************************************** \n")

fun selcomplain le (t,i :int) =
       (clickerror();
        say "SE *** select **** Type conflicting in lexp =====> \n    ";
        say "selecting "; say (makestring i); 
        say "-th component from the type \n     "; 
        LT.printLty t; say "\n \n";
        MCprint.printLexp le;
        say "***************************************************** \n")

fun simplify(le,0) = STRING "<dummy>"
  | simplify(le,n) = 
       let fun h le = simplify(le,n-1)
        in case le 
            of FN(v,t,le) => FN(v,t,h le)
             | APP(e1,e2) => APP(h e1,h e2)
             | CON(l,e) => CON(l,h e)
             | DECON(l,e) => DECON(l,h e)
             | FIX(lv,lt,le,b) => FIX(lv,lt,map h le, h b)
             | SWITCH(le,l,dc,opp) => 
                (let fun g(c,e) = (c,h e)
                     fun z(NONE) = NONE
                       | z(SOME e) = SOME(h e)
                  in SWITCH(h le,l,map g dc,z opp)
                 end)
             | RECORD le => RECORD (map h le)
             | SRECORD le => SRECORD (map h le)
             | VECTOR le => VECTOR (map h le)
             | SELECT(i,e) => SELECT(i,h e)
             | HANDLE(e1,e2) => HANDLE(h e1,h e2)
             | WRAP(t,e) => WRAP(t,h e)
             | UNWRAP(t,e) => UNWRAP(t,h e)
             | _ => le
       end

fun checklty le s (t1,t2) = 
     if LT.equivLty(t1,t2) then ()
     else (let val le' = simplify(le,3)
            in complain le' (t1,t2,"EQ *** "^s)
           end)


fun whichlty d = case d 
  of DATAcon(_,_,t) => (case LT.out t of LT.ARROW(a,b) => b | _ => t)
   | INTcon _ => LT.injINT
   | WORDcon _ => LT.injINT
   | WORD32con _ => LT.inj LT.INT32
   | REALcon _ => LT.injREAL
   | STRINGcon _ => LT.injBOXED
   | VLENcon _ => LT.injINT

fun checklam le = 
  let exception MM
      val typtable : LT.lty Intmap.intmap = Intmap.new(32,MM)
      val addty = Intmap.add typtable
      fun getty v = (Intmap.map typtable v) handle MM =>
                          (say ("Can't find the variable "^(makestring v)
                               ^" in the typtable ***** \n");
                          raise MM)
      fun checkaux(le) = case le
        of VAR v =>  getty(v)
         | INT i => LT.injINT
         | REAL r => LT.injREAL
         | STRING s => LT.injBOXED
         | EXNF (s,t) => (* t *) LT.injBOXED
         | EXNC s => LT.injBOXED
         | PRIM (p,t) => t

         | APP(FN(v,t,e1),e2) => 
             let val t1 = checkaux e2
                 val _ = addty(v,t1)
              in checkaux(e1)
             end
         | FN(v,t,e) => (addty(v,t); LT.injARROW(t,checkaux(e)))
         | FIX(vl,tl,el,e) => 
             let val _ = app2(addty,vl,tl)
                 val tl' = map checkaux el
                 val _ = app2(checklty le "in FIX",tl,tl')
              in checkaux e
             end
         | APP(e1,e2) => 
             let val t1 = checkaux e1
                 val t2 = checkaux e2
              in case LT.out t1 
                  of LT.ARROW(a,b) => (checklty le "in APP" (a,t2);b)
                   | _ => (complain le (t1,t2,"AP *** in APP"); t1)
             end                 

         | SWITCH(e,rep,cl,opp) => 
             let val t1 = checkaux e
                 fun h(c,le) = 
                    let val ct = whichlty c
                        val _ = checklty e "in SWT1" (t1,ct)
                     in checkaux le
                    end
                 val tt = map h cl
                 val t = hd tt 
                 val r = tl tt
                 val _ = app (fn x => checklty le "in SWT2" (x,t)) r
              in case opp of NONE =>  t
                           | SOME z => (checklty z "in SWT3" (checkaux z,t); t)
             end               

         | CON((_,_,t),e) =>
             let val t1 = checkaux e
              in case LT.out t 
		  of LT.ARROW(a,b) => (checklty le "in CON" (t1,a); b)
		   | _ => t
             end

         | DECON((_,_,t),e) => 
             let val t1 = checkaux e
              in case LT.out t 
		  of LT.ARROW(a,b) => (checklty le "in DECON" (t1,b); a)
		   | _ => t
             end   

         | RECORD el => 
             let val tt = map checkaux el
              in case tt of nil => LT.injINT
                           | _ => LT.injRECORD tt
             end

         | SRECORD el => 
             let val tt = map checkaux el
              in case tt of nil => LT.injINT
                          | _ => LT.inj (LT.SRECORD tt)
             end

         | VECTOR el  => 
             let val tt = map checkaux el
              in case tt 
                  of nil => LT.injBOXED
                   | (a::r) => ((app (fn x => checklty le "in VEC" (a,x)) r);
                                LT.injBOXED)
             end
         | SELECT(i,e) => 
             let val t = checkaux e

                 exception Notfound
                 fun look [] = raise Notfound
                   | look ((k,t)::r) = if k = i then t else look r 
 
		 fun bogus() = (selcomplain le (t,i); LT.BOGUS)

              in case LT.out t  
                  of LT.RECORD tt => (List.nth(tt,i) handle Subscript => bogus())
                   | LT.SRECORD tt => (List.nth(tt,i) handle Subscript => bogus())
                   | LT.GREC l => (look l handle Notfound => bogus())
                   | _ => bogus()
             end

         | RAISE(e,t) => 
             let val t1 = checkaux e
                 val _ = checklty le "in RAISE" (LT.injBOXED,t1)
              in t
             end
         | HANDLE(e1,e2) => 
             let val t1 = checkaux e1
                 val t2 = checkaux e2
              in case LT.out t2 
                   of LT.ARROW(a,b) => (checklty le "in HAN" (t1,b);t1)
                    | _ => (complain le (t1,t2,"HD *** HANDLE");t1)
             end

         | WRAP(t,e) => (* (checklty le "in WRAP" (checkaux e,t); LT.injBOXED) *)
                (checkaux e; LT.injBOXED)
         | UNWRAP(t,e) => (checklty le "in UNWRAP" (checkaux e,LT.injBOXED); t)

    in checkaux le
   end

fun check le = (anyerror := false; checklam le; !anyerror)

end (* CheckLty *)
