(* Copyright 1989 by AT&T Bell Laboratories *)
(* basics/pputil.sml *)

structure PPUtil : PPUTIL =
struct

  structure Symbol : SYMBOL = Symbol
  structure PP = PrettyPrint

  fun ppSequence0 ppstream (sep:PP.ppstream->unit,pr,elems) =
      let fun prElems [el] = pr ppstream el
	    | prElems (el::rest) =
	        (pr ppstream el;
		 sep ppstream;
                 prElems rest)
	    | prElems [] = ()
       in prElems elems
      end

  fun ppSequence ppstream {sep:PP.ppstream->unit, pr:PP.ppstream->'a->unit, 
                           style:PP.break_style} (elems: 'a list) =
      (PP.begin_block ppstream style 0;
       ppSequence0 ppstream (sep,pr,elems);
       PP.end_block ppstream)

  fun ppClosedSequence ppstream{front:PP.ppstream->unit,sep:PP.ppstream->unit,
                               back:PP.ppstream->unit,pr:PP.ppstream->'a->unit,
                                style:PP.break_style} (elems:'a list) =
      (PP.begin_block ppstream PP.CONSISTENT 0;
       front ppstream;
       PP.begin_block ppstream style 0;
       ppSequence0 ppstream (sep,pr,elems); 
       PP.end_block ppstream;
       back ppstream;
       PP.end_block ppstream)

  fun ppSym ppstream (s:Symbol.symbol) = PP.add_string ppstream (Symbol.name s)

  val stringDepth = Control.Print.stringDepth

(** NOTE: this duplicates code in basics/printutil.sml **)
  fun escape i = let
	val m = makestring: int->string 
	in
	  concat ["\\", m(i div 100), m((i div 10)mod 10), m(i mod 10)]
	end
  val offset = Char.ord #"A" - Char.ord #"\^A"
  fun ml_char #"\n" = "\\n"
    | ml_char #"\t" = "\\t"
    | ml_char #"\\" = "\\\\"
    | ml_char #"\"" = "\\\""
    | ml_char c = if ((c >= #"\^A") andalso (c <= #"\^Z"))
	  then "\\^" ^ String.str(Char.chr(Char.ord c + offset))
	else if ((#" " <= c) andalso (c <= #"~"))
	  then String.str c
	  else escape(Char.ord c)

  fun mlstr s = concat["\"", concat(map ml_char (explode s)), "\""]

  fun pp_mlstr ppstream s =
      let val depth = !stringDepth
          val add_string = PP.add_string ppstream
	  fun pr i =
	      if i=depth then add_string "#"
	      else (let val ch = String.sub(s,i)
		    in  add_string (ml_char ch); pr (i+1)
		    end handle Substring => ())
       in add_string "\""; pr 0; add_string "\""
      end

  fun ppvseq ppstream ind (sep:string) pr elems =
      let fun prElems [el] = pr ppstream el
	    | prElems (el::rest) = (pr ppstream el; 
                                    PP.add_string ppstream sep; 
                                    PP.add_newline ppstream;
                                    prElems rest)
	    | prElems [] = ()
       in PP.begin_block ppstream PP.CONSISTENT ind;
          prElems elems;
          PP.end_block ppstream
      end

  fun ppvlist ppstrm (header,separator,pr_item,items) =
      case items
	of nil => ()
	 | first::rest =>
	     (PP.add_string ppstrm header;
	      pr_item ppstrm first;
	      app (fn x => (PP.add_newline ppstrm;
			    PP.add_string ppstrm separator;
			    pr_item ppstrm x))
		   rest)

  (* debug print functions *)
  fun ppIntPath ppstream =
      ppClosedSequence ppstream 
	{front=(fn pps => PP.add_string pps "["),
	 sep=(fn pps => (PP.add_string pps ","; PP.add_break pps (0,0))),
	 back=(fn pps => PP.add_string pps "]"),
	 style=PP.INCONSISTENT,
	 pr=(fn pps => PP.add_string pps o (makestring:int->string))}

  fun ppSymPath ppstream (SymPath.SPATH path: SymPath.path) = 
      ppSequence ppstream
        {sep=(fn pps => PP.add_string pps "."),
	 style=PP.INCONSISTENT,
	 pr=ppSym}
        path

  fun ppInvPath ppstream (InvPath.IPATH path: InvPath.path) =
      ppClosedSequence ppstream 
	{front=(fn pps => PP.add_string pps "<"),
	 sep=(fn pps => (PP.add_string pps ".")),
	 back=(fn pps => PP.add_string pps ">"),
	 style=PP.INCONSISTENT,
	 pr=ppSym}
        path

  fun nl_indent ppstrm i = PP.add_break ppstrm (127,i)
      (* !!! 127 should be !line_width+1 *)

end (* structure PPUtil *)
