(* Copyright 1992 by AT&T Bell Laboratories *)
(* modules/sigmatch.sml *)

signature SIGMATCH =
sig
  val match :
    {abstract: bool,
     arg_option: Modules.structureVar option,
     err: ErrorMsg.complainer,
     scope: Stamps.scope,
     rpath: InvPath.path,
     printEnv: Modules.env,
     self: bool,
     sign: Modules.Signature,
     str: Modules.Structure} 
    -> Modules.Structure * Modules.absWrapper * Modules.thinning
  val matchFarg :
    {abstract: bool,
     arg_option: Modules.structureVar option,
     err: ErrorMsg.complainer,
     scope: Stamps.scope,
     rpath: InvPath.path,
     printEnv: Modules.env,
     self: bool,
     sign: Modules.Signature,
     str: Modules.Structure} 
    -> Modules.Structure * Modules.absWrapper * Modules.thinning
  val matchFct :
    {abstract: bool,
     err: ErrorMsg.complainer,
     scope: Stamps.scope,
     rpath: InvPath.path,
     printEnv: Modules.env,
     self: bool,
     fsig: Modules.FctSignature,
     fct: Modules.Functor} 
    -> Modules.Functor * Modules.fctThinning option
  val debugging : bool ref
end

(* ASSUMPTION: match is only used with TOP signatures. 
 * This condition ensures a correct determination of the parent of functors
 * which must contain the structure resulting from the matching.
 *)

structure SigMatch : SIGMATCH =
struct

open Modules ModuleUtil Types TypesUtil Variables PrettyPrint ErrorMsg Access
     Instantiate Extern

val say = Control.Print.say

(* the last element of a list -- should be in List *)
fun last [x] = x
  | last (_ :: t) = last t
  | last nil = ErrorMsg.impossible "last"

val debugging = ref false

val BogusTy = UNDEFty (*** this is just a place holder in VALtrans when 
			   the access is not INLINE and the 3rd component
			   is NONE.  ***)

(* a couple of utility functions *)

fun isExn (DATACON{rep,...}) =
    case rep
      of VARIABLE _ => true
       | VARIABLEc _ => true
       | _ => false

(* a special hack on unrolled list done by zsh *)
(*  fun special_rep(LISTCONS|LISTNIL) = true
      | special_rep _ = false

    fun newrep(DATACON{rep=_,name,const,typ,orig,sign},nr,nsign) =
	DATACON{rep=nr,name=name,const=const,typ=typ,orig=orig,sign=nsign}
*)

(* for type abbreviations - Cregut *)
fun transDef str (DEFtyc{tyfun=TYFUN{body,arity},strict,path}) = 
      DEFtyc{tyfun=TYFUN{body=transType str body,arity=arity},
	     strict=strict,path=path}
  | transDef str tyc = tyc


(* checkSharing: checks sharing constraints.  Takes a structure
   environment, a list of structure sharing constraints, a list
   of type sharing constraints, an error function, and a symbolic
   path as arguments.  The symbolic path is the path to this
   structure: it is prefixed to symbolic paths within this
   structure when error messages are printed. *)

fun checkSharing(str : Structure, fullEnv : Modules.env,
		 sConstraints:{internal: SymPath.path list,
			       external: Structure option} list,
		 tConstraints:{internal: SymPath.path list,
			       external: Types.tycon option} list,
		 abbreviations,
		 err) =
    let fun f (lookup : SymPath.path ->'a, printName : ppstream -> 'a -> unit,
	       eq : 'a * 'a -> bool, errName:string)
	      ({internal: SymPath.path list, external : 'a option}) =
	    let fun g first =
		let fun check x =
		    if eq(lookup x,first)
		    then ()
		    else (err COMPLAIN (errName^" sharing violation")
			   (fn ppstrm =>
			      (add_newline ppstrm; printName ppstrm first;
			       add_string ppstrm " # ";
			       PPUtil.ppSymPath ppstrm x)))
		 in app check internal
		end
	     in case (internal,external)
		  of (l,SOME i) => g i
		   | (h::t,NONE) => g (lookup h)
		   | (nil,NONE) => ()
	    end
	val checkStructure =
	      f (fn x => lookBindingSTR(str,x),
		 fn ppstrm => fn x => PPBasics.ppStructureName ppstrm (fullEnv,x),
		 eqOrigin,"structure")
	val checkType =
	      f (fn x => lookBindingTYC(str,x),
		 fn ppstrm => fn tyc => PPType.ppTycon fullEnv ppstrm tyc,
		 equalTycon,"type")
	 fun checkAbbreviations {internal,external} = 
	     let val t1 = transPosTycon str [internal]
		 val t2 = transDef str external
	      in if equalTycon(t1,t2) then () 
		 else err COMPLAIN ("abbreviation definition violation : ")
			(fn ppstrm => PPType.ppTycon fullEnv ppstrm t1)
	     end
     in app checkStructure sConstraints;
	app checkType tConstraints;
        app checkAbbreviations abbreviations
    end

fun compareTypes (env:Modules.env,err) (spec: ty, actual:ty,name) : unit =
    if TypesUtil.compareTypes{spec=spec,actual=actual} then ()
    else err COMPLAIN "value type in structure doesn't match signature spec"
	   (fn ppstrm =>
	    (PPType.resetPPType();
	     add_newline ppstrm;
	     app (add_string ppstrm) ["  name: ", Symbol.name name];
	     add_newline ppstrm;
	     add_string ppstrm "spec:   ";
	     PPType.ppType env ppstrm spec;
	     add_newline ppstrm;
	     add_string ppstrm "actual: ";
	     PPType.ppType env ppstrm actual))


fun conforming(INSTANCE{sign=SIG{stamp,kind=ref(TOP _),...},...},
      SIG{stamp=stamp',kind=ref(TOP _),...}) = Stamps.eq(stamp,stamp')
  | conforming _ = false

fun match1 (_,fctm) {str=ERROR_STR,...} = (ERROR_STR,NONE,NONE)
  | match1 (_,fctm) {str,sign=FULL_SIG,...} = (str,NONE,NONE)
  | match1 (_,fctm) {str,sign=ERROR_SIG,...} = (str,NONE,NONE)
  | match1 (context: (Structure Array.array * Functor Array.array
		      * tycon Array.array) option,
            fctm) 
	    {sign as SIG{symbols,env,kind,...},str,rpath: InvPath.path,
	     scope,err,printEnv,abstract,self,parent_sig,arg_option}
		  : Structure * absWrapper * thinning =
    if conforming(str,sign)
    then (if abstract then 
	    let val str' = instantiate((rpath,scope,err),sign)
		val wrapper = SOME(TransBinding.transStrLty str',
				   TransBinding.transStrLty str)
	     in (str',wrapper,NONE)
	    end
	  else (str,NONE,NONE)) 
    else
    let val v = Access.mkLvar() (* local lvar for accessing str *)
	val fullEnv = Env.atop(makeEnv(str,LVAR v),printEnv)
	     (* used for printing error msgs *)
	val newContext as (subStrs,subFcts,types) = 
	      case (context,!kind)
		of (_, TOP{strcount,typecount,fctcount,...}) =>
		   (Array.array(strcount,ERROR_STR),
		    Array.array(fctcount,ERROR_FCT),
		    Array.array(typecount,ERRORtyc))
		 | (SOME st, EMBEDDED) => st
		 | _ => impossible "Sigmatch.match1"
	val (self',oldEnv) = 
	      if self then
		case str
		  of INSTANCE{sign=SIG{env=str_env,...},subStrs,...} =>
			(false,SOME (str_env,!env,subStrs))
		   | INSTANCE _ => (false,NONE)
		   | _ => (true,NONE)
	      else (false,NONE)
	val newstr = INSTANCE{sign=sign,
			      subStrs=subStrs,
			      subFcts=subFcts,
			      types=types,
			      path=rpath,
			      origin =
				 if self'
				 then STAMP_ORIGIN(Stamps.newStamp scope ())
				 else STAMP_ORIGIN(getStrStamp str)}

	val new_parent_sig = 
	      case !kind
		of TOP _ => newstr
		 | EMBEDDED => parent_sig
		 | IRRELEVANT => newstr

	fun complain s = err COMPLAIN s nullErrorBody
	val compare = compareTypes(fullEnv,err)
	val transType' = ModuleUtil.transType newstr

	fun complainMissing (UnboundComponent _,name,namespace) =
	      (complain("unmatched "^namespace^" spec: "^
			Symbol.name name);
	       raise Error)
	  | complainMissing (ErrorStructure,_,_) = raise Error
	  | complainMissing (exn,_,_) = raise exn


	(* findComponent: Given a binding specification, find the actual
	   binding in a structure.  If the binding is not there, print an
	   error message and raise the exception Error.

	   We must handle exception bindings specially, since they are
	   in the name space for constructors.  When we search for an
	   actual exception binding, we may find a constructor binding
	   instead.  For bindings in other namespaces, we will never
	   accidentally find bindings in other name spaces. *)

	(* declare type *)

	fun findComponent spec =
	  case spec
	   of CONbind(spec as (DATACON{rep=r1,name,sign=ns1,...})) =>
		((case lookBinding(str,SymPath.SPATH[name],LVAR v)
		   of binding as (CONbind actual) => 
		      if isExn spec=isExn actual then binding
 (* Unrolled list hack - Zhong Shao
			(if special_rep(r1) then 
				  CONbind(newrep(actual,r1,ns1))
			  else binding)
 *)
		      else raise UnboundComponent(SymPath.SPATH [])
		    | _ => raise UnboundComponent(SymPath.SPATH []))
		 handle exn => complainMissing(exn,name,
				     if isExn spec
				     then "exception"
				     else "data constructor"))
	      (* we never check infix bindings; may return anything here *)
	    | (spec as FIXbind _) => spec
	    | (spec as STRbind(STRvar{name=id,...})) =>
		if Symbol.eq(id,argumentId) then
		     (* note that the access is not modified so that we
			have a true LVAR and not a slot *)
		    (case arg_option
		       of NONE => impossible "sigmatch: can't find the arg"
			| SOME arg => STRbind arg)
		else (lookBinding (str,SymPath.SPATH[id],LVAR v)
		      handle exn => complainMissing(exn,id,"structure"))
	    | spec =>
		let val (name,namespace) =
		      case spec
		       of (FCTbind(FCTvar{name=id,...})) =>
			     (id,"structure")
			| (TYCbind(FORMtyc{spec=sigTycon,...})) =>
			     (tycName sigTycon,"type")
			| (CONbind(DATACON{name,...})) =>
			     (name,"data constructor")
			| (VARbind(VALvar{path,...})) =>
			    (case  path
			       of SymPath.SPATH [id] => (id,"val")
				| _ => impossible "SigMatch.findComponent 1")
			| _ => impossible "SigMatch.findComponent 2"
		 in lookBinding (str,SymPath.SPATH[name],LVAR v)
		     handle exn => complainMissing(exn,name,namespace)
		end

      exception BadBinding

      fun checkDataconSign(name,d1,d2) =
	  let fun friend(UNTAGGEDREC 2,LISTCONS) = true
		| friend(CONSTANT 0,LISTNIL) = true
		| friend(r1,r2) = (r1 = r2)

              fun compatDom(d1, d2) =
                    let val lt1 = Transtypes.transTyLty d1
                        val lt2 = Transtypes.transTyLty d2
                     in LambdaType.compatLty(lt1,lt2)
                    end

	      fun ck(DATACON{rep=r1,name=n1,typ=t1,...},
                     DATACON{rep=r2,typ=t2,...}) =
		  if (compatDom(t1,t2) orelse (not fctm))
                     andalso friend(r1,r2) then ()
		  else (err COMPLAIN
			 ("The constructor "^Symbol.name n1^
			  " of datatype "^Symbol.name name^
			  "\nhas different representations in \
			   \the signature and the structure.  \n\
			   \Change the definition of the types \
			   \carried by the constructors in the\n\
			   \functor formal parameter and the functor \
			   \actual parameter so that\n\
			   \they are both abstract, or so that \
			   \neither is abstract.\n")
			  nullErrorBody;
		  raise BadBinding)
	   in List2.app2 ck (d1,d2)
	  end

       (* compare datacon names of spec and actual datatype.  This
	  uses the fact that datacons have been sorted by name. *)
       fun compareDcons(spec,actual) =
	   let fun comp(l1 as dc1::r1, l2 as dc2::r2, s_only, a_only) =
		     if Symbol.eq(dc1,dc2) then comp(r1,r2,s_only,a_only)
		     else if Symbol.symbolGt(dc1,dc2) then
		       comp(l1,r2,s_only,dc2::a_only)
		     else comp(r1,l2,dc1::s_only,a_only)
		 | comp([], [], s_only, a_only) = (rev s_only, rev a_only)
		 | comp([], r, s_only, a_only) = (rev s_only, rev a_only @ r)
		 | comp(r, [], s_only, a_only) = (rev s_only @ r, rev a_only)
	    in comp(spec,actual,[],[])
	   end

       fun checkTycBinding(specTycon,strTycon) =
	   let val name = fn () => Symbol.name(tycName specTycon)
	       fun complain' x = (complain x; raise BadBinding)
	       fun symbolsToString [] = ""
		 | symbolsToString [n] = Symbol.name n
		 | symbolsToString (n::r) =
		    concat(Symbol.name n ::
			    foldr (fn(n,b) => (","::Symbol.name n::b)) [] r)
	       fun dconName(DATACON{name,...}) = name
	    in case specTycon
		 of GENtyc {stamp=s,arity,kind,eq=ref eq,...} =>
		     (if arity <> tyconArity strTycon
		      then complain' ("tycon arity for "^name()
				     ^ " does not match specified arity")
		      else case (!kind, strTycon)
			     of (DATAtyc dcons,
				 GENtyc{arity=a',kind=ref (DATAtyc dc'),...})=>
				  (case compareDcons(map dconName dcons,
						     map dconName dc')
				     of ([],[]) =>
					  checkDataconSign(tycName specTycon,
							   dcons,dc')
				      | (s_only, a_only) =>
					 complain'(concat(List.concat
					  [["datatype ",name(),
					    " does not match specification\n"],
					   case s_only
					     of [] => []
					      | _  =>
					       ["  constructors in spec only: ",
						symbolsToString s_only, "\n"],
					   case a_only
					     of [] => []
					      | _  =>
					       ["  constructors in actual only: ",
						symbolsToString a_only, "\n"]])))
			       | (DATAtyc _, _) => 
				   complain'("type "^name()^" must be a datatype")
			       | ((FORMtyck | FORMDEFtyc _), _) =>
				  if eq=YES andalso
				     not(EqTypes.isEqTycon strTycon)
				  then complain'("type "^name()^
						 " must be an equality type")
				  else ()
			       | _ => impossible "checkTycBinding 1")
		  | ERRORtyc => raise BadBinding
		  | _ => ErrorMsg.impossible "checkTycBinding 2"
	   end

	(* checkSpec:  Check that a binding specification is matched by an
	   actual binding in a structure.  Fill in the instantiation vectors
	   for types and structures.*)

	fun checkSpec spec =
	    let val result = findComponent spec
	     in case (spec,result) 
		  of (STRbind(STRvar{name=id,binding=STR_FORMAL{pos,spec,...},
				     ...}),
		      STRbind(STRvar{access,binding=str',...})) =>
		       let val (str,_,thin) =
			      if Symbol.eq(id,parentId) then (str',NONE,NONE)
			      else match1 (SOME newContext, fctm)
				    {sign=spec, str=str',
				     rpath = InvPath.extend(rpath,id),
				     scope=scope, err=err, abstract=false, 
				     parent_sig=new_parent_sig, self=false, 
				     printEnv = printEnv, arg_option = NONE}
			 in Array.update(subStrs,pos,str);
			    if (hidden id) then []
			    else
			     (case thin
			       of NONE => [VALtrans(access,BogusTy,NONE)]
				| SOME(v,transl) => 
				    [THINtrans(access,v,transl)])
			end
		   | (FCTbind(FCTvar{name=id,binding=FCT_FORMAL{pos,spec,...},
				     ...}),
		      FCTbind(FCTvar{access,binding=fct',...})) =>
		       let val (fct,thin) = matchFct1
			     {fsig=spec,
			      fct=fct',
			      rpath = InvPath.extend(rpath,id),
			      scope = scope,
			      err=err,
			      parent_sig = new_parent_sig,
			      printEnv = printEnv,
			      abstract=false,
			      self=false}
			in Array.update(subFcts,pos,fct);
			   case thin 
			    of NONE => [VALtrans(access,BogusTy,NONE)]
			     | SOME fctThin => 
				 (case spec 
				  of FSIG{argument,...} => 
				       let val lambdaty = 
					    TransBinding.transFctLty(fct)
					in [FCTtrans(access,lambdaty,fctThin)]
				       end
				   | _ => impossible "sigmatch check-spec 324")
		       end
		   | (TYCbind(FORMtyc{pos,spec=sigTycon,...}),
		      TYCbind(strTycon)) =>
		       ((checkTycBinding(sigTycon,strTycon);
			 Array.update(types,pos,strTycon))
			handle BadBinding => Array.update(types,pos,ERRORtyc);
			nil)
		   | (CONbind(DATACON{name,typ,const,...}),
		      CONbind(DATACON{typ=typ',rep,...})) =>
		       let val typ1 = transType' typ
			   val _ = compare(typ1,typ',name)
			   val typ2 = NONE (* case typ' of POLYty _ => (SOME typ1)
						    | _ => NONE *)
			in case rep
			     of VARIABLE access => 
				 [VALtrans(access,typ',typ2)]
			      | VARIABLEc access => 
				 [VALtrans(access,typ',typ2)]
			      | _ => nil
		       end
		   | (VARbind(VALvar{path,typ,...}),a) =>
		      (case a
			 of VARbind(VALvar{access,typ=typ',...}) =>
			     let val typ1 = transType'(!typ)
				 val typ2 = !typ'
				 val _ = compare(typ1,typ2,SymPath.first path)
				 val typ3 = (case (headReduceType typ2)
                                              of POLYty _ => (SOME typ1)
					       | _ => NONE)
			      in [VALtrans(access,typ2,typ3)]
			     end
			  | CONbind(dcon as DATACON{typ=typ',orig,name,
                                              const,rep,sign}) =>
			     let val typ1 = transType'(!typ)
				 val _ = compare(typ1,typ',SymPath.first path)
                                 val newtyp = case orig of NONE => typ'
                                                         | SOME z => z
                              in case rep 
                                  of VARIABLE acc => 
                                       [VALtrans(acc, newtyp, NONE)]
                                   | VARIABLEc acc => 
                                       [VALtrans(acc, newtyp, NONE)]
                                   | _ => (let val dcon' = 
                                                 DATACON{typ=newtyp,orig=orig,
                                                         name=name,const=const,
                                                         rep=rep,sign=sign}
                                            in [CONtrans(dcon',SOME typ1)]
			                   end)
                             end
                   | _ => impossible "sigmatch.sml: 122")
		   | (FIXbind _,_) => nil (* nonchecked binding *)
		   | _ => impossible "sigmatch.sml: 124"
	    end

	fun checkList (a::rest) =
	     (checkSpec (Env.look (!env,a))
	      handle Error => nil | Env.Unbound => impossible "checkList")
	      @ checkList rest
	  | checkList nil = nil

	val trans = checkList (!symbols)
	val _ = case !kind
		  of TOP{sConstraints,tConstraints,abbreviations,...} =>
		       checkSharing(newstr,fullEnv,sConstraints,tConstraints,
				    abbreviations,err)
		   | EMBEDDED => ()
		   | IRRELEVANT => ()

 (* this optimization causes a bug -- (which one?)
	val _ =
	  if self then (
	    case oldEnv
	      of SOME (env,sigenv,strs) => 
		   (* The meaning of this hack is the following:
		      If the structure is a self but belongs to a functor
		      then it may contain a useless and big INSTANCE used
		      as an intermediate structure for open.
		      This function destroys such instances and replace them
		      by a dummy structure. *)
		   let fun clrOpen n = (
			 case Array.sub(strs,n) 
(* type error *)	   of INSTANCE{path as _ :: _, ...} =>
			      if Symbol.eq(openId,last path) then 
				(Array.update(strs,n,ERROR_STR); clrOpen (n+1))
			      else clrOpen (n+1)
			    | _ =>  clrOpen (n+1)) handle Array.Subscript => ()
		       val thinenv = ref Env.empty
		       (* Note: it would be nice if Env.map gave also the
			  symbol to the argument function *)
		       val _ = 
			 Env.app 
			   (fn (name,_) =>
			      thinenv := Env.bind(name,Env.look(!env,name),
						      !thinenv)
			      handle Env.Unbound => ())
			   sigenv
		    in env := Env.consolidate(!thinenv);
		       clrOpen 0
		   end
	       | NONE => ())
	  else () *)

     in if abstract then 
	  let val str' = instantiate((rpath,scope,err),sign)
	      val wrapper = SOME(TransBinding.transStrLty str',
				 TransBinding.transStrLty newstr)
	   in (str', wrapper, SOME(v,trans))
	  end
	else (newstr, NONE, SOME(v,trans))
    end

and matchFct1 {abstract,err,scope,rpath,printEnv,self,
	       parent_sig,fct,
	       fsig=fsig as FSIG{paramName,argument=sig_arg,
				 body=sig_body,...}} =
    let 
       (* externalize the sharing constraints with the parent in the
	* argument signature so that we can use instantiate *)
	val arg_final = externalize_sharing(parentId,parent_sig,sig_arg)
       (* instance of the argument *)
	val inst_sig_arg =
	      Instantiate.instantiate((InvPath.IPATH[],scope,err),arg_final)
	val arg_var = STRvar{access=NO_ACCESS, name=argumentId,
			     binding=inst_sig_arg}
	val _ = update_structure(parentId,parent_sig,inst_sig_arg)
       (* the corresponding version with the parent of the structure *)
	val val_X = lookBindingSTR(inst_sig_arg,SymPath.SPATH[parameterId])
	val arglty = TransBinding.transStrLty val_X

       (* we don't keep thinnings: thinin is a thinning against a dummy
	  structure we usually never build and thinout is discarded (if
	  we used one it would come with res_match *)
	val (res,thinIn,realArg) =
	  ApplyFunctor.applyFunctorFull(fct,val_X,scope,rpath,err,printEnv,match)

	val realArg' = lookBindingSTR(realArg,SymPath.SPATH[parameterId])
	val fctWrap = SOME (TransBinding.transFctLty fct,
		       LambdaType.injARROW(TransBinding.transStrLty realArg',
					   TransBinding.transStrLty res))

       (* matches the result against the body specification of the functor
	* signature. The specif of the functor body must be a TOP to get
	* a correct notion of parent_sig 
	* we keep the thinning obtained *)
	val (res_match,_,thinning) = 
	      match1 (NONE, true)
		{abstract=abstract, err=err, scope=scope, rpath=rpath,
		 printEnv=printEnv, self=false, parent_sig=ERROR_STR,
		 sign=sig_body, str=res, arg_option=SOME arg_var}

	val reslty = TransBinding.transStrLty res_match
	val lambdaty = LambdaType.injARROW(arglty,reslty)

	val raw_fct = 
	    case fct 
	      of FCT_INSTANCE{fct,...} => fct
	       | fct => fct
	val new_fct = 
	  FCT_INSTANCE{fct=raw_fct,fsig=fsig,parent=parent_sig,
		       lambdaty=lambdaty}
     in (new_fct,SOME(thinIn,thinning,fctWrap))
    end
  | matchFct1 {fct,...} = (fct,NONE)

and match {abstract: bool,
	   err: ErrorMsg.complainer,
	   arg_option: structureVar option,
	   scope: Stamps.scope,
	   rpath: InvPath.path,
	   printEnv: Modules.env,
	   self: bool,
	   sign: Modules.Signature,
	   str: Modules.Structure} =
    match1 (NONE,false) {abstract=abstract,err=err,scope=scope,rpath=rpath,
  		 printEnv=printEnv,self=self,arg_option=arg_option,
		 parent_sig = ERROR_STR, sign=sign,str=str}

and matchFarg {abstract: bool,
	   err: ErrorMsg.complainer,
	   arg_option: structureVar option,
	   scope: Stamps.scope,
	   rpath: InvPath.path,
	   printEnv: Modules.env,
	   self: bool,
	   sign: Modules.Signature,
	   str: Modules.Structure} =
    match1 (NONE,true) {abstract=abstract,err=err,scope=scope,rpath=rpath,
  		 printEnv=printEnv,self=self,arg_option=arg_option,
		 parent_sig = ERROR_STR, sign=sign,str=str}

and matchFct {abstract,err,scope,rpath,printEnv,self,fsig,fct} =
    matchFct1{abstract=abstract,err=err,scope=scope,printEnv=printEnv,
	      self=self,rpath=rpath,fsig=fsig,fct=fct, parent_sig=ERROR_STR}

end  (* structure SigMatch *)
