(* Copyright 1992 by AT&T Bell Laboratories *)
(* moduleutil.sml *)

structure ModuleUtil : MODULEUTIL = struct

open Modules ErrorMsg Variables Access Types TypesUtil PrintUtil Symbol
structure SP = SymPath
structure IP = InvPath

val say = Control.Print.say

(* reducing a type scheme into a type *)
fun reduce ty =
  case TypesUtil.headReduceType ty
   of POLYty{tyfun=TYFUN{body,...},...} => reduce body
    | ty => ty

(* error given back by looking up function when they fail *)
exception UnboundComponent of SP.path
(* error given back by the same functions when an error structure is found *)
exception ErrorStructure

(* compiler bugs *)
fun error s = impossible ("ModuleUtil: "^s)

(* a symbol for an undefined structure variable *)
val bogusSTRvar =
  STRvar{name=Symbol.strSymbol "Bogus", access=SLOT 0, binding=ERROR_STR}

(* a symbol for an undefined functor variable *)
val bogusFCTvar = 
  FCTvar{name=Symbol.fctSymbol "Bogus", access=SLOT 0, binding=ERROR_FCT}

fun root(PATH(_,p)) = root p
  | root(LVAR v) = SOME v
  | root _ = NONE

val bogusCONbind = CONbind bogusCON

(* gets the stamp of a structure *)
fun getStrStamp (SIMPLE{stamp,...}) = stamp
  | getStrStamp (INSTANCE{origin,...})=
      (case origin
	 of STAMP_ORIGIN s => s
	  | _ => error "getStrStamp - bad origin")
  | getStrStamp (APPLY{res,...}) = getStrStamp res
  | getStrStamp ERROR_STR = Stamps.null 
  | getStrStamp (STR_OPEN{path,...}) =
      (say (SP.makestring path); say "\n";
       error "getStrStamp - STR_OPEN")
  | getStrStamp (STR_FORMAL _) = error "getStrStamp - STR_FORMAL"
  | getStrStamp (STR_ABSFB _) = error "getStrStamp - STR_ABSFB"
  | getStrStamp (STR_EXTERN(ModuleId.SIMPLEid stamp)) = stamp
  | getStrStamp (STR_EXTERN(ModuleId.INSTANCEid{origin,...})) = origin
  | getStrStamp _ = error "getStrStamp"

(* gets the stamp of a signature *)
fun getSignStamp (SIG{stamp,...}) = stamp
  | getSignStamp (EXTERN_SIG(ModuleId.SIGid stamp)) = stamp
  | getSignStamp _ = Stamps.null

(* gets the stamp of a functor *)
fun getFctStamp (FCT{stamp,...}) = stamp
  | getFctStamp (FCT_INSTANCE{fct,...}) = getFctStamp fct
  | getFctStamp ERROR_FCT = Stamps.null
  | getFctStamp _ = error "getFctStamp"

(* equality of signatures *)
fun eqSign (a,b) = Stamps.eq(getSignStamp a,getSignStamp b)

(* equality of functor signatures (for printing only) *)
fun eqFsig (FSIG{argument=a1,body=b1,...},FSIG{argument=a2,body=b2,...}) =
      eqSign(a1,a2) andalso eqSign(b1,b2)
  | eqFsig _ = false

(* gets the origin of a structure *)
fun getOrigin str = str

(* equality of the origins of two structures (always true if one of the
   argument is unelaborated (ERROR_STR) *)
fun eqOrigin (x,y) =
    case (getOrigin x,getOrigin y)
      of (ERROR_STR,_) => true
       | (_,ERROR_STR) => true
       | (ox,oy) => Stamps.eq(getStrStamp ox,getStrStamp oy)
 
exception Id

fun onlyFree st = if Stamps.isFree st then st else raise Id

fun sigId(SIG{stamp,...}) = ModuleId.SIGid (onlyFree stamp)
  | sigId(EXTERN_SIG id) = id
  | sigId _ = ErrorMsg.impossible "ModuleUtil.sigId"

fun strId(SIMPLE{id,...}) = id
  | strId(INSTANCE{sign,origin,...}) =
      ModuleId.INSTANCEid{origin=case origin
				   of STAMP_ORIGIN stamp => onlyFree stamp
				    | _ => raise Id,
			  sign=case sign 
				of SIG{stamp,...} => onlyFree stamp
				 | EXTERN_SIG(ModuleId.SIGid st) => st
				 | _ => raise Id}
  | strId (STR_EXTERN id) = id
  | strId _ = raise Id

fun fsigId(FSIG{argument,body,...}) =
      ModuleId.FSIGid{argument=onlyFree(getSignStamp(argument)), 
		      body=onlyFree(getSignStamp(body))}
  | fsigId(EXTERN_FSIG id) = id
  | fsigId _ = raise Id

fun fctId(FCT{stamp,...}) = ModuleId.FCTid (onlyFree stamp)
  | fctId(FCT_INSTANCE{fsig,fct,...}) =
      ModuleId.FCT_INSTANCEid{fsig=fsigId fsig, fct=fctId fct}
  | fctId(FCT_EXTERN id) = id
  | fctId _ = raise Id

fun tycId(GENtyc{stamp,...}) = ModuleId.TYCid (onlyFree stamp)
  | tycId(EXTERNtyc id) = id
  | tycId _ = raise Id

(* basic function to make a new structure *)
fun mkStructure (env,path)=
  let val stamp = Stamps.newFree()
   in SIMPLE {stamp=stamp,env=env,path=path, id=ModuleId.SIMPLEid stamp}
  end

(* appends two dynamic access path *)
fun appendAccess (acc as SLOT s,NO_ACCESS) = acc  (* tranforming INSTANCE env
						     to SIMPLE env *)
  | appendAccess (SLOT s, l) = PATH(s,l)
  | appendAccess (acc as INLINE _, l) = acc
  | appendAccess (acc,_) =
       (* hack so that we can print structures w/ bogus access paths *)
	    if !Control.internals then acc
	    else error "appendAccess"

(* translate a numeric path to the corresponding type in a given structure *)
fun transPosTycon str path =
    let fun f ([pos],INSTANCE{types,...}) = Array.sub(types,pos)
	  | f (h::t,INSTANCE{subStrs,...}) = f(t,Array.sub(subStrs,h))
	  | f (_,ERROR_STR) = ERRORtyc
	  | f (path,APPLY{res,...}) = f(path,res)
	  | f _ = error "transPosTycon 1"
     in f (path,str) handle General.Subscript =>
           (Control.Print.say "path: ";
	    PrintUtil.prIntPath path;
	    error "transPosTycon 2")
    end

(* translate a numeric path to the corresponding functor *)
fun transPosFct str path =
    let fun f ([pos],INSTANCE{subFcts,...}) = Array.sub(subFcts,pos)
	  | f (h::t,INSTANCE{subStrs,...}) =  f(t,Array.sub(subStrs,h))
	  | f (_,ERROR_STR) = ERROR_FCT
	  | f (path,APPLY{res,...}) = f(path,res)
	  | f _ = error "transPosFct 1"
     in f (path,str) handle General.Subscript =>
	    (Control.Print.say "path: ";
	     PrintUtil.prIntPath path;
	     error "transPosFct 2")
    end

(* translate a numeric acces path into the corresponding structure *)
fun transPosStr str path =
    let fun f ([],str) = str
	  | f (h::t,INSTANCE{subStrs,...}) = f(t,Array.sub(subStrs,h))
	  | f (p,APPLY{res,...}) = f(p,res)
	  | f (_,ERROR_STR) = ERROR_STR
	  | f _ = error "transPosStr 1"
     in f (path,str) handle General.Subscript =>
	    (Control.Print.say "path: ";
	     PrintUtil.prIntPath path;
	     error "transPosStr 2")
    end

(* translate a type in a given context *)
fun transType (str as INSTANCE {subStrs, types,...}) =
    let fun findFormTyc(path, tycIndex) = transPosTycon str (path@[tycIndex])

	fun transTycon (RELtyc {pos=tycAddress,...}) = findFormTyc tycAddress
	  | transTycon tyc = tyc

	fun transType0 ty =
	    case ty
	    of VARty _ => ty
	     | IBOUND _ => ty
	     | CONty (tc, tl) =>
		 mkCONty (transTycon tc, map transType0 tl)
	     | POLYty {sign, tyfun=TYFUN {arity, body}, abs} =>
		 POLYty{sign=sign,
			tyfun=TYFUN{arity=arity,body=transType0 body},
			abs=abs}
	     | UNDEFty => ty
	     | WILDCARDty => ty
     in transType0
    end
  | transType (SIMPLE _) = (fn ty => ty)
  | transType (APPLY{res,...}) = transType res
  | transType _ = error "transtype"

(* transBindingINSTANCE: Structure array * tycon array * int list
			   -> Modules.binding -> Modules.binding

   The binding argument is assumed to be from the environment of
   signature of the structure (which must be an INSTANCE), so its access,
   if any, will be a SLOT, an OPEN, or INLINE.
   transBindingINSTANCE interprets types of value and constructor bindings,
   and adjusts access info to make it absolute (i.e. relative to the
   top-level dynamic environment).  The int list argument is the access
   path of the structure argument. *)

fun transBindingINSTANCE (str,subStrs,types,subFcts,apath:Access.access) binding =
    let val transType = transType str
	(* invariant: Any structure binding in the sign of an
	   INSTANCE structure is a FORMAL *)
     in case binding
	  of VARbind (VALvar {access, path, typ}) =>
		  VARbind (VALvar {access=appendAccess (access, apath),
				   path=path,typ=ref(transType(!typ))})
	   | VARbind (OVLDvar _) => error "Modules.transBindingINSTANCE"
	   | VARbind ERRORvar => binding
           | CONbind (DATACON {name, const, typ, rep, sign, orig}) =>
                         (* ask zsh why he did the following hack ... *)
                (let fun lookupOrig([],n,d) = d
                       | lookupOrig((DATACON{name=n',orig=z,...})::r,n,d) =
                           if Symbol.eq(n',n) then z else lookupOrig(r,n,d)

                     val typ' = transType typ
                     val rep' = (case rep
                                  of VARIABLE access =>
				       VARIABLE (appendAccess(access,apath))
				   | VARIABLEc access =>
				       VARIABLEc (appendAccess (access,apath))
				   | _ => rep)
                     val dty = 
                        (case ((reduce typ'),const)
                          of (x,true) => x
                           | (CONty(_,[_,y]),false) => reduce y
                           | _ => error "impossible dcon type in ModuleUtil")
                     val orig' = 
                        (case dty 
                          of CONty(GENtyc({kind=ref(DATAtyc dcons),...}),_) =>
                                                 lookupOrig(dcons,name,orig)
                           | _ => orig)
                  in CONbind (DATACON {name=name,const=const,rep=rep',
                                       sign=sign,orig=orig',typ=typ'})
                 end)
	   | TYCbind (FORMtyc {pos, ...}) => TYCbind (Array.sub(types,pos))
	   | TYCbind (OPENFORMtyc {pos=(path,pos), ...}) =>
		TYCbind (transPosTycon str (path @ [pos]))
	   | SIGbind _ => binding
	   | STRbind (STRvar {name, access, binding=STR_FORMAL{pos, ...}}) =>
		 STRbind (STRvar {access=appendAccess (access, apath),
				  name=name,binding=Array.sub(subStrs,pos)})
	   | STRbind (STRvar {name, access, binding=STR_OPEN{pos, ...}}) =>
		 STRbind (STRvar {access=appendAccess (access, apath),
				  name=name,binding=transPosStr str pos})
	   | FCTbind (FCTvar {name, access, binding=FCT_FORMAL{pos, ...}}) =>
		 FCTbind (FCTvar {access=appendAccess (access, apath),
				  name=name,binding=Array.sub(subFcts,pos)})
	   | FCTbind (FCTvar {name, access, binding=FCT_OPEN{pos, ...}}) =>
		 FCTbind (FCTvar {access=appendAccess (access, apath),
				  name=name,binding=transPosFct str pos})
	   | _ => binding
   end

(* transBindingSIMPLE: int list -> Modules.binding -> Modules.binding
 * just adjusts access fields of bindings.  bindings assumed to come
 * from a SIMPLE structure, and the int list is its top-level access path 
 *)

fun transBindingSIMPLE apath binding =
    case binding
      of VARbind (VALvar {access, path, typ}) =>
	   VARbind (VALvar {access=appendAccess (access, apath),
			    path=path,typ=typ})
       | CONbind (DATACON {name, const, typ, rep, sign, orig}) =>
	   CONbind (DATACON {name=name, const=const, sign=sign, orig=orig,
			     rep=
			     case rep
			     of VARIABLE access =>
				  VARIABLE(appendAccess (access, apath))
			      | VARIABLEc access =>
				  VARIABLEc(appendAccess(access,apath))
			      | rep => rep,
			     typ=typ})
       | STRbind (STRvar {name, access, binding}) =>
	   STRbind (STRvar {access=appendAccess (access, apath),
			    name=name,binding=binding})
       | FCTbind (FCTvar {name, access, binding}) =>
	   FCTbind (FCTvar {access=appendAccess (access, apath),
			    name=name,binding=binding})
       | binding => binding

(* convert formal bindings to opened, doesn't recompute types. *)

fun transBindingFORMAL (relpath,spath: SP.path) binding = 
    case binding
      of TYCbind(FORMtyc{name,pos,spec}) =>
	   TYCbind(OPENFORMtyc{pos=(relpath,pos),spec=spec,
			       path=SP.extend(spath,name)})
       | STRbind (STRvar{name,access,binding=STR_FORMAL{pos,spec}}) =>
	   STRbind(STRvar{name=name,access=access,
			  binding=STR_OPEN{pos=relpath@[pos],spec=spec,
					   path=SP.extend(spath,name)}})
       | FCTbind (FCTvar{name,access,binding=FCT_FORMAL{pos,spec}}) =>
	   FCTbind(FCTvar{name=name,access=access,
			  binding=FCT_OPEN{pos=relpath@[pos],spec=spec,
					   path=SP.extend(spath,name)}})
       | _ => binding

(* err: raise an exception when an unbound component in 
   the symbolic path is found.  It is passed the remainder
   of the symbolic path, including the unbound component.
   From this it computes the symbolic path to the unbound
   component. *)

fun errorUnbound (SP.SPATH spath: SP.path) (SP.SPATH suffix: SP.path) =
    (* assumption: suffix is a suffix of spath and length(suffix) > 0 *)
    case suffix  (* check that length suffix > 0 *)
      of (h::t) =>
	  let val diffPath =
		  SP.SPATH
		    (ListUtil.prefix(length spath - length suffix + 1, spath))
	   in raise UnboundComponent(diffPath)
	  end
       | _ => error "spath"

(* find a binding, adjust its access paths and interpret its types.*)

fun lookBinding (topStr, spath: SP.path, apath: Access.access) : binding =
    let val err' = errorUnbound(spath)
	fun get (str, SP.SPATH[sym], apath: Access.access) =
	     ((case str
		 of SIMPLE {env,...} =>
		      (transBindingSIMPLE apath (Env.look(env,sym)))
		  | INSTANCE {subStrs,types,subFcts,sign=SIG{env,...},...} =>
		      (transBindingINSTANCE(str,subStrs,types,subFcts,apath) 
		         (Env.look(!env,sym)))
		  | APPLY{res,...} => get (res,SP.SPATH[sym],apath)
		  | ERROR_STR => raise ErrorStructure
		  | _ => (error ("lookBinding 1 "^Symbol.name sym)))
	       handle Env.Unbound => raise UnboundComponent spath)
	  | get (str, spath as SP.SPATH(h::t), apath) =
	      let fun get_str(str,path) =
		     (case str
		       of SIMPLE {env,...} =>
			   (case Env.look(env,h)
			      of STRbind(STRvar
				   {access=SLOT slot,binding=str,...}) =>
				     (str,PATH(slot,path))
			       | _ => error "lookBinding 2 ")
			| INSTANCE{sign=SIG{env,...},subStrs,...} =>
			   (case Env.look(!env,h)
			      of STRbind(STRvar{binding=STR_FORMAL{pos,...},
						access=SLOT slot, ...}) =>
				   (Array.sub(subStrs,pos),PATH(slot,path))
			       | STRbind(STRvar{binding=STR_OPEN{pos,...},
						access=SLOT slot, ...}) =>
				   (transPosStr str pos,PATH(slot,path))
			       | _ => (error "lookBinding 3"))
			| APPLY{res,...} => get_str(res,path)
			| ERROR_STR => raise ErrorStructure
			| _ => error "lookUnadjusted 2")
		      handle Env.Unbound => err' spath
		  val (str', apath') = get_str(str,apath)
	       in get(str', SP.SPATH t, apath')
	      end
	  | get _ = error "Modules.lookBinding DD95"
      in get(topStr,spath,apath)
     end

local
fun lookBinding' (topStr, spath: SP.path) : binding =
    let val err' = errorUnbound(spath)
	fun get (str, SP.SPATH [sym]) =
	     ((case str
		 of SIMPLE {env,...} =>
		      Env.look(env,sym)
		  | INSTANCE {subStrs,types,subFcts,sign=SIG{env,...},...} =>
		      transBindingINSTANCE(str,subStrs,types,subFcts,NO_ACCESS) 
						(Env.look(!env,sym))
		  | APPLY{res,...} => get (res,SP.SPATH[sym])
		  | ERROR_STR => raise ErrorStructure
		  | _ => (error ("lookBinding' 1 "^Symbol.name sym)))
	       handle Env.Unbound => raise UnboundComponent spath)
	  | get (str, spath' as SP.SPATH(h::t)) =
	      let fun get_str str =
		   (case str
		     of SIMPLE {env,...} =>
			 (case Env.look(env,h)
			  of STRbind(STRvar{binding=str,...}) => str
			   | _ => error "lookBinding' 4")
		      | INSTANCE{sign=SIG{env,...},subStrs,...} =>
			 (case Env.look(!env,h)
			  of STRbind(STRvar{binding=STR_FORMAL{pos,...},...}) =>
			       Array.sub(subStrs,pos)
			   | STRbind(STRvar{binding=STR_OPEN{pos,...}, ...}) =>
			       transPosStr str pos
			   | _ => error "lookBinding' 3")
		      | APPLY{res,...} => get_str res
		      | ERROR_STR => raise ErrorStructure
		      | _ => error "lookBinding' 2")
		    handle Env.Unbound => err' spath'
		  val str' = get_str str
	       in get(str', SP.SPATH t)
	      end
	  | get _ = error "Modules.lookBinding DD95"
     in get(topStr,spath)
    end

in

(* lookBindingSTR: look up a structure binding, return only the Structure *)
fun lookBindingSTR (str,spath) =
    (case lookBinding' (str,spath)
     of STRbind(STRvar{binding,...}) => binding
      | _ => error "lookBindingSTR")
    handle ErrorStructure => ERROR_STR

(* lookBindTYC: look up a type binding *)

fun lookBindingTYC (str,spath) =
    (case lookBinding' (str,spath)
     of TYCbind tyc => tyc
      | _ => error "lookBindingTYC")
    handle ErrorStructure => ERRORtyc
end

(* builds an environment from a structure *)
fun makeEnv (str as INSTANCE{sign=SIG{env,...},subStrs,types,subFcts, ...}, apath) =
      Env.open'(!env, transBindingINSTANCE(str,subStrs,types,subFcts,apath),
		Env.empty)
  | makeEnv (str as SIMPLE{env, ...}, apath) =
      Env.open'(env, transBindingSIMPLE(apath), Env.empty)
  | makeEnv (ERROR_STR, _) = Env.empty
  | makeEnv (INSTANCE{sign=FULL_SIG,...}, _) = 
      error "makeEnv 1"
  | makeEnv (APPLY{res,...},apath) = makeEnv(res,apath)
  | makeEnv _ = error "makeEnv 2"

(* should be in Symbol *)
val symbolToName = fn s => Symbol.nameSpaceToString(Symbol.nameSpace s)

(* look for a signature (necessaraly top) *)
fun lookSIG (env,id,err) = 
    (case Env.look(env,id) 
       of SIGbind(SIGvar{binding,...}) => binding
        | _ => error "lookSIG")
    handle Env.Unbound =>
	    (err COMPLAIN ("unbound signature: "^Symbol.name id) nullErrorBody;
	     ERROR_SIG)
	 | Bind =>
	     error 
	       ("lookSIG: bind exception looking up "
		^ Symbol.name id^" in name space "^symbolToName id)

(* look for a functor signature *)
fun lookFSIG (env,id,err) = 
    (case Env.look(env,id) 
       of  FSIGbind(FSIGvar{binding,...}) => binding
        | _ => error "lookFSIG")
    handle Env.Unbound =>
	     (err COMPLAIN ("unbound funsig: "^Symbol.name id) nullErrorBody;
	      ERROR_FSIG)
	 | Bind =>
	     error
	       ("lookFSIG: bind exception looking up "
		^ Symbol.name id^" in name space "^symbolToName id)

(* fixity bindings *)
fun lookFIX (env,id) =
    let val binding = 
	  case Env.look(env,id)
	    of FIXbind(FIXvar{binding,...}) => binding
	     | _ => error "lookFIX"
     in binding
    end
    handle Env.Unbound =>  Fixity.NONfix
	 | Bind => error ("lookFix: bind exception looking up "^Symbol.name id
			  ^" in name space "^symbolToName id)


(* lookFormalBinding: given a symbolic path, find a formal binding.
   Also return a relative path to it.*)
fun lookFormalBinding(env,spath: SP.path) : binding * int list =
    let val err' = errorUnbound(spath)
	fun get (env,SP.SPATH[sym],p) =
	      ((Env.look (env,sym),rev p)
	       handle Env.Unbound => raise (UnboundComponent spath))
	  | get (env,spath as SP.SPATH(first::rest),p) =
	     ((case Env.look (env,first)
		 of STRbind(STRvar{binding=STR_FORMAL{pos,
						      spec=SIG{env,kind,...}},
				   ...}) =>
		      get(!env, SP.SPATH rest,
			  case !kind of EMBEDDED => p | _ => pos::p)
		  | STRbind(STRvar{binding=STR_FORMAL{spec=ERROR_SIG,...},...}) =>
		      raise ErrorStructure
		  | STRbind(STRvar{binding=ERROR_STR,...}) =>
		      raise ErrorStructure
		  | _ => error "lookFormalBinding 1")
	      handle Env.Unbound => err' spath)
	  | get _ = error "lookFormalBinding 2"
     in get (env,spath,[])
    end

(* lookGen: generic lookup function for identifiers which may occur 
   in:
       1. environments
       2. actual structure environments
       3. signature parsing environments *)

fun lookGen (extract,errorVal) (env,path,err) =
    (case path
       of SP.SPATH [id] => extract ((Env.look(env,id),nil,path)
			   handle Env.Unbound => raise UnboundComponent path)
	| SP.SPATH(first::rest) =>
	    let val rest = SP.SPATH rest
		val strvar = 
		  (case Env.look(env,first)
		     of STRbind(STRvar strvar) => strvar
		      | obj =>  error "lookGen 3")
		  handle Env.Unbound =>
		    raise UnboundComponent(SP.SPATH[first])
	     in case strvar
		  of {binding=STR_FORMAL{pos,spec=SIG{env,kind,...}},...} =>
		       let val (binding,relpath) = lookFormalBinding(!env,rest)
			in extract (binding,
				    case !kind 
				      of EMBEDDED => relpath
				       | _ => pos :: relpath,path)
		       end
		   | {binding=STR_OPEN{pos,spec=SIG{env,...},path=path'},...} =>
		       let val (binding,relpath) = lookFormalBinding(!env,rest)
		        in extract(binding,pos@relpath,SP.append(path',rest))
		       end
		   | {binding=STR_FORMAL{spec=ERROR_SIG,...},...} =>
		       raise ErrorStructure
		   | {binding,access,...} =>
		       extract(lookBinding(binding,rest,access), nil, path)
	    end
	| _ => error "lookGen 2")
    handle UnboundComponent spath => 
	     let val badsym = SP.last spath
	      in err COMPLAIN ("unbound "^symbolToName badsym^": "^
			       Symbol.name badsym^
			       (if SP.length path > 1
			        then " in path "^(SP.makestring path)
			        else ""))
		     nullErrorBody;
		 errorVal
	     end
	 | ErrorStructure => errorVal
	 | Bind => error ("bind exception: lookGen: looking up "
			  ^(SP.makestring path)
			  ^" as a "^symbolToName (SP.last path))
	 | exn => raise exn

(* look for a variable or a constructor (simple path) *)
fun lookShortVARCON (arg as (env,name,err)) =
    Env.look(env,name)
    handle Env.Unbound => 
	     (err COMPLAIN ("unbound "^symbolToName name^" "^
			    Symbol.name name)
		  nullErrorBody;
	      bogusCONbind)

(* look for a variable or a constructor (complete path) *)
val lookVARCON = lookGen((fn (x,_,_) => x), bogusCONbind)

(* look for a structure *)
val lookSTR = lookGen ((fn (STRbind sv,_,_) => sv
			 | _ => error "lookSTR"),
		       bogusSTRvar)

(* look for a functor *)
val lookFCT = lookGen ((fn (FCTbind sv,_,_) => sv
			 | _ => error "lookFCT"),
		       bogusFCTvar)

(* look for a type *)
val lookTYC =
    lookGen (fn (TYCbind(FORMtyc{pos,spec,...}),relpath,spath) =>
		   RELtyc{path=ConvertPaths.invertSPath spath,
			  pos=(relpath,pos),spec=spec}
	       | (TYCbind tyc,_,_)=> tyc
	       | _ => error "lookTYC",
	     ERRORtyc)

(* tycon lookup with arity checking *)

fun checkArity(tycon, arity,err,result) =
    case tycon
      of ERRORtyc => result
       | _ =>
	 if tyconArity(tycon) <> arity
	 then (err COMPLAIN ("type constructor "^
			     (SP.makestring
			        (ConvertPaths.invertIPath(tycPath(tycon))))^
		             " given " ^ makestring arity ^ 
			     " arguments, wants "
			     ^ makestring (tyconArity tycon))
		   nullErrorBody;
	      ERRORtyc)
	 else result

fun lookArTYC ((env,normMap),qid: SP.path, arity: int, err) =
    let val normQid = Normalize.normalize(normMap,qid)
     in lookGen (fn (TYCbind (FORMtyc {pos,spec,...}),relpos,spath) =>
	             checkArity(spec,arity,err,
				RELtyc{path=ConvertPaths.invertSPath spath,
				       pos=(relpos,pos),spec=spec})
                  | (TYCbind (OPENFORMtyc {pos,spec,path,...}),[],_) =>
		     checkArity(spec,arity,err,
				RELtyc{path=ConvertPaths.invertSPath path,
				       pos=pos,spec=spec})
		  | (TYCbind (OPENFORMtyc _),_,_) =>
		     error "lookArTyc 1"
		  | (TYCbind tyc,_,_) => checkArity(tyc,arity,err,tyc)
		  | _ => error "lookArTyc 2",
		 ERRORtyc)
	        (env,normQid,err)
    end

(* looking for an exception *)
fun lookEXN (env,path,err) =
    let val binding =
	  case path
	    of SP.SPATH [id] =>
		 (Env.look(env,id)
		  handle Env.Unbound => raise UnboundComponent path)
	     | SP.SPATH(first::rest) =>
		 ((case Env.look(env,first)
		    of STRbind(STRvar {binding,access,...}) =>
			 lookBinding(binding, SP.SPATH rest, access)
		     | _ => error "ModuleUtl.lookExn 3")
		   handle Env.Unbound =>
		     raise UnboundComponent(SP.SPATH[first]))
	     | _ => error "lookExn 2"
    in case binding
	 of CONbind c =>
	      (case c
	       of DATACON {rep=VARIABLE _,...} => c
		| DATACON {rep=VARIABLEc _,...} => c
		| _ => (err COMPLAIN ("found data constructor \
				     \instead of exception")
			    nullErrorBody;
			bogusEXN))
	  | VARbind _ =>
	      (err COMPLAIN ("found variable instead of exception")
	           nullErrorBody;
	       bogusEXN)
	  | _ => error("lookEXN: looking up " ^
		       (SP.makestring path) ^
		       " as a " ^ symbolToName (SP.last path))
   end
   handle UnboundComponent spath => 
	    (err COMPLAIN ("unbound " ^
			  (if SP.length path = SP.length spath
			   then "exception "
			   else "structure ") ^
			  Symbol.name(SP.last spath)^
			  (if SP.length path > 1
			   then " in path "^(SP.makestring path)
			   else ""))
		 nullErrorBody;
	     bogusEXN)
	| ErrorStructure => bogusEXN
	| exn => raise exn

fun openSigStructure (bindEnv,spath: SP.path,baseEnv,complainer) =
    let fun makeEnv (str as INSTANCE{sign=SIG{env,...},subStrs,types,subFcts, ...},
		     _,_) =
	      Env.open'(!env,transBindingINSTANCE(str,subStrs,types,subFcts,
						  NO_ACCESS),
			baseEnv)
	  | makeEnv (str as SIMPLE{env, ...},_,_) =
	      Env.open' (env, fn x =>x, baseEnv)
	  | makeEnv (ERROR_STR, _,_) = Env.empty
	  | makeEnv (STR_FORMAL{pos,spec=SIG {env,kind,...}},relpath,spath) =
	      let val relpath' = case !kind
				   of EMBEDDED => relpath
				    | _ => relpath @ [pos]
	       in Env.open'(!env, transBindingFORMAL(relpath',spath), baseEnv)
	      end
	  | makeEnv (STR_FORMAL{spec=ERROR_SIG,...},_,_) = baseEnv
	  | makeEnv (STR_OPEN{pos,spec=SIG{env,...},path},nil,_) =
	      Env.open'(!env,transBindingFORMAL(pos,path),baseEnv)
	  | makeEnv (STR_OPEN{spec=ERROR_SIG,...},nil,_) = baseEnv
	  | makeEnv (STR_OPEN _,_,_) =
	      error "openSigStructure.makeEnv.STR_OPEN"
	  | makeEnv _ = error "openSigStructure.makeEnv"
     in lookGen(fn (STRbind(STRvar{binding,...}),relpath,spath':SP.path) =>
		      makeEnv(binding,relpath,spath')
		 | _ => error "openSigStructure",
		bindEnv) (bindEnv,spath,complainer)
    end

fun openStructureVar (env,STRvar{access=p,binding=str,...}) : env =
      Env.atop (makeEnv (str, p), env)
     (* will generate spurious error messages unless we give up completely *)

(* findPath:  convert inverse symbolic path names to a printable string in the
  context of an environment.

  Its arguments are the inverse symbolic path, a static semantic value,
  an equality function on static semantic values, and a lookup function
  mapping paths to their bindings (if any) in an environment.   The second
  argument of the lookup function is a function which is called if there
  is no binding for a path name in the environment.

  It looks up each suffix of the path name, going from shortest to longest
  suffix,in the current environment until it finds one whose lookup value
  equals the static semantic value argument.  It then converts that suffix
  to a string.  If it doesn't find any suffix, it returns "?" concatenated
  with the full path name.

  Example:
	 Given A.B.t as a path, and a lookup function for an
	 environment, this function tries:
		   t
		   B.t
		   A.B.t
	 If none of these work, it returns ?.A.B.t

  Note: the symbolic path is passed in reverse order because that is
  the way all symbolic path names are stored within static semantic objects.
 *)
    
fun findPath (p: IP.path, elem0, eq, look): string =
    let fun try(IP.IPATH(name::untried),tried) =
	      (let val elem = look(SP.SPATH(name :: tried),
				   fn _ => raise Env.Unbound)
	        in if eq(elem0,elem)
		   then SP.makestring(SP.SPATH(name::tried))
		   else try(IP.IPATH untried,name::tried)
	       end handle Env.Unbound =>
		     try(IP.IPATH untried,name::tried))
	  | try(IP.IPATH[],tried) =
	      "?." ^(SP.makestring(SP.SPATH tried))
     in try(p,[])
    end

(* sortEnvBindings: sort the bindings in an environment for printing
  purposes.  The bindings are sorted in the following order:
	     signatures
	     functors
	     structures
	     types
	     constructors
	     values
	     fixity declarations
 It is only correct to sort environments which have no duplicate bindings.
 All routines which build structure environments maintain this
 invariant, so it is ok to sort any structure environment using
 this function.
*)

local
  open Symbol
  fun binderGt(bind1: symbol * Modules.binding,
		bind2: symbol * Modules.binding) =
      case (bind1,bind2)
	of ((n1,FIXbind _),(n2,FIXbind _)) => symbolGt(n1,n2)
	 | ((_,FIXbind _),_) => true
	 | (_,(_,FIXbind _)) => false
	 | ((n1,VARbind _),(n2,VARbind _)) => symbolGt(n1,n2)
	 | ((_,VARbind _),_) => true
	 | (_,(_,VARbind _)) => false
	 | ((n1,CONbind _),(n2,CONbind _)) => symbolGt(n1,n2)
	 | ((_,CONbind _),_) => true
	 | (_,(_,CONbind _)) => false
	 | ((n1,TYCbind _),(n2,TYCbind _)) => symbolGt(n1,n2)
	 | ((_,TYCbind _),_) => true
	 | (_,(_,TYCbind _)) => false
	 | ((n1,STRbind _),(n2,STRbind _)) => symbolGt(n1,n2)
	 | ((_,STRbind _),_) => true
	 | (_,(_,STRbind _)) => false
	 | ((n1,FCTbind _),(n2,FCTbind _)) => symbolGt(n1,n2)
	 | ((_,FCTbind _),_) => true
	 | (_,(_,FCTbind _)) => false
	 | ((n1,SIGbind _),(n2,SIGbind _)) => symbolGt(n1,n2)
	 | ((_,SIGbind _),_) => true
	 | (_,(_,SIGbind _)) => false
	 | ((n1,FSIGbind _),(n2,FSIGbind _)) => symbolGt(n1,n2)
  (*     | (_,(_,FSIGbind _)) => false
	 | ((_,FSIGbind _),_) => true   redundant*)
in
  fun sortEnvBindings env =
       let val bl : (Symbol.symbol * Modules.binding) list ref = ref nil
        in Env.app(fn b => bl := b :: !bl) env;
	   Sort.sort binderGt (!bl)
       end
end

  (* notInitialLowerCase:  this function not currently used.  It could be
     used to detect anomalous noncapitalization of constructors. *)

  fun notInitialLowerCase string =
      (* string does NOT start with lower-case alpha *)
      let val firstchar = String.sub(string,0)
       in firstchar < #"a" orelse firstchar > #"z"
      end

fun getStrPath (SIMPLE {path, ...}) = path
  | getStrPath (INSTANCE {path,...}) = path
  | getStrPath _ = IP.IPATH []
       (* bogus empty path returned -- should be error or exception? *)

fun getStrPos str (sym: Symbol.symbol) = 
    case lookBindingSTR(str, SP.SPATH [sym])
      of (STR_FORMAL {pos,...}) => pos
       | _ => error "ModuleUtil:getStrPos"

fun getStrTPos str sym =
    case lookBindingTYC (str, SP.SPATH [sym])
      of (FORMtyc {pos, ...}) => pos
       | _ => error "ModuleUtil:getStrTPos"

fun getSigPosGen (SIG {env, ...}) sym =
      (Env.look(!env,sym)
       handle Env.Unbound => error "ModuleUtil:getSigPos 1")
  | getSigPosGen _ _ = error "ModuleUtil:getSigPos 2"

fun getSigTPos (SIG {env, ...}) sym =
    (case Env.look(!env,sym)
       of TYCbind (FORMtyc {pos, ...}) => pos
	| _ => error "ModuleUtil:getSigTPos.1")
  | getSigTPos _ _ = error "ModuleUtil:getSigPos.2"

fun getSigPos (sign as SIG {env, ...}) sym =
    ((case Env.look(!env,sym)
	of (STRbind (STRvar {binding=STR_FORMAL {pos, ...},...})) => pos
	 | _ => error "ModuleUtil:getSigPos.1")
     handle Env.Unbound => 
       error ("ModuleUtil:getSigPos.2"^(Symbol.name sym)))
  | getSigPos _ _ = error "ModuleUtil:getSigPos.2"

fun eqSignature (SIG {stamp=s1,...}, SIG {stamp=s2,...}) = 
      Stamps.eq(s1,s2)
  | eqSignature _ = false


end  (* structure ModuleUtil *)
