(* hppaProps.sml
 *
 * COPYRIGHT (c) 1996 Bell Laboratories.
 *)

structure HppaProps = struct
  structure I = HppaInstruction
  structure C = HppaCells

  val itow      = Word.fromInt
  val wtoi      = Word.toInt
  
  fun error msg = ErrorMsg.impossible ("HppaProps." ^ msg)

  fun low11(n) = wtoi(Word.andb(itow n, 0wx7ff))
  fun high21(n) = wtoi(Word.~>>(itow n, 0w11))

  datatype kind = IK_JUMP | IK_NOP | IK_INSTR
  datatype target = LABELLED of Label.label | FALLTHROUGH | ESCAPES

  (* Note: Even though a CALL instruction expands to BLE 
   * it is not viewed as a branch instruction.
   *)  
  fun instrKind(I.BCOND _)  = IK_JUMP
    | instrKind(I.BCONDI _) = IK_JUMP
    | instrKind(I.B _)      = IK_JUMP
    | instrKind(I.FBCC _)   = IK_JUMP
    | instrKind(I.BV _)     = IK_JUMP
    | instrKind(I.NOP)      = IK_NOP
    | instrKind _	    = IK_INSTR

  fun branchTargets(I.BCOND{t, ...})    = [LABELLED t, FALLTHROUGH]
    | branchTargets(I.BCONDI{t, ...})   = [LABELLED t, FALLTHROUGH]
    | branchTargets(I.B(lab))           = [LABELLED lab]
    | branchTargets(I.FBCC{t,...})      = [LABELLED t, FALLTHROUGH]
    | branchTargets(I.BV{labs=[],...})  = [ESCAPES]
    | branchTargets(I.BV{labs,...})     = map LABELLED labs
    | branchTargets _ = error "branchTargets"

  fun nop() = I.NOP

  fun defUseR instr = let
    fun trap((I.ADDO | I.SUBO | I.SH1ADDO), d, u) = (d, C.exnptrR::u)
      | trap(_, d, u) = (d, u)
    fun trapi((I.ADDIO | I.SUBIO), d, u) = (d, C.exnptrR::u)
      | trapi(_, d, u) = (d, u)
  in
    case instr
     of I.STORE {b, r,...}          => ([],  [b,r])
      | I.ARITH {a, r1, r2, t, ...} => trap(a, [t], [r1,r2])
      | I.ARITHI {ai, r, t, ...}    => trapi(ai, [t], [r])
      | I.COMCLR{r1, r2, t, ...}    => ([t], [r1, r2])
      | I.SHIFTV {r, t, ...}        => ([t], [r])
      | I.SHIFT {r, t, ...}         => ([t], [r])
      | I.BCOND {r1, r2, ...}       => ([],  [r1,r2])
      | I.BCONDI {r2, ...} 	    => ([],  [r2])
      | I.BV {x, b, ...}	    => ([],  [x,b])
      | I.CALL{t, defs, uses}       => (#1 defs, t :: #1 uses)
      | I.LDIL{i, t}		    => ([t], [])
      | I.LDO{b, t, ...}	    => ([t], [b])
      | I.LADDR{t, ...}		    => ([t], [])
      | I.MTCTL{r, t}		    => ([],  [r])
      | I.FSTORE {b, ...}	    => ([],  [b])
      | I.FSTOREX {b, x, ...}  	    => ([],  [b,x])
      | I.FLOAD {b, ...}	    => ([],  [b])
      | I.FLOADX{b, x, ...} 	    => ([],  [b,x])
      | _   => ([],[])
  end

  fun defUseF instr = 
    case instr
      of I.FSTORE {r, ...}  	   => ([], [r])
       | I.FSTOREX{r, ...}	   => ([], [r])
       | I.FLOAD{t, ...}	   => ([t], [])
       | I.FLOADX{t, ...}	   => ([t], [])
       | I.FARITH {r1, r2, t, ...} => ([t], [r1,r2])
       | I.FUNARY {f, t, ...}      => ([t], [f])
       | I.FCMP  (_, f1, f2)	   => ([], [C.exnptrR, f1, f2])
       | I.CALL{defs, uses, ...}   => (#2 defs, #2 uses)
       | _ => ([],[])

  fun defUseM instr = let
    val MEM   = 64
    val STACK = 65
    val C     = 66
    val CR11  = 67
    (* it is not easy to distinguish between STACK and MEM *)
  in
    case instr 
     of I.STORE _   => ([MEM], [])
      | I.FSTORE _  =>  ([MEM], [])
      | I.FSTOREX _ => ([MEM], [])
      | I.ARITH{a=I.LDWX, ...} => ([], [MEM]) 
      | I.ARITH{a=I.LDHX, ...} => ([], [MEM]) 
      | I.ARITH{a=I.LDBX, ...} => ([], [MEM]) 
      | I.ARITHI{ai=I.LDW, ...} => ([], [MEM]) 
      | I.ARITHI{ai=I.LDH, ...} => ([], [MEM]) 
      | I.ARITHI{ai=I.LDB, ...} => ([], [MEM]) 
      | I.MTCTL _ =>  ([CR11], [])
      | I.SHIFTV _ => ([], [CR11])
      | I.FCMP _  => ([C], [])
      | I.FTEST => ([], [C])
      | _ => ([], [])
  end

  fun defUse instr = let
    val (rd, ru) = defUseR instr
    val (fd, fu) = defUseF instr
    val (md, mu) = defUseM instr
  in
    ((rd,fd,mu), (ru,fu,mu))
  end

  local
    val initialSpillOffset = 112	(* from runtime system *)
    val spillOffset = ref initialSpillOffset
    fun newOffset n = 
      if n > 4096 then error "incOffset - spill area too small"
      else spillOffset := n
  in
    fun spillInit () = spillOffset := initialSpillOffset

    fun spill1 r = let
      val offset = !spillOffset
    in
      newOffset(offset+4);
      ([I.STORE{st=I.STW, b=C.stackptrR, d= ~offset, r=r}],
       [I.ARITHI{ai=I.LDW, i= ~offset, r=C.stackptrR, t=r}])
    end

    fun spill2 f = let
      val n = !spillOffset
      val aligned = Word.toInt (Word.andb(itow (n+7), itow ~8))
    in
      newOffset(aligned+8);
      if aligned < 16 andalso aligned >= ~16 then
        ([I.FSTORE{fst=I.FSTDS, b=C.stackptrR, d= ~aligned, r=f}],
	 [I.FLOAD{fl=I.FLDDS, b=C.stackptrR, d= ~aligned, t=f}])
      else let
	  val tmpR = C.newReg()
	  val spill =
	    [I.LDIL{i=I.IMMED(high21(~aligned)), t=tmpR},
	     I.LDO{i=I.IMMED(low11(~aligned)), b=tmpR, t=tmpR},
	     I.FSTOREX{fstx=I.FSTDX, b=C.stackptrR, x=tmpR, r=f}]
	  val reload =
	    [I.LDIL{i=I.IMMED(high21(~aligned)), t=tmpR},
	     I.LDO{i=I.IMMED(low11(~aligned)), b=tmpR, t=tmpR},
	     I.FLOADX{flx=I.FLDDX, b=C.stackptrR, x=tmpR, t=f}]
	in
	  (spill, reload)
	end
    end

    fun spill3 _ = error "spill3"
  end

  structure GR = GetReg(val nRegs = 32 val available = C.availRegs)
  structure FR = GetReg(val nRegs = 32 val available = C.availFRegs)

  val getreg1 = GR.getreg
  val getreg2 = FR.getreg
  val getreg3 = fn _ => error "getreg3"

  val branchDelayedArch = false

  fun latency _ = 1

  fun isSdi(I.BCOND _)  = true
    | isSdi(I.BCONDI _) = true
    | isSdi(I.B _)      = true
    | isSdi(I.FBCC _)   = true
    | isSdi(I.LADDR _)  = true
    | isSdi _           = false

  fun im12 n = ~2048 <= n andalso n < 2048
  fun im17 n = ~65536 <= n andalso n < 65536

  fun sdiSize(instr, labMap, loc) = let
    fun branchOffset lab = ((labMap lab) - loc - 8) quot 4
    fun branch lab = let
      val offset = branchOffset lab
    in
      if im12 offset then 4 else if im17 offset then 8 else 20
    end
  in
    case instr 
     of I.LADDR{i=I.LabExp lexp, ...} => let
	  fun lexpVal(I.POSLAB(lab,k)) = k + labMap lab
	    | lexpVal(I.NEGLAB(lab,k)) = k - labMap lab
	  val labVal = lexpVal lexp
	in
	  if labVal >= ~8192 andalso labVal < 8192 then 4 else 12
	end
      | I.BCOND{t, ...}  => branch t
      | I.BCONDI{t, ...} => branch t
      | I.B(lab)         => if im17 (branchOffset lab) then 4 else 16
      | I.FBCC{t, ...}   => if im17 (branchOffset t) then 4 else 16
      | _		 => error "sdiSize"
  end

  fun longJump lab = let
    val baseDisp =  I.POSLAB(lab, ~8192) (* consBaseRegOffset *)
  in
    [I.LDIL{i=I.HILabExp baseDisp, t=C.asmTmpR},
     I.LDO{i=I.LOLabExp baseDisp, b=C.asmTmpR, t=C.asmTmpR},
     I.ARITH{a=I.ADD, r1=C.baseptrR, r2=C.asmTmpR, t=C.asmTmpR},
     I.BV{x=0, labs=[lab], b=C.asmTmpR}]
  end

  fun expand(I.LADDR{i=I.LabExp lexp, t, b}, size, lookup) = 
      (case size 
        of 4 => [I.LDO{i=I.LabExp lexp, b=b, t=t}]
         | 12 => [I.LDIL{i=I.HILabExp lexp, t=t},
	          I.LDO{i=I.LOLabExp lexp, b=t, t=t},
		  I.ARITH{a=I.ADD, r1=t, r2=b, t=t}]
      (*esac*))
    | expand(instr as I.BCOND{cmp,bc, t, f, r1, r2}, size, lookup) = let
	fun rev I.COMBT = I.BCOND{cmp=I.COMBF, bc=bc, t=f, f=f, r1=r1, r2=r2}
	  | rev I.COMBF = I.BCOND{cmp=I.COMBT, bc=bc, t=f, f=f, r1=r1, r2=r2}
      in
	(case size 
	  of 4 => [instr]
	   | 8 => [rev cmp, I.B(t)]
	   | 20 => rev cmp :: longJump t
	(*esac*))
      end
    | expand(instr as I.BCONDI{cmpi, bc, t, f, i, r2}, size, lookup) = let
        fun rev I.COMIBT = I.BCONDI{cmpi=I.COMIBF, bc=bc, i=i, r2=r2, t=f, f=f}
	  | rev I.COMIBF = I.BCONDI{cmpi=I.COMIBT, bc=bc, i=i, r2=r2, t=f, f=f}
      in
	(case size 
	  of 4 => [instr]
	   | 8 => [rev cmpi, I.B(t)]
	   | 20 => rev cmpi :: longJump t
	(*esac*))
      end
    | expand(instr as I.B(lab), size, lookup) =
      (case size 
	of 4 => [instr]
         | 16 => longJump lab
      (*esac*))
    | expand(instr as I.FBCC{t, f}, size, lookup) =
      (case size 
	of 4 => [I.B(t)]
         | 16 => 
	     (* lets hope this sequence never gets generated sequence:
			FTEST
			allways trapping instruction
			B (f)
			longJmp
	      *)
	        error "FBCC"
      (*esac*))
    | expand _ = error "expand"

  fun moveInstr(I.ARITH{a=I.OR, r2=0, ...}) = true
    | moveInstr(I.FUNARY{fu=I.FCPY, ...}) = true
    | moveInstr _ = false

  fun minSize _ = 4

  fun maxSize (I.LADDR _)  = 12
    | maxSize (I.BCOND _)  = 20
    | maxSize (I.BCONDI _) = 20
    | maxSize (I.B _)	   = 16
    | maxSize (I.FBCC _)   = 16
    | maxSize _		   = 4

  fun mayNeedNops _ = 0
  fun needsNop _ = 0

  val numResources = 32 + 32 + 4

  local
    fun fregRIds([], acc) = acc
      | fregRIds(f::fregs, acc) = fregRIds(fregs, 32+f::acc)
  in
    fun bdefUse [rMap, fMap] insn = let
      val mapR = map (fn r => Array.sub(rMap,r))
      val mapF = map (fn f => Array.sub(fMap,f))
      val (def, use) = defUse insn
      fun doit (r,f,e) = e @ fregRIds(mapF f, mapR r)
    in
      (doit def, doit use)
    end
  end
end
