(* hppaCells.sml
 *
 * COPYRIGHT (c) 1996 Bell Laboratories.
 *)

signature HPPACELLS = sig

  include CELLS

  val maxReg       : unit -> int
  val maxFreg      : unit -> int
  val maskRegR     : int
  val exhaustedR   : int
  val stackptrR    : int
  val baseptrR     : int
  val exnptrR      : int
  val returnPtr    : int

  val asmTmpR      : int 
  val milliarg1    : int
  val milliarg2    : int  
  val milliret	   : int

  val availRegs    : int list
  val availFRegs   : int list
  val numRegs      : int
  val numFloatRegs : int
end

structure HppaCells : HPPACELLS = struct
  structure T = MLTree
  structure SL = SortedList

  (* HPPA register conventions 
     0     zero
     1	   caller-saves
     2     return-pointer and scratch
     3-18  callee-saves
     19-22 caller-saves
     23    arg3
     24    arg2
     25    arg1
     26    arg0
     27    reserved
     28    ret0
     29    ret1
     30    stack pointer
     31    millicode return and scratch.
   *)

  (** these registers are always live and have dedicated uses **)
  (* Note: If we are really desperate for an extra register, we
   *   can make span dependent instructions use a pseudo registers that
   *   will be used as asmTmp where necessary.
   *)
  val allocptr				= T.REG 3
  val limitptr				= T.REG 4
  val storeptr				= T.REG 5
  val exnptr as T.REG exnptrR		= T.REG 6
  val varptr				= T.REG 7
  val baseptr as T.REG baseptrR		= T.REG 8
  val stackptr as T.REG stackptrR	= T.REG 30
  val asmTmpR				= 29

  (** these registers have dedicated uses but may be used as 
   ** temporaries where possible.
   **)
  val returnPtr				= 31
  val stdlink				= T.REG 9
  val stdclos				= T.REG 10
  val stdarg				= T.REG 11
  val stdcont				= T.REG 12
  val gclinkreg				= T.REG returnPtr
  val maskreg as T.REG maskRegR		= T.REG 20
  val exhausted as T.REG exhaustedR	= T.REG 21

  val milliarg1                         = 26
  val milliarg2				= 25
  val milliret				= asmTmpR

  fun fromto(n, m) = if n>m then [] else n :: fromto(n+1, m)

  val miscregs  = 
    map T.REG [1, 13, 14, 15, 16, 17, 18, 19, 22, 23, 24, 25, 26, 28, 2]
  val calleesave = Array.fromList miscregs
  val availRegs = map (fn T.REG r => r)
    ([stdlink, stdclos, stdarg, stdcont, gclinkreg, maskreg, exhausted] @ miscregs)

  (* Note: We need at least one register for shuffling purposes. *)
  val floatregs = map T.FREG (fromto(6, 31))
  val savedfpregs = []
  val availFRegs = fromto(6, 31)

  val numRegs = length miscregs + 3
  val numFloatRegs = length floatregs

  local 
    fun dedicate(lo, hi, avail) = 
      SL.difference(SL.uniq(fromto(lo,hi)), SL.uniq avail)
  in
   val dedicated = (dedicate(0, 31, availRegs), dedicate(0, 31, availFRegs), [])
  end

  type cell = int

  local
    val nRegs = 32
    val nFregs = 32
    val regCnt = ref nRegs
    val fregCnt = ref nFregs
    fun bump (r as ref c) = (r := c+1; c)
  in
    fun newReg () = bump regCnt
    fun newFreg() = bump fregCnt
    fun maxReg()  = !regCnt
    fun maxFreg() = !fregCnt
    fun resetRegs() = (regCnt := nRegs; fregCnt := nFregs)
  end

  fun regMap(T.REG r) = r
  fun fregMap(T.FREG f) = f

  type cellset  = (int list * int list * int list)

  val empty = ([],[],[])
  fun cellset2string _ = ""
  fun addReg(r, (rc,fc,mc)) = (SL.enter(r,rc), fc, mc)
  fun addFreg(f, (rc,fc,mc)) = (rc, SL.enter(f,fc), mc)
end
