(* raBitset.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *
 * Imperative bitsets.
 *
 * This has been written specially for the register allocator.
 * The computation of n(n+1)/2 very quickly overflows in practice.
 *)

(** This has been written specially for the register allocator.
 ** We use a hash table representation, because it performs better
 ** than a linear representation except for small numbers of live 
 ** ranges.
 **)

signature BITMATRIX = sig
    type bitMatrix
    val new         : int -> bitMatrix
    val add         : bitMatrix -> (int * int) -> bool
    val member      : bitMatrix -> (int * int) -> bool
    val delete      : bitMatrix -> (int * int) -> unit
    val clear 	    : bitMatrix * int -> unit
end


abstraction TriangularBitMatrix : BITMATRIX = 
struct

  datatype bucket = NIL | B of (int * int * bucket)

  datatype bitMatrix = 
      INTPAIRMAP of {table:bucket Array.array ref, 
		     elems:int ref, 
		     size:int ref, 
		     original : bucket Array.array}
  fun roundsize size = let
      fun f x = if x >= size then x else f(x*2)
    in f 64
    end

  fun new size = let 
      val tblSize = roundsize size   
      val tbl = Array.array(tblSize,NIL)
    in 					(* note: size is offset by 1 *)
       INTPAIRMAP{table    = ref tbl,
		  elems    = ref 0, 
		  size     = ref(tblSize-1),
		  original = tbl}
    end

  fun moduloSize(i, sz) = 
    Word.toInt(Word.andb(Word.fromInt i, Word.fromInt sz))

  fun member(INTPAIRMAP{table,elems,size,...}) (i,j) = let
	fun find NIL = false
	  | find(B(i',j',b)) = (i=i' andalso j=j') orelse find b
      in find(Array.sub(!table, moduloSize(i+j, !size)))
      end

  fun add (t as INTPAIRMAP{table,elems,size,...}) (v as (i,j)) = let
	val ref tbl = table
	val ref sz = size
      in
	if !elems <> sz then let
	    val indx = moduloSize(i+j, sz)
	    fun find NIL = false
	      | find(B(i',j',r)) = (i=i' andalso j=j') orelse find r
	    val b = Array.sub(tbl,indx)
	  in 
	     if find b then false
	     else (System.Unsafe.update(tbl,indx,B(i,j,b)); 
		   elems := !elems + 1;
		   true)
	  end
	else let 
	     val newsize=sz+sz+2
	     val new = Array.array(newsize,NIL)
	     val newsize1 = newsize-1
	     fun redo n = let
		 fun add'(a,b,B(i,j,r)) = 
		     if moduloSize(i+j, newsize1) = n then add'(B(i,j,a),b,r)
		     else add'(a,B(i,j,b),r)
		   | add'(a,b,NIL) = 
		       (Array.update(new,n,a); 
			Array.update(new,n+sz+1,b);
			redo(n+1))
	       in add'(NIL, NIL, Array.sub(tbl,n))
	       end
	  in 
	     table:=new;
	     size:=newsize-1;
	     redo 0 handle _ => ();
	     add t v
	  end
      end

  fun delete(INTPAIRMAP{table=ref table,elems,size=ref size,...}) (i,j) = let
	fun find NIL = NIL
	  | find(B(i',j',b)) =
	      if i=i' andalso j=j' then (elems := !elems-1; b) else B(i',j',find b)
	val indx = moduloSize(i+j, size)
      in System.Unsafe.update(table, indx, find(Array.sub(table,indx)))
      end

  fun clear (INTPAIRMAP{table,elems,original,size}, _) = let
      fun init n = (Array.update(original,n,NIL); init(n+1))
    in 
      elems:=0; 
      size:=(Array.length original - 1);
      table:=original;
      init 0 handle _ => ()
    end
end

