(** Copyright 1993,1994,1995 AT&T Bell Laboratories **)

(** Graph coloring register allocation.
 ** Implements the 'iterated register coalescing' scheme described in POPL'96
 **)


signature RA_ARG = sig
    structure Liveness  : LIVENESS
    structure InsnProps : INSN_PROPERTIES

    sharing Liveness.F.I = InsnProps.I    

    val getreg         : int list * int list -> int
    val spill	       : int -> (InsnProps.I.instruction list * 
				       InsnProps.I.instruction list)
    val regSet         : InsnProps.C.cellset -> int list
    val defUse	       : InsnProps.I.instruction -> (int list * int list)
    val nFreeRegs      : int
    val firstPseudoR   : int
    val maxPseudoR     : unit -> int
    val regMap	       : int Array.array list -> int Array.array
end



signature RA = sig
    structure F : FLOWGRAPH
    val ra : F.cluster -> unit
end



functor RegAllocator(structure Ra : RA_ARG
		     structure Emitter : EMITTER_NEW
		     sharing Emitter.I = Ra.Liveness.F.I
		     val codegen : Ra.Liveness.F.cluster -> unit) : RA =
struct

  structure F = Ra.Liveness.F
  structure P = Ra.InsnProps
  structure C = F.C
  structure SL = SortedList
  structure BM = TriangularBitMatrix

  val misc4 = Control.CG.misc4

  fun error msg = ErrorMsg.impossible ("RegAllocator." ^ msg)
  fun assert(msg, true) = () | assert(msg, false) = error msg

  val phase = Stats.doPhase o Stats.makePhase

		(*---------printing------------ *)
  fun prList (l:int list,msg:string) = let
      fun pr [] = print "\n"
	| pr (x::xs) = (print (makestring x ^ " "); pr xs)
  in
	print msg; pr l
  end

  fun printBlocks([],regmaps) = print"\n"
    | printBlocks(F.BBLOCK{blknum,insns,liveOut,liveIn,succ,pred,...}::blocks,
		  regmaps) = let
        fun pr [] = prList(Ra.regSet(!liveOut),"liveOut: ")
	  | pr (instr::rest) = (Emitter.emitInstr(instr,regmaps); pr rest)
      in
	print("BLOCK" ^ Int.toString blknum ^ "\n");
	prList(Ra.regSet(!liveIn),"LiveIn :");
	prList(!pred,"predecessors: ");
	case !insns of [] => print "empty instruction sequence\n"
                     |  l  => pr(rev l)
	(*esac*);
	prList(!succ,"successors: ");
	printBlocks(blocks,regmaps)
      end
    | printBlocks(F.LABEL lab::blocks,regmaps) = 
        (print(Label.nameOf lab^":\n"); 
	 printBlocks(blocks,regmaps)) 
    | printBlocks(_::blocks,regmaps) = printBlocks(blocks,regmaps)


		(*-------------------------------*)
  (* set of dedicated registers *)
  val dedicatedSet      = Ra.regSet C.dedicated   
  val defUse		= Ra.defUse
  fun rmvDedicated rset = SL.difference (rset,dedicatedSet)

  (* many clusters have fewer than liveSetUniverse live-ranges and
   * can use fixed data structures allocated once.
   *)
  val liveSetUniverse = 512
  val liveSet	      = SparseSet.set liveSetUniverse
  val bitMatrix       = BM.new liveSetUniverse

  (* register mapping functions *)
  fun uniqMap(f, l) = let
    fun map([], acc) = acc
      | map(x::xs, acc) = map(xs, SL.enter(f x, acc))
  in
    map(l, [])
  end

  (* reduce the length of chains in register aliases *)
  fun normalizeRegMap regMap = let
    fun chase(x,also) = let 
      val r = Array.sub(regMap,x)
    in 
      if x<>r then chase(r,x::also)
      else app (fn y => System.Unsafe.update(regMap,y,r)) also
    end	      

    fun normalize n = let
      val r = Array.sub(regMap,n)
    in
      if n = r then normalize (n+1)
      else (chase(r,[n]); normalize(n+1))
    end
  in
    normalize 0 handle _ => ()
  end

		(*------------------------------*)
  fun graphColoring(blocks,cblocks,blockDU,prevSpills,regmaps,spillCount) = let
    val K = Ra.nFreeRegs
    val numOfBlocks = Array.length cblocks
    val maxR   = Ra.maxPseudoR()
    val regMap = Ra.regMap regmaps
    val graph  = Array.array(maxR, []:int list)
    fun rNum r = Array.sub(regMap, r)
    fun chase(r:int) = 
      let val x = Array.sub(regMap,r) in if r=x then r else chase x end


    datatype moveStatus = MOVE | COALESCED | CONSTRAINED | LOST | WORKLIST
    datatype move = 
      MV of {src : int,			(* source register of move *)
	     dst : int,			(* dest register of move *)
	     indx : int,		(* index into basic block *)
	     block : int,		(* block number (index) *)
	     status : moveStatus ref	(* coalesced? *)
	     }

    datatype worklists = 
      WKL of {count : int,		(* number of nodes remaining *)
	      simplifyWkl: int list,	(* nodes that can be simplified *)
	      moveWkl : move list,	(* moves to be considered for coalescing *)
	      freezeWkl : int list,	(* all n, s.t. degree(n)<K and moveRelated(n) *)
	      spillWkl : int list,	(* all n, s.t. degree(n)>=K  *)
	      stack : int list}		(* nodes removed from the graph *)
      
					
    val spillFlag = ref false		(* has (chaitin) spilling occurred *)
    val undoInfo : (int * int * moveStatus ref) list ref  = ref []

    (* degree is used to maintain the number of edges connected to a node.
     * A value of ~1 indicates that the node has been removed from
     * the graph. Physical registers are treated specially.
     *)
    val degree = let
	val a = Array.array(maxR, 0)
	fun init n = (Array.update(a, n, ~1); init(n-1))
      in 
	init(Ra.firstPseudoR - 1) handle _ => (); a
      end

    val (liveSet, bitMatrix) = 
      if maxR > liveSetUniverse then (SparseSet.set maxR, BM.new(maxR-1))
      else (BM.clear(bitMatrix, maxR-1); (liveSet, bitMatrix))
    val addBitMatrix = BM.add bitMatrix
    val memBitMatrix = BM.member bitMatrix


    (* moveList -- a mapping of nodes to lists of moves they are 
     *   associated with. Built once but conservatively maintained,
     *	 i.e., a list may contain more moves than necessary.
     *)
    exception MoveLists
    val moveListTbl  : move list Intmap.intmap = Intmap.new(16, MoveLists)
    val moveList = Intmap.map moveListTbl
    val enterMoveList = Intmap.add moveListTbl
    fun addMoveList(x, mv) = enterMoveList(x, mv::(moveList x handle _ => []))

    (* mvCntTbl contains the number of times a node is involved in a move
     * instruction. 
     *)
    exception MvCnt
    val mvCntTbl : int Intmap.intmap = Intmap.new (16, MvCnt)
    val mvCnt = Intmap.map mvCntTbl
    val rmvMvCnt = Intmap.rmv mvCntTbl
    val addMvCnt = Intmap.add mvCntTbl
    fun isMoveRelated v = ((mvCnt v); true) handle _ => false

	    (*--------interference graph construction--------*)

    (* add an edge to the interference graph.
     * note --- adjacency lists for machine registers are not maintained.
     *)
    fun addEdge(x:int, y) = let
      val (u,v) = if x < y then (x,y) else (y,x)
      fun add(r,s) = 
	if r < Ra.firstPseudoR then () 
	else (System.Unsafe.update(graph, r, s::Array.sub(graph, r));
	      System.Unsafe.update(degree, r, 1+Array.sub(degree, r)))
    in
      if v=u then () 
      else if addBitMatrix(v,u) then (add(v,u); add(u,v)) else ()
    end


    (* Builds the interference graph and initialMove list *)
    fun mkInterferenceGraph() = let
      val insert = SparseSet.insert liveSet
      val delete = SparseSet.delete liveSet
      val forall = SparseSet.forall liveSet

      fun forallBlocks(~1, mvs) = mvs
	| forallBlocks(n, mvs) = let
	    val F.BBLOCK{insns,liveOut,...} = Array.sub(cblocks, n)
	    val bdu = Array.sub(blockDU, n)
	    fun doBlock([], _, _, mvs) = forallBlocks(n-1, mvs)
	      | doBlock(instr::rest, indx, (def',use')::bdu, mvs) = let
		  val def = map rNum def'
		  val use = map rNum use'
		  val moves = 
		    if P.moveInstr instr then
		      (case (def, use)
			of ([d], [u]) => 
			     (delete u; 
			      MV{src=u, dst=d, indx=indx, block=n,
				 status=ref WORKLIST}::mvs)
			 |  _ => mvs
		      (*esac*))
		     else mvs
		in
		  app (fn d => forall (fn lr => addEdge(d, lr)))  def;
		  app delete def;
		  app insert use;
		  doBlock(rest, indx+1, bdu, moves)
		end
	    val lout = map rNum (rmvDedicated(Ra.regSet(!liveOut)))
	  in
	    SparseSet.clear liveSet;
	    app insert lout;
	    doBlock(!insns, 0, !bdu, mvs)
	  end

      (* Filter moves that already have an interference.
       * Also create moveList and mvCnt at this time.
       *)
      fun incMvCnt v = addMvCnt(v, 1+(mvCnt v handle _ => 0))
      fun filter([]) = []
	| filter((mv as MV{src,dst,...})::rest) = let
	    val (x,y) = if src > dst then (src, dst) else (dst, src)
	  in
	    if (memBitMatrix(x, y) orelse x < Ra.firstPseudoR) then 
	      filter(rest)
	    else let
	        fun info u = 
		  if u < Ra.firstPseudoR then ()
		  else (addMoveList (u, mv); incMvCnt u)
	      in
		info x; info y; mv::filter(rest)
	      end
	  end
    in
      filter(forallBlocks(numOfBlocks-1, []))
    end (* mkInterferenceGraph *)


		    (*--------build worklists----------*)

    (* make initial worklists and count of nodes remaining in 
     * the graph. Note: register aliasing may have occurred due
     * to previous rounds of graph-coloring.
     *)
    fun mkInitialWorkLists iMoves = let
      fun iter(n, count, simplifyWkl, freezeWkl, spillWkl) = 
	if n = maxR then 
	  {count       = count, 
	   simplifyWkl = simplifyWkl, 
	   freezeWkl   = freezeWkl,
	   spillWkl    = spillWkl, 
	   moveWkl     = iMoves,
	   stack       = []}
	else if n <> rNum(n) then
	  (Array.update(degree, n, ~1);
	   iter(n+1, count, simplifyWkl, freezeWkl, spillWkl))
	else let
	    val c = length(Array.sub(graph, n))
	  in
	    if c >= K then 
	      iter(n+1, count+1, simplifyWkl, freezeWkl, n::spillWkl)
	    else if isMoveRelated n then 
	      iter(n+1, count+1, simplifyWkl, n::freezeWkl, spillWkl)
	    else
	      iter(n+1, count+1, n::simplifyWkl, freezeWkl, spillWkl)
	  end
    in
      iter(Ra.firstPseudoR, 0, [], [], [])
    end

(*
    val mkInterferenceGraph = 
      phase "Compiler 127 interference graph" mkInterferenceGraph
*)

    val initialMoves = mkInterferenceGraph()
    val initialWkls = mkInitialWorkLists initialMoves


		    (*---------simplify-----------*)

    (* activate moves associated with a node and its neighbors *)
    fun enableMoves(node, moveWkl) = let
      fun addMvWkl([], wkl) = wkl
	| addMvWkl((mv as MV{status, ...})::rest, wkl) = 
	   (case !status
	     of MOVE => 
	         (status := WORKLIST; addMvWkl(rest, mv::wkl))
	      | _ => addMvWkl(rest, wkl)
	   (*esac*))

      fun add([], wkl) = wkl
	| add(v::vs, wkl) = 
	   if Array.sub(degree, v) = ~1 then add(vs, wkl)
	   else if isMoveRelated v then add(vs, addMvWkl(moveList v, wkl))
		else add(vs, wkl)
    in
      add(node::Array.sub(graph,node), moveWkl)
    end

    (* decrement the degree associated with a node returning a potentially
     * new set of worklists --- simplifyWkl, freezeWkl, and moveWkl.
     *)
    fun decrementDegree(node, simplifyWkl, freezeWkl, moveWkl) = let
      val d = Array.sub(degree, node)
    in
      System.Unsafe.update(degree, node, d-1);
      if d = K then let
	  val moveWkl = enableMoves(node, moveWkl)
        in
	  if isMoveRelated(node) then
	    (simplifyWkl, node::freezeWkl, moveWkl)
	  else
	    (node::simplifyWkl, freezeWkl, moveWkl)
	end
      else
	(simplifyWkl, freezeWkl, moveWkl)
    end

    (* repeatedly remove low-degree (non-move-related) nodes from the graph *)
    fun simplify(WKL{count,simplifyWkl,freezeWkl,spillWkl,moveWkl,stack}) = let
      fun iter([], v::vs, count, fzWkl, mvWkl, stack) = 
	  if (Array.sub(degree, v) = ~1) then
	    iter([], vs, count, fzWkl, mvWkl, stack)
	  else
	    (System.Unsafe.update(degree, v, ~1);
	     iter(Array.sub(graph, v), vs, count-1, fzWkl, mvWkl, v::stack))
	| iter([], [], count, fzWkl, mvWkl, stack) = 
	    WKL{simplifyWkl=[], spillWkl=spillWkl,
		count=count, 
		freezeWkl=fzWkl, moveWkl=mvWkl, stack=stack}
	| iter(n::rest, wkl, count, fzWkl, mvWkl, stack) = 
	    if Array.sub(degree, n) = ~1 then
	      iter(rest, wkl, count, fzWkl, mvWkl, stack)
	    else let
		val (wkl, fzWkl, mvWkl) = 
		  decrementDegree(n, wkl, fzWkl, mvWkl)
	      in
		iter(rest, wkl, count, fzWkl, mvWkl, stack)
	      end
    in
      iter([], simplifyWkl, count, freezeWkl, moveWkl, stack)
    end

		    (*-----------coalesce-------------*)

    fun coalesce(WKL{count,moveWkl,simplifyWkl,freezeWkl,spillWkl,stack}) = let
      (* v is being replaced by u *)
      fun combine(v, u, mvWkl, simpWkl, fzWkl) = let
	(* merge moveList entries, taking the opportunity to prune the lists *)
	fun mergeMoveLists([], [], mvs) = mvs
	  | mergeMoveLists([], xmvs, mvs) = mergeMoveLists(xmvs, [], mvs)
	  | mergeMoveLists((mv as MV{status,...})::rest, other, mvs) = 
	     (case !status
	       of (MOVE | WORKLIST) => 
		     mergeMoveLists(rest, other, mv::mvs)
		| _ => mergeMoveLists(rest, other, mvs)
	     (*esac*))

	(* form combined node *)
	fun union([], mvWkl, simpWkl, fzWkl) = (mvWkl, simpWkl, fzWkl)
	  | union(t::ts, mvWkl, simpWkl, fzWkl) = 
	    if t < Ra.firstPseudoR then
	      (addEdge(t,u);
	       union(ts, mvWkl, simpWkl, fzWkl))
	    else if Array.sub(degree, t) = ~1 then
	      union(ts, mvWkl, simpWkl, fzWkl) 
	    else 
	      (addEdge(t, u);
	       let val (wkl, fzWkl, mvWkl) = 
			          decrementDegree(t, simpWkl, fzWkl, mvWkl)
		 in union(ts, mvWkl, wkl, fzWkl)
		 end)
      in
	Array.update(regMap, v, u);
	Array.update(degree, v, ~1);
	rmvMvCnt v;
	if u < Ra.firstPseudoR then ()
	else enterMoveList (u, mergeMoveLists(moveList u, moveList v, []));
	union(Array.sub(graph, v), mvWkl, simpWkl, fzWkl)
      end (* combine *)

      (* If a node is no longer move-related as a result of coalescing,
       * it can become candidate for the  next round of simplification.
       * The handler is invoked when node is a machine register.
       *)
      fun addWkl(node, c, wkl) = let
	val ncnt = mvCnt node - c
      in
	if ncnt <> 0 then (addMvCnt (node, ncnt); wkl)
	else (rmvMvCnt node;
	      if Array.sub(degree, node) >= K then wkl
	      else node::wkl)
      end handle _ => wkl

      (* heuristic used to determine if a pseudo and machine register
       * can be coalesced.
       *)
      fun safe(r,t) = let
	fun f [] = true
	  | f (x::xs) = 
	     (x < Ra.firstPseudoR orelse 
	           Array.sub(degree, x) < K orelse memBitMatrix (x, r)) 
	      andalso 
	      f xs
      in 
	f (Array.sub(graph, t))
      end (* safe *)

      (* return true if Briggs et.al. conservative heuristic applies *)
      fun conservative(x, y) = let
	val dx = Array.sub(degree, x)
	val dy = Array.sub(degree, y)
      in
	dx + dy < K 
	orelse let 
	    val insert = SparseSet.insert liveSet
	    val _ = SparseSet.clear liveSet
	    fun g(0, _, k) = k
	      | g(_, _, 0) = 0
	      | g(d, v::vs, k) = 
	        if SparseSet.member(liveSet, v) then g(d-1, vs, k)
		else if v < Ra.firstPseudoR then 
		  (insert v; g(d-1, vs, k-1))
	        else let val dv = Array.sub(degree, v)
		  in
		     if dv = ~1 then g(d, vs, k)
		     else if dv < K then g(d-1, vs, k)
		     else (insert v; g(d-1, vs, k-1))
                  end
	    val k' = g(dx, Array.sub(graph, x), K)
	  in
	    k' > 0 andalso g(dy, Array.sub(graph, y), k') > 0
	  end
      end

      (* iterate over move worklist *)
      fun doMoveWkl((mv as MV{src,dst,status,...})::rest, wkl, cnt, fzWkl) = let
	    val x = chase src
	    val y = chase dst
	    val (u,v) = if x < y then (x,y) else (y,x)
	    fun addUndoInfo() = 
	      if !spillFlag then undoInfo := (v, u, status) :: (!undoInfo)
	      else ()
	  in
	    if u = v then
	      (status := COALESCED;
	       addUndoInfo ();
	       doMoveWkl(rest, addWkl(u, 2, wkl), cnt, fzWkl))
	    else if (v < Ra.firstPseudoR) then
	      (status := CONSTRAINED;
	       doMoveWkl(rest, wkl, cnt, fzWkl))
	    else if memBitMatrix (v, u) then
	      (status := CONSTRAINED;
	       doMoveWkl(rest, addWkl(v,1,addWkl(u,1,wkl)), cnt, fzWkl))
	    else if u < Ra.firstPseudoR then
	      if safe(u,v) then let
		  val (mvWkl, simpWkl, fzWkl) = combine(v, u, rest, wkl, fzWkl)
		in
		  status := COALESCED;
		  addUndoInfo();
		  doMoveWkl(mvWkl, simpWkl, cnt-1, fzWkl)
		 end
	      else
		(status := MOVE;
		 doMoveWkl(rest, wkl, cnt, fzWkl))
	    else if conservative(u, v) then let
	        val vCnt = mvCnt v
		val (mvWkl, wkl, fzWkl) = 
			combine(v, u, rest, wkl, fzWkl)
	      in
		status := COALESCED;
		addUndoInfo();
		doMoveWkl(mvWkl, addWkl(u, 2-vCnt, wkl), cnt-1, fzWkl)
	      end
	    else 
	      (status := MOVE;
	       doMoveWkl(rest, wkl, cnt, fzWkl))
	  end
	| doMoveWkl([], wkl, cnt, fzWkl) = 
	  (* Note. The wkl is not uniq, because decrementDegree may have
	   * added the same node multiple times.
	   *)
	    WKL{count = cnt,  simplifyWkl = wkl,   freezeWkl = fzWkl, 
		moveWkl = [], spillWkl = spillWkl, stack = stack}
    in
      doMoveWkl(moveWkl, simplifyWkl, count, freezeWkl)
    end (* coalesce *)



		    (*-----------freeze------------*)

    (* When a move is frozen in place, the operands of the move may
     * be simplified. One of the operands is node (below).
     *)
    fun wklFromFrozen(node) = let
      fun mkWkl(MV{status, src, dst, ...}::rest) = let
	    val s = chase src and  d = chase dst
	    val y = if s = node then d else s
	  in
	    case !status 
	    of MOVE  => 
	      (status := LOST;
	       if y < Ra.firstPseudoR then mkWkl(rest)
	       else let
		   val ycnt = mvCnt y
		 in
		   if ycnt = 1 then
		     (rmvMvCnt y;
		      if Array.sub(degree, y) < K then
			y::mkWkl rest
		      else
			mkWkl rest)
		   else
		      (addMvCnt(y, ycnt-1);
		       mkWkl rest)
		 end)
	     | WORKLIST => error "wklFromFrozen"
	     | _ => mkWkl(rest)
	  end
	| mkWkl [] = []
    in
      rmvMvCnt node;
      mkWkl(moveList node)
    end

    (* freeze a move in place *)
    fun freeze(WKL{freezeWkl,count,simplifyWkl,spillWkl,moveWkl,stack}) = let
      fun find([], acc) =  (NONE, acc)
	| find(n::ns, acc) = let
	    val d = Array.sub(degree, n)
	  in
	    if d = ~1 then find(ns, acc)
	    else if d >= K then find(ns, d::acc)
		 else (SOME n, acc@ns)
          end

      fun mkWorkLists(NONE, fzWkl) = 
	   WKL{freezeWkl=fzWkl, count=count, simplifyWkl=simplifyWkl, 
	       spillWkl=spillWkl, moveWkl=moveWkl, stack=stack}
	| mkWorkLists(SOME n, fzWkl) = let
	    val wkl = n::wklFromFrozen n
          in
	    WKL{freezeWkl=fzWkl, count=count, simplifyWkl=wkl,
		spillWkl=spillWkl, moveWkl=moveWkl, stack=stack}
	  end
    in
      mkWorkLists(find(freezeWkl,[]))
    end

	    (*----------select spill node--------------*)
    type info  = int list Intmap.intmap

   (* remainInfo: blocks where spill nodes are defined and used. *)
    val remainInfo : (info * info) option ref	= ref NONE

    fun cleanupSpillInfo() = remainInfo := NONE

    fun selectSpillNode
	  (WKL{simplifyWkl, spillWkl, count, stack, moveWkl, freezeWkl}) = let

       (* duCount: compute the def/use points of spilled nodes. *)
	fun duCount [] = error"selectSpillNode.duCount"
	  | duCount spillable = let
	      val size = length spillable
	      exception Info
	      val defInfo : info = Intmap.new(size,Info)
	      val useInfo : info = Intmap.new(size,Info)
	      val addDef = Intmap.add defInfo 
	      val addUse = Intmap.add useInfo
	      fun getDefs n = (Intmap.map defInfo n) handle _ => []
	      fun getUses n = (Intmap.map useInfo n) handle _ => []
	      fun rNum r = Array.sub(regMap,r)

	      (* doblocks --- updates the defInfo and useInfo tables to indicate
	       * 	the blocks where spillable live ranges are defined and used.
	       *)
	      fun doblocks blknum = 
		  if blknum=numOfBlocks then ()
		  else let
		      val bdu = Array.sub(blockDU,blknum)
		      fun iter [] = ()
			| iter((def',use')::rest) = let
			    val def = uniqMap(rNum,def')
			    val use = uniqMap(rNum,use')
			    val d = SL.intersect(def,spillable)
			    val u = SL.intersect(use,spillable)
			  in
			    case(d,u)
	 		     of ([],[]) => ()
			      | _ => let
				  fun updateDef n =
				        addDef(n,SL.enter(blknum,getDefs n))
				  fun updateUse n =
				        addUse(n,SL.enter(blknum,getUses n))
			        in
				  app updateDef d; app updateUse u
			        end
			    (*esac*);
			    iter rest
			  end
		    in
		      iter(!bdu);
		      doblocks(blknum+1)
		    end

	      (* If a node is live going out of a block we will record this
	       * as a definition. We need to consider these blocks as it may 
	       * be terminated by an escaping branch. Too much trouble to 
	       * bother with this. This may offset the Chaitan hueristic but ...
	       *)
	      fun doBBlocks n = let
		  val F.BBLOCK{blknum,liveIn,liveOut,...} = Array.sub(cblocks,n)
		  val liveout = uniqMap(rNum,rmvDedicated(Ra.regSet(!liveOut)))
	      in 
		case SL.intersect(spillable,liveout)
		 of [] => doBBlocks (n+1)
		  | some => 
		       (app (fn n => addDef(n,SL.enter(blknum,getDefs n))) some;
			doBBlocks (n+1))
	      end (* doBBlocks *) 
	    in
	      doblocks 0;
	      doBBlocks 0 handle _ => ();
	      (defInfo,useInfo)
	    end (* duCount *)

	(* Since the spillWkl is not actively maintained, the set of
	 * spillable nodes for which def/use info is needed is a subset
	 * of spillWkl. {\tiny Intense cheating going on with the representation
	 * of spillWkl!}
	 *)
	fun remainingNodes() = let
	  fun prune [] = []
	    | prune (x::xs) = if Array.sub(degree,x) = ~1 then prune xs
				      else x::prune xs
        in
	  case !remainInfo 
	   of SOME info => prune spillWkl
	     | NONE => let
		 (* first time spilling *)
		 val spillable = prune (rev spillWkl)
	       in 
		 remainInfo:=SOME(duCount spillable);
		 spillable
	       end
        end

       (** apply the chaitan hueristic to find the spill node **)
	fun chaitanHueristic(spillable) = let
	      val infinity = 1000000.0
	      val infinityi= 1000000
	      val SOME(dinfo,uinfo) = !remainInfo
	      val getdInfo = Intmap.map dinfo
	      val getuInfo = Intmap.map uinfo
	      fun iter([],node,_) = 
		  if node <> ~1 then node
		  else let
		      fun dump [] = ()
			| dump (node::rest) = let
			    val cost = 
				  (length(getdInfo node) + 
				   (length(getuInfo node) handle _ => infinityi))
			    val degree = Array.sub(degree,node)
			    val hueristic = real cost / real degree
			  in
			    print(concat["node =", Int.toString node,
					  " cost =", Int.toString cost,
					  " degree = ", Int.toString degree,
					  "hueristic = ", Real.toString hueristic,
					  "\n"]);
			    dump rest
			  end
		    in
			prList(prevSpills, "prevSpills ");
			dump spillable;
			node
		    end
		| iter(node::rest,cnode,cmin) = let
		   (* An exeception will be raised if the node is defined
		    * but not used. This is not a suitable node to spill.
		    *)
		    val cost = ((length(getdInfo node) +
			       (length(getuInfo node) handle _ => infinityi)))
		    val degree = Array.sub(degree,node)
		    val hueristic = real cost / real degree 
		  in
		    if hueristic < cmin andalso not(SL.member prevSpills node)
		    then iter(rest,node,hueristic)
		    else iter(rest,cnode,cmin)
		  end
	    in
	      iter(spillable,~1,infinity)
	    end
	val spillWkl = remainingNodes()
	val spillNode = chaitanHueristic(spillWkl)
	val simpWkl = 
	      if isMoveRelated spillNode then spillNode::wklFromFrozen(spillNode)
	      else [spillNode]
    in
      spillFlag:=true;
      WKL{simplifyWkl=simpWkl,
	    count=count,
	    spillWkl = spillWkl,
	    freezeWkl = freezeWkl,
	    stack = stack,
	    moveWkl = moveWkl}
    end (* selectSpillNode *)



	       (*---------rerun algorithm-------------*)

   (** rerun(spillList) - an unsuccessful round of coloring as taken
    **   place with nodes in spillList having been spilled. The
    **   flowgraph must be updated and the entire process repeated. 
    **)
    fun rerun spillList = let
	type blockInfo = (P.I.instruction list ref * 
			  (int list * int list) list ref *
			  int list)
	exception Affected
	val affected : blockInfo Intmap.intmap = Intmap.new(32,Affected)
	val addAffected = Intmap.add affected

	val SOME(dInfo,uInfo) = !remainInfo
	fun rNum r = Array.sub(regMap,r)
       (** affectedInstrs - finds blocks of instructions that require 
	** 	spill code. 
	**)
	fun affectedInstrs () = let
	    (** list of blocks that require spill code **)
	    fun affectedBlocks() = let
		fun doNodes([],[]) = 
		      error"rerun.affectedInstrs.affectedBlocks.doNodes"
		  | doNodes([],acc) = acc
		  | doNodes(n::nodes,acc) = let
		      val duBlocks = SL.merge(Intmap.map dInfo n,
					      Intmap.map uInfo n handle _ => [])
		    in
		      doNodes(nodes,SL.merge(duBlocks,acc))
		    end (* doNodes *)
	    in doNodes(spillList,[])
	    end (* affectedBlocks *)

	    fun gather [] = ()
	      | gather(blk::rest) = let
		  val F.BBLOCK{blknum,insns,liveOut,...} = Array.sub(cblocks,blk)
		  val blockDU = Array.sub(blockDU,blk)
		  val liveout = uniqMap(rNum, Ra.regSet(!liveOut))
		in addAffected(blknum, (insns,blockDU,liveout));
		   gather rest
		end
	in gather(affectedBlocks())
	end (* affectedInstrs *)

       (* returns list of block info associated with a spill nodes *)
	fun assocInfo node = let
	    fun iter([],acc) = acc
	      | iter(blk::rest,acc) = iter(rest,SL.enter(blk,acc))
	  in
	    map (fn blk => (blk,Intmap.map affected blk))
		(SL.merge(Intmap.map dInfo node,
			  Intmap.map uInfo node handle _ => []))
	  end
	fun doBlocks([],_,_,_,_) = ()
	  | doBlocks((blknum,(instrs,bdu,liveOut))::rest,node,spill,
							      reload,toSpill) = let

	      fun rNum r = Array.sub(regMap,r)
	      fun newdu instr = let 
		  val (d',u') = defUse instr
		  fun rmv set = 
			uniqMap(rNum, SL.difference(SL.uniq(set), dedicatedSet))
		in (rmv d',rmv u')
		end
	      val spillDU  = map newdu spill
	      val reloadDU = map newdu reload

	      (* doSpillNodes - is simple and avoids the most embarrassing 
	       *	spill placement decisions.
	       *)
	      fun doSpillNode() = let
		  fun doInstrs(instr::rest,(du as(d',u'))::blockDU,
					   lastInstr,newInstrs,newBdu) = let

			val d=uniqMap(rNum,d')
			val u=uniqMap(rNum,u')
			fun spillIt() =
			    if SL.member d node then
			       (instr::(spill@newInstrs),du::(spillDU@newBdu))
			    else
			      (case lastInstr 
				of SOME(rld,rldDU) => 
				   (instr::(rld@newInstrs),du::(rldDU@newBdu))
				 | NONE => (instr::newInstrs,du::newBdu)
			      (*esac*))
			fun reloadIt() = 
			    if SL.member u node then
			      SOME(reload,reloadDU)
			    else NONE
			val (newInstrs',newBdu') = spillIt()
		      in
			doInstrs(rest,blockDU,reloadIt(),newInstrs',newBdu')
		      end
		    | doInstrs([],[],NONE,newInstrs,newBdu) = 
			(rev newInstrs,rev newBdu)
		    | doInstrs([],[],SOME(rld,rldDU),newInstrs,newBdu) =
			(rev(rld@newInstrs),rev(rldDU@newBdu))
		in
		  doInstrs(!instrs,!bdu,NONE,[],[])
		end (* doSpillNode *)

	     (* special action if the last instruction is an escaping
	      * branch and the node is live across the branch.
	      * We discover if the node needs to be spilled or reloaded,
	      * and what the new liveOut list should be.
	      * The new liveOut is not used anymore, but is left in here
	      * since it may be useful at some future date.
	      *)
	      fun blockEnd(i as (instr::instrs),d as (du::bdu)) = let
		    fun escapes [] = false
		      | escapes (P.ESCAPES::_) = true
		      | escapes (_::targets) = escapes targets
		  in
		    if SL.member liveOut node then
			(case P.instrKind instr
			 of P.IK_JUMP =>
			     if escapes(P.branchTargets instr) then
			       ((instr::(rev reload@instrs),
				 du::(rev reloadDU@bdu),
				 liveOut))
			     else (i,d,SL.rmv(node,liveOut))
			  | _ => (i,d,SL.rmv(node,liveOut))
			(*esac*))
		    else (i,d,liveOut)
		  end
		| blockEnd([],[]) = ([],[],liveOut)
	      val (newInstrs,newBdu,newLiveOut) = blockEnd(doSpillNode())
	    in
	      instrs:=newInstrs;
	      bdu:=newBdu;
	      addAffected (blknum,(instrs,bdu,newLiveOut));
	      doBlocks(rest,node,spill,reload,toSpill)
	    end (* doBlocks *)

	(* The optimistic coloring selection may come up with a node
	 * that has already been spilled. Must be careful not to spill
	 * it twice.
	 *)
	fun glue [] = ()
	  | glue(node::rest) = 
	    if SL.member prevSpills node then glue rest
	    else let
	        val (spill,reload) = Ra.spill node
  	      in 
		doBlocks(assocInfo node,node,spill,reload,rest);
		glue rest
	      end

	fun redoAlgorithm() = let
	    val spills = SL.merge(spillList,prevSpills)
	in
	  graphColoring(blocks,cblocks,blockDU,spills,regmaps,spillCount+1)
	end

      in
	 affectedInstrs();
	 glue spillList;
	 Ra.Liveness.liveness(blocks,regMap);
	 redoAlgorithm()
      end (* rerun *)

(*    val rerun = phase "Compiler 129 spilling" rerun *)

	    (*--------remove coalesced upto first spill---------*)

    (* Remove moves coalesced.
     * Two version exist: one maintains the def/use information and the other
     * does not.
     *)
    local
      fun rmCoalesced(_, []) = ()
	| rmCoalesced(rmvMoves, moves as (MV{block=currblk, ...}::_)) = let
	    fun collect(moves as 
			  (MV{indx, block, status=ref COALESCED,...}::rest),
		        acc) = 
		if block = currblk then 
		  collect(rest, indx::acc)
		else 
		  (rmvMoves(currblk, acc);  rmCoalesced(rmvMoves, moves))
	      | collect(MV{status=ref WORKLIST, ...}::_, _) = 
		  error "rmCoalesced.collect"
	      | collect(_::rest, acc) = collect(rest, acc)
	      | collect([], acc) = rmvMoves(currblk, acc)
	  in
	    collect(moves, [])
	  end (* rmvCoalesced *)

	  fun rmvMoves(currblk, []) = ()
	    | rmvMoves(currblk, moves) = let
		val F.BBLOCK{insns, ...} = Array.sub(cblocks, currblk)
		fun rmv([], _, instrs, newI) = (insns := rev newI @ instrs)
		  | rmv(cl as (indx::indices), pos, instr::instrs, newI) = 
		    if indx = pos then 
		      (assert("removing move", P.moveInstr instr);
		       rmv(indices, pos+1, instrs, newI))
		    else
		      rmv(cl, pos+1, instrs, instr::newI)
	      in
		rmv(moves, 0, !insns, [])
	      end (* rmvMoves *)

	  (* rmvMovesDU --- identical to rmvMoves except that it also 
	   *   updates the def/use list.
	   *)
	  fun rmvMovesDU (currblk, []) = ()
	    | rmvMovesDU (currblk, moves) = let
		val F.BBLOCK{insns, ...} = Array.sub(cblocks, currblk)
		val blockdu = Array.sub(blockDU, currblk)
		fun rmv([], _, instrs, bdu, newI, newB) = 
		     (insns := rev newI @ instrs;
		      blockdu := rev newB @ bdu)
		  | rmv(cl as(indx::indices), pos, instr::il, du::dl, 
							newI, newB) =
		    if indx = pos then
		      rmv(indices, pos+1, il, dl, newI, newB)
		    else
		      rmv(cl, pos+1, il, dl, instr::newI, du::newB)
	      in
		rmv(moves, 0, !insns, !blockdu, [], [])
	      end (* rmvMovesDU *)
    in
      fun rmvCoalesced moves = rmCoalesced(rmvMoves, moves)
      fun rmvCoalescedDU moves = rmCoalesced(rmvMovesDU, moves)
    end

		    (*-----------select-------------*)
    (* spilling has occurred, and we retain coalesces upto to first
     * potential (chaitin) spill. Any move that was coalesced after 
     * the spillFlag was set, is undone.
     *)
    fun undoCoalesced [] = ()
      | undoCoalesced((v,u,status)::rest) =
         (Array.update(regMap, v, v);
	  status := MOVE;
	  undoCoalesced rest)

    (* assigns colors to nodes in the stack using 
     * the Briggs's optimistic spilling.
     *)
    fun assignColors(WKL{stack, spillWkl, ...}) = let 
      fun optimistic([], spills) = spills
	| optimistic(n::ns, spills) = let
	    fun neighbors [] = []
	      | neighbors(r::rs) = let 
	          val col = chase r
		in
		  if col < Ra.firstPseudoR then col::neighbors rs
		  else neighbors rs
                end
	    fun getcolor () = Ra.getreg([], neighbors(Array.sub(graph, n)))
	  in
	    let val col = getcolor()
	    in
	      Array.update(regMap, n, col);
	      optimistic(ns, spills)
	    end
	      handle _ => optimistic(ns, SL.enter(n,spills))
          end
    in
      case optimistic(stack, [])
       of [] =>
 	     (normalizeRegMap regMap;
	      rmvCoalesced initialMoves)
	| spills  =>
	     (app (fn r => Array.update(regMap, r, r)) stack;
	      undoCoalesced (!undoInfo);
	      normalizeRegMap regMap;
	      rmvCoalescedDU initialMoves;
	      rerun spills)
    end (* assignColors *)


		    (*---------main------------*)
    (* iterate (WKL{count,simplifyWkl,freezeWkl,spillWkl,moveWkl,stack})
     * Note: freezeWkl or spillWkl are maintained lazily.
     *)
    fun iterate(wkls as WKL{count=0, ...}) = assignColors wkls
      | iterate(wkls as WKL{simplifyWkl= _::_, ...}) = iterate(simplify wkls)
      | iterate(wkls as WKL{moveWkl= _::_, ...}) = iterate(coalesce wkls)
      | iterate(wkls as WKL{freezeWkl= _::_, ...}) = iterate(freeze wkls)
      | iterate(wkls as WKL{spillWkl= _::_, ...}) = iterate(selectSpillNode wkls)
      | iterate _ = error "interference graph is uncolorable!"
      
(*    val iterate = phase "Compiler 128 simplify + coalesce" iterate *)
  in
    iterate (WKL initialWkls)
  end (* graphColoring *)


  fun ra(cluster as (F.CLUSTER{blocks, regmaps})) = 
    if (Ra.firstPseudoR = Ra.maxPseudoR()) then codegen cluster
    else let
	val regMap = Ra.regMap regmaps

	fun numOfCBlocks(F.BBLOCK _::rest) = 1 + numOfCBlocks rest
	  | numOfCBlocks(_::rest) = numOfCBlocks rest
	  | numOfCBlocks [] = 0
	val nCBlks = numOfCBlocks blocks
	val blockDU = Array.array(nCBlks, ref ([]: (int list * int list) list))
	val cblocks = Array.array(nCBlks, F.MARK)

	val isDedicated = SL.member dedicatedSet

	fun blockDefUse((blk as F.BBLOCK{insns, ...})::blks, n) = let
	      fun insnDefUse insn = let 
		  val (d,u) = defUse insn
		  fun rmv [] = []
		    | rmv (l as [x]) = if isDedicated x then [] else l
		    | rmv set = SL.difference(SL.uniq(set), dedicatedSet)
		in (rmv d, rmv u)
		end
	    in
	      System.Unsafe.update(cblocks, n, blk);
	      System.Unsafe.update(blockDU, n, ref(map insnDefUse(!insns)));
	      blockDefUse(blks, n+1)
	    end
	  | blockDefUse(_::blks, n) = blockDefUse(blks, n)
	  | blockDefUse([], _) = ()

	fun debug msg = 
	  if Word.andb(Word.fromInt(!misc4), 0w32768) = 0w0 then ()
	  else (print ("------------------" ^ msg ^ " ----------------\n");
		printBlocks(blocks,regmaps))

	val graphColoring = phase "Compiler 128 graph coloring" graphColoring
      in
	blockDefUse(blocks, 0);
	Ra.Liveness.liveness(blocks,regMap);
	debug "before register allocation";
	graphColoring(blocks,cblocks,blockDU,[],regmaps,0);
	debug "after register allocation";
	codegen cluster
      end 

  val ra = phase "Compiler 127 ra init" ra
end (* functor *)


