(* bbsched2.sml
 *
 * COPYRIGHT (c) 1996 Bell Laboratories.
 *)

(** bbsched2.sml - invoke scheduling after span dependent resolution **)
functor BBSched2
    (structure Flowgraph : FLOWGRAPH
     structure InsnProps : INSN_PROPERTIES
     structure Emitter : EMITTER_NEW
     val icount : int -> InsnProps.I.instruction

     sharing Flowgraph.I = InsnProps.I = Emitter.I) : BBSCHED =

struct

  structure F = Flowgraph
  structure I = F.I
  structure P = InsnProps
  structure C = InsnProps.C
  structure E = Emitter

  structure Scheduler = Scheduler(structure InsnProps=P val icount=icount)

  fun error msg = ErrorMsg.impossible ("BBSched."^msg)

  (* Note: posn does not include previous sdi instructions *)
  datatype sdiInfo = 
      SDI of {size : int ref, 		(* size of instruction *)
	      indx : int,		(* index in instruction vector *)
	      posn : int,		(* offset in code string*)
	      insn : I.instruction}  	(* instruction *)


  datatype compressed = 
      DATA of int
    | STRING of Label.label * int
    | REAL   of Label.label
    | LABEL  of Label.label
    | JMPTBL of Label.label * int
    | CODE of {baseSize : int,
	       sdis : sdiInfo list}	(* in order of increasing posn *)

  val clusterList : F.cluster list ref = ref []
  fun cleanUp() = clusterList := []

  (* bbsched --- get rid of hairy BBLOCK constructor *)
  fun bbsched(F.CLUSTER{blocks,regmaps}) = let
      fun f(F.BBLOCK{insns,...}) = F.CODE(ref(Vector.fromList(!insns)))
	| f x = x
  in clusterList:= F.CLUSTER{blocks=map f blocks,regmaps=regmaps}::(!clusterList)
  end

  fun finish() = let
    fun compress(F.CLUSTER{blocks,regmaps}::rest) = let
	  fun f(F.MARK::rest) = DATA 4::f rest
	    | f(F.LABEL lab::rest) = LABEL lab:: f rest
	    | f(F.REALconst(l,r)::rest) = REAL l:: f rest
	    | f(F.STRINGconst(l,_,s)::rest) = 
	        STRING(l,String.size s)::f rest
	    | f(F.JMPtable(l,labs)::rest) = 
	        JMPTBL(l,4 * length labs)::f rest
	    | f(F.CODE insns::rest) = let
		val instrs = !insns
		val len = Vector.length instrs
		fun mkSdiList(~1, sdis, baseSize, pos) = (rev sdis, baseSize)
		  | mkSdiList(n, sdis, baseSize, pos) = let
		      val instr = Vector.sub(instrs, n)
		      val size = P.minSize instr
		    in
		      if P.isSdi instr then
			mkSdiList(n-1,
				  SDI{indx=n, posn=pos, size=ref size,
				      insn=instr}::sdis,
				  baseSize,
				  pos)
		      else
			mkSdiList(n-1, sdis, baseSize+size, pos+size)
		    end

		val (sdis, baseSize) = mkSdiList(len-1, [], 0, 0)
	      in
		CODE{baseSize=baseSize, sdis=sdis}:: f rest
	      end
	    | f(F.BBLOCK _::_) = error "finish.compress.f: BBLOCK"
	    | f [] = compress rest
	in
	  f blocks
	end
      | compress [] = []

    fun align n = Word.toInt(Word.andb(Word.fromInt n + 0w7, 
				       Word.notb 0w7))

    fun labels(DATA n::rest, pos) = labels(rest, pos+n)
      | labels(REAL lab::rest, pos) = let val aligned = align pos
	in Label.setAddr(lab,aligned+8); labels(rest,aligned+16)
	end
      | labels(STRING(lab,size)::rest, pos) = 
	 (Label.setAddr(lab,pos+8); labels(rest,pos+8+size))
      | labels(JMPTBL(lab,size)::rest, pos) = 
	 (Label.setAddr(lab,pos); labels(rest, pos+size))
      | labels(LABEL lab::rest, pos) = 
	 (Label.setAddr(lab,pos); labels(rest, pos))
      | labels(CODE{baseSize,sdis}::rest, pos) = let
	  fun sdiDelta(SDI{size, ...}::rest, delta) = 
		sdiDelta(rest, delta+(!size))
	    | sdiDelta([], delta) = delta
	in 
	  labels(rest, pos+baseSize+sdiDelta(sdis,0))
	end
      | labels([], pos) = pos

    fun adjust(DATA n::rest,pos,changed) = adjust(rest,pos+n,changed)
      | adjust(REAL _::rest,pos,changed) = adjust(rest,(align pos)+16,changed)
      | adjust(STRING(_,s)::rest,pos,changed) = adjust(rest,pos+s+8,changed)
      | adjust(LABEL _::rest,pos,changed) = adjust(rest,pos,changed)
      | adjust(JMPTBL(_,s)::rest,pos,changed) = adjust(rest,pos+s,changed)
      | adjust(CODE{baseSize,sdis}::rest, pos, changed) = let
	  fun doSdis(SDI{size,posn,insn,...}::rest, delta, changed) = let
		val sdiPosn = pos + posn + delta
		val newSize = P.sdiSize(insn, Label.addrOf, sdiPosn)
	      in 
		if newSize <= !size then
		  doSdis(rest, delta+(!size), changed)
		else (size := newSize;
		      doSdis(rest, delta+newSize, true))
	      end
	    | doSdis([], delta, changed) = (delta, changed)
	  val (delta,changed') = doSdis(sdis, 0, changed)
	in 
	  adjust(rest, pos+baseSize+delta, changed')
	end
      | adjust([],_,changed) = changed

    fun fixpoint zl = let val size = labels(zl,0)
	in if adjust(zl,0,false) then fixpoint zl else size
	end

    fun emitCluster(F.CLUSTER{blocks,regmaps}::clusters,zl) = let
	  fun emit(F.MARK::rest,zl) = (E.mark(); emit(rest,zl))
	    | emit(F.LABEL lab::rest,zl) = (E.defineLabel lab; emit(rest,zl))
	    | emit(F.REALconst arg::rest,zl) = (E.emitReal arg; emit(rest,zl))
	    | emit(F.STRINGconst (arg as (l,_,s))::rest,zl) = 
		(E.emitString arg; emit(rest,zl))
	    | emit(F.JMPtable (arg as (_,ls))::rest,zl) = 
		(E.emitJmpTable arg; emit(rest,zl))
	    | emit(F.CODE insns::rest,zl) = let
		val instrs = !insns
		fun findCode((z as CODE _)::zl') = (z,zl')
		  | findCode(_::zl) = findCode zl
		  | findCode [] = error "emitCluster.emit.findCode"
		val len = Vector.length instrs
		fun collect(~1,[],acc) = Vector.fromList acc
		  | collect(n,[],acc) = collect(n-1,[],Vector.sub(instrs,n)::acc)
		  | collect(n, sdis as (SDI{indx,size,insn,...}::rest), acc) =
		    if n<>indx then collect(n-1,sdis,Vector.sub(instrs,n)::acc)
		    else let
			 val sdiCode = P.expand(insn,!size,fn x => x)
			 fun add([],acc) = acc
			   | add(i::instrs,acc) = add(instrs,i::acc)
		      in collect(n-1, rest, add(sdiCode,acc))
		      end
		val (CODE{sdis,...},zl') = findCode zl
		val scheduled =
		    (case sdis
		      of [] => Scheduler.schedule (instrs,regmaps)
		       | _ => Scheduler.schedule(collect(len-1,sdis,[]), regmaps)
		     (*esac*)) 
	      in
		app (fn i => E.emitInstr(i,regmaps)) scheduled;
		emit(rest,zl')
	      end
	    | emit(F.BBLOCK _::_,_) = error "emitCluster.emit: BBLOCK"
	    | emit([],zl) = zl
	in emitCluster(clusters,emit(blocks,zl))
	end
      | emitCluster([],_) = ()

    val clusters = (rev (!clusterList)) before cleanUp()
    val compressed = compress clusters
  in
    E.init(fixpoint compressed);
    emitCluster(clusters,compressed)
  end (*finish*)

  val finish = Stats.doPhase (Stats.makePhase "Compiler 130 Schedule") finish
end (* bbsched2 *)

