(* Copyright 1989 by AT&T Bell Laboratories *)
structure Prim : sig val primEnv : StaticEnv.staticEnv
		     val inLineName : Access.primop -> string
		     val pure : Access.primop -> bool
		     val mayRaise : Access.primop -> bool
		     val special : Access.access -> bool
                     val assignVar : Variables.var
		 end = 
struct
   open Access Modules Variables Types BasicTypes Symbol Fixity
   structure P = Access.P (* to avoid confusing SourceGroup *)

(* primTypes structure *)

   val env =
	foldr (fn ((s, t), e) => Env.bind(tycSymbol s, TYCbind t, e)) Env.empty 
	   [("bool", 	boolTycon),
	    ("list", 	listTycon),
	    ("ref", 	refTycon),
	    ("unit", 	unitTycon),
	    ("int", 	intTycon),
	    ("real", 	realTycon),	
	    ("word",	wordTycon),
	    ("word8",	word8Tycon),
	    ("word32",	word32Tycon),
	    ("cont", 	contTycon),
	    ("array", 	arrayTycon),
	    ("vector", 	vectorTycon),
	    ("string", 	stringTycon),
	    ("char", 	charTycon),
	    ("exn", 	exnTycon),
            ("frag", 	fragTycon)
	   ]

   val env = foldr (fn ((s,c),e) => Env.bind(varSymbol s, CONbind c, e)) env [
	    ("true", trueDcon),
	    ("false", falseDcon),
	    ("::", consDcon),
	    ("nil", nilDcon),
	    ("ref", refDcon),
            ("QUOTE", QUOTEDcon),
            ("ANTIQUOTE", ANTIQUOTEDcon)
	  ]

   val primTypes = ModuleUtil.mkStructure(Env.consolidate env,
					  InvPath.IPATH [])

(* uList structure *)
   
   val env = Env.bind(tycSymbol "list", TYCbind ulistTycon, Env.empty)

   val env = Env.bind(varSymbol "nil", CONbind unilDcon,
                    Env.bind(varSymbol "::", CONbind uconsDcon, env))

   val uList = ModuleUtil.mkStructure(Env.consolidate env,
				      InvPath.IPATH [])

(* inLine structure *)

   val bottom = POLYty{sign=[{weakness=infinity,eq=false}], abs=0,
		       tyfun=TYFUN{arity=1,body=IBOUND 0}}

   local
     fun bits size oper = P.ARITH{oper=oper, overflow=false, kind=P.INT size}
     val bits31 = bits 31		

     fun int size oper = P.ARITH{oper=oper, overflow=true, kind=P.INT size}
     val int31 = int 31

     fun word size oper = P.ARITH{oper=oper, overflow=false, kind=P.UINT size}
     val word32 = word 32
     val word31 = word 31
     val word8  = word 8

     fun float size oper = P.ARITH{oper=oper, overflow=true, kind=P.FLOAT size}
     val float64 = float 64

     fun purefloat size oper = P.ARITH{oper=oper,overflow=false,kind=P.FLOAT size}
     val purefloat64 = purefloat 64	

     fun cmp kind oper = P.CMP{oper=oper, kind=kind}
     val int31cmp = cmp (P.INT 31)

     val word32cmp = cmp (P.UINT 32)
     val word31cmp = cmp (P.UINT 31)
     val word8cmp  = cmp (P.UINT 8)

     val float64cmp = cmp (P.FLOAT 64)

     fun sub kind = P.NUMSUBSCRIPT{kind=kind, checked=false, immutable=false}
     fun chkSub kind = P.NUMSUBSCRIPT{kind=kind, checked=true, immutable=false}

     fun subv kind = P.NUMSUBSCRIPT{kind=kind, checked=false, immutable=true}
     fun chkSubv kind = P.NUMSUBSCRIPT{kind=kind, checked=true, immutable=true}

     fun update kind = P.NUMUPDATE {kind=kind, checked=false}
     fun chkUpdate kind = P.NUMUPDATE {kind=kind, checked=true}
   in
     val primopNames = 
         [("capture",	P.CAPTURE),
	  ("callcc",	P.CALLCC),
	  ("throw",	P.THROW),
	  ("!",		P.DEREF),
	  (":=",	P.ASSIGN),
	  ("makeref",	P.MAKEREF),
	  ("boxed",	P.BOXED),
	  ("unboxed",	P.UNBOXED),
	  ("cast",	P.CAST),
	  ("=",		P.POLYEQL),
	  ("<>",	P.POLYNEQ),
	  ("inlnot",	P.INLNOT),
	  ("ptreql",	P.PTREQL),
	  ("ptrneq",	P.PTRNEQ),
	  ("getvar",	P.GETVAR),
	  ("setvar",	P.SETVAR),
	  ("setpseudo",	P.SETPSEUDO),
	  ("getpseudo",	P.GETPSEUDO),
	  ("mkspecial", P.MKSPECIAL),
	  ("getspecial",P.GETSPECIAL),
	  ("setspecial",P.SETSPECIAL),
	  ("gethdlr",	P.GETHDLR),
	  ("sethdlr",	P.SETHDLR),
	  ("boxedupdate",P.BOXEDUPDATE),
	  ("unboxedupdate",P.UNBOXEDUPDATE),
	  ("getrunvec",	P.GETRUNVEC),
	  ("gettag", 	P.GETTAG),
	  ("setmark",	P.SETMARK),
	  ("dispose",	P.DISPOSE),
	  ("uselvar",	P.USELVAR),
	  ("deflvar",	P.DEFLVAR),
	  ("compose",	P.INLCOMPOSE),
	  ("before",	P.INLBEFORE),
	  
	  (* integer 31 primops *)
	  ("i31mul",	int31 P.* ),
	  ("i31quot",	int31 P./),
	  ("i31add", 	int31 P.+),
	  ("i31sub",	int31 P.-),
	  ("i31orb",	bits31 P.ORB),
	  ("i31andb",	bits31 P.ANDB),
	  ("i31xorb",	bits31 P.XORB),
	  ("i31notb",	bits31 P.NOTB),
	  ("i31neg",	int31 P.~),
	  ("i31lshift",	bits31 P.LSHIFT),
	  ("i31rshift",	bits31 P.RSHIFT),
	  ("i31lt",	int31cmp P.<),
	  ("i31le",	int31cmp P.<=),
	  ("i31gt",	int31cmp P.>),
	  ("i31ge", 	int31cmp P.>=),
	  ("i31ltu",	int31cmp P.LTU), 
	  ("i31geu",	int31cmp P.GEU),
	  ("i31mod",    P.INLMOD),
	  ("i31div",	P.INLDIV),
	  ("i31rem",	P.INLREM),
	  ("i31max",	P.INLMAX),
	  ("i31min",	P.INLMIN),
	  ("i31abs",	P.INLABS),
	  ("i31eq",	int31cmp P.EQL),
	  ("i31ne",	int31cmp P.NEQ),

	  (* float 64 primops *)
	  ("f64add", 	float64 P.+),
	  ("f64sub",	float64 P.-),
	  ("f64div", 	float64 P./),
	  ("f64mul",	float64 P.* ),
	  ("f64eq",	float64cmp P.EQL),
	  ("f64ge",	float64cmp P.>=),
	  ("f64gt",	float64cmp P.>),
	  ("f64le",	float64cmp P.<=),
	  ("f64lt",	float64cmp P.<),
	  ("f64ne",	float64cmp P.NEQ),
	  ("f64neg",	purefloat64 P.~),
	  ("f64abs",	purefloat64 P.ABS),

	  (* float64 array *)	
	  ("f64Sub",	sub (P.FLOAT 64)),
	  ("f64chkSub",	chkSub (P.FLOAT 64)),
	  ("f64Update",	update (P.FLOAT 64)),
	  ("f64chkUpdate",chkUpdate (P.FLOAT 64)),

	  (* word8 primops *)
	  (* In the long run, we plan to represent 
              WRAPPED word8 tagged, and UNWRAPPED untagged.
             But right now, we represent both of them tagged, with
             23 high-order zero bits and 1 low-order 1 bit.
             In this representation, we can use the comparison and 
             (some of the) bitwise
             operators of word31; but we cannot use the shift and arithmetic
             operators.

	     THIS IS A TEMPORARY HACKJOB until all the word8 primops are
             correctly implemented.

         *)


(*	  ("w8mul",	word8 P.* ),
	  ("w8div",	word8 P./),
	  ("w8add",	word8 P.+),
	  ("w8sub",	word8 P.-),
*)
	  ("w8orb",	word31 P.ORB),
	  ("w8xorb",	word31 P.XORB),
	  ("w8andb",	word31 P.ANDB),
(*	  ("w8notb",	word31 P.NOTB),
	  ("w8rshift",	word8 P.RSHIFT),
	  ("w8rshiftl",	word8 P.RSHIFTL),
	  ("w8lshift",	word8 P.LSHIFT),
*)
	  ("w8gt",	word8cmp P.>),
	  ("w8ge",	word8cmp P.>=),
	  ("w8lt",	word8cmp P.<),
	  ("w8le",	word8cmp P.<=),
	  ("w8eq",	word8cmp P.EQL),
	  ("w8ne",	word8cmp P.NEQ),
(*
	  ("w8toint",	P.ROUND{floor=true, fromkind=P.UINT 8, tokind=P.INT 31}),
	  ("w8fromint",P.REAL{fromkind=P.INT 31,tokind=P.UINT 8}),
*)

	  (* word8 array and vector *)
	  ("w8Sub",	sub (P.UINT 8)),		
	  ("w8chkSub",	chkSub (P.UINT 8)),
	  ("w8subv",	subv (P.UINT 8)),
	  ("w8chkSubv",	chkSubv (P.UINT 8)),
	  ("w8update",	update (P.UINT 8)),
	  ("w8chkUpdate",chkUpdate (P.UINT 8)),

	(* word31 primops *)
	  ("w31mul",	word31 P.* ),
	  ("w31div",	word31 P./),
	  ("w31add",	word31 P.+),
	  ("w31sub",	word31 P.-),
	  ("w31orb",	word31 P.ORB),
	  ("w31xorb",	word31 P.XORB),
	  ("w31andb",	word31 P.ANDB),
	  ("w31notb",	word31 P.NOTB),
	  ("w31rshift",	word31 P.RSHIFT),
	  ("w31rshiftl",word31 P.RSHIFTL),
	  ("w31lshift",	word31 P.LSHIFT),
	  ("w31gt",	word31cmp P.>),
	  ("w31ge",	word31cmp P.>=),
	  ("w31lt",	word31cmp P.<),
	  ("w31le",	word31cmp P.<=),
	  ("w31eq",	word31cmp P.EQL),
	  ("w31ne",	word31cmp P.NEQ),
	  ("w31toint",	P.REAL{fromkind=P.UINT 31, tokind=P.INT 31}),
	  ("w31fromint",P.REAL{fromkind=P.INT 31, tokind=P.UINT 31}),
	  ("w31tow32",	P.REAL{fromkind=P.UINT 31, tokind=P.UINT 32}),
	  ("w31fromw32",P.REAL{fromkind=P.UINT 32, tokind=P.UINT 31}),

	(* word32 primops *)
	  ("w32mul",	word32 P.* ),
	  ("w32div",	word32 P./),
	  ("w32add",	word32 P.+),
	  ("w32sub",	word32 P.-),
	  ("w32orb",	word32 P.ORB),
	  ("w32xorb",	word32 P.XORB),
	  ("w32andb",	word32 P.ANDB),
	  ("w32notb",	word32 P.NOTB),
	  ("w32rshift",	word32 P.RSHIFT),
	  ("w32rshiftl",word32 P.RSHIFTL),
	  ("w32lshift",	word32 P.LSHIFT),
	  ("w32gt",	word32cmp P.>),
	  ("w32ge",	word32cmp P.>=),
	  ("w32lt",	word32cmp P.<),
	  ("w32le",	word32cmp P.<=),
	  ("w32eq",	word32cmp P.EQL),
	  ("w32ne",	word32cmp P.NEQ),
	  ("w32toint",	P.ROUND{floor=true, fromkind=P.UINT 32, tokind=P.INT 31}),
	  ("w32fromint",P.REAL{fromkind=P.INT 31,tokind=P.UINT 32}),

	  ("length",	P.LENGTH),
	  ("objlength",	P.OBJLENGTH),


	  (* polymorphic array and vector *)
	  ("arrSub", 	P.SUBSCRIPT),
	  ("arrChkSub",	P.INLSUBSCRIPT),
	  ("vecSub",	P.SUBSCRIPTV),
	  ("vecChkSub",	P.INLSUBSCRIPTV),
	  ("arrUpdate",	P.UPDATE),
	  ("arrChkUpdate",P.INLUPDATE),

	  ("floor",P.ROUND{floor=true,fromkind=P.FLOAT 64,tokind=P.INT 31}),
	  ("round",P.ROUND{floor=false,fromkind=P.FLOAT 64,tokind=P.INT 31}),
	  ("real",P.REAL{fromkind=P.INT 31,tokind=P.FLOAT 64}),

	  ("ordof",P.NUMSUBSCRIPT{kind=P.INT 8,checked=false,immutable=true}),
	  ("store",P.NUMUPDATE{kind=P.INT 8,checked=false}),
	  ("inlbyteof",P.NUMSUBSCRIPT{kind=P.INT 8,checked=true,immutable=false}),
	  ("inlstore",P.NUMUPDATE{kind=P.INT 8,checked=true}),
	  ("inlordof",P.NUMSUBSCRIPT{kind=P.INT 8,checked=true,immutable=true})
	 ]
   end

   fun enter((s : string, p : primop), env) =
       let val name = varSymbol s
        in Env.bind(name,
		    VARbind(VALvar{access=INLINE p, path=SymPath.SPATH[name],
				   typ= ref bottom}),
		    env)
       end

   val assignVar =
         VALvar{access=INLINE P.ASSIGN,
                path=SymPath.SPATH [Symbol.varSymbol ":="],
                typ= ref bottom}

   val inLine =
      ModuleUtil.mkStructure(Env.consolidate(foldr enter Env.empty primopNames),
			     InvPath.IPATH [])

  (* priming structures: PrimTypes and InLine *)
   val nameofPT = Symbol.strSymbol "PrimTypes"
   val varofPT = STRvar{name=nameofPT,access=NO_ACCESS,binding=primTypes}
   val nameofUL = Symbol.strSymbol "UnrolledList"
   val varofUL = STRvar{name=nameofUL,access=NO_ACCESS,binding=uList}
   val nameofIL = Symbol.strSymbol "InLine"
   val varofIL = STRvar{name=nameofIL,access=NO_ACCESS,binding=inLine}


   val primEnv =
         Env.bind(nameofIL,STRbind varofIL, 
           Env.bind(nameofUL,STRbind varofUL,
  	     Env.bind(nameofPT,STRbind varofPT,
		      ModuleUtil.openStructureVar (Env.empty,varofPT))))


   fun inLineName p =
       let fun find [] = ErrorMsg.impossible "Prim.inLineName - bad primop name"
	     | find ((s,p1)::rest) = if p1=p then s else find rest
        in find primopNames
       end

   val pure =
     fn P.DEREF => false
      | P.ASSIGN => false (* this should probably should never be called on ASSIGN *)
      | P.SUBSCRIPT => false
      | P.BOXEDUPDATE => false
      | P.UNBOXEDUPDATE => false
      | P.UPDATE => false
      | P.CAPTURE => false
      | P.CALLCC => false
      | P.ARITH{overflow,...} => not overflow
      | P.NUMSUBSCRIPT{immutable,...} => immutable
      | P.GETSPECIAL => false
      | P.SETSPECIAL => false
      | _ => true
  
   val mayRaise =
     fn P.ARITH{overflow,...} => overflow
      | P.ROUND _ => true
      | P.INLSUBSCRIPT => true
      | P.INLUPDATE => true
      | P.INLSUBSCRIPTV => true
      | P.NUMSUBSCRIPT{checked,...} => checked
      | P.NUMUPDATE{checked,...} => checked
      | _ => false

   fun special (INLINE P.POLYEQL) = true
     | special (INLINE P.POLYNEQ) = true
     | special (INLINE P.ASSIGN) = true
     | special (INLINE P.UPDATE) = true
     | special (INLINE P.INLUPDATE) = true
     | special (INLINE P.INLSUBSCRIPT) = true
     | special (INLINE P.INLSUBSCRIPTV) = true
     | special (INLINE(P.NUMSUBSCRIPT{checked,...})) = checked
     | special (INLINE(P.NUMUPDATE{checked,...})) = checked
     | special (INLINE P.INLDIV) = true
     | special (INLINE P.INLMOD) = true
     | special (INLINE P.INLREM) = true
     | special (INLINE P.INLMIN) = true
     | special (INLINE P.INLMAX) = true
     | special (INLINE P.INLABS) = true
     | special (INLINE P.INLNOT) = true
     | special (INLINE P.INLCOMPOSE) = true
     | special (INLINE P.INLBEFORE) = true
     | special _ = false

end (* structure Prim *)

