(* pickle.sml
 *
 * COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 *)

signature PICKLE_ENV =
sig
  type pickledEnv
  exception Rehydrate
  val pickleEnv : {env:StaticEnv.staticEnv, context: SCStaticEnv.staticEnv}
                  -> pickledEnv
  val unPickleEnv : {env:pickledEnv, context: SCStaticEnv.staticEnv}
                    -> StaticEnv.staticEnv
  val debugging: bool ref
end

abstraction PickleEnv: PICKLE_ENV =
struct

  fun arrayapp f a =
      let val n = Array.length a
	  fun g i = if i<n then (f (Array.sub(a,i)); g(i+1)) else ()
       in g 0
      end

  val debugging = ref false

  fun ifdebug f x = if !debugging then f x else ()

  open ModuleId Modules ModuleUtil Types Variables
  structure LT = LambdaType

  exception Rehydrate

  val printst = print o Stamps.stampToString

  fun printid(SIMPLEid st) = (print "SIMPLEid("; printst st; print ")")
    | printid(INSTANCEid{origin,sign}) =
        (print "INSTANCEid{origin="; printst origin; print ",sign=";
	 printst sign; print ")")
    | printid(SIGid st) = (print "SIGid("; printst st; print ")")
    | printid(FCTid st) = (print "FCTid("; printst st; print ")")
    | printid(FCT_INSTANCEid{fct,fsig}) =
        (print "FCT_INSTANCEid{fct="; printid fct; print ",fsig=";
	 printid fsig; print "}")
    | printid(FSIGid{argument,body}) =
        (print "FSIGid{argument="; printst argument; print ",body=";
         printst body; print ")")
    | printid(TYCid st) = (print "TYCid("; printst st; print ")")

  fun for(l,h) f = if l<=h then (f l; for(l+1,h) f) else ();
  fun map1 f NONE = NONE | map1 f (SOME x) = SOME(f x)

(*
  fun printpath [sym] = (print(Symbol.name sym))
    | printpath (a::rest) = (print(Symbol.name a); print "."; printpath rest)
    | printpath nil = print "EMPTY PATH"

  val anonsigsym = [Symbol.sigSymbol "SIG"]
  val anonfsigsym = [Symbol.sigSymbol "FSIG"]
  val functorsym = [Symbol.fctSymbol "FCT"]
  val functorisym = [Symbol.fctSymbol "FCT_INSTANCE"]

  fun showModuleTable(key,path) = 
      (print "  "; printid key; print " -> "; printpath path; print "\n")

  fun showsize x = x (* (if !dosizes then print(Size.size x) else (); x) *)

*)

  datatype mode = DEHYDRATE | REHYDRATE

  fun pickle mode {env,context} =
      let val env = Env.consolidate env

	  fun unique(table,SClooker,stubcons) (id, make) =
	      case ModuleTable.look(id, !table)
		of NONE =>
		    let val v =
			    case (mode,SClooker context id)
			      of (DEHYDRATE, SOME _) => 
				     (ifdebug (fn()=>(print "       extern: ";
						      printid id;
						      print "\n")) ();
				      stubcons id)
			       | (REHYDRATE, SOME object) => 
				     (ifdebug (fn()=>(print "       extern: ";
						      printid id;
						      print "\n")) ();
				      object)
			       | (_, NONE) => 
				     (ifdebug(fn()=>(print "local: "; 
						     printid id;
						     print "  [\n"))();
				      make() before 
				       ifdebug print "]\n")
                     in table := ModuleTable.bind(id,v,!table);
			v
		    end
	         | SOME v =>
		    (ifdebug (fn()=>(print "       intern: \n"))
		             (); 
		     v)

          val uniqSig = unique (ref ModuleTable.empty, 
				SCStaticEnv.lookSIG,EXTERN_SIG)
          val uniqFSig = unique (ref ModuleTable.empty,
				SCStaticEnv.lookFSIG, EXTERN_FSIG)
          val uniqStr = unique (ref ModuleTable.empty,
				SCStaticEnv.lookSTR, STR_EXTERN)
          val uniqFct = unique (ref ModuleTable.empty,
				SCStaticEnv.lookFCT, FCT_EXTERN)
	  
	  val uniqTyc = unique (ref ModuleTable.empty,
				SCStaticEnv.lookTYC, EXTERNtyc)

fun eqTyc(DEFtyc{path=p1,strict=s1,tyfun=TYFUN{arity=a1,body=ty1}},
	  DEFtyc{path=p2,strict=s2,tyfun=TYFUN{arity=a2,body=ty2}}) =
          InvPath.equal(p1,p2) andalso s1=s2 andalso a1=a2
	   andalso eqTy(ty1,ty2)
  | eqTyc(FORMtyc{pos=p1,spec=s1,name=n1},FORMtyc{pos=p2,spec=s2,name=n2}) =
      p1=p2 andalso n1=n2 andalso eqTyc(s1,s2)
  | eqTyc(OPENFORMtyc{pos=p1,spec=s1,path=n1},
	  OPENFORMtyc{pos=p2,spec=s2,path=n2}) =
      p1=p2 andalso SymPath.equal(n1,n2) andalso eqTyc(s1,s2)
  | eqTyc(ABSFBtyc(p1,s1),ABSFBtyc(p2,s2)) =
      p1=p2 andalso eqTyc(s1,s2)
  | eqTyc(RELtyc{path=n1,pos=p1,...},RELtyc{path=n2,pos=p2,...}) =
      InvPath.equal(n1,n2) andalso p1=p2
  | eqTyc _ = false

and eqTy(IBOUND i1, IBOUND i2) = i1=i2
  | eqTy(VARty v1, VARty v2) = v1=v2
  | eqTy(CONty(t1,a1),CONty(t2,a2)) = 
          eqTyc (t1,t2) andalso List2.all2 eqTy(a1,a2)
  | eqTy(POLYty{sign=s1,tyfun=TYFUN{arity=a1,body=t1},abs=b1},
	 POLYty{sign=s2,tyfun=TYFUN{arity=a2,body=t2},abs=b2}) =
        s1=s2 andalso a1=a2 andalso b1=b2 andalso eqTy(t1,t2)
  | eqTy _ = false

	  val tycs = ref (nil: (tycon * tycon) list)

	  fun tycHack (t: tycon,make): tycon =
	      let fun look((a,b)::rest) = if eqTyc(a,t)
					      then b else look rest
		    | look nil = 
		      let val b  = make()
	               in tycs := (t,b):: !tycs;
			   b
		      end
               in look (!tycs)
              end

	  fun fail() = raise Rehydrate

          type 'a pair = 'a * 'a
          val strAmap : Structure Array.array pair list ref = ref nil
          val fctAmap : Functor Array.array pair list ref = ref nil
          val tycAmap : tycon Array.array pair list ref = ref nil
          fun uniqA(amap,from,transform) =
             let fun look((a,b)::rest) = if a=from then b else look rest
                   | look nil = 
		  let open Array infix 9 sub
		      val new = if length from = 0 
				  then from
		      		  else array(length from, from sub 0)
		   in amap := (from,new) :: !amap;
		      for(0,length(new)-1)
			(fn i => update(new,i,transform(from sub i)));
		      new
		  end
	      in look (!amap)
	     end

	  val dconrefs = ref (nil: (Types.tyckind ref * 
				    Types.datacon list) list)

          fun addref dcons = 
	      let val r = ref(DATAtyc nil)
	       in dconrefs := (r,dcons) :: !dconrefs;
		   r
              end

	  fun transLty t = case mode 
	      of DEHYDRATE => t
               | REHYDRATE => LT.rehashcons t

          fun resolveref(r,dcons) =   r := DATAtyc(map transdcon dcons)

          and transEnv e = Env.map transBind e

          and transStr(SIMPLE{stamp,path,env,id}) =
	        uniqStr(id,
		    fn()=>SIMPLE{stamp=stamp,path=path,env=transEnv env,id=id})
            | transStr(INSTANCE{sign,origin as STAMP_ORIGIN s,
				subStrs,subFcts,types,path}) =
                uniqStr(INSTANCEid{origin=s, sign=getSignStamp sign},
			(fn () =>
			  INSTANCE{sign=transSig sign,
				   origin=origin,
				   subStrs=uniqA(strAmap,subStrs,transStr),
				   subFcts=uniqA(fctAmap,subFcts,transFct),
				   types=  uniqA(tycAmap,types,transTyc),
				   path=path}))
            | transStr(INSTANCE{sign,origin,subStrs,subFcts,types,path}) =
		INSTANCE{sign=transSig sign,
			 origin=origin,  (* ABS_ or NULL_ORIGIN *)
			 subStrs=uniqA(strAmap,subStrs,transStr),
			 subFcts=uniqA(fctAmap,subFcts,transFct),
			 types=  uniqA(tycAmap,types,transTyc),
			 path=path}
	    | transStr(STR_OPEN{pos,spec,path}) =
		STR_OPEN{pos=pos,spec=transSig(spec),path=path}	
	    | transStr(STR_FORMAL{pos,spec}) =
		STR_FORMAL{pos=pos,spec=transSig(spec)}
	    | transStr(APPLY{fct,arg,res,path}) =
	        APPLY{fct=transFct(fct),arg=transStr(arg),res=transStr(res),
		      path=path}
            | transStr(STR_EXTERN id) = (* unpickling *)
		uniqStr(id,fail)
	    | transStr s = s

	  and transFct(FCT{stamp,parent,paramName,lambdaty,argument,
			   body={strseq,fctseq,tyseq,str}}) =
                uniqFct(FCTid stamp,
			fn()=>FCT{stamp=stamp,
				  parent=transStr parent,
				  paramName=paramName,
				  lambdaty=transLty lambdaty,
				  argument=transSig argument,
				  body={strseq=map transStr strseq,
					fctseq=map transFct fctseq,
					tyseq=map transTyc tyseq,
					str=transStr str}})
            | transFct(FCT_FORMAL{pos,spec}) =
	        FCT_FORMAL{pos=pos, spec=transFSig spec}
	    | transFct(FCT_OPEN{pos,spec,path}) =
	        FCT_OPEN{pos=pos,path=path,spec=transFSig spec}
            | transFct(f as FCT_INSTANCE{fsig,fct,parent,lambdaty}) =
	      let fun g() = 
		     let val fct = transFct fct
		      in FCT_INSTANCE{fsig=transFSig fsig,
					   fct=fct,
					   parent=transStr parent,
					   lambdaty=transLty lambdaty}
		     end
	       in case SOME(FCT_INSTANCEid{fct=fctId fct, fsig=fsigId fsig})
		       handle Id => NONE
		   of SOME id => uniqFct(id, g)
  		    | NONE => g()
	       end
	    | transFct(FCT_EXTERN id) = uniqFct(id,fail)
	    | transFct f = f

	  and transSigKind(TOP{strcount,fctcount,typecount,slotcount,
			       sConstraints,tConstraints,abbreviations}) =
	        TOP{strcount=strcount,fctcount=fctcount,
		    typecount=typecount,slotcount=slotcount,
		    sConstraints = map (fn{internal,external}=>
					{internal=internal,
					 external=map1 transStr external})
					sConstraints,
		    tConstraints = map (fn{internal,external}=>
					{internal=internal,
					 external=map1 transTyc external})
		                       tConstraints,
		    abbreviations = map (fn{internal,external}=>
					 {internal=internal,
					  external=transTyc external})
		                       abbreviations}
            | transSigKind k = k

	  and transSig(SIG{stamp,env,symbols,name,kind}) =
                uniqSig(SIGid stamp,
			fn()=>SIG{stamp=stamp,
				  env=ref(transEnv(!env)),
				  symbols=symbols,
				  name=name,
				  kind= ref(transSigKind(!kind))})
	    | transSig(EXTERN_SIG id) = uniqSig(id,fail)
            | transSig s = s

	  and transFSig(FSIG{name,paramName,argument,body}) =
	       uniqFSig(FSIGid{argument=getSignStamp argument,
			       body=getSignStamp body},
			fn()=> FSIG{name=name,paramName=paramName,
				    argument=transSig argument,body=transSig body})
	    | transFSig(EXTERN_FSIG id) = uniqFSig(id,fail)
	    | transFSig s = s

          and transdcon(DATACON{name,const,typ,rep,orig=SOME t,sign}) =
              DATACON{name=name,const=const,rep=rep,sign=sign,
                      orig=SOME (transty t),typ=transty typ}
            | transdcon(DATACON{name,const,typ,rep,orig,sign}) =
              DATACON{name=name,const=const,rep=rep,sign=sign,
                      orig=orig,typ=transty typ}

          and transty(CONty(tyc,tyl)) = CONty(transTyc tyc,map transty tyl)
            | transty(POLYty{sign,tyfun,abs}) =
		POLYty{sign=sign,tyfun=transtyfun tyfun, abs=abs}
	    | transty(VARty(ref(INSTANTIATED ty))) = transty ty
	    | transty t = t

          and transtyfun(TYFUN{arity,body}) =
	              TYFUN{arity=arity,body=transty body}
	  and transTyc(GENtyc{stamp,path,arity,eq,kind=ref(DATAtyc dcons)}) =
	      uniqTyc(TYCid stamp,
		      fn()=>GENtyc{stamp=stamp,path=path,arity=arity,
				   eq=eq,kind=addref dcons})

	    | transTyc(GENtyc{stamp,path,kind=ref(FORMDEFtyc _),arity,eq}) =
	         transTyc(GENtyc{stamp=stamp,path=path,kind=ref FORMtyck,
				 arity=arity,eq=eq})
	    | transTyc(t as GENtyc{stamp,path,...}) =
	      uniqTyc(TYCid stamp,fn()=> t)

	    | transTyc(EXTERNtyc id) = uniqTyc(id,fail)
	    | transTyc(t as DEFtyc{path,strict,tyfun}) =
	      tycHack(t,fn()=> 
	        DEFtyc{path=path,strict=strict,tyfun=transtyfun tyfun})
	    | transTyc(t as FORMtyc{pos,spec,name}) =
	      tycHack(t,fn()=> 
	        FORMtyc{pos=pos,spec=transTyc spec,name=name})
	    | transTyc(t as OPENFORMtyc{pos,spec,path}) =
	      tycHack(t,fn()=> 
		       OPENFORMtyc{pos=pos,spec=transTyc spec,path=path})
	    | transTyc(t as ABSFBtyc(pos,tyc)) = 
	      tycHack(t,fn()=> ABSFBtyc(pos,transTyc tyc))
	    | transTyc(t as RELtyc{spec=ERRORtyc,...}) = t
	    | transTyc(t as RELtyc{spec,path,pos}) =
	      (* the "spec" part of a RELtyc is needed for the translation
	         into lambda-language.  But for this compilation unit,
		 the translation has already been done.  Other compilation
		 unit won't need the spec. *)

	         tycHack(t,fn()=>RELtyc{spec=ERRORtyc,path=path,pos=pos})

	    | transTyc t = t

	  and transVar(VALvar{access,path,typ=ref t}) =
	           VALvar{access=access,path=path,typ=ref(transty t)}
	    | transVar(OVLDvar{name,options=ref options,scheme}) =
	        let fun f{indicator,variant} =
	                   {indicator=transty indicator,
	 		    variant=transVar variant}
	         in OVLDvar{name=name,options=ref(map f options),
			    scheme=transtyfun scheme}
                end
	    | transVar v = v 

          and transCon(DATACON{name,const,typ,rep,sign,orig=SOME t}) =
              DATACON{name=name,const=const,typ=transty typ,rep=rep,
                        sign=sign,orig=SOME (transty t)}
            | transCon(DATACON{name,const,typ,rep,sign,orig}) =
              DATACON{name=name,const=const,typ=transty typ,rep=rep,
                      sign=sign,orig=orig}

	  and transBind(STRbind(STRvar{name,access,binding})) = 
	       STRbind(STRvar{name=name,access=access,
                              binding=transStr binding})
	    | transBind(FCTbind(FCTvar{name,access,binding})) =
	       FCTbind(FCTvar{name=name,access=access,
                    binding=transFct binding})
	    | transBind(SIGbind(SIGvar{name,binding})) =
	       SIGbind(SIGvar{name=name,binding=transSig binding})
	    | transBind(FSIGbind(FSIGvar{name,binding})) =
	       FSIGbind(FSIGvar{name=name,binding=transFSig binding})
	    | transBind(TYCbind binding) = 
	       TYCbind(transTyc binding)
	    | transBind(VARbind v) = VARbind(transVar v)
	    | transBind(CONbind v) = CONbind(transCon v)
	    | transBind b = b

    in transEnv env 
	before app resolveref (!dconrefs)
   end

type pickledEnv = StaticEnv.staticEnv

val pickleEnv = Stats.doPhase(Stats.makePhase "Compiler 170 Pickle")
	            (pickle DEHYDRATE)

val unPickleEnv = Stats.doPhase(Stats.makePhase "Compiler 004 UnPickle")
		    (pickle REHYDRATE)
(* val _ = (Stats.sayBegin := true; Stats.sayEnd:=true(*; debugging:=true*))*)
end (* structure PickleEnv *)
