(* normalize.sml
 *
 * COPYRIGHT (c) 1996 Bell Laboratories.
 *)

(* 
   Normalize: NORMALIZE

   This structure defines path normalization maps that transform
   a user path to an internal path by replacing the initial symbol
   with a path.  The essential information is represented by a
   a mapping of type symbol -> SymPath.path that is applied to the
   initial symbol of a path.

   As it is necessary in functor signatures to prefix a whole map with
   a symbol (i.e., add a level of indirection at the head of each of
   its elements) we provide a function to express this prefixing without
   redefining the whole map.

   openStr is a primitive mechanism to handle include specifications.
   The other operations available are the basic operations on finite
   maps (empty, apply, extend).

*)

signature NORMALIZE =
  sig
     exception Unbound
     type normMap (* normalization map *)
     val empty : normMap
     val extend : Symbol.symbol * InvPath.path * normMap -> normMap
     val prefix : Symbol.symbol * normMap -> normMap
     val apply : normMap * SymPath.path -> SymPath.path
     val normalize : normMap option * SymPath.path -> SymPath.path
     val openStr : normMap * SymPath.path * Modules.Signature * normMap
	           -> normMap
     val openSubStr : Symbol.symbol * Modules.Signature * normMap
	              -> normMap
  end

structure Normalize : NORMALIZE = struct

open Modules
structure SP = SymPath
structure IP = InvPath
structure CP = ConvertPaths

exception Unbound

type normEnv = IP.path Env.env
  (* an environment used to map symbols to inverse paths *)

abstype normMap
  = EMPTY
  | LAYER of {inner: normEnv, outer: normMap}
  | PREFIX of {name: Symbol.symbol, nmap : normMap}
     (* prefix all mapped paths with name *)
with

  (* the empty normalization map *)
  val empty = EMPTY

  (* apply a normalization map to a path *)

  fun apply (EMPTY, spath: SP.path) = raise Unbound
    | apply (PREFIX{name,nmap}, spath) = SP.prepend(name, apply(nmap,spath))
    | apply (LAYER{inner,outer}, spath) =
      ((case spath
	  of SP.SPATH nil => ErrorMsg.impossible "Normalize.apply"
	   | SP.SPATH [h] => CP.invertIPath(Env.look(inner,h))
	   | SP.SPATH (h :: t) =>
	       SP.append(CP.invertIPath(Env.look(inner,h)),
			 SP.SPATH t))
       handle Env.Unbound => apply(outer,spath))

  (* same as apply but the normMap is optional and we get back the
   * original path if it is not mapped *)
  fun normalize(NONE: normMap option, spath: SP.path) = spath
    | normalize(SOME nmap, spath) =
	apply(nmap ,spath) handle Unbound => spath

  (* extends a normalization map by mapping a symbol to an inverse path *)
  fun extend (sym,rpath,LAYER{inner,outer}) =
      LAYER{inner=Env.bind(sym,rpath,inner),outer=outer}
    | extend (sym,rpath,nmap) =
      LAYER{inner=Env.bind(sym,rpath,Env.empty),outer=nmap}

  (* takes a symbol name and a normMap and produces a new map that
   *  prepends name to each result path *)
  fun prefix(name,nmap) = PREFIX{name=name,nmap=nmap}

  (* extent a normEnv that maps each symbol s in symbols to extend(rpath,s)
   * i.e., s |-> rpath^s *)
  fun mapSymbols(rpath: IP.path, symbols: Symbol.symbol list): normEnv =
      foldr (fn (s,e) => Env.bind(s,IP.extend(rpath,s),e)) Env.empty symbols

  (* normalizes spath relative to bindMap to get spath', extends second
     normMap by mapping symbols of sign by prepending spath' to them *)
  fun openStr (bindMap: normMap, spath: SP.path, sign: Signature,
	       nmap: normMap) =
      let val symbols =
	      case sign
		of SIG{symbols,...} => !symbols
		 | ERROR_SIG => []
		 | _ => ErrorMsg.impossible "Normalize.openStr"
	  val rpath = CP.invertSPath(apply(bindMap,spath))
	  val newenv = mapSymbols(rpath,symbols)
       in LAYER{inner=newenv, outer=nmap}
      end

  (* openSubStr(name,sign,nmap)
   *  let sign' be the signature of the name component of sign and add mappings
   *  s |-> name^s to nmap for each symbol s in sign' *)
  fun openSubStr(name: Symbol.symbol, sign: Signature, nmap: normMap) : normMap =
      let val symbols =  (* get the bound symbol list from the signature *)
	      (case sign 
		 of SIG{env,...} =>
		     (case Env.look(!env,name)
			of STRbind(STRvar{binding=STR_FORMAL{spec=SIG{symbols,...},
							     ...},
					  ...}) =>
			     !symbols
			 | STRbind(STRvar{binding=STR_FORMAL{spec=ERROR_SIG,...},
					  ...}) =>
			     []
			 | _ => ErrorMsg.impossible "Normalize.openSubStr 1")
		  | ERROR_SIG => []
		  | _ => ErrorMsg.impossible "Normalize.openSubStr 2")
	      handle Env.Unbound => ErrorMsg.impossible "Normalize.openSubStr 3"
	  val newenv = mapSymbols(IP.IPATH[name],symbols)
       in LAYER{inner=newenv, outer=nmap}
      end

end (* abstype normMap *)

end (* structure Normalize *)
