(* Copyright 1992 by AT&T Bell Laboratories *)

structure ElabSig : ELABSIG =
struct

open Extern Symbol Absyn Ast PrintUtil AstUtil Types BasicTypes TyvarSet
     Modules EqTypes ModuleUtil TypesUtil Variables Misc ElabUtil ErrorMsg
     Access ElabCore
structure SP = SymPath
structure IP = InvPath

infix -->
val say = Control.Print.say;

val debugging = ref false

fun debugmsg (msg: string) = if (!debugging) then (say msg; say "\n") else ()

(* type context:
       strs: the number of structures declared in the signature
       fcts: the number of functors declared in the signature
       tycons: the number of types declared in the signature,
       slots: the number of slots for values in the structure record
       inner: the current local environment
       total: the complete environment
       names: a stack of the names of the enclosing structures.
              There is no name for the outermost environment
	      enclosing the declaration of a top-level signature.
       s: a list of structure sharing constraints.  All sharing
          contraints for embedded signatures are moved to the
	  top-level signatures enclosing them.   When this is done,
	  symbolic constraints are normalized relative to the
	  enclosing top-level signature.
       t: a list of type sharing constraints, 
       a: list of type abbreviations

invariants: |names| = |enclosing|-1
*)

type context =
  {strs: int ref,
   tycons: int ref,
   slots: int ref,
   fcts: int ref,
   inner: env ref,
   parent: Signature,
   normTotal: Normalize.normMap ref,
   names: IP.path,
   makeStamp: unit->Stamps.stamp,
   total: env ref,
   s: {internal:SP.path list,external:Structure option} list ref,
   t: {internal:SP.path list,external:Types.tycon option} list ref,
   a: {internal:int,external:Types.tycon} list ref}

fun next (ri as ref i) = (ri := i+1; i)
val functorId = Symbol.fctSymbol "<functor>"

fun bind (sym,binding,{inner,total,normTotal,names,...} : context, err) =
    (Env.look (!inner,sym);
     err COMPLAIN ("duplicate specifications for "
		   ^Symbol.nameSpaceToString(Symbol.nameSpace sym)
		   ^" "^Symbol.name sym^" in signature"))
         nullErrorBody
     handle Env.Unbound =>
       (inner := Env.bind(sym,binding,!inner);
	total := Env.bind(sym,binding,!total);
	normTotal := Normalize.extend(sym,IP.extend(names,sym),!normTotal))

fun addExistingConstraints ({t,s,a,names,...}: context,
			    {tConstraints=t',sConstraints=s'}) =
    let val names' = ConvertPaths.invertIPath names
        val prefix = fn {internal: SP.path list,external: 'a} =>
	      {internal = map (fn spath => SP.append(names',spath)) internal,
	       external=external}
     in t := ((map prefix t') @ !t);
        s := ((map prefix s') @ !s)
    end

(* adjustBinding: adjust all the slots and relative positions
   of bindings in an included SIGNATURE.*)

fun adjustBinding (basetype,basestr,basefct,baseslot,makeStamp,redef) =
    let fun adjust baseslot binding =
	    case binding
	      of VARbind v =>
		  VARbind (case v
			     of VALvar {access=SLOT i,typ,path} =>
				 VALvar {access=SLOT (i+baseslot),
					 typ=ref(adjustType (!typ)),path=path}
			      | ERRORvar => ERRORvar
			      | _ => impossible "Sign.adjustbinding/VARbind")
 	       | CONbind (DATACON{name,const,typ,rep,sign,orig}) =>
  		  CONbind (DATACON{name=name,const=const,
                                     typ=adjustType typ,orig=orig,
				   rep=case rep
					 of VARIABLE(SLOT i) =>
					     VARIABLE(SLOT(i+baseslot))
					  | VARIABLEc(SLOT i) =>
					     VARIABLEc(SLOT(i+baseslot))
					  | _ => rep,sign=sign})
	       | TYCbind tyc => TYCbind (adjustTycon tyc)
	       | STRbind(STRvar{name,access=SLOT k,binding=STR_FORMAL{pos,spec}}) =>
		  let val spec =
		          case spec
			    of SIG {symbols,env,kind=ref EMBEDDED,...} =>
				SIG{symbols=symbols,
				    name=NONE,
				    stamp=makeStamp(),
				    env=ref (Env.map (adjust 0) (! env)),
				    kind=ref EMBEDDED}
			     | _ => spec
		      val binding =
		          STR_FORMAL{pos=pos+basestr,spec=spec}
		   in STRbind(STRvar{name=name,access=SLOT (k+baseslot),
				     binding=binding})
		  end
	       | STRbind(STRvar{binding=ERROR_STR,...}) => binding
	       | FCTbind(FCTvar{name,access=SLOT k,binding=FCT_FORMAL{pos,spec}}) =>
		  FCTbind(FCTvar{name=name,access=SLOT (k+baseslot),
				 binding=FCT_FORMAL{pos=pos+basefct,spec=spec}})
	       | FCTbind(FCTvar{binding=ERROR_FCT,...}) => binding
	       | FIXbind _ => binding
	       | _ => impossible "adustbinding"
	and relocate pos =
	    let fun reloc [] = pos+basetype
		  | reloc ((porig,pdest)::l) =
		      if pos = porig then pdest else reloc l
	     in reloc redef
	    end
	and adjustType ty =
	    case ty
	      of (CONty (tycon,tylist)) =>
		   mkCONty(adjustTycon tycon,map adjustType tylist)
	       | (POLYty{sign,tyfun=TYFUN{arity,body},abs}) =>
		   POLYty{sign=sign,
			  tyfun=TYFUN{arity=arity,body=adjustType body},
			  abs=abs}
	       | ty => ty
	and adjustTycon tycon =
	    case tycon
	      of FORMtyc{pos,name,spec as GENtyc{stamp,arity,eq,path,
						 kind=ref FORMtyck}} =>
		  FORMtyc{pos=relocate pos, name=name, spec=spec}
	       | FORMtyc{pos,name,spec=GENtyc{stamp,arity,eq,path,
					      kind = ref(DATAtyc dl)}} =>
		  let val kind =
			  ref (DATAtyc
 				(map (fn DATACON {name,const,typ,
                                                    rep,sign,orig} =>
				      (DATACON{name=name,const=const,
 					       typ=adjustType typ,orig=orig,
					       rep=rep,sign=sign})) dl))
		   in FORMtyc{pos=relocate pos, name=name,
			      spec=GENtyc{stamp=stamp,arity=arity,eq=eq,
					  path=path,kind=kind}}
		  end
	       | FORMtyc _ => impossible "ElabSig.adjustTycon 2"
	       | RELtyc {path,pos=([],offset),spec} =>
		  RELtyc {path=path,pos=([],relocate offset),spec=spec}
	       | RELtyc {path,pos=(h::t,offset),spec} =>  
		   (* drt: fix: we need to adjust this by the structure offset *)
		  RELtyc {path=path,pos=((h+basestr)::t,offset),spec=spec}
	       | tyc => tyc
     in adjust baseslot
    end


fun make_includespec (ID,err) 
                     (context as {strs,tycons,fcts,slots,total,makeStamp,...}) =
    case lookSIG (!total,ID,err)
      of SIG{symbols,kind=ref(TOP{sConstraints=s,tConstraints=t,abbreviations=a,
	 			  fctcount,strcount,typecount,slotcount}),
	     env,...} =>
	  let val accu = ref [] 
	      fun foo (name,
		       TYCbind(FORMtyc
				 {pos, name=name',
				  spec as GENtyc{kind=ref (FORMtyck |
							   FORMDEFtyc _),
						 arity,eq=ref eq ,...}})) =
		    ((case Env.look (!total,name)
			of TYCbind(FORMtyc{pos=pos2,
					   spec=GENtyc{arity=arity2,
						       eq=ref eq2,...},...}) =>
			    if arity=arity2 
			    then accu := (pos,pos2) :: !accu
			    else bind(name,TYCbind(FORMtyc{pos=pos+ !tycons,
							   spec=spec, name=name'}),
				      context,err)
			  | _ =>
			    bind(name,TYCbind(FORMtyc{pos=pos+ !tycons,spec=spec,
						      name=name'}),
				 context,err))
		      handle Env.Unbound => 
			bind(name,TYCbind(FORMtyc{pos=pos+ !tycons,spec=spec,
						  name=name'}),
			     context,err))
	        | foo _ = ()

	      val redef = (Env.app foo (!env); !accu)
	      val adjust = adjustBinding (!tycons,!strs,!fcts,!slots,makeStamp,
					  redef)
	      fun mem_assoc name [] = false
		| mem_assoc name ((name',_)::l) = 
		   (name=name') orelse (mem_assoc name l)
	   in Env.app (fn (name,binding as TYCbind(FORMtyc
				{spec=GENtyc{kind=ref (FORMtyck |FORMDEFtyc _),
					     ...},
				 ...})) => ()
	                | (name,binding) => bind(name,adjust binding,context,err))
	        (!env);
	      addExistingConstraints(context,{sConstraints=s,tConstraints=t});
	      strs := strcount + !strs;
              fcts := fctcount + !fcts;
              tycons := typecount + !tycons;
              slots := slotcount + !slots;
              rev(!symbols)
	  end
       | ERROR_SIG => nil
       | _ => impossible "make_includespec"

         
fun make_openspec (strpaths: SP.path list, err) 
                  ({total=total as ref env,
		    normTotal=normTotal as ref symenv,...}: context) =
    (app (fn p => total := openSigStructure (env,p,!total,err)) strpaths;
     app (fn p => (case lookSTR(env,p,fn _ => raise Normalize.Unbound)
		     of STRvar{binding=STR_FORMAL{spec,...},...} =>
			 normTotal := Normalize.openStr(symenv,p,spec,!normTotal)
		      | STRvar{binding=STR_OPEN{spec,...},...} =>
			 normTotal := Normalize.openStr(symenv,p,spec,!normTotal)
                      | _ => ())
	           handle Normalize.Unbound => ())
         strpaths;
     nil) (* no bindings returned *)


fun make_tyspec(eq,tyvars,name,abbrev,error,region)
	       (context as {tycons,a,total,normTotal,...}:context)
      : symbol =
    let val err = error region
(*	val _ = checkbound(no_tyvars,tyvars,err)
 *         -- useless, since uniqueness already checked *)
	val pos = next tycons
	val eq = if eq then YES else UNDEF
	val kind = case abbrev
		  of SOME def =>
		       let val (ty,tvs) =
			     elabType error region (!total,SOME(!normTotal)) def
			   val arity = length tyvars
			   val _ = 
			     (checkbound(tvs,tyvars,err);
			      TypesUtil.bindTyvars tyvars;
			      compressTy ty)
			   val rep = DEFtyc{path=IP.IPATH [name],
					    strict=[],
					    tyfun=TYFUN{arity=arity, body=ty}}
(*			   val _ = checkbound (tvs,tyvars,err)  ?? *)
		        in a := {internal = pos, external = rep} :: !a;
			   FORMDEFtyc rep
		       end
		   | NONE => FORMtyck
	val spec = GENtyc{stamp = Stamps.null,
			  path = IP.IPATH [name], arity = length tyvars,
			  eq = ref eq, kind = ref kind}
	val binding = TYCbind(FORMtyc{pos=pos,spec=spec,name=name})
     in bind(name,binding,context,err);
        name
    end

fun make_dtyspec (error,region) (context as {tycons,total,normTotal,...}: context) 
		 (db,tbs) =
    let fun predefine region (Db{tyc=id,tyvars,def}) = 
	      let val r = ref(DATAtyc nil) 
		  val spec = GENtyc{path=IP.IPATH [id],arity=length tyvars,
				    stamp=Stamps.null,
				    eq=ref DATA,kind=r}
		  val pos = next tycons
		  val tvs = elabTyvList error region tyvars
		  val binding = TYCbind (FORMtyc{pos=pos,spec=spec,name=id})
	       in bind(id,binding,context,error region); ((tvs,id,def,region),r)
	      end
	  | predefine region (MarkDb(db,region')) = predefine region' db

        (* add each constructor to the environment, checking whether env'
	   contains constructors whose names were used before in a val
	   spec or datatype spec *)
 
	fun redefine ((db as (tvs,name,parse_rhs,region),r),(names,conlists)) = 
	    let val (r',env') = 
		    elabDB (Env.empty,(!total,SOME(!normTotal)),IP.IPATH[],error) db
		val conlist = map (fn (DATACON{name,...}) => name) r'
	     in r := DATAtyc(r');
                Env.app (fn (name,binding) =>
			    (bind(name,binding,context,error region)))
			env';
		(name :: names, conlist @ conlists)
	    end
 
	val pre = map (predefine region) db

        fun tyspec _ (MarkTb(tb,region)) = tyspec region tb
          | tyspec region (Tb{tyc,def,tyvars}) =
             make_tyspec(false,elabTyvList error region tyvars,
			 tyc,SOME def,error,region) context

        val withtycs = map (tyspec region) tbs

	val (tycbinds,alldconvals) = foldr redefine ([],[]) pre

     in tycbinds @ alldconvals @ withtycs
    end


fun make_valspec(name,(ty,tv),err)
                (context as {slots,total,normTotal,...}:context) =
    let val typ = case get_tyvars tv
		   of [] => ty
		    | tvs => let val sign = TypesUtil.bindTyvars1 tvs
			      in POLYty{sign = sign, abs=0,
					tyfun = TYFUN{arity = length tvs, 
						      body = ty}}
			     end
	val _ = TypesUtil.compressTy typ
	val binding = 
	    VARbind(VALvar{path=SymPath.SPATH [name],
			   typ= ref typ,access=SLOT(next slots)})
     in bind(name,binding,context,err);
        [name]
    end


fun make_exnspec (name,err) (context as {slots,...}:context) =
    let val binding = CONbind(DATACON{name=name,const=true,typ=exnTy,sign=[],
                   	      orig=NONE,rep=VARIABLEc(SLOT(next slots))})
     in bind(name,binding,context,err);
        [name]
    end


fun make_exnspecOF(name,(body,tv),err) 
                  (context as {slots,total,normTotal,...}:context) =
    let val typ = case get_tyvars tv
		    of nil => body --> exnTy
		     | _ => (err COMPLAIN ("type variable in exception spec: " ^
					   Symbol.name name)
			       nullErrorBody;
			     WILDCARDty)
	val _ = TypesUtil.compressTy typ
	val binding = CONbind(DATACON{name=name, const=false, typ= typ,sign=[],
 			            orig=NONE,rep=VARIABLE(SLOT(next slots))})
     in bind(name,binding,context,err);
        [name]
    end


fun normalizeConstraint 
     (look : Modules.env * SP.path * ErrorMsg.complainer -> 'a,
      eq : 'a * 'a -> bool,
      deferr : ErrorMsg.complainer -> SP.path * SP.path -> unit)

     (qids : SP.path list,
      totalenv : Modules.env,
      normenv : Normalize.normMap,
      err : ErrorMsg.complainer) =

    (* scan: scan a list of qualified identifiers, dividing them into local
       and definitional constraints.  Keep only one copy of the definitional
       constraints, since they've all got to be the same anyway.*)
    let fun scan(nil,internal,definitional,_) =
	      {internal=internal,external=definitional}
          | scan (qid::r,internal,definitional,defqid) =
             (* look up the qualified identifer in the total environment
	        to make sure it exists and hasn't been hidden.*)
	     let val errflag = ref false
                 val nQid =  Normalize.apply(normenv,qid)
              in look(totalenv,nQid,(fn a => (errflag := true; err a)));
		 if !errflag then scan(r,internal,definitional,qid)
                 else scan(r,nQid::internal,definitional,defqid)
             end
	     handle Normalize.Unbound =>
              let val errflag = ref false
		  val global = look(totalenv,qid,(fn a => (errflag := true; err a)))
               in if !errflag then scan(r,internal,definitional,qid)
	          else case definitional
			 of NONE => scan(r,internal,SOME global,qid)
			  | SOME existing =>
		             (if eq(global,existing) then ()
			      else deferr err (defqid,qid);
			      scan(r,internal,definitional,defqid))
	      end
     in scan(qids,[],NONE,SP.SPATH []) 
    end

val normalizeStrConstraint =
    let fun lookSTR' arg =
	    let val STRvar{binding=b1,...} = lookSTR arg
	     in b1
	    end
     in normalizeConstraint 
	 (lookSTR',eqOrigin,
	  fn err => fn (qid1,qid2) =>
	      (err COMPLAIN 
		 ("definitional sharing constraint " ^ SP.makestring qid1 ^ 
		  " = " ^ SP.makestring qid2 ^ " can never be satisfied")
		 nullErrorBody))
    end

val normalizeTypeConstraint = 
    normalizeConstraint (lookTYC,equalTycon,
			 fn err => fn (qid1,qid2) =>
			    (err COMPLAIN 
			      ("definitional type sharing constraint " ^
			       SP.makestring qid1 ^ " = " ^ SP.makestring qid2 ^
			       " can never be satisfied")
			      nullErrorBody))


fun addStrConstraint({total,normTotal,s,...} : context,qids,err) =
    let val constraints=
	   normalizeStrConstraint (qids,!total,!normTotal,err)
     in s := constraints :: !s
    end


fun addTypeConstraint({total,normTotal,t,...} : context,qids,err) =
    let val constraints =
	   normalizeTypeConstraint (qids,!total,!normTotal,err)
     in t := (constraints :: !t)
    end


fun make_type_sharespec (patheqn,err) context =
    (addTypeConstraint(context,patheqn,err); nil)


fun make_str_sharespec (patheqn,err) context =
    (addStrConstraint(context,patheqn,err); nil)


fun elabSpec (error,region) (context as {total,normTotal,inner,...}:context)
             (spec,names) =
    case spec
      of StrSpec lspec =>
	   foldl 
	     (fn ((name,sign),lnames) => 
		make_strspec (name,sign,error,region) context @ lnames)
	     names lspec
       | FctSpec lspec =>
	   foldl
	     (fn ((name,fsig),lnames) => 
		make_fctspec (error,region) context (name,fsig) @ lnames)
	     names lspec
       | TycSpec (lspec,eqprop) =>
	   foldl 
	     (fn ((name,tvs,abbrev),lnames) => 
		make_tyspec (eqprop,elabTyvList error region tvs,
			     name,abbrev,error,region)
		            context
		:: lnames)
	     names lspec
       | ValSpec lspec =>
	   foldl
	     (fn ((name,ty),lnames) => 
		let val ety = elabType error region (!total,SOME(!normTotal)) ty
		in make_valspec (name,ety,error region) context @ lnames end)
	     names lspec
       | DataSpec{datatycs=dbs,withtycs=tbs} =>
	    rev (make_dtyspec (error,region) context (dbs,tbs)) @ names
       | ExceSpec (lspec) =>
	   foldl 
	     (fn ((name,topt),lnames) => 
		case topt
		of SOME ty =>
		     let val ety = elabType error region 
			             (!total,SOME(!normTotal)) ty
		      in make_exnspecOF (name,ety,error region) context @ lnames
		     end
		 | NONE => make_exnspec (name,error region) context @ lnames)
	     names lspec
       | MarkSpec (s,region') => elabSpec (error,region') context (s,names)
       | FixSpec {ops,fixity} => (
	   error region WARN "Fixity specification in signatures are not supported"
		 nullErrorBody;
	   app (fn id => inner := Env.bind(id,FIXbind(FIXvar{name=id,
				    binding=fixity}),!inner)) ops;
	   names)
       | ShareSpec pl =>
	   make_str_sharespec (map SP.SPATH pl,error region) context @ names
       | ShatycSpec pl =>
	   make_type_sharespec (map SP.SPATH pl,error region) context @ names
       | IncludeSpec s => make_includespec (s,error region) context @ names
       | OpenSpec pl => make_openspec (map SP.SPATH pl, error region) context
	                @ names 
       | LocalSpec(sp1,sp2) => 
	  (error region WARN "LOCAL specs are only partially implemented"
		 nullErrorBody;
	   elabSpecList (error,region) context sp1;
	   (elabSpecListRev (error,region) context sp2) @ names)

and elabSpecListRev (error,region) context specs =
      foldl (elabSpec (error,region) context) [] specs

and elabSpecList ctxt1 ctxt2 specs =
      rev (elabSpecListRev ctxt1 ctxt2 specs) 

and elabEMBEDsig (error,region) 
		 (context as {strs,fcts,tycons,slots,makeStamp,inner,
			      parent,total,normTotal,names,t,s,a},
		  name)
		 dsc =
    case dsc
      of VarSig name' => 
	   let val err = error region
	       val sgn = lookSIG(!total,name',err)
	       val strbind =
		 STRbind (STRvar{name=name, 
				 access=if Extern.hidden name' then NO_ACCESS
					else SLOT(next slots),
				 binding=STR_FORMAL{pos=next strs, spec=sgn}})
	    in bind(name,strbind,context,err);
	       sgn
	   end
       | SigSig specs =>
	   let val err = error region
	       val inner' = ref Env.empty
	       val symbols = ref ([]:Symbol.symbol list)
	       val sgn = SIG {symbols=symbols, env=inner', kind=ref EMBEDDED,
			      stamp=makeStamp(), name=NONE}
	       val binding = 
		 STRbind (STRvar{name=name,
				 access=if Extern.hidden name then NO_ACCESS
					else SLOT (next slots),
				 binding=STR_FORMAL{pos=next strs,spec=sgn}})
	       val _ = bind(name,binding,context,err)
	       val context' = ({strs=strs, fcts=fcts, tycons=tycons,
				slots=ref 0, parent=parent, makeStamp=makeStamp,
				inner=inner', total=ref (!total),
				normTotal=ref (!normTotal),
				names=IP.extend(names,name),
				t=t,s=s,a=a} : context)
	       val symbollist = elabSpecList (error,region) context' specs
	    in symbols := symbollist;
	       sgn
	   end
       | MarkSig(sign,region') =>
	   elabEMBEDsig (error,region') (context,name) sign

and elabTOPsig (error,region) (env,makeStamp,name) dsc =
    case dsc
      of VarSig name => lookSIG(env,name,error region)
       | SigSig specs =>
	   let val _ = debugmsg ">> elabTOPsig SigSig\n"
	       val err = error region
	       val inner = ref Env.empty
	       val symbols = ref []
	       val kind = ref IRRELEVANT
	       val sgn = SIG {symbols = symbols, kind=kind, env = inner, name=name,
			      stamp=makeStamp()}
	       val strs = ref 0 and fcts = ref 0 and tycons = ref 0 and 
		   slots = ref 0 and s = ref [] and t = ref [] and a = ref []
	       val context = ({strs=strs, fcts=fcts, tycons=tycons, slots=slots,
			       parent = sgn, makeStamp = makeStamp, inner = inner,
			       total = ref env, normTotal = ref Normalize.empty,
			       names = IP.IPATH[], t = t, s = s, a = a} : context)
	       val symbollist = elabSpecList (error,region) context specs
	       val _ = debugmsg ">> elabTOPsig after elabSpecList\n"
	    in symbols := symbollist;
	       kind := TOP{strcount = !strs, fctcount = !fcts, typecount = !tycons,
			   slotcount = !slots, sConstraints = !s, tConstraints = !t,
			   abbreviations = !a};
	       if !Control.instSigs 
	       then (debugmsg ">> elabTOPsig instantiate\n";
		     Instantiate.instantiate
		       ((IP.IPATH [],Stamps.newBoundScope(),err),sgn);
		     ())
	       else ();
	       debugmsg "<< elabTOPsig SigSig\n";
	       sgn
	   end
       | MarkSig(sign,region') =>
	   elabTOPsig (error,region') (env,makeStamp,name) sign

and elabPARTIALsig (error,region) 
		   (ctx as (nameArg,sgnArg,total,normTotalArg,makeStamp)) dsc =
    case dsc
      of VarSig name => lookSIG(total,name,error region)
       | SigSig specs =>
	   let val symbols = ref ([]:Symbol.symbol list)
	       val bindArg = 
		 STRbind (STRvar{name=nameArg,access=SLOT 0,
				 binding=STR_FORMAL{pos=0, spec=sgnArg}})
	       val totalArg = Env.bind(nameArg,bindArg,total)
	       val inner = ref (Env.bind(nameArg,bindArg,Env.empty))
	       val kind = ref IRRELEVANT
	       val sgn = SIG {symbols = symbols, env = inner, name = NONE,
			      stamp = makeStamp (), kind = kind}
	       val context =
		     {strs = ref 1, fcts = ref 0,tycons = ref 0, slots = ref 0,
		      inner = inner, parent = sgn, total = ref totalArg,
		      normTotal = normTotalArg, a = ref [],
		      names = IP.IPATH [], makeStamp = makeStamp,
		      s = ref [], t = ref []}
	       val symbollist = elabSpecList (error,region) context specs
	    in symbols := nameArg :: symbollist;
	       kind := TOP{strcount= !(#strs context), fctcount= !(#fcts context),
			   typecount= !(#tycons context),
			   slotcount= !(#slots context),
			   sConstraints= !(#s context),
			   tConstraints= !(#t context),
			   abbreviations = !(#a context)};
	       sgn
	   end
       | MarkSig(sign,region') =>
	   elabPARTIALsig (error,region') ctx sign

and make_strspec (name,sign,error,region) (context as {inner,...} : context) =
    let val sgn = elabEMBEDsig (error,region) (context, name) sign
        val bind = Env.look(!inner,name)
     in [name]
    end

and make_fsigexp (error,region) name
                 (context as {total,normTotal,makeStamp,parent,...} : context)
                 (FsigFsig{param=[param],def}) =
      let val (nameParam,specParam) =
	      case param of (SOME n,s) => (n,s) | (NONE,s) => (parameterId,s)
	  val normTotalParent = Normalize.prefix(parentId,!normTotal)
	  val sgnArg = 
	      elabPARTIALsig (error,region)
			     (parentId,parent,!total,ref normTotalParent,makeStamp)
			     (SigSig [StrSpec[(parameterId,specParam)]])
	  val normTotalArg =
	      case param
		of (NONE,_) =>
		     Normalize.openSubStr(parameterId,sgnArg,normTotalParent)
		 | (SOME _ ,_) => 
		     Normalize.extend(nameParam,IP.IPATH[parameterId],
				      normTotalParent)
	  val sgnBody =
	      elabPARTIALsig 
		(error,region)
		(argumentId,sgnArg,!total,
		 ref(Normalize.prefix(argumentId,normTotalArg)),makeStamp)
		def
	  val fsig =
	      FSIG{name = name,paramName = nameParam, argument = sgnArg,
		   body = sgnBody}
       in fsig
      end
  | make_fsigexp (error,region) name ctxt (FsigFsig{param = a1 :: larg,def}) =
      make_fsigexp (error,region) name ctxt
                  (FsigFsig{param=[a1],
                            def=SigSig[FctSpec[
                                 (functorId,FsigFsig{param=larg,def=def})]]})
  | make_fsigexp err name ctxt (FsigFsig{param = [],def}) =
      impossible "make_fsig"
  | make_fsigexp (error,region) name ({total,...}: context) (VarFsig name') =
      lookFSIG(!total,name',error region)
  | make_fsigexp (error,region) name context (MarkFsig(fsig,region')) =
      make_fsigexp (error,region') name context fsig

and make_fctspec (error,region) (context as {fcts,slots,...} : context)
		 (name,fsig) =
    let val fsgn = make_fsigexp (error,region) NONE context fsig
	val binding = 
            FCTbind (FCTvar{name=name,access=SLOT (next slots),
			    binding=FCT_FORMAL{pos=next fcts,spec=fsgn}})
     in bind(name,binding,context,error region);
        [name]
    end

and elabFSIGB (error,region)  (arg as (env:env,scope)) (name,fsig) =
    let val context = 
	  {strs=ref 0, fcts=ref 0, tycons=ref 0, slots=ref 0,  total = ref env,
	   makeStamp = Stamps.newStamp scope, inner = ref Env.empty,
	   parent = ERROR_SIG, a = ref [],
	   normTotal = ref Normalize.empty, names = IP.IPATH [],
	   t = ref [], s = ref []}
	val binding = make_fsigexp (error,region) (SOME name) context fsig
	val fsigvar = FSIGvar{name=name,binding=binding}
     in ([fsigvar],Env.bind(name,FSIGbind fsigvar,Env.empty):env)
    end


end (* structure ElabSig *)
