(* instrumast.sml
 *
 * COPYRIGHT (c) 1996 Bell Laboratories.
 *)

(* DebugInstrum

   Instrument user code.
   Instrumented code needs to gain access to a variety of structures at 
     run-time, some of them in a dirty fashion.  This is done via
     a special system ref set up in DebugInterface.

*)

signature DEBUGINSTRUM' = sig
   val instrumDec:  {ast:Ast.dec,
		    firstPlace:DebugStatic.place,  (* = firstEvn *)
		    lastBindTime:DebugStatic.time} ->
         {ast: Ast.dec,
	  events:DebugStatic.event Vector.vector,
	  evns:DebugStatic.place Vector.vector Vector.vector}
   val instrumLevel : int ref
end

structure DebugInstrum : DEBUGINSTRUM =
struct

open Vector Array List DebugUtil DebugStatic Ast ErrorMsg
structure P = Access.P (* to avoid confusing SourceGroup *)
infix 9 sub
   
val instrumLevel = ref 3 
(* Controls instrumentation method.  Possible values:
   2 - simplistic:
         - no sequentialization analysis; 
	 - event at top of every function
   3 - a limited sequentialization approach
   4 - and its variant.
*)


fun makevar s = 

local
val count = ref 0
in
fun makenvar str =
     [Symbol.varSymbol(str ^ makestring(!count))]
     before inc count
fun makenstrvar str =
       [Symbol.strSymbol (str ^ makestring(!count))]
       before inc count	      
end

(* Generate a "use" for an object with the given lvar, which need not be
   a simple object. *)
fun fakeuse name = VarExp[name]

val xdebug = Symbol.strSymbol "XDebug"
val assop = [xdebug,Symbol.varSymbol ":="]
val subop = [xdebug,Symbol.varSymbol "subscript"]
val derefop = [xdebug,Symbol.varSymbol "!"]
val addop = [xdebug,Symbol.varSymbol "+"]
val updateop = [xdebug,"unboxedupdate"]
val ieqlop = [xdebug,Symbol.varSymbol "ieql"]
val specialop = [xdebug,Symbol.varSymbol "mkspecial"]
val consDcon = [xdebug,Symbol.varSymbol "::"]

fun App(f,a) = AppExp{function=f,argument=a}

fun instrumDec {ast:dec,firstPlace:int,lastBindTime:time} =

let 
val nextPlace = ref firstPlace  (* next event ev to use *) 
val eventList = ref ([]: event list)  (* events generated *)
val nextEvn = ref firstPlace (* next evn to use *)
val evnList = ref ([]: place vector list) (* evns generated *)

fun makeEvent event : place =
  (eventList := event::(!eventList);
   !nextPlace before
   inc nextPlace)

fun makeEvn places : evn =
  (evnList := fromList places::(!evnList);
   !nextEvn before
   inc nextEvn)

val timesvar =  makevar "_debugtimes"
val eventtimes = makevar "_eventtimes"
val breakentry = makevar "_breakentry"
val arrayvar = makevar "_array"

fun maketimearr len =
  let val timearr = makenvar "_timearr"
  in (ValDec[Vb{pat=VarPat timearr,
		exp=App(VarExp arrayvar, TupleExp [IntExp len, IntExp 0])}],
      VarExp timearr)
  end

val GETTIMEexp = App(VarExp subop, TupleExp[VarExp timesvar, IntExp 0])

fun INCexp exp = App(VarExp addop, TupleExp[exp, IntExp 1])

fun makent label = 
  let val nt = makenvar label
  in (ValDec[Vb{pat=VarPat nt, exp=INCexp GETTIMEexp}],
      VarExp nt)
  end

fun SETTIMEexp ntimeexp = App(VarExp updateop,
				 TupleExp[VarExp timesvar, IntExp 0, ntimeexp])

fun NOTETIMEexp(ntimeexp,evn) = 
    App(VarExp updateop, 
	   TupleExp[VarExp eventtimes,IntExp evn-firstPlace, ntimeexp])

fun BREAKexp(ntimeexp, evn, args) =
      IfExp{test=App(VarExp ieqlop,
			TupleExp[ntimeexp,
				 App(VarExp subop,
					TupleExp[VarExp timesvar, IntExp 1])]),
	    thenCase=App(VarExp breakentry,TupleExp(IntExp evn :: args)),
	    elseCase=unitExp}

fun EVENTexp (evn,lbt,args) = 
     let val (ntimedec,ntimeexp) = makent "_newtime"
     in LetExp{dec=ntimedec,
	       exp=SeqExp [SETTIMEexp ntimeexp,
			   BREAKexp(ntimeexp, evn, lbt::args),
			   NOTETIMEexp(ntimeexp,evn),
			   ntimeexp]}
     end

fun FEVENTexp (evn,lbt,args) =
     let val (ntimedec,ntimeexp) = makent "_newtime"
     in LetExp{dec=ntimedec,
	       exp=SeqExp[SETTIMEexp ntimeexp,
			  App(VarExp breakentry,
				 TupleExp(IntExp evn::lbt::args)),
			  NOTETIMEexp(ntimeexp,evn),
			  ntimeexp]}
     end

(* Because ref is a constructor that cannot be replaced at top level, 
    we replace it explicitly here: *)

val hcreater = makevar "_hcreater"
val HREFexp = VarExp hcreater

(* For efficiency, we implement

fun hass (objr,value) =
        (updatedRList := (weak objr) :: (!updatedList);
	 objr := value)
	
in-line. Note we maintain type information in opr used by translate to
determine whether to use boxed or unboxed update. *)

val weakvar = makevar "_weak"
val udrl = makevar "_udrl"
fun HASSIGNexp opr = 
     let val objvar = makenvar "_obj"
	 val valvar = makenvar "_val"
	 val objexp = VarExp objvar
         val valexp = VarExp valvar
      (* val newobj = App(VarExp weakvar,objexp)  *)
         val newobj = App(VarExp specialop,
			   TupleExp[IntExp System.Tags.special_weak, objexp]) 
	 val oldlist = App(VarExp derefop,VarExp udrl)
     in FnExp[Rule{pat=TuplePat[VARpat objvar,VARpat valvar],
		   exp=SeqExp[App(VarExp opr, 
				 TupleExp[objexp,valexp]),
			  App(VarExp assop,
				 TupleExp[VarExp udrl,
					  App(VarExp consDcon,
						 TupleExp[newobj,oldlist])])]},
	       Rule{pat=WildPat, exp=IntExp 0}]
     end

val pconsvar = makevar "_pcons"
val udal = makevar "_udal"
fun HUPDATEexp opr =
     let val objvar = makenvar "_obj"
	 val offvar = makenvar "_off"
	 val valvar = makenvar "_val"
	 val objexp = VarExp objvar
	 val offexp = VarExp offvar
         val valexp = VarExp valvar
      (* val newobj = App(VarExp weakvar,objexp)  *)
         val newobj = App(VarExp specialop, 
			   TupleExp[IntExp System.Tags.special_weak, objexp]) 
	 val oldlist = App(VarExp derefop,VarExp udal)
     in FnExp[Rule{pat=TuplePat[VARpat objvar,VARpat offvar, VARpat valvar],
	           exp=SeqExp[App(VarExp opr, 
				     TupleExp[objexp,offexp,valexp]),
			      App(VarExp assop,
				     TupleExp[VarExp udal,
					      App(VarExp pconsvar,
						     TupleExp[newobj,offexp,oldlist])])]},
	       Rule{pat=WildPat, exp=IntExp 0}]
     end


fun simplebind ((btexp,bsites,bvars,bhasfn),ev,vars) =
       let val evn = makeEvn (ev::bsites)
	   val bevvar = makevar("_bind" ^ makestring evn)
	   val evndec = ValDec[Vb{pat=VARpat bevvar,
				  exp=EVENTexp(evn,btexp,vars@bvars)}]
       in ((VarExp bevvar,nil,nil,false),evndec)
       end

(* Variable naming conventions:
     btexp:exp  represents time of last binding event
     bsites:place list  places of accumulated events to be discharged
     bvars:exp list     associated accumulated variables 
     bhasfn:bool       bsites includes a FNev or HANDLEev *)

fun instrexp2(b as (btexp:exp,bsites:place list,bvars:exp list,bhasfn:bool),
	      exp:exp) : 
     exp (* instrumented expression *) =
  let 
    fun instr (RecordExp l) = RecordExp (map (fn(lab,exp)=>(lab,instr exp)) l)
      | instr (VectorExp l) = VectorExp(map instr l)
      | instr (SeqExp expl) = SeqExp(map instr expl)
(*      | instr (VarExp(assopr as VALvar{access=INLINE(P.ASSIGN),
                                           ...}),_)) =
		HASSIGNexp assopr
      | instr (VarExp(ref(updopr as VALvar{access=INLINE(P.UPDATE),
                                           ...}),_)) =
		HUPDATEexp updopr
      | instr (VarExp(ref(updopr as VALvar{access=INLINE(P.INLUPDATE),
                                           ...}),_)) =
		HUPDATEexp updopr
*)
(* ZZZZZZZ *)
      | instr (exp as AppExp{function=opr,argument=arg}) =
	  let fun strip (MarkExp(exp,_,_)) = strip exp
		| strip (ConstraintExp{expr,constraint}) = strip expr
		| strip (SeqExp[exp]) = strip exp
		| strip exp = exp
	      fun normal () =
		 let val opr' = instr opr
		     val arg' = instr arg
		     val evn = makeEvn(makeEvent(APPev exp)::bsites)
		     val oprtmp = makenvar "_oprvar"
		     val argtmp = makenvar "_argvar"
		 in LetExp{dec=ValDec[Vb{pat = VarPat oprtmp, exp = opr'}, 
				      Vb{pat = VarPat argtmp, exp = arg'}],
			   exp=SeqExp [EVENTexp(evn,btexp,bvars), 
				       App(VarExp oprtmp, VarExp argtmp)]}
		 end
	  in case  strip opr
	      of VarExp(VALvar{access=INLINE(P.ASSIGN),...}),_) => normal()
	     | VarExp(ref(VALvar{access=INLINE(P.UPDATE),...}),_) => normal()
	     | VarExp(ref(VALvar{access=INLINE(P.INLUPDATE),
                                 ...}),_) => normal()
	     | (opr as VarExp(ref(VALvar{access=INLINE prim,...}),_)) =>
		 let val arg' = instr arg
		 in App(opr,arg')
                 end
	     | CONexp(DATACON{rep=REF,...},_) => normal()
	     | CONexp _ => App(opr, instr arg)
             | FNexp(body,t) =>  (* really a CASE or equivalent *)
		 let val arg' = instr arg
		     val body' = 
			 instrrules (b,fn r => makeEvent(CASEev(arg,r))) body
		 in App(FNexp(body',t),arg')
		 end
	     | _ => normal()
	  end
      | instr (ConstraintExp {expr,constraint}) = 
	     ConstraintExp{expr=instr expr, constraint=constraint}
      | instr (exp as MarkExp(RaiseExp arg,_,_)) =
		(* N.B. Must be marked *)
	  let val arg' = instr arg
	      val evn = makeEvn(makeEvent(RAISEev exp)::bsites)
	      val argtmp = makenvar "_argvar"
	  in LetExp{dec=ValDec [Vb{pat = VarPat argtmp, exp = arg'}],
		    exp=SeqExp [EVENTexp(evn,btexp,bvars),
				RaiseExp(VarExp argtmp)]}
	  end
      | instr (exp as MarkExp(LetExp (ldec,lexp),_,_)) = 
		(* note: must be marked *)
          let val ((btexp',bsites',bvars',bhasfn'),ldec') = instrdec (b,ldec)
	      val evn = makeEvn(makeEvent(LETev exp)::bsites')
	      val bevvar = makevar("_bind" ^ makestring evn)
	      val lexp' = instrexp2((VarExp bevvar,
                                     nil,nil,false),lexp)
	  in LETexp(SEQdec[ldec',
			   VALdec[VB{pat=VARpat bevvar,
				     exp=EVENTexp(evn,btexp',bvars'),
				     tyvars=ref nil}]],
		    lexp')
          end
      | instr (CASEexp(exp,rl)) = 
	  let val exp' = instr exp
	      val rl' =  instrrules (b,fn r => makeEvent(CASEev(exp,r))) rl
	  in CASEexp(exp',rl')
	  end
      | instr (HANDLEexp (e, HANDLER(FNexp(body,t)))) =
     	    let val e' = instr e
     	        val body' = instrrules(b,makeEvent o HANDLEev) body
     	    in HANDLEexp(e',HANDLER(FNexp(body',t)))
            end
      | instr (FNexp(body,t)) = 
	    let val body' = instrrules (b,makeEvent o FNev) body
	    in FNexp(body',t)
	    end
      | instr (MarkExp (exp,s,e)) =
	    let val exp' = instr exp
	    in MarkExp(exp',s,e)
	    end
      | instr exp = exp
    and instrrules (b as (btexp,bsites,bvars,bhasfn),evf) = 
      let 
        fun f (rule as RULE(pat,exp as MarkExp(_))) = 
	     let val vars = (patvars (fn v => VarExp v) pat)
		 val bsites' = (evf rule)::bsites
		 val bvars' = vars@bvars
		 val evn = makeEvn bsites'
		 val bevvar = makevar("_bind" ^ makestring evn)
		 val exp' = instrexp2((VarExp bevvar,
                                       nil,nil,false),exp)
	     in RULE(pat, LETexp(VALdec[VB{pat=VARpat bevvar,
					   exp=EVENTexp(evn,btexp,bvars'),
					   tyvars=ref nil}],
				 exp'))
	     end
	  | f (RULE(pat,CONSTRAIntExp(exp,_))) =  f (RULE(pat,exp))
          | f (RULE(pat,exp)) =
	     let val exp' = instrexp2(b,exp)
	     in RULE(pat,exp')
             end
      in map f 
      end
  in 
    instr exp
  end

(* The idea of levels 3/4 is as follows: for every instrumented expression
   we return the flag (d:bool), which is set iff the expr discharges
   (any and all) events on the bsites list (even if that list is empty).
   If d is true, the instrumented expression returns a pair (lbt,value);
   otherwise it just returns the value.
   It is important that every event containing a FNev is executed immediately
   after the event containing the matching APPev.  This implies that each
   event must contain at most one FNev, and that each FNev is contained
   in at most one event in any possible path through the instrumented code.
   We arrange the stronger condition that, if there is a FNev to be 
   discharged, every expression either discharges it or executes no 
   events at all; this means that forcing FNevs at the top of rules 
   should never be necessary.
   We do this by forcing events before LETs, HANDLEs and CASEs where necessary.
   (Note that this means a FNev can never be pushed inside a CASE or LET,
   lest we need to discharge the FNev again in a subsequent event.  This
   is unnecessaary if there are no subsequent events, but our present
   analysis is too stupid to notice this.)
   We also force discharge of FNevs before FNs, to prevent the two FNs in
   one list problem.
   The remaining issue is testing for backstop events in LET bodies and
   FN/CASE rule bodies.  Clearly, if the discharge flag is set, no backstop is
   needed; but this is not a sufficient test because
   LETs, CASEs, and HANDLEs all execute events (so no backstop is necessary),
   but don't in general discharge. (Actually, it is sufficient for FN
   rule bodies, where there will always be a FNev to discharge.)
   Tentative Soln #1: Strengthen the forced discharge before LET,CASE, and 
   HANDLE to insert an event if there are any events *at all* in the
   bsites list.  If we're in the body of a  LET (of any conceivable interest) 
   or a CASE (or of course a FN) there will be such an event.
   Then backstop needed can be equated with d false. THIS is level #3
   Tentative Soln #2: Just allow the extra backstop events. This may be
   a winner on the whole. This is level #4.
*)
and instrexp3(b as (btexp:exp,bsites:place list,bvars:exp list,bhasfn:bool),
	      exp:exp) : bool * exp =
(* bool: d = true iff instrumented expression discharges (any and all) 
             events (even if none to discharge 
   exp: exp' = instrumented expression. Consists of a tuple (lbt,value)
                iff d = true, else just a value.
*)
  let
    fun discharge (allsites:bool,
		   f:(exp * place list * exp list * bool) -> exp) =
	(* Discharge any function events (any events at all if allsites true)
	   before executing expression constructed by f. *)
	  let val (btopt,b as (btexp,bsites,bvars,bhasfn)) =
	         if (allsites andalso (!instrumLevel = 3)
		     andalso (not (null bsites))) 
		   orelse bhasfn then
		   let val btvar = makenvar "_btvar"
		       val evn = makeEvn bsites
		       val btdec = 
			   VALdec[VB{pat=VARpat btvar,
				     exp=EVENTexp(evn,btexp,bvars),
				     tyvars=ref nil}]
		   in (SOME btdec,(VarExp btvar,nil,nil,false))
		   end
		 else (NONE,b)
	      val exp' = f b
          in case btopt of
	       SOME btdec =>
		 (true,LETexp(btdec,TupleExp[btexp,exp']))   (* !! *)
	     | NONE => (false,exp')
          end
    fun instr (RECORDexp l) = 
          let fun f ((lab as LABEL{name,...},exp)::rest,accv,accd) = 
 	          let val fieldtmp =
		              makevar("_field:" ^ Symbol.name name)
		  in f (rest,
		        (lab,VarExp fieldtmp) :: accv,
			(fieldtmp,exp) :: accd)
		  end
	        | f (nil,accv,accd) = (rev accv, rev accd)
              val (lv,ld) = f (l,nil,nil)
		  (* lv is list of (label,varexp) pairs
		     ld is list of (var,exp) pairs *)
	      fun g (b as (btexp,bsites,bvars,bhasfn),(fieldtmp,exp)::rest) = 
		    let val (d,exp') = instrexp3(b,exp)
		    in if d then
			 let val btvar = makenvar "_btvar"
		             val dec' = VALdec[VB{pat=TuplePat[VARpat btvar,
							       VARpat fieldtmp],
						  exp=exp',
						  tyvars=ref nil}]
			     val (_,rest') = 
				 g((VarExp btvar,nil,nil,false),rest)
			 in (SOME(VarExp btvar),dec'::rest')
                         end
		       else
			 let val dec' = VALdec[VB{pat=VARpat fieldtmp,
						  exp=exp',
						  tyvars=ref nil}]
			     val (btopt,rest') = g(b,rest)
			 in (btopt,dec'::rest')
			 end
		    end
                | g (_,nil) = (NONE,nil)
	      val (btopt,ld') = g(b,ld)
	      (* avoid decl's when possible, for cosmetic reasons *)
	      fun h ((lab,_)::rl,(VALdec[VB{exp,...}])::re) = 
		         (lab,exp)::(h(rl,re))
		| h (nil,nil) = nil
		| h _ = impossible "DebugInstrum.instrexp RECexp"
	  in case btopt of
	       SOME (btexp') => (true,
				 LETexp(SEQdec ld',
					TupleExp[btexp',
						 RECORDexp lv]))  (* !! *)
	     | NONE => (false, RECORDexp(h(lv,ld')))
	  end 
      | instr (VECTORexp(expl,_)) = 
	 (* NEED SOMETHING HERE *)
	 debugPanic "Can't use instrumLevel 3/4 with VECTORexp"
      | instr (SeqExp expl) =
	  let 
	    fun g(usebt,b as (btexp,bsites,bvars,bhasfn),exp::rest) =
		  let val (d,exp') = instrexp3(b,exp)
		  in if d then
		       let val btvar = makenvar "_btvar"
			   val valvar = makenvar "_valvar"
			   val (_,rest') = 
			       g(false,(VarExp btvar,
                                        nil,nil,false),rest)
			   val useexp = 
			      if usebt then
				TupleExp[VarExp btvar,
					 SeqExp(VarExp valvar::
						rest')]
			      else SeqExp(VarExp valvar::rest')
		       in (true,
			   [LETexp(VALdec[VB{pat=TuplePat[VARpat btvar,
							  VARpat valvar],
					    exp=exp',
					    tyvars=ref nil}],
				   useexp)])
		       end
		     else 
		       let val (d',rest') = g(usebt,b,rest)
		       in (d',exp'::rest')
		       end
		  end
	      | g(_,_,nil) = (false,nil)
	    val (d,expl') = g(true,b,expl)
          in (d,SeqExp expl')                          (* !! *)
          end
      | instr (CONexp(DATACON{rep=REF,...},_)) = (false,HREFexp)
      | instr (VarExp(assopr as VALvar{access=INLINE(P.ASSIGN),
                                           ...}),_)) =
	        (false,HASSIGNexp assopr)
      | instr (VarExp(ref(updopr as VALvar{access=INLINE(P.UPDATE),
                                           ...}),_)) =
		(false,HUPDATEexp updopr)
      | instr (VarExp(ref(updopr as VALvar{access=INLINE(P.INLUPDATE),
                                           ...}),_)) =
		(false,HUPDATEexp updopr)
      | instr (exp as App(opr,arg)) =
	  let fun strip (MarkExp(exp,_,_)) = strip exp
		| strip (CONSTRAIntExp(exp,_)) = strip exp
		| strip (SeqExp[exp]) = strip exp
		| strip exp = exp
	      fun normal () =
		 let val oprtmp = makenvar "_oprvar"
		     val argtmp = makenvar "_argvar"
		     val btvar = makenvar "_btvar"
		     val (dopr,opr') = instr opr
		     val oprpat =
			if dopr then
			  TuplePat[VARpat btvar,VARpat oprtmp]
			else VARpat oprtmp
		     val (darg,arg') =
		       if dopr then
			 instrexp3((VarExp(ref btvar,nil,nil,false),arg)
		       else instr arg
		     val argpat = 
		       if darg then
                         if dopr then
			   TuplePat[WildPat, VARpat argtmp]
			 else TuplePat[VARpat btvar,VARpat argtmp]
		       else VARpat argtmp
		     val appexp = App(VarExp oprtmp,
					 VarExp argtmp)
		     val fullexp =
		        if dopr orelse darg then 
			  let val evn = makeEvn[makeEvent(APPev exp)]
			  in TupleExp[VarExp btvar,
				      SeqExp[EVENTexp(evn,VarExp btvar
                                                                 ,nil),
					     appexp]]
			  end
			else 
			  let val evn = 
			        makeEvn(makeEvent(APPev exp)::bsites)
			  in TupleExp[EVENTexp(evn,btexp,bvars),
				      appexp]
			  end
		 in (true,
		     LETexp(SEQdec[VALdec[VB{pat = oprpat,
					     exp = opr', 
					     tyvars = ref nil}],
				   VALdec[VB{pat = argpat,
					     exp = arg', 
					     tyvars = ref nil}]],
			    fullexp))
		 end
 	  in case (strip opr) of
	       VarExp(VALvar{access=INLINE(P.ASSIGN),...}),_) => normal()
	     | VarExp(ref(VALvar{access=INLINE(P.UPDATE),...}),_) => normal()
	     | VarExp(ref(VALvar{access=INLINE(P.INLUPDATE),
                                 ...}),_) => normal()
	     | (opr as VarExp(ref(VALvar{access=INLINE prim,...}),_)) =>
		 let val (darg,arg') = instr arg
		     val argtmp = makenvar "_argvar"
		 in if darg then
		      let val btvar = makenvar "_btvar"
		      in (true,LETexp(VALdec[VB{pat=TuplePat[VARpat btvar,
							     VARpat argtmp],
						exp=arg',
						tyvars=ref nil}],
				      TupleExp[VarExp(ref btvar,
					       App(opr,
						  VarExp argtmp)]))
		      end
		    else if bhasfn andalso (Prim.mayRaise prim) then
		      let val evn = makeEvn(makeEvent(APPev exp)::bsites)
		      in (true,LETexp(VALdec[VB{pat=VARpat argtmp,
						exp=arg',
						tyvars=ref nil}],
				      TupleExp[EVENTexp(evn,btexp,bvars),
					       App(opr,
						  VarExp argtmp)]))
		      end
  		    else (false,App(opr,arg'))
		 end
	     | CONexp(DATACON{rep=REF,...},_) => normal()
	     | CONexp _ => 
		 let val (darg,arg') = instr arg
		 in if darg then
		      let val btvar = makenvar "_btvar"
			  val argtmp = makenvar "_argtmp"
		      in (true,LETexp(VALdec[VB{pat=TuplePat[VARpat btvar,
							     VARpat argtmp],
						exp=arg',
						tyvars=ref nil}],
				      TupleExp[VarExp btvar,
					       App(opr,
						  VarExp argtmp)]))
		      end
		    else (false,App(opr,arg'))
		 end
             | FNexp(body,t) =>  (* really a CASE or equivalent *)
		 let val (darg,arg') = instr arg
		 in if darg then
		      let val valvar = makenvar "_valtmp"
			  val btvar = makenvar "_btvar"
			  val body' = 
			      instrrules((VarExp btvar,
                                          nil,nil,false),
					 fn r => makeEvent(CASEev(arg,r)),
					 false) body
		      in (true,
			  LETexp(VALdec[VB{pat=TuplePat[VARpat btvar,
							VARpat valvar],
					   exp=arg',
					   tyvars=ref nil}],
				 TupleExp[VarExp btvar,
					  App(FNexp(body',t),
						 VarExp valvar)]))
		      end
		    else 
		      discharge (true,fn b =>
		        let val body' = 
			    instrrules(b,
				       fn r => makeEvent(CASEev(arg,r)),
				       false) body
			in App(FNexp(body',t),arg')
			end)
		 end
	     | _ => normal()
	  end
      | instr (CONSTRAIntExp (e,c)) =
	  let val (d,e') = instr e
          in (d,CONSTRAIntExp(e',c))
	  end
      | instr (exp as MarkExp(RAISEexp (arg,t),_,_)) = 
		(* N.B. Must be marked *)
	  let val argtmp = makenvar "_argvar"
	      val (darg,arg') = instr arg
	  in if darg then
	       let val evn = makeEvn[makeEvent(RAISEev exp)]
		   val btvar = makenvar "_btvar"
	       in (true,
		   LETexp(VALdec[VB{pat=TuplePat[VARpat btvar,
						 VARpat argtmp],
				    exp=arg',
				    tyvars=ref nil}],
			  TupleExp[VarExp btvar,
			           SeqExp[EVENTexp(evn,VarExp btvar,nil),
					  RAISEexp(VarExp argtmp,t)]]))
               end
             else 
	       let val evn = makeEvn(makeEvent(RAISEev exp)::bsites)
               in (true,
		   LETexp(VALdec[VB{pat=VARpat argtmp,
			 	    exp=arg',
				    tyvars=ref nil}],
			  TupleExp[EVENTexp(evn,btexp,bvars),
				   RAISEexp(VarExp argtmp,t)]))
	       end
	   end
      | instr (exp as MarkExp(LETexp (ldec,lexp),_,_)) = 
		(* note: must be marked *)
	  (* We discharge any events before executing LET. *)
	  discharge (true,fn (b as (btexp,bsites,bvars,bhasfn)) =>
	      (* assume instrdec returns a btexp' valid in context of ldec' *)
	    let val ((btexp',bsites',bvars',bhasfn'),ldec') =
		           instrdec ((btexp,bsites,bvars,bhasfn),ldec)
		val (d',lexp') = instrexp3((btexp',bsites',bvars',bhasfn'),
					   lexp)
	    in if d' then
		   let val valvar = makenvar "_valtmp"
		   in LETexp(ldec',
			     LETexp(VALdec[VB{pat=TuplePat[WildPat,
							   VARpat valvar],
					      exp=lexp',
					      tyvars=ref nil}],
				    VarExp (ref valvar,NONE)))
		   end
	       else (* If anything at all was bound by ldec, then if 
		       any event is executed in body it would have discharged.
		       So at this point, we need a backstop event. *)
		   let val evn = makeEvn(makeEvent(LETev exp)::bsites')
		   in LETexp(SEQdec[ldec',
				    VALdec[VB{pat=WildPat,
					      exp=EVENTexp(evn,btexp',bvars'),
					      tyvars=ref nil}]],
			     lexp')
		   end
	    end)
      | instr (CASEexp(exp,rl)) = 
	  let val (d,exp') = instr exp
	  in if d then
	       let val valvar = makenvar "_valtmp"
		   val btvar = makenvar "_btvar"
		   val rl' = instrrules((VarExp btvar,nil,nil,false),
					fn r => makeEvent(CASEev(exp,r)),
				        false) rl
	       in (true,
		   LETexp(VALdec[VB{pat=TuplePat[VARpat btvar,VARpat valvar],
				    exp=exp',
				    tyvars=ref nil}],
			  TupleExp[VarExp btvar,
				   CASEexp(VarExp valvar,
					   rl')]))
	       end
	     else 
               discharge (true,fn b =>		 
		 let val rl' = 
			instrrules(b,fn r => makeEvent(CASEev(exp,r)),false) rl
		 in CASEexp(exp',rl')
		 end)
	  end
	
      | instr (HANDLEexp (e, HANDLER(FNexp(body,t)))) =
	  (* We discharge any events before executing HANDLEexp *)
	  discharge (true,fn b => 
	    let val (d',e') = instrexp3(b,e)
		val e' = 
		    if d' then
		      let val valvar = makenvar "_valtmp"
		      in LETexp(VALdec[VB{pat=TuplePat[WildPat,VARpat valvar],
					  exp=e',
					  tyvars=ref nil}],
				VarExp (ref valvar,NONE))
		      end
		    else e'
		val body' = instrrules(b,makeEvent o HANDLEev,true) body
	    in HANDLEexp(e',HANDLER(FNexp(body',t)))
	    end)
      | instr (FNexp(body,t)) =
	  (* We discharge any function events before executing FNexp *)
	  discharge (false,fn b =>
	    let val body' = instrrules(b,makeEvent o FNev,true) body
	    in FNexp(body',t)
	    end)
      | instr (MarkExp (exp,s,e)) =
          let val (d,exp') = instr exp
          in (d, MarkExp(exp',s,e))
          end
      | instr exp = (false,exp)
    and instrrules (b as (btexp,bsites,bvars,bhasfn),evf,addsfn) =
	(* always return just a value *)
      let 
        fun f (rule as RULE(pat,exp as MarkExp(_))) = 
	     let val vars = (patvars (fn v => VarExp v) pat)
		 val bsites' = (evf rule)::bsites
		 val bvars' = vars@bvars
		 val bhasfn' = bhasfn orelse addsfn
		 val (d,exp') = instrexp3((btexp,bsites',bvars',bhasfn'),exp)
	     in if d then
		  let val valvar = makenvar "_valtmp"
		  in RULE(pat,
			  LETexp(VALdec[VB{pat=TuplePat[WildPat,
							VARpat valvar],
					   exp=exp',
					   tyvars=ref nil}],
				 VarExp valvar))
		  end
		else (* need backstop *)
		  let val evn = makeEvn bsites'
		  in RULE(pat,
			  SeqExp[EVENTexp(evn,btexp,bvars'),
				 exp'])
		  end
	      end
	  | f (RULE(pat,CONSTRAIntExp(exp,_))) =  f (RULE(pat,exp))
          | f (RULE(pat,exp)) =
	     let val (d,exp') = instrexp3(b,exp)
	     in if d then
 	          let val valvar = makenvar "_valtmp"
		  in RULE(pat,
			  LETexp(VALdec[VB{pat=TuplePat[WildPat,
							VARpat valvar],
					   exp=exp',
					   tyvars=ref nil}],
				 VarExp valvar))
		  end
		else RULE(pat,exp')
             end
      in map f 
      end
  in
    instr exp
  end



	    in (b', MARKdec (dec',region))
	    end
      | instr dec = (b, dec)

   and instrstrbl timearrexp strbl = 
         let 
           fun dostrb (STRB{strvar,abslty,def,thin,constraint},
		       (n,lasttimedec,strbl))= 
	        let val def' = instrstrexp(b,def)
		    val strb' = STRB{strvar=strvar,
				     def=LETstr(lasttimedec,def'),
				     thin=thin,abslty=abslty,
				     constraint=constraint}
		    val lasttimedec' = 
			VALdec[VB{pat=WildPat,
				  exp=App(VarExp updateop,
					     TupleExp[timearrexp,
						      IntExp n,
						      GETTIMEexp]),
				  tyvars=ref nil}]
		in (n+1,
		    lasttimedec',
	            strb' :: strbl)
		end
	   val (_,lasttimedec,strbl') = 
	       revfold dostrb strbl (0,SEQdec nil,[])
  	   fun tovar (STRB{strvar=STRvar{name=n,access=LVAR lv,...},...}) =
		 fakeuse(n,lv)
	   val strvl = map tovar strbl
	 in (rev strbl',lasttimedec,strvl)
         end
    in instr dec
   end

and instrstrexp (b,mstrexp as MARKstr(strexp,_)) =
     (case strexp of 
        STRUCTstr{body,locations,str} =>
          let val (b',body') = instrlist(b,body)
              val (_,enddec) = 
       	       simplebind(b',makeEvent(STRENDev mstrexp),nil)
          in STRUCTstr{body=body'@[enddec],locations=locations,
		       str=str}
          end
      | APPstr{oper,instancelty,argexp,argthin,str} => 
          let val argexp' = instrstrexp (b,argexp)
	      val strvar = makenstrvar "_AnonStruct"
	      val strb = STRB{strvar=strvar,def=argexp',abslty=NONE,
			      thin=argthin (*??*),constraint=NONE}
	      val STRvar{access=LVAR lv,...} = strvar
	      val paramv = fakeuse (Symbol.varSymbol "param",lv)
              val (_,appdec) = 
       		simplebind(b,makeEvent(FCTAPPev mstrexp),[paramv])
          in LETstr(SEQdec[STRdec[strb],appdec], 
		    APPstr{oper=oper,argexp=VARstr strvar,
			   instancelty=instancelty,argthin=NONE,str=str})
          end
      | LETstr(dec,strexp) =>
          let val (b',dec') = instrdec(b,dec)
              val strexp' = instrstrexp(b',strexp)
          in LETstr(dec',strexp')
          end
      | VARstr _ =>
          let val (_,defdec) = 
       		simplebind(b,makeEvent(STRVARev mstrexp),nil)
          in LETstr(defdec,strexp)
          end
      | MARKstr _ => debugPanic "instrstrexp: double MARKstr")
  | instrstrexp (_)= debugPanic "instrstrexp: unmarked strexp"

and instrlist (b, decl) = 
   let fun g (dec::rest, b, acc) =
     		let val (b',dec') = instrdec(b,dec)
	        in g(rest,b',dec' :: acc)
	        end
	 | g (nil,b,acc) = (b,rev acc)
   in g (decl,b,nil)
   end

val absyn' =
  let val startevn = makeEvn [makeEvent(STARTev absyn)]
      val bevvar = makevar ("_bind" ^ makestring startevn)
      val STARTexp = EVENTexp(startevn,IntExp lastBindTime,nil)
      val STARTdec = VALdec[VB{pat=VARpat(bevvar),exp=STARTexp,tyvars=ref nil}]
      val ((btexp,bsites,bvars,bhasfn),absyn') = 
 	     instrdec((VarExp (ref bevvar,NONE),nil,nil,false),absyn)
      val endevn = makeEvn(makeEvent(ENDev absyn)::bsites)
      val ENDexp = FEVENTexp(endevn,btexp,VarExp bevvar::bvars)
      val ENDdec = VALdec[VB{pat=WildPat,exp=ENDexp,tyvars=ref nil}]
  in  SEQdec[STARTdec,
	     absyn',
	     ENDdec]
  end


     val VARbind getDebugVar = ModuleUtil.lookVARCON 
	 (corenv,[Symbol.strSymbol "Core", Symbol.varSymbol "getDebug"],
	  fn _ => fn s => fn _ => ErrorMsg.impossible "222 in instrum")

in 
  {absyn=LOCALdec(VALdec [VB {pat = TuplePat [VARpat timesvar,
					      VARpat eventtimes,
					      VARpat breakentry,
					      VARpat hcreater,
					      VARpat weakvar,
					      VARpat udrl,
					      VARpat pconsvar,
					      VARpat udal,
					      VARpat arrayvar],
                              exp = App(VarExp getDebugVar,
					   IntExp (firstPlace)),
  	                      tyvars = ref nil}],
                  absyn'),
   events = fromList(rev(!eventList)),
   evns = fromList(rev(!evnList))}
end  (* fun instrumDec *)

end  (* structure debugInstrum *)
