(* Copyright 1989,1992 by AT&T Bell Laboratories *)

signature DEBUGGER = 
sig
 type charno (* = int *)
 type location (* = filename * charno *)
 type filename (* = string *)
 type time (* = int *)
 type place (* = int *)
 type value (* = System.Unsafe.object *)
 type ty  (* = System.Unsafe.object *)
 datatype 'a outcome = COMPLETED of 'a | INTERRUPTED of 'a | NOTRUNNING
 datatype ('a,'b) plist = PNIL | PCONS of 'a * 'b * ('a,'b) plist
 datatype debuglevel = 
       FULL
     | LIVE of ((string * instream) option * (unit -> unit) * (unit -> unit))
     | INTERPOLATION

 val times : time array
 val eventtimes : int -> time array
 val break : System.Unsafe.object array -> unit
 val hcreater: System.Unsafe.object -> (System.Unsafe.object array)
 val updatedRList: System.Unsafe.object array System.Weak.weak list ref
 val updatedAList: (System.Unsafe.object array System.Weak.weak,int) plist ref
 val weak : 'a -> 'a System.Weak.weak
 val array : int * '1a -> '1a array

 (* Naming conventions:

  X routines are heavy-weight: they take care of establishing and resetting
    times as appropriate, and generally return outcomes to note interrupts.

  Y routines typically take a time as argument, and are more fragile; they
    must be called under the protection of an X routine 
    (such as XwithEstablishedTime) to reset times and deal with interrupts.

  Z routines generally operate independent of current time/context.

  W routines are strictly private, for debugging the debugger.
 *)

 val ZdebugPervEnv: Environment.environment ref
 val Xuse_file: (debuglevel*string)->unit
 val Xuse_stream: (debuglevel*instream)->unit
 val XwithEstablishedTime: (time->'a) -> 'a outcome
 val YcurrentTime: unit->time
 val YcurrentPlaces: unit->place list
 val YboundingTimes: unit->(time*time)
 val YlastTimes: place->time*time
 val Xjump: time-> (place * time) outcome
 val XbinSearch: (unit->time) * time * bool -> (place*time) outcome
 val YcallTrace: int->time->(((place*time)*(place*time)*(((string*ty)*value) list)) list)
 val YgetVal: string->time->(value*ty*(place*time)) option
 val YprintVal: (value*ty)->unit
 val ZisFn: ty->bool
 val YprintBind: ((place*time)*int)->unit
 val Wdd:bool ref
 val hcreatea: (int * value) -> (value array)
 val ZeventsAfterLocation: location -> place list
 val ZeventsBeforeLocation: location -> place list
 val setHandler: (System.Signals.signal * ((int * unit cont) -> unit cont) option) -> unit
 val inqHandler: System.Signals.signal -> ((int * unit cont) -> unit cont) option
 val maskSignals: bool->unit
 val pause:unit->unit
 val Xcomplete: unit->unit outcome
 val Xabort: unit->unit outcome
 val ZinDebug: unit->bool
 val Yexception: unit->exn option
 val logit : ((unit->'a)*('a->unit)*bool)->'a
 val ZeventDesc: place ->(string*bool*location*bool) option
 val WmaxTimeDelta:int ref
 val Wtimes: int array
 val Ycaller: time->((place*time)*(place*time))
 val Zinfinity:int
 val ZsetEnvTime: time->unit
 val YatCall: time->bool
 val ZcharnoForLinepos: (filename * int * int) -> charno
 val Wsizereport: (string->unit) ref
 val WinstrumLevel:int ref
 val WmemoLevel:int ref
 val WdumpCache: unit -> unit
 val Wdfactor:real ref
 val WexecTime: unit -> int ref
 val WmaxStates: int ref
 val WpreCachingEnabled: bool ref
 val WcpCost: int ref
 val Wpcfactor: (real ref)
 val WzapFactor: (real ref)
 val WstrictLru : (bool ref)
 val WcacheRatio : (int ref)
 val WzapCount: (int ref)
 val Xinterpolate_stream: (instream->unit)
 val ZdebugCommandsEnv: Environment.environment ref
 val XsetSignal: System.Signals.signal -> unit
 val XclearSignal: System.Signals.signal -> unit
 val Ysignal: unit -> (System.Signals.signal option)
 val ZhaltOnSignal : (System.Signals.signal * bool) -> unit
 val YexnArg: exn -> ((value * ty) option)
 val ZlineposForCharno : location -> (int * int)
end

funsig DEBUGGERFUN (structure Machm : CODEGENERATOR) = DEBUGGER

functor RealDebugger(structure Machm : CODEGENERATOR) : DEBUGGER =
struct
  open ErrorMsg CompUtil Environment PrettyPrint DebugStore
      DebugMotions

 type charno  = int 
 type filename  = string 
 type location  = filename * charno 
 type place  = int 
 type value  = System.Unsafe.object 
 type ty   = Types.ty
  type time = DebugKernel.time
  structure U = System.Unsafe

  val times = DebugKernel.times
  val eventtimes = DebugStatic.evnTimesArray
  val break = DebugKernel.break
  val hcreater = DebugStore.hcreater
  val weak = System.Weak.weak
  val updatedRList = DebugStore.updatedRList
  val updatedAList = DebugStore.updatedAList
  val array = Array.array

  structure EvalLoop = EvalLoopF(CompileF(Machm))

  val say = Control.Print.say

  (* shared with dbguser/interface.sml *)
  datatype debuglevel = 
       FULL
     | LIVE of ((string * instream) option * (unit -> unit) * (unit -> unit))
     | INTERPOLATION

  val _ = DebugUtil.debugStatEnv := #static (#get pervasiveEnvRef ())

  fun instrument (source: Source.inputSource,
	          statEnv: StaticEnv.staticEnv,
	          absyn: Absyn.dec) : Absyn.dec =
	let fun dump label absyn =
		if !Control.debugging then
  		  (with_pp (ErrorMsg.defaultConsumer())
  		     (fn ppstrm =>
  		       (add_string ppstrm ("\n" ^ label ^ "\n"))))
		else ()
            val _ = dump "BEFORE:" absyn
	    val firstPlace = DebugStatic.nextPlace()
	    val lastBindTime  = 
	       if DebugExec.inCompUnit() then  (* interpolation *)
		 DebugExec.currentTime()
	       else DebugStatic.lastUnitTime()
	    val {absyn,events,evns} =
	       DebugInstrum.instrumDec (#static(#get Environment.topLevelEnvRef ()))
	         {absyn=absyn,
		  firstPlace=firstPlace,
		  lastBindTime=lastBindTime}
        in dump "AFTER:" absyn;
           DebugStatic.install{inputSource=source,
			       firstPlace=firstPlace,
			       events=events,
			       evns=evns};
	   absyn
        end
		
  val instrument = Stats.doPhase(Stats.makePhase "Compiler 035 DebugInstrum")
                       instrument

  (* Transformation function guarantees that every function argument
     and every function rule pattern has a "simple" type, i.e., has
     an obvious base type or an explicit type constraint. 
     This notion of simplicity must match that used in DebugBindings. *)
  val constrainer : Absyn.dec -> Absyn.dec =
      let open Absyn Variables TypesUtil
	  fun constrainDec dec =
	     (case dec of
		VALdec vbl => VALdec(map constrainVb vbl)
	      | VALRECdec rvbl => VALRECdec(map constrainRvb rvbl)
	      | ABSTYPEdec{abstycs,withtycs,body} =>
		    ABSTYPEdec{abstycs=abstycs,withtycs=withtycs,
			       body=constrainDec body}
	      | STRdec strbl => STRdec(map constrainStrb strbl)
	      | ABSdec strbl => ABSdec(map constrainStrb strbl)
	      | FCTdec fctbl => FCTdec(map constrainFctb fctbl)
	      | LOCALdec(decin,decout) =>
		    LOCALdec(constrainDec decin,constrainDec decout)
	      | SEQdec decl => SEQdec(map constrainDec decl)
	      | MARKdec(dec,region) => MARKdec(constrainDec dec,region)
	      | dec => dec)
          and constrainExp exp =
	     (case exp of
		RECORDexp lexpl =>
		    RECORDexp(map (fn (l,exp) => (l,constrainExp exp)) lexpl)
	      | SEQexp expl => SEQexp (map constrainExp expl)
	      | APPexp (exp1,exp2) =>
		  let fun simple exp =
		       (case exp of
			  INTexp _ => exp
			| REALexp _ => exp
			| STRINGexp _ => exp
			| CHARexp _ => exp
			| CONSTRAINTexp(exp,ty) => 
			      CONSTRAINTexp(constrainExp exp,ty)
			| MARKexp(exp,region) => MARKexp(simple exp,region)
			| _ => CONSTRAINTexp(constrainExp exp,mkMETAty()))
		  in APPexp(constrainExp exp1,simple exp2)
		  end
	      | CONSTRAINTexp (exp,ty) => CONSTRAINTexp(constrainExp exp,ty)
	      | HANDLEexp(exp1,HANDLER exp2) =>
		    HANDLEexp(constrainExp exp1,HANDLER (constrainExp exp2))
	      | RAISEexp(exp,t) => RAISEexp(constrainExp exp,t)
	      | LETexp(dec,exp) => LETexp(constrainDec dec,constrainExp exp)
	      | CASEexp(exp,rl) => CASEexp(constrainExp exp,
					   map constrainRule rl)
	      | FNexp(rl,t) => FNexp(map constrainRule rl,t)
	      | MARKexp(exp,region) => MARKexp(constrainExp exp,region)
	      | _ => exp)
	  and constrainRule(RULE(pat,exp)) =
  	        RULE(constrainPat pat,constrainExp exp)
	  and constrainPat pat =
	       (case pat of 
		  WILDpat => pat
		| VARpat(VALvar _) => pat
		| INTpat _ => pat
		| REALpat _ => pat
		| STRINGpat _ => pat
		| CHARpat _ => pat
	        | CONSTRAINTpat _ => pat
		| _ => CONSTRAINTpat(pat,mkMETAty()))
          and constrainVb (VB{pat,exp,tyvars}) =
	        VB{pat=pat,exp=constrainExp exp,tyvars=tyvars}
	  and constrainRvb (RVB{var,exp,resultty,tyvars}) =
	        RVB{var=var,exp=constrainExp exp,
		    resultty=resultty,tyvars=tyvars}
	  and constrainStrb (STRB{strvar,abslty,def,thin,constraint}) =
	        STRB{strvar=strvar,def=constrainStrexp def,thin=thin,
		     abslty=abslty,constraint=constraint}
	  and constrainStrexp strexp =
	       (case strexp of
		  VARstr sv => VARstr sv
		| STRUCTstr{body,str,locations} =>
		      STRUCTstr{body=map constrainDec body,str=str,
				locations=locations}
		| APPstr{oper,instancelty,argexp,argthin,str} =>
		      APPstr{oper=oper,argexp=constrainStrexp argexp,
			     instancelty=instancelty,argthin=argthin,str=str}
		| LETstr(dec,strexp) =>
		      LETstr(constrainDec dec,constrainStrexp strexp)
		| MARKstr(strexp,region) =>
		      MARKstr(constrainStrexp strexp,region))
	  and constrainFctb (FCTB{fctvar,def}) =
	        FCTB{fctvar=fctvar,def=constrainFctexp def}
	  and constrainFctexp fctexp =
	     (case fctexp
		of VARfct vfct => VARfct vfct
	         | FCTfct{param,def,thin,constraint} =>
		     FCTfct{param=param,def=constrainStrexp def,
			    thin=thin,constraint=constraint}
  	         | LETfct (dec,fct) => 
  		     LETfct(constrainDec dec, constrainFctexp fct))
       in constrainDec
      end

  (* Filter printing of top-level declarations from instrumented code. 
     Debugger declarations begin with an underscore (_). *)
  fun debugPrintDec (env as {static=statenv,dynamic,symbolic}) ppstrm absyn =
      let open Absyn Variables
	  fun cleanVb (VB{pat,exp,tyvars}) =
	        let fun cleanPat(pat as VARpat(VALvar{path=SymPath.SPATH[n],
						      ...})) =
			  if substring(Symbol.name n,0,1) = "_"
			  then WILDpat
			  else pat
		      | cleanPat(RECORDpat{fields,flex,typ}) =
			  RECORDpat{fields=map (fn (l,p) => (l,cleanPat p)) 
				               fields,
				    flex=flex,typ=typ} (* ?? *)
		      | cleanPat(APPpat(con,t,pat)) =
			  APPpat(con,t,cleanPat pat)
		      | cleanPat(CONSTRAINTpat(pat,ty)) =
			  CONSTRAINTpat(cleanPat pat,ty)
		      | cleanPat(LAYEREDpat(pat1,pat2)) =
			  LAYEREDpat(cleanPat pat1, cleanPat pat2)
		      | cleanPat pat = pat
		in VB{pat=cleanPat pat,exp=exp,tyvars=tyvars}	  
		end
          fun cleanDec (VALdec vbs) = VALdec (map cleanVb vbs)
	    | cleanDec (ABSTYPEdec{abstycs,withtycs,body}) =
               ABSTYPEdec{abstycs=abstycs,withtycs=withtycs,body=cleanDec body}
	    | cleanDec (LOCALdec(decIn,decOut)) =
	                    LOCALdec(decIn, cleanDec decOut)
	    | cleanDec (SEQdec decs) = SEQdec(map cleanDec decs)
	    | cleanDec (MARKdec(dec,region)) = MARKdec(cleanDec dec,region)
	    | cleanDec dec = dec
     in PPDec.ppDec env ppstrm (cleanDec absyn)
     end


  val debugEnv = DebugEnv.debugEnvironment

  val debuggerCommandsEnvRef = ref emptyEnv  (* to speed up lookups *)
  val debugMonitorEnvRef = ref emptyEnv

  fun init FULL f =
      let open DebugMotions DebugStatic DebugUtil
      in case runCompUnit complete f of
           NORMAL r =>  
	       (say "[debugging support included]\n";
		r)
	 | EXCEPTION e => 
	       (rollback(); raise e)
	 | ABORT => (* shouldn't happen *)
	       (rollback(); raise ErrorMsg.Error)
	 | INTERRUPT =>
	       (rollback(); raise Interrupt)
      end
   | init (LIVE(control,startUp,abortShutDown)) f =
      let open DebugMotions DebugStatic Control DebugUtil
	  val oldPrimaryPrompt = !primaryPrompt
	  and oldSecondaryPrompt = !secondaryPrompt
	  fun debugMonitor() = 
	      (debugMonitorEnvRef := !debuggerCommandsEnvRef;
	       say "[ready to execute under debugger]\n";
	       startUp();
	       let val baseEnv0 = layerEnv(#get topLevelEnvRef (),
					   #get pervasiveEnvRef ())
		   val baseEnv = layerEnv(debugEnv,baseEnv0)
		   val b = ref baseEnv
		   val innerLoopParams = 
		       {compManagerHook = ref NONE,
			baseEnvRef = {get=(fn()=> !b),set=(fn x=> b:=x)},
			localEnvRef={get=(fn()=> !debugMonitorEnvRef),
				     set=(fn x=> debugMonitorEnvRef:=x)},
			transform = (fn x => x),
			instrument = (fn _ => fn a => a),
			perform=(fn x => x),
			isolate=(fn f => fn x => f x),
			printer=PPDec.ppDec}
	       in primaryPrompt := "[dbg]" ^ oldPrimaryPrompt;
		  secondaryPrompt := "[dbg]" ^ oldSecondaryPrompt;
		  debugStatEnv := #static baseEnv0;
		  case control of
		    SOME (fname,stream) => 
		       EvalLoop.eval_stream innerLoopParams (fname,stream)
		  | NONE => EvalLoop.interact innerLoopParams;
		  (* return only via ctrl/d or stream error *)
		  abortShutDown();
		  abort();
		  debugPanic "Returned from abort"
	       end)
	  fun reset() = 
	     (#set topLevelEnvRef 
	          (layerEnv(!debugMonitorEnvRef,#get topLevelEnvRef ()));
	      primaryPrompt := oldPrimaryPrompt;
	      secondaryPrompt := oldSecondaryPrompt)
      in case runCompUnit debugMonitor f of 
	   NORMAL r =>
	       (say "[completing normal execution]\n";
		reset();
		r) 
	 | EXCEPTION e => 
	       (say "[execution terminated by exception]\n";
		rollback();
		reset();
		raise e)
	 | ABORT => 
	       (say "[execution aborted]\n";
		rollback();
		reset();
		raise ErrorMsg.Error)
	 | INTERRUPT =>
	       (say "[execution interrupted]\n";
		rollback();
		reset();
		raise Interrupt)
     end
    | init INTERPOLATION f =
         (DebugMotions.interpolateCompUnit (fn () => (f(); ()));
	  raise ErrorMsg.Error)
	  
  val init = fn level => fn f => fn v => init level(fn()=>f v)

  val debuggerPervasiveEnvRef = ref emptyEnv

  fun dbgParams level =
      {compManagerHook = ref NONE,
       baseEnvRef= {get=(fn()=> !debuggerPervasiveEnvRef),
		    set=(fn x=> debuggerPervasiveEnvRef:=x)},
       localEnvRef=topLevelEnvRef,
       transform = constrainer,
       instrument = (fn{source,compenv} => fn absyn=> 
		     instrument(source,compenv,absyn)),
       perform=init level,
       isolate=CompUtil.isolate,
       printer=debugPrintDec} 

  fun use_file_dbg (level:debuglevel,fname:string) : unit = 
      (DebugStatic.hideFile fname;
       EvalLoop.eval_stream (dbgParams level)
              (fname,(open_in fname
		        handle e as Io _ =>
			    (say(concat["[use failed: ",General.exnMessage e,
					"]\n"]);
			     raise ErrorMsg.Error))))


  fun use_stream_dbg (level:debuglevel,stream:instream) : unit = 
      EvalLoop.eval_stream (dbgParams level) ("<instream>",stream)
      
  fun interpolate_stream(stream:instream) : unit =
      let val dummyEnvRef = ref emptyEnv
          val baseEnv = layerEnv(!debugMonitorEnvRef,
				 layerEnv(debugEnv,
					  layerEnv(#get topLevelEnvRef(),
						   !debuggerPervasiveEnvRef)))
	  val b = ref baseEnv
      in EvalLoop.eval_stream 
	     {compManagerHook = ref NONE,
	      baseEnvRef = {get=(fn()=> !b),set=(fn x=> b:=x)},
	      localEnvRef = {get=(fn()=> !dummyEnvRef),
			     set=(fn x=> dummyEnvRef:=x)},
	      transform = constrainer,
	      instrument = (fn{source,compenv} => fn absyn=> 
			    instrument(source,compenv,absyn)),
	      perform = init INTERPOLATION,
	      isolate = (fn f => fn x => f x),
	      printer = (fn env => fn ppstrm => fn absyn => ())}
	    ("<interpolation>",stream)
	    handle ErrorMsg.Error => ()
      end
	 

 val ZdebugPervEnv =  debuggerPervasiveEnvRef
 val Xuse_file =  use_file_dbg
 val Xuse_stream =  use_stream_dbg
 val XwithEstablishedTime =  DebugMotions.withEstablishedTime
 val YcurrentTime =  DebugExec.currentTime
 val YcurrentPlaces =  (fn() => DebugStatic.immediatePlaces(
			      DebugStatic.placesFor(DebugExec.currentEvn())))
 val YboundingTimes =  (fn () => (!DebugExec.initialTime,!DebugExec.finalTime))
 val YlastTimes =  DebugQueries.lastTimes
 val Xjump =  DebugMotions.jump
 val XbinSearch =  DebugMotions.binSearch
 val YcallTrace =  DebugQueries.callTrace
 val YgetVal =  DebugQueries.getVal
 val YprintVal =  DebugQueries.printVal
 val ZisFn =  DebugUtil.isFn
 val YprintBind =  DebugQueries.printBind 
 val Wdd =  DebugUtil.debugdebug
 val updatedAlist = DebugStore.updatedAList
 val hcreatea = DebugStore.hcreatea
 val ZeventsAfterLocation =  DebugStatic.eventPlacesAfter
 val ZeventsBeforeLocation =  DebugStatic.eventPlacesBefore
 val setHandler = DebugMotions.setHandler
 val inqHandler = DebugSignals.inqHandler
 val maskSignals = DebugSignals.maskSignals
 val pause = DebugSignals.pause
 val Xcomplete =  DebugMotions.complete
 val Xabort =  DebugMotions.abort
 val ZinDebug =  DebugExec.inCompUnit
 val Yexception =  (fn () => !DebugExec.blockingExn)
 val logit =  DebugIO.logit
 val ZeventDesc =  DebugQueries.eventDesc
 val WmaxTimeDelta =  DebugRun.maxTimeDelta
 val Wtimes =  DebugKernel.times
 val Ycaller =  DebugQueries.caller
 val Zinfinity =  DebugUtil.infinity
 val ZsetEnvTime =  DebugEnv.setEnvTime
 val YatCall =  DebugQueries.atCall
 val ZcharnoForLinepos =  DebugStatic.charnoForLinepos
 val Wsizereport =  DebugUtil.sizereport
 val WinstrumLevel =  DebugInstrum.instrumLevel
 val WmemoLevel =  DebugRun.memoLevel
 val WdumpCache =  DebugRun.dumpCache
 val Wdfactor =  DebugRun.dfactor
 val WexecTime =  (fn () => DebugKernel.execTime)
 val WmaxStates =  DebugRun.maxStates
 val WpreCachingEnabled =  DebugRun.preCachingEnabled
 val WcpCost =  DebugMotions.cpCost
 val Wpcfactor =  DebugRun.pcfactor
 val WzapFactor =  DebugRun.zapFactor
 val WstrictLru  =  DebugRun.strictLru
 val WcacheRatio  =  DebugRun.cacheRatio
 val WzapCount =  DebugRun.zapCount
 val Xinterpolate_stream =  interpolate_stream
 val ZdebugCommandsEnv =  debuggerCommandsEnvRef
 val XsetSignal =  DebugMotions.setSignal
 val XclearSignal =  DebugMotions.clearSignal
 val Ysignal =  (fn () => 
		       case DebugSignals.deliverableSignal() of
			 SOME(signal,_) => SOME signal
		       | NONE => NONE)
 val ZhaltOnSignal  =  DebugSignals.setHalting
 val YexnArg =  DebugQueries.exnArg
 val ZlineposForCharno  =  DebugStatic.lineposForCharno

end

