(* interface.sml
 *
 * COPYRIGHT (c) 1996 Bell Laboratories.
 *)

structure Debugger = Compiler.Debugger(structure Machm = Compiler.Machm)

structure UserDebugInterface = struct
local  open UserDebugUtil 
       structure U = System.Unsafe 
in

 open Debugger

 type wherewhen = place * time 

 (** Useful functions on events, built up from interface functions. *)

 fun interruptableQuery (f:time->unit)  =
   (* Suitable for operations that play with time and do text-style output *)
   case (XwithEstablishedTime f) of
     COMPLETED x => x
   | INTERRUPTED _ => Compiler.Control.Print.say "(Interrupted)\n"
   | NOTRUNNING => printNotUnder()

 fun safeQuery f  =
   (* Suitable for operations that don't change the time *)
   case (XwithEstablishedTime f) of
     COMPLETED x => x
   | INTERRUPTED x => x
   | NOTRUNNING => raise (DebugUserError "safeQuery")

   
 fun establishedTime() = safeQuery (fn t => t)
 fun establishedPlace() = safeQuery (fn _ => hd(YcurrentPlaces()))

 fun eventText ev =
     #1 (ensureD(ZeventDesc ev, "eventText"))

 fun eventLocation ev : location option =
     case ZeventDesc ev of
       SOME (_,pseudo,filpos,visible) =>
	   if (not pseudo) andalso visible then
             SOME filpos
	   else NONE
     | NONE => NONE

 fun traceEvent (ww:wherewhen) (n:int) : wherewhen option = 
 (* Return the nth caller above the given location, counting that location
  * as 0th. *)
   let fun trace (ww as (_,t:time)) =
         if t > 0 then
	   fn 0 => SOME ww
	    | n => 
	       let val (_,ww) = Ycaller t
	       in trace ww (n-1)
	       end
	 else fn _ => NONE
   in case (XwithEstablishedTime (fn _ => trace ww n)) of
        COMPLETED wwop => wwop
      | INTERRUPTED _ => NONE
      | NOTRUNNING => raise (DebugUserError "traceEvent")
   end

 local 
   exception NotAvailable
   fun findEv ev =
   (* Return file, character position for event, "eventsAfter" list 
    * containing event and its position in that list. *)
       case eventLocation ev of
	 SOME (loc as (file,cp)) =>
	   let val elist = ZeventsAfterLocation loc
	       val index =
		   case (first (fn x => x = ev) elist) of
		     SOME i => i
   (* It is possible that an event will not be found in the list: fine-grained
    * events are sometimes not indexed. In that case, simply move to an
    * event near the fine-grained event. *)
		   | NONE => ~1
	   in (file,cp,elist,index)
	   end
        | _ => raise NotAvailable
 in
 fun prevEvent  ev =
 (* Return the event that lexically  precedes the given event, i.e. the
  * previous event in the source text.  This allows us to sequentially
  * traverse all the events in a compilation unit. *)
     let val (file,cp,elist,index) = findEv ev
     in	
	 SOME (List.nth (elist, index-1 ))
	 handle Subscript =>
		 (SOME (foot (ZeventsBeforeLocation (file, cp - 1)))
		  handle Hd => NONE)
     end handle NotAvailable => NONE

 fun nextEvent ev =
 (* Return the event that lexically follows the given event, i.e. the next
  * event in the source text.  This allows us to sequentially
  * traverse all the events in a compilation unit. *)
     let val (file,cp,elist,index) = findEv ev
     in
	 SOME (List.nth (elist, index+1))
	 handle Subscript =>
		 (SOME (hd (ZeventsAfterLocation (file, cp + 1)))
		  handle Hd => NONE)
     end handle NotAvailable => NONE
 end (* local *)

end (* local *)
end (* structure *)

