(** mltreeGen.sml
 **
 ** COPYRIGHT (c) 1995 AT&T Bell Laboratories.
 **
 **	- Generates the mltree from the ctree
 **	- requires the knowledge table to discover if known functions
 **	  have been generated.
 **)

(*
 * Control.CG.misc4 := 64; for timing the labelling phase
 * Control.CG.misc4 := 128; for printing the sequences
 *)

signature MLTREE_GEN = sig
    datatype gcInfo = 
      GCINFO of {lab: Label.label, 
		 maskRegs: MLTree.mlrisc list,		
		 i32Regs: MLTree.mlrisc list,
		 fRegs: MLTree.mlrisc list,
		 ret: MLTree.mlrisc}

    val mltreeGen : 
      (CPS.lvar * Label.label * CTree.var list * CTree.funCtree
      * CPS.cty list
      * CPS.cty Intmap.intmap
      * (CPS.lvar -> Label.label)
      * (MLTree.mltree -> unit)
      * (CPS.lvar -> Frag.frag)
      * int)  
            -> gcInfo list

   val standardcont  : CPS.cty list -> MLTree.mlrisc list
   val standardescape: CPS.cty list -> MLTree.mlrisc list
   val callGc : (MLTree.mltree -> unit) -> gcInfo -> unit
end

(** MLTREEgen 
 ** 	Takes a ctree and generates an mltree. This module largely
 **	contains the semantic actions for the first BURG phase.
 **)

functor MLTREEgen (structure C : CELLS
		   structure R: MACH_SPEC
		     sharing C.T = MLTree
		       )  : MLTREE_GEN = struct
  structure M = MLTree
  structure Burm = BurmGen (CTreeIn)
  structure T = System.Tags
  structure CG   = Control.CG
  open CTreeIn

  (* WARNING: WARNING:
   * 	This constant is machine specific and should be passed as a component
   *    of regspec.
   *)
  val firstPseudoR = 32
  val firstPseudoF = 32

  structure CT = CTree					(* XXX DEBUG *)

  fun error msg = ErrorMsg.impossible ("MLTreeGen." ^ msg)

  fun List2app2 f ([],[]) = ()				(* XXX *)
    | List2app2 f (a::l,b::m) = (f(a,b); List2app2 f (l,m))

  (*
   * This datatype is used for
   * - keeping the registers associated with a given "variable" (regkFromVar)
   * - passing information during the walk of the tree (=> RLIST,ALIST)
   *)
  datatype arg
    = GPR of M.mlrisc						 (* register *)
    | FPR of M.mlrisc				  (* floating point register *)
    | NOTHING of M.mlrisc		       (* for side effect operations *)
    | ALIST of alarg list			      (* arguments of an app *)
    | NEVER				     (* for a flow control primitive *)

  and alarg
    = AL_LABEL of CPS.lvar
    | AL_INT of int
    | AL_REAL of Label.label
    | AL_STRING of Label.label
    | AL_ARG of arg

  datatype regkind
    = GPR_K of MLTree.mlrisc
    | FPR_K of MLTree.mlrisc

  (* GcInfo contains information to generate code to invoke gc *)
  datatype gcInfo = 
    GCINFO of {lab: Label.label,  
	       maskRegs: MLTree.mlrisc list, 
	       i32Regs: MLTree.mlrisc list,
	       fRegs: MLTree.mlrisc list,
	       ret: MLTree.mlrisc}

  (* The call to gc defines gclinkreg, and the exhausted register. 
   * It uses and defines the maskRegs and i32Regs registers.
   * Live floating point registers are put into a record and 
   * are pointed to by one of the pseudoregs.
   * SPACE LEAK: The pseudoreg is not cleared on return from a gc
   * call. Since this space does not grow and is a small constant, 
   * we will let this slip by.
   *)
  fun callGc addtoseq (GCINFO{lab, maskRegs, i32Regs, fRegs, ret}) = let
    fun invokeGC () = addtoseq (M.CODE
	  (M.SEQ (M.LDREGMASK maskRegs,
		  M.CALL (M.LOAD32 (M.ADD (C.stackptr,
					   M.LI R.startgcOffset)),
			  C.exhausted::C.gclinkreg::(maskRegs@i32Regs),
			  C.maskreg::(maskRegs@i32Regs)))))
    fun gcReturn () = (addtoseq(M.CODE ret);
		       addtoseq(M.ESCAPEBLOCK(C.exhausted::maskRegs)))
  in
    addtoseq (M.LABEL lab);
    case fRegs
     of [] => (invokeGC(); gcReturn())
      | _ => let
	  val k = length fRegs
	  val desc = T.make_desc(k, T.tag_realdarray)
	  val base = M.REG(C.newReg())

	  fun deposit([], _) = ()
	    | deposit(fpr::rest, i) = let
		fun storeD() = M.STORED (M.ADD (C.allocptr, M.LI i), fpr)
	      in
		 addtoseq(M.CODE(storeD())); 
		 deposit(rest,i+8)
	      end

	  fun restore([],_,_) = ()
	    | restore(fpr::rest, base, i) = let
		fun loadD() =  M.FMV(fpr, M.LOADD (M.ADD (base, M.LI i)))
	      in
		addtoseq(M.CODE (loadD ())); 
		restore(rest, base, i+8)
	      end
	in
	   (* align allocation pointer *)
	   addtoseq(M.CODE(M.MV(C.allocptr, M.ORB(C.allocptr, M.LI 4))));
	   deposit(fRegs, 4);
	   addtoseq(M.CODE(
	       M.SEQ(M.STORE32(C.allocptr, M.LI desc),
	       M.SEQ(M.STORE32(M.ADD(C.stackptr, M.LI R.pseudoRegOffset),
			       M.ADD(C.allocptr, M.LI 4)),
		     M.MV(C.allocptr, M.ADD(C.allocptr, M.LI(k * 8 + 4)))))));
	   invokeGC();
	   addtoseq(M.CODE(
	      M.MV(base, M.LOAD32(M.ADD(C.stackptr, 
				      M.LI R.pseudoRegOffset)))));
	   restore(fRegs, base, 0);
	   gcReturn()
	end
  end (* gcJump *) 

  val noinfoarray = Array.array (0, {ctree=EX(INT 0,[],0), deptype= ~1})

  val k = R.numCalleeSaves
  val kf = R.numFloatCalleeSaves

  local
    val gpregs = C.stdlink::C.stdclos::C.stdarg::C.stdcont::C.miscregs
    val fpregs = C.savedfpregs @ C.floatregs

    fun fromto(i, j, regs) = let
	fun from(0, acc) = acc
	  | from(n, x::xs) = from(n-1, xs)
	fun to(k, rl) =
	    if k > j then []
	    else (case rl of [] => error "fromto"
			   | r::rs => r :: to(k+1, rs))
    in to(i, from(i,regs))
    end
  in
    fun gprfromto(i, j) = fromto(i, j, gpregs)
    fun fprfromto(i, j) = fromto(i, j, fpregs)
    val calleesaveregs = gprfromto(4, k+3) @ fprfromto(0, kf-1)
  end

  fun cut_head(n,l as (a::r)) = if n=0 then l else cut_head(n-1,r)
    | cut_head _ = error "mltreeGen.cut_head"

  fun isFlt CPS.FLTt = true  | isFlt _ = false

  fun scan(t::z, gp, fp) = 
      if isFlt t then (hd fp)::(scan(z,gp,tl fp)) else (hd gp)::(scan(z,tl gp,fp))
    | scan([], _, _) = []

  fun standardescape args = let
      val rest = cut_head(k+kf+3, args)
      val len = length(args)
      val gpr = C.stdarg :: gprfromto(k+4, len)
      val fpr = fprfromto(kf,len)
  in 
     C.stdlink::C.stdclos::C.stdcont :: calleesaveregs @ scan(rest,gpr,fpr)
  end

  fun standardcont args = let
      val rest = if k > 0 then cut_head(k+kf+1,args) else cut_head(2,args)
      val len = length(args)
      val gpr = C.stdarg::gprfromto(k+4, 1+len)
      val fpr = fprfromto(kf,len)
  in 
     if k > 0 then C.stdcont::(calleesaveregs @ scan(rest,gpr,fpr))
     else C.stdlink::C.stdcont::scan(rest,gpr,fpr)
  end


  fun mltreeGen (funVar, lab, varfmls, functree, regtyps, typTable, labmap, 
		 addtoseq, know, alloc) = 
    let
      val addCode = addtoseq o M.CODE
      val typmap = Intmap.map typTable
      fun iscont v = case (typmap v) of CPS.CNTt => true | _ => false

      (* given a var, give me the corresponding arg (GPR/FPR) *)
      exception RegBind
      val regtable : regkind Intmap.intmap	= Intmap.new (32, RegBind)
      val addRegBinding				= Intmap.add regtable
      val regkFromVar				= Intmap.map regtable
      fun clearRegBindings () 			= Intmap.clear regtable

      val curinfoarray = ref noinfoarray

      val curseq = ref (0,0)				       (* XXX DEBUG *)

      fun addrOfReal r = M.LADDR (r, 0); 

      fun addrOfString s = M.LADDR (s, 0);
	
      fun addrOfLabel l = M.LADDR (labmap l, 0);

      fun storeFpr (fp) =
	let
	  val tmp = M.REG(C.newReg ())
	in
 	  (* alignment code added by Ken Cline *)
 	  M.SEQ(M.MV(C.allocptr, M.ORB(C.allocptr, M.LI 4)),
	  M.SEQ(M.STORED(M.ADD(C.allocptr, M.LI 4),fp),
 	  M.SEQ(M.STORE32(C.allocptr, M.LI T.desc_reald),
  	  M.SEQ(M.MV(tmp,M.ADD(C.allocptr,M.LI 4)),
	  M.SEQ(M.MV(C.allocptr, M.ADD(C.allocptr, M.LI 12)),
		tmp)))))
	end

      fun regFromVar v =
	case regkFromVar v of
	  GPR_K gp => gp
	| _ => error ("regFromVar " ^ Int.toString v)

      fun fregFromVar v =
	case regkFromVar v of
	  GPR_K gp => error "fregFromVar:GPR"
	| FPR_K fp => fp

      fun regFromArg arg =
	case arg of
	  GPR reg   => reg
	| FPR fp    => error "regFromArg:FPR"
	| NOTHING _ => error "regfromArg.NOTHING"
	| NEVER     => error "regfromArg.NEVER"
	| ALIST _   => error "regfromArg.ALIST"

      fun fregFromArg arg =
	case arg 
	 of FPR fp => fp
	  | _ => error "fregFromArg"

      val stdGcBlocks = ref ([] : gcInfo list)
      val knwGcBlocks = ref ([] : gcInfo list)
  
      local 
	(* (+ 8) for descriptor and possible alignment *)
	val falloc = R.numFloatRegs * 8 + 8 
	fun checkLimit (maxAlloc, regfmls, regtys) = let
          val lab = Label.newLabel ""
	  val max_alloc = maxAlloc + falloc
	in
	  if max_alloc < 4096 then () 
	  else addtoseq (M.CODE 
			 (M.TESTLIMIT (M.ADD (C.allocptr,
					      M.LI (max_alloc-4096)),
				       C.limitptr)));
	    addtoseq(M.CODE(M.CHECKLIMIT lab));
	    lab
	end
	(* floating point registers and INT32t paramaters do not 
	 * appear in the mask.
	 *)
	fun maskList([],[], mask, i32mask, fregs) = (mask, i32mask, fregs)
	  | maskList(r::rl, CPS.INT32t::tl, mask, i32mask, fregs) = 
	     maskList(rl, tl, mask, r::i32mask, fregs)
	  | maskList(r::rl, CPS.FLTt::tl, mask, i32mask, fregs) = 
	     maskList(rl, tl, mask, i32mask, r::fregs)
	  | maskList(r::rl, _::tl, mask, i32mask, fregs) = 
	     maskList(rl, tl, r::mask, i32mask, fregs)


      in
	fun stdCheckLimit(maxAlloc, regfmls, regtys, return) = let
	  val lab = checkLimit(maxAlloc, regfmls, regtys)
	  val (maskRegs, i32Regs, fregs) = maskList(regfmls, regtys, [], [], [])
	in
	   stdGcBlocks := 
	      GCINFO{lab=lab, maskRegs=maskRegs,
		     i32Regs=i32Regs, fRegs=fregs, ret=return} :: (!stdGcBlocks)
        end

	fun knwCheckLimit(maxAlloc, regfmls, regtys, return) = let
	  val lab = checkLimit(maxAlloc, regfmls, regtys)
	  val (maskRegs, i32Regs, fregs) = maskList(regfmls, regtys, [], [], [])
	in
	   knwGcBlocks := 
	       GCINFO{lab=lab, maskRegs=maskRegs,
		      i32Regs=i32Regs, fRegs=fregs, ret=return} :: (!knwGcBlocks)
	end
      end (*local*)


      fun recordwhat (r, CPS.SELp (0,p)) =  recordwhat (M.LOAD32 (r), p)
	| recordwhat (r, CPS.SELp (n,p)) =
	    recordwhat (M.LOAD32 (M.ADD (r, M.LI (n*4))), p)
	| recordwhat (r, CPS.OFFp 0) = r
	| recordwhat (r, CPS.OFFp n) = M.ADD (r, M.LI (n*4))

      fun frecordwhat(r, CPS.SELp(0, CPS.OFFp 0)) = M.LOADD(r)
	| frecordwhat(r, CPS.SELp(j, CPS.OFFp 0)) = M.LOADD(M.ADD(r, M.LI (j*4)))
	| frecordwhat(r, CPS.SELp(0,p)) = frecordwhat(M.LOAD32(r),  p)
	| frecordwhat(r, CPS.SELp(j,p)) = 
	    frecordwhat(M.LOAD32(M.ADD(r, M.LI(j*4))),p)
	| frecordwhat _ = error "frecordwhat"

      fun paramPassingConvention paramTypes = let
	  fun f CPS.FLTt = M.FREG(C.newFreg())
	    | f _ = M.REG(C.newReg())
      in map f paramTypes
      end

      (*   One entry to a function, the parameters will be in formal
       * parameter passing registers. Within the body of the function, they
       * are moved immediately to fresh temporary registers. This ensures
       * that the life time of the formal paramters is restricted to the 
       * function body and is critical in avoiding artificial register
       * interferences.
       *)
      fun initialRegBindings arg = let
	  fun f (x, r as M.REG v) = let
	        val t = C.newReg()
	      in
	        addtoseq(M.CODE(M.MV (M.REG t, M.REG v)));
	        addRegBinding(x, GPR_K (M.REG t))
	      end
	    | f (x, r as M.FREG v) = let
	        val t = C.newFreg()
	      in
		addtoseq(M.CODE(M.FMV (M.FREG t, M.FREG v)));
		addRegBinding(x, FPR_K (M.FREG t))
	      end
	    | f _ = error "initialRegBindings"
      in
	clearRegBindings();
	List2app2 f arg
      end

      fun callSetup(alist,formals) = let
	(* setupX - evaluates expressions into fresh temporaries *)
	  fun setupR((AL_ARG a,dst)::rest, flush, remain) = let
	        fun save exp = let 
	 	  val tmpR = C.newReg()
		in 
		  addtoseq (M.CODE(M.MV(M.REG tmpR, exp))); tmpR
		end
		val  gp = regFromArg a
	      in 
		case gp
		 of M.REG r => 
		      if dst = r then setupR(rest, flush, remain)
		      else setupR(rest, (dst,r)::flush, remain)
	          | exp => setupR(rest, (dst,save exp)::flush, remain)
	      end
	    | setupR(ad::rest, flush, remain) =  setupR(rest, flush, ad::remain)
	    | setupR([],flush,remain) = (rev flush, rev remain)

	  fun setupF((AL_ARG a,dst)::rest, flush, remain) = let
	        fun save exp = let val tmpR = C.newFreg()
		in addtoseq (M.CODE(M.FMV(M.FREG tmpR, exp))); tmpR
		end
		val fp = fregFromArg a 
	      in
		case fp
		 of M.FREG r => 
		      if dst = r then setupF(rest, flush, remain)
		      else setupF(rest, (dst,r)::flush, remain)
	          | exp => setupF(rest, (dst,save exp)::flush, remain)
	      end
	    | setupF(ad::rest, flush, remain) =  setupF(rest, flush, ad::remain)
	    | setupF([],flush,remain) = (rev flush, rev remain)

	 (* moveX -- performs the actual register shuffle. Since all 
	  * expressions have already been evaluated and formal parameters
	  * are moved into fresh temporaries, there is  guaranteed not to 
	  * be any interference between the source and  destination registers.
	  * The moves can therefore be performed blindly.
	  *)
	  fun moveR(fl, rl) = let
	    fun emitMove(dst,src) = addtoseq(M.CODE(M.MV(M.REG dst, M.REG src)))
	    fun remain(arg, dst) = let
	      val thing = 		      
		(case arg 
		  of AL_LABEL l => addrOfLabel l
		   | AL_INT i => M.LI (i+i+1)
		   | AL_REAL r => addrOfReal r
		   | AL_STRING s => addrOfString s
		   | _ => error "callSetup.moveR.remain"
		(*esac*))
	    in
	      addtoseq(M.CODE(M.MV(M.REG dst,thing)))
	    end
	  in
	    app emitMove fl;
	    app remain rl
	  end

	  fun moveF(fl, rl) = let
	    fun emitMove(dst, src) = 
	      addtoseq(M.CODE(M.FMV(M.FREG dst, M.FREG src)))
	    fun remain(arg, dst) = let
	      val thing =  
		(case arg 
		  of AL_REAL r => M.LOADD(addrOfReal r)
		   | _ => error "callSetup.moveF.remain"
		(*esac*)) 
	    in 
	      addtoseq(M.CODE(M.FMV(M.FREG dst,thing)))
	    end
	  in
	    app emitMove fl;
	    app remain rl
	  end

	  (* partition into general and floating point parameters  *)
	  (* Note: arguments are evaluated from left to right *)
	  fun partition(a::args,  M.FREG f::rl, gpl, fpl) =
	        partition(args, rl, gpl, (a,f)::fpl)
	    | partition(a::args, M.REG r::rl, gpl, fpl) = 
		partition(args, rl, (a,r)::gpl, fpl)
	    | partition([], [], gpl, fpl) = (rev gpl, rev fpl)

	  (* Important to evaluation arguments before shuffling registers *)
	  val (gpl, fpl) = partition(alist, formals, [], [])
	  val gpregs = setupR(gpl, [], [])
	  val fpregs = setupF(fpl, [], [])
      in
	  moveR(gpregs);  
	  moveF(fpregs)
      end

      
      (*
       * This generates the code for an expression, by first calling
       * the iburg-generated matcher to get the labelled tree,
       * then walking the tree and taking the appropriate
       * semantic actions.
       *)
      fun genexpr ex = let
	fun addOp (CT.ADD{signed=true}) = M.ADDT
	  | addOp (CT.ADD{signed=false})= M.ADD

	fun subOp (CT.SUB{signed=true}) = M.SUBT
	  | subOp (CT.SUB{signed=false})= M.SUB

	fun mulOp (CT.MUL{signed=true}) = M.MULT
	  | mulOp (CT.MUL{signed=false}) = M.MULU


	fun divOp (CT.DIV{signed=true}) = M.DIVT
	  | divOp (CT.DIV{signed=false})= M.DIVU

	(* for commutative operations *)
	fun walk2c((t1, t2), order) = 
	    case order 
	     of 2 => (walk t1, walk t2)
	      | 3 => (walk t2, walk t1)
	      | _ => error "walk2c"

	and walk2 ((t1,t2), order) =
	    case order
	     of 2 => (walk t1, walk t2, M.LR)
	      | 3 => (walk t1, walk t2, M.RL)
	      | _ => error "walk2"

	(* ignores order *)
	and walk2i (t1,t2) = (walk t1, walk t2)

	(* ignores order on t3. returns order for t1 and t2 *)
	   
	and walk3i ((t1,t2,t3), order) =
	    case order
	     of 4 => (walk t1, walk t2, walk t3, M.LR)
	      | 5 => (walk t1, walk t2, walk t3, M.LR)
	      | 6 => (walk t1, walk t2, walk t3, M.RL)
	      | 7 => (walk t1, walk t2, walk t3, M.RL)
	      | 8 => (walk t1, walk t2, walk t3, M.LR)
	      | 9 => (walk t1, walk t2, walk t3, M.RL)
	      | _ => error "walk3i"

	and walk3 ((t1,t2,t3),4) = (walk t1, walk t2, walk t3)
	  | walk3 ((t1,t2,t3),5) = let
		val r1 = walk t1
		val r3 = walk t3
		val r2 = walk t2
	      in (r1,r2,r3) 
              end
	  | walk3 ((t1,t2,t3),6) = let
		val r2 = walk t2
		val r3 = walk t3
		val r1 = walk t1
	      in (r1,r2,r3) 
              end
	  | walk3 ((t1,t2,t3),7) = let
		val r2 = walk t2
		val r1 = walk t1
		val r3 = walk t3
	      in (r1,r2,r3) 
	      end
	  | walk3 ((t1,t2,t3),8) = let
		val r3 = walk t3
		val r1 = walk t1
		val r2 = walk t2
	      in (r1,r2,r3) 
              end
	  | walk3 ((t1,t2,t3),9) = let
		val r3 = walk t3
		val r2 = walk t2
		val r1 = walk t1
	      in (r1,r2,r3) 
	      end
	  | walk3 _ = error "walk3"

	and doswitch (offset, seqlist) =
	  let
	    fun genSwitch ([],[]) = ()
	      | genSwitch (lab::labs, seq::seql) =
		(addtoseq (M.LABEL lab);
		 parallel_gen seq;
		 genSwitch (labs, seql))
	      | genSwitch _ = error "gen.genSwitch"
	    val lab = Label.newLabel ""
	    val labs = map (fn _ => Label.newLabel "") seqlist
	    val tmpR = M.REG (C.newReg ())
	    val docall = M.CODE
	      (M.SEQ (M.MV (tmpR, M.LADDR (lab, 0)),
		      M.GOTO (M.ADD (tmpR, M.LOAD32 (M.ADD(tmpR, offset))),
			      labs)))
	    val jmptbl = M.JMPTABLE {base=lab, targets=labs}
	  in
	    addtoseq (docall);
	    addtoseq (jmptbl);
	    genSwitch (labs, seqlist);
	    NEVER
	  end

	and GPRwalk tree = let val GPR gp = walk tree in gp end
	and ALISTwalk tree = let val ALIST alist = walk tree in alist end

	and walk (ruleandtree) =
	  case ruleandtree of
	    (Burm.CAST r, _) => walk r
	  | (Burm.reg_VAR, EX(VAR v,_,_)) => 
	      GPR (regFromVar v)
	  | (Burm.reg_LABEL, EX(LABEL l,_,_)) =>
	      GPR (addrOfLabel l)
	  | (Burm.reg_INT, EX(INT i,_,_)) =>
	      ((GPR (M.LI (i+i+1)))
	       handle Overflow => error "reg_INT")
	  | (Burm.reg_INT32, EX(INT32 i,_,_)) =>
	      GPR (M.LI32 i)
	  | (Burm.sreg_INT, EX(INT i,_,_)) =>
	      ((GPR (M.LI (i+i)))
	       handle Overflow => error "sreg_INT")
	  | (Burm.ureg_INT, EX(INT i,_,_)) =>
	      GPR (M.LI i)
	  | (Burm.reg_REAL, EX(REAL r,_,_)) =>
	      GPR (addrOfReal r)
	  | (Burm.reg_STRING, EX(STRING s,_,_)) =>
	      GPR (addrOfString s)
	  | (Burm.freg_VAR, EX(VAR v,_,_)) =>
	      FPR (fregFromVar v)
	  | (Burm.freg_REAL, EX(REAL r,_,_)) =>
	      FPR (M.LOADD (addrOfReal r))
	  | (Burm.freg_reg r, _) => error "freg_reg"
	  | (Burm.reg_freg fr, _) => let
		(* with better type information associated with non-terminals,
		 * this rule should not be required. It is present because our
		 * start non-terminal is reg (see ctreeRules.burg)
		 *)
		val arg = walk fr
	      in arg
	      end
	  | (Burm.reg_sreg sr, _) =>
	      GPR (M.ADD (GPRwalk sr, M.LI 1))
	  | (Burm.sreg_reg r, _) =>
	      GPR (M.SUB (GPRwalk r, M.LI 1, M.LR))
	  | (Burm.sreg_ureg ur, _) =>
	      GPR (M.SLL (GPRwalk ur, M.LI 1, M.LR))
	  | (Burm.ureg_sreg_or_reg r, _) =>
	      GPR (M.SRA (GPRwalk r, M.LI 1, M.LR))
	  | (Burm.r_ADD_r_r kids, EX(add,_,order)) =>
	      let val (GPR gp1, GPR gp2) = walk2c (kids, order) 
	      in GPR (addOp add (gp1, gp2))
	      end
	  | (Burm.r_ADD_2im_r reg, EX(add,[EX (INT i,_,_),_],_)) =>
	      GPR (addOp add (GPRwalk reg, M.LI (i+i-1)))
	  | (Burm.r_ADD_r_2im reg, EX(add,[_,EX (INT i,_,_)],_)) =>
	      GPR (addOp add (GPRwalk reg, M.LI (i+i-1)))
	  | (Burm.r_SUB_r_r kids, EX(minus,_,order)) =>
	      let val (GPR gp1, GPR gp2, ord) = walk2 (kids, order) in
		GPR (subOp minus (gp1, gp2, ord))
	      end
	  | (Burm.r_SUB_2ipp_r reg, EX(minus,[EX (INT i,_,_),_],_)) =>
	      GPR (subOp minus (M.LI (i+i+2), GPRwalk reg, M.RL))
	  | (Burm.r_SUB_r_2im reg, EX(minus,[_,EX (INT i,_,_)],_)) =>
	      GPR (subOp minus (GPRwalk reg, M.LI (i+i-1), M.LR))
	  | (Burm.r_MUL_r_r kids, EX(mul,_,order)) =>
	      let val (GPR gp1, GPR gp2) = walk2c (kids, order) in
		GPR (mulOp mul (gp1, gp2))
	      end
	  | (Burm.r_DIV_r_r kids, EX(dv,_,order)) =>
	      let val (GPR gp1, GPR gp2, ord) = walk2 (kids, order) in
		GPR (divOp dv (gp1, gp2, ord))
	      end
	  | (Burm.r_NEG_r reg, _)=>
	      GPR (M.SUBT (M.LI 0, GPRwalk reg, M.RL))
	  | (Burm.r_NEG_r_p_2 reg, _) =>
	      GPR (M.SUBT (M.LI 2, GPRwalk reg, M.RL))
	  | (Burm.r_ORB_r_r kids, EX(_,_,order)) =>
	      let val (GPR gp1, GPR gp2) = walk2c (kids, order) in
		GPR (M.ORB (gp1, gp2))
	      end
	  | (Burm.r_ANDB_r_r kids, EX(_,_,order)) =>
	      let val (GPR gp1, GPR gp2) = walk2c (kids, order) in
		GPR (M.ANDB (gp1, gp2))
	      end
	  | (Burm.r_XORB_r_r kids, EX(_,_,order)) =>
	      let val (GPR gp1, GPR gp2) = walk2c (kids, order) in
		GPR (M.XORB (gp1, gp2))
	      end
	  | (Burm.r_NOTB_r reg, _) =>
	      GPR (M.SUB (M.LI ~1, GPRwalk reg, M.RL))
	  | (Burm.r_NOTB_r_p_1 reg, _) =>
	      GPR (M.SUB (M.LI 0, GPRwalk reg, M.RL))
	  | (Burm.r_RSHIFTL_r_u kids, EX(_,_,order)) =>
	      let val (GPR gp1, GPR gp2, ord) = walk2 (kids, order) in
		GPR (M.ORB(M.SRL (gp1, gp2, ord), M.LI 1))
	      end
	  | (Burm.r_RSHIFTL_r_r kids, EX(_,_,order)) =>
	      let val (GPR gp1, GPR gp2, ord) = walk2 (kids, order) in
		GPR (M.SRL (gp1, gp2, ord))
	      end
	  | (Burm.r_RSHIFT_r_r kids, EX(_,_,order)) =>
	      let val (GPR gp1, GPR gp2, ord) = walk2 (kids, order) in
		GPR (M.SRA (gp1, gp2, ord))
	      end
	  | (Burm.r_RSHIFT_r_r_o kids, EX(_,_,order)) =>
	      let val (GPR gp1, GPR gp2, ord) = walk2 (kids, order) in
		GPR (M.ORB (M.SRA (gp1, gp2, ord), M.LI 1))
	      end
	  | (Burm.r_RSHIFT_r_r_a kids, EX(_,_,order)) =>
	      let val (GPR gp1, GPR gp2, ord) = walk2 (kids, order) in
		GPR (M.ANDB (M.SRA (gp1, gp2, ord), M.LI ~2))
	      end
	  | (Burm.r_LSHIFT_r_r kids, EX(_,_,order)) =>
	      let val (GPR gp1, GPR gp2, ord) = walk2 (kids, order) in
		GPR (M.SLL (gp1, gp2, ord))
	      end
	  | (Burm.r_W32ADD_r_r kids, EX(_,_,order)) =>
	      let val (GPR gp1, GPR gp2) = walk2c (kids, order) in
		GPR (M.ADD (gp1, gp2))
	      end
	  | (Burm.r_W32SUB_r_r kids, EX(_,_,order)) =>
	      let val (GPR gp1, GPR gp2, ord) = walk2 (kids, order) in
		GPR (M.SUB (gp1, gp2, ord))
	      end
	  | (Burm.r_W32MUL_r_r kids, EX(_,_,order)) =>
	      let val (GPR gp1, GPR gp2) = walk2c (kids, order) in
		GPR (M.MULU (gp1, gp2))
	      end
	  | (Burm.r_W32DIV_r_r kids, EX(_,_,order)) =>
	      let val (GPR gp1, GPR gp2, ord) = walk2 (kids, order) in
		GPR (M.DIVU (gp1, gp2, ord))
	      end
	  | (Burm.r_W32FROMWORD_r kid, _) => 
	      GPR(M.SRL(GPRwalk kid, M.LI 1, M.LR))
	  | (Burm.f_FADD_f_f kids, EX(_,_,order)) =>
	      let
		val (arg1,arg2) = walk2c (kids, order)
		val (fp1,fp2) = (fregFromArg arg1, fregFromArg arg2)
	      in
		FPR (M.FADDD (fp1, fp2))
	      end
	  | (Burm.f_FSUB_f_f kids, EX(_,_,order)) =>
	      let
		val (arg1,arg2,ord) = walk2 (kids, order)
		val (fp1,fp2) = (fregFromArg arg1, fregFromArg arg2)
	      in
		FPR (M.FSUBD (fp1, fp2, ord))
	      end
	  | (Burm.f_FMUL_f_f kids, EX(_,_,order)) =>
	      let
		val (arg1,arg2) = walk2c (kids, order)
		val (fp1,fp2) = (fregFromArg arg1, fregFromArg arg2)
	      in
		FPR (M.FMULD (fp1, fp2))
	      end
	  | (Burm.f_FDIV_f_f kids, EX(_,_,order)) =>
	      let
		val (arg1,arg2,ord) = walk2 (kids, order)
		val (fp1,fp2) = (fregFromArg arg1, fregFromArg arg2)
	      in
		FPR (M.FDIVD (fp1, fp2, ord))
	      end
	  | (Burm.f_FNEG_f fr, _) =>
	      FPR (M.FNEGD (fregFromArg (walk fr)))
	  | (Burm.f_FABS_f fr, _) =>
	      FPR (M.FABSD (fregFromArg (walk fr)))
	  | (Burm.f_REALOFINT_ureg ur, _) =>
	      FPR (M.CVTI2D (GPRwalk ur))
	  | (Burm.reg_MKSPECIAL_ureg_reg kids, EX(_,_,order)) =>
	      let
		val (GPR r1, arg2, ord) = walk2 (kids, order)
		val r2 = regFromArg arg2
		val sw = if ord = M.LR then fn x => x else fn (a,b) => (b,a)
		val stdesc = 
		      M.STORE32(C.allocptr,
				M.ORB (M.SLL (r1, M.LI T.width_tags, M.LR),
				       M.LI T.desc_special))
		val stval = M.STORE32(M.ADD(C.allocptr,M.LI 4),r2)
		val (first, second) = sw (stdesc, stval)
		val tmp = M.REG (C.newReg ())
	      in
		GPR (M.SEQ (first,
		     M.SEQ (second,
 		     M.SEQ (M.MV (tmp, M.ADD (C.allocptr, M.LI 4)),
		     M.SEQ (M.MV (C.allocptr, M.ADD (C.allocptr, M.LI 8)),
			    tmp)))))
	      end
	  | (Burm.ureg_LENGTH_reg reg, _) =>
	      let
		val GPR gp = walk reg		  (* reg is a pointer => GPR *)
	      in
		GPR (M.SRA (M.LOAD32 (M.SUB (gp, M.LI 4, M.LR)),
			    M.LI T.width_tags,
			    M.LR))
	      end
	  | (Burm.sreg_LENGTH_reg reg, _) =>
	      GPR (M.ANDB (M.SRA (M.LOAD32 (M.SUB (GPRwalk reg, M.LI 4, M.LR)),
				  M.LI (T.width_tags - 1),
				  M.LR),
			   M.LI ~2))
	  | (Burm.reg_LENGTH_reg reg, _) =>
	      GPR (M.ORB (M.SRA (M.LOAD32 (M.SUB (GPRwalk reg, M.LI 4, M.LR)),
				 M.LI (T.width_tags - 1),
				 M.LR),
			  M.LI 1))
	  | (Burm.ureg_GETTAG_reg reg, _) =>
	      GPR (M.ANDB (M.LOAD32 (M.SUB (GPRwalk reg, M.LI 4, M.LR)),
			   M.LI (T.power_tags - 1)))
	  | (Burm.reg_SUBSCRIPTV_reg_i reg, EX(_,[_,EX(INT i,_,_)],_)) =>
	      GPR (M.LOAD32 (M.ADD (GPRwalk reg, M.LI (i*4))))
	  | (Burm.reg_SUBSCRIPTV_reg_ureg kids, EX(_,_,order)) =>
	      let
		val (GPR gp1, GPR gp2, ord) = walk2 (kids, order)
		(*gp1 is a pointer, gp2 an int *)
		val sw = if ord = M.LR then (fn x => x) else (fn (a,b) => (b,a))
	      in
		GPR (M.LOAD32 (M.ADD (sw (gp1, M.SLL (gp2, M.LI 2, M.LR)))))
	      end
	  | (Burm.reg_SUBSCRIPTV_reg_sreg kids, EX(_,_,order)) =>
	      let
		val (GPR gp1, GPR gp2, ord) = walk2 (kids, order)
		(*gp1 is a pointer, gp2 an int *)
		val sw = if ord = M.LR then (fn x => x) else (fn (a,b) => (b,a))
	      in
		GPR (M.LOAD32 (M.ADD (sw (gp1, M.SLL (gp2, M.LI 1, M.LR)))))
	      end
	  | (Burm.reg_SUBSCRIPTV_reg_reg kids, EX(_,_,order)) =>
	      let
		val (GPR gp1, GPR gp2, ord) = walk2 (kids, order)
		(*gp1 is a pointer, gp2 an int *)
		val sw = if ord = M.LR then (fn x => x) else (fn (a,b) => (b,a))
	      in
		GPR (M.LOAD32 (M.SUB (M.ADD (sw (gp1,
					       M.SLL (gp2, M.LI 1, M.LR))),
				      M.LI 2,
				      M.LR)))
	      end
	  | (Burm.reg_DEREF_reg reg, _) =>
	      GPR (M.LOAD32 (GPRwalk reg))
	  | (Burm.freg_SUBSCRIPTF_reg_i reg, EX(_,[_,EX(INT i,_,_)],_)) =>
	      FPR (M.LOADD (M.ADD (GPRwalk reg, M.LI (i*8))))
	  | (Burm.freg_SUBSCRIPTF_reg_ureg kids, EX(_,_,order)) =>
	      let
		val (GPR gp1, GPR gp2, ord) = walk2 (kids, order)
		(*gp1 is a pointer, gp2 an int *)
		val sw = if ord = M.LR then (fn x => x) else (fn (a,b) => (b,a))
	      in
		FPR (M.LOADD (M.ADD (sw (gp1, M.SLL (gp2, M.LI 3, M.LR)))))
	      end
	  | (Burm.freg_SUBSCRIPTF_reg_sreg kids, EX(_,_,order)) =>
	      let
		val (GPR gp1, GPR gp2, ord) = walk2 (kids, order)
		(*gp1 is a pointer, gp2 an int *)
		val sw = if ord = M.LR then (fn x => x) else (fn (a,b) => (b,a))
	      in
		FPR (M.LOADD (M.ADD (sw (gp1, M.SLL (gp2, M.LI 2, M.LR)))))
	      end
	  | (Burm.freg_SUBSCRIPTF_reg_reg kids, EX(_,_,order)) =>
	      let
		val (GPR gp1, GPR gp2, ord) = walk2 (kids, order)
		(*gp1 is a pointer, gp2 an int *)
		val sw = if ord = M.LR then (fn x => x) else (fn (a,b) => (b,a))
	      in
		FPR (M.LOADD (M.SUB (M.ADD (sw (gp1,
					       M.SLL (gp2, M.LI 2, M.LR))),
				     M.LI 4,
				     M.LR)))
	      end
	  | (Burm.ureg_ORDOF_reg_ureg kids, EX(_,_,order)) => let
		val (GPR gp1, GPR gp2, ord) = walk2 (kids, order)
		(*gp1 is a pointer, gp2 an int *)
		val sw = if ord = M.LR then (fn x => x) else (fn (a,b) => (b,a))
	      in
		GPR (M.LOAD8 (M.ADD (sw (gp1, gp2))))
	      end
	  | (Burm.ureg_GETSPECIAL_reg reg, _) =>
	      GPR (M.SRA (M.LOAD32 (M.SUB (GPRwalk reg, M.LI 4, M.LR)),
			  M.LI T.width_tags,
			  M.LR))

	  | (Burm.reg_GETPSEUDO_reg mt, _) =>
	      (print "getpseudo -- not implemented"; GPR(M.REG(C.newReg())))
	  | (Burm.reg_SETPSEUDO_reg_reg mt, _) => 
	      (print "setpseudo -- not implemented"; GPR(M.REG(C.newReg())))

	  | (Burm.reg_GETVAR, _) =>
	      let val tmpR = M.REG (C.newReg ()) in
		GPR (M.SEQ (M.MV (tmpR, C.varptr), tmpR))
	      end
	  | (Burm.reg_GETHDLR, _) =>
	      let val tmpR = M.REG (C.newReg ()) in
		GPR (M.SEQ (M.MV (tmpR, C.exnptr), tmpR))
	      end
	  | (Burm.reg_RSTORE_reg reg, EX(RSTORE (i,p),_,_)) => let
		val what = recordwhat(regFromArg(walk reg), p)
	      in
		NOTHING (M.STORE32 (M.ADD (C.allocptr, M.LI (i)), what))
	      end
	  | (Burm.reg_RSTOREF_reg reg, EX(RSTOREF(n,p),_,_)) => let
	      val what = frecordwhat(regFromArg(walk reg), p)
	    in
	      NOTHING(M.STORED(M.ADD(C.allocptr, M.LI (n)), what))
	    end
	  | (Burm.reg_RSTOREF0_freg freg, EX(RSTOREF0(n),_,_)) => let
	      val fp = fregFromArg(walk freg)
	    in NOTHING(M.STORED(M.ADD(C.allocptr, M.LI(n)), fp))
	    end
	  | (Burm.reg_RSTOREDESC, EX(RSTOREDESC(false,i,d),_,_)) => 
	      NOTHING (M.STORE32 (M.ADD (C.allocptr, M.LI i), M.LI d))
	  | (Burm.reg_RSTOREDESC, EX(RSTOREDESC(true,i,d),_,_)) => let
	      val align = M.MV(C.allocptr, M.ORB(C.allocptr, M.LI 4))
	      val storedesc = 
		if i = 0 then M.STORE32 (C.allocptr, M.LI d)
		else M.STORE32 (M.ADD (C.allocptr, M.LI (i)), M.LI d)
	    in
	      NOTHING(M.SEQ(align, storedesc))
	    end
	  | (Burm.reg_RSTORESTORE, EX(RSTORESTORE i,_,_)) =>
	      NOTHING (M.STORE32 (M.ADD (C.allocptr, M.LI (i)), C.storeptr))
	  | (Burm.reg_RSET, EX(RSET i,_,_)) =>
	      GPR (M.ADD (C.allocptr, M.LI (i)))
	  | (Burm.reg_RSETSTORE, EX(RSETSTORE i,_,_)) =>
	      NOTHING (M.MV (C.storeptr,M.ADD (C.allocptr, M.LI (i))))
	  | (Burm.reg_RINC, EX(RINC i,_,_)) =>
	     (* Due to alignment considerations the allocptr optimization
	        is not currently performed. The write_allocate_hack is 
		temporarily commented out.
              let fun prefetch j =
                if j>=i orelse not(R.write_allocate_hack)
                then M.MV (C.allocptr,M.ADD (C.allocptr, M.LI (i)))
                else M.SEQ(M.LOAD32(M.ADD(C.allocptr,M.LI(j*4+96))),
                           prefetch(j+8))
               in NOTHING(prefetch 0)
              end
	      *) 
	     NOTHING(M.MV (C.allocptr,M.ADD (C.allocptr, M.LI (i))))

	  | (Burm.reg_SELECT_reg reg, EX(SELECT n,_,_)) =>
	      GPR (M.LOAD32 (M.ADD (GPRwalk reg, M.LI (n*4))))
	  | (Burm.freg_SELECTF_reg reg, EX(SELECTF n,_,_)) => 
	      FPR (M.LOADD (M.ADD (GPRwalk reg, M.LI (n*8))))
	  | (Burm.reg_OFFSET_reg reg, EX(OFFSET n,_,_)) =>
	      GPR (M.ADD (GPRwalk reg, M.LI (n*4)))

	  | (Burm.reg_STORE_reg_ureg_ureg kids, EX(_,_,order)) =>
	      let
		(* base, offset, byte *)
		val (GPR r1, GPR r2, GPR r3, ord) = walk3i (kids, order)
		val sw = if ord = M.LR then (fn x => x) else (fn (a,b) => (b,a))
	      in
		NOTHING (M.STORE8 (M.ADD (sw (r1, r2)), r3))
	      end
	  | (Burm.reg_UNBOXEDUPDATE_reg_i_reg kids, EX(_,[_,EX(INT i,_,_),_],_)) =>
	      let
		(* base, offset, word *)
		val (GPR r1, arg3) = walk2i (kids)
		val r3 = regFromArg arg3
	      in
		NOTHING (M.STORE32 (M.ADD (r1, M.LI (i*4)), r3))
	      end
	  | (Burm.reg_UNBOXEDUPDATE_reg_ureg_reg kids, EX(_,_,order)) =>
	      let
		(* base, offset, word *)
		val (GPR r1, GPR r2, arg3, ord) = walk3i (kids, order)
		val r3 = regFromArg arg3
		val sw = if ord = M.LR then (fn x => x) else (fn (a,b) => (b,a))
	      in
		NOTHING (M.STORE32 (M.ADD (sw (r1,
					       M.SLL (r2, M.LI 2, M.LR))),
				    r3))
	      end
	  | (Burm.reg_UNBOXEDUPDATE_reg_sreg_reg kids, EX(_,_,order)) =>
	      let
		(* base, offset, word *)
		val (GPR r1, GPR r2, arg3, ord) = walk3i (kids, order)
		val r3 = regFromArg arg3
		val sw = if ord = M.LR then (fn x => x) else (fn (a,b) => (b,a))
	      in
		NOTHING (M.STORE32 (M.ADD (sw (r1,
					       M.SLL (r2,M.LI 1, M.LR))),
				    r3))
	      end
	  | (Burm.reg_UNBOXEDUPDATE_reg_reg_reg kids, EX(_,_,order)) =>
	      let
		(* base, offset, word *)
		val (GPR r1, GPR r2, arg3, ord) = walk3i (kids, order)
		val r3 = regFromArg arg3
		val sw = if ord = M.LR then (fn x => x) else (fn (a,b) => (b,a))
	      in
		NOTHING (M.STORE32 (M.SUB (M.ADD (sw (r1,
						      M.SLL (r2, M.LI 1, M.LR))),
					   M.LI 2,
					   M.LR),
				    r3))
	      end
 	  | (Burm.reg_CALCADDR_reg_i reg, EX(_,[_,EX(INT i,_,_)],_)) =>
 	      GPR (M.ADD (GPRwalk reg, M.LI (i*4)))
 	  | (Burm.reg_CALCADDR_reg_ureg kids, EX(_,_,order)) =>
 	      let
 		val (GPR gp1, GPR gp2, ord) = walk2 (kids, order)
 		(*gp1 is a pointer, gp2 an int *)
 		val sw = if ord = M.LR then (fn x => x) else (fn (a,b) => (b,a))
 	      in
 		GPR (M.ADD (sw (gp1, M.SLL (gp2, M.LI 2, M.LR))))
 	      end
 	  | (Burm.reg_CALCADDR_reg_sreg kids, EX(_,_,order)) =>
 	      let
 		val (GPR gp1, GPR gp2, ord) = walk2 (kids, order)
 		(*gp1 is a pointer, gp2 an int *)
 		val sw = if ord = M.LR then (fn x => x) else (fn (a,b) => (b,a))
 	      in
 		GPR (M.ADD (sw (gp1, M.SLL (gp2, M.LI 1, M.LR))))
 	      end
 	  | (Burm.reg_CALCADDR_reg_reg kids, EX(_,_,order)) =>
 	      let
 		val (GPR gp1, GPR gp2, ord) = walk2 (kids, order)
 		(*gp1 is a pointer, gp2 an int *)
 		val sw = if ord = M.LR then (fn x => x) else (fn (a,b) => (b,a))
 	      in
 		GPR (M.SUB (M.ADD (sw (gp1, M.SLL (gp2, M.LI 1, M.LR))),
 			    M.LI 2, M.LR))
 	      end
	  | (Burm.reg_UPDATEF_reg_i_freg kids, EX(_,[_,EX(INT i,_,_),_],_)) =>
	      let
		(* base, offset, float *)
		val (GPR r1, arg3) = walk2i (kids)
		val fp = fregFromArg arg3
	      in
		NOTHING (M.STORED (M.ADD (r1, M.LI (i*8)), fp))
	      end
	  | (Burm.reg_UPDATEF_reg_ureg_freg kids, EX(_,_,order)) =>
	      let
		(* base, offset, float *)
		val (GPR r1, GPR r2, arg3, ord) = walk3i (kids, order)
		val fp = fregFromArg arg3
		val sw = if ord = M.LR then (fn x => x) else (fn (a,b) => (b,a))
	      in
		NOTHING (M.STORED (M.ADD (sw (r1,
					     M.SLL (r2, M.LI 3, M.LR))),
				  fp))
	      end
	  | (Burm.reg_UPDATEF_reg_sreg_freg kids, EX(_,_,order)) =>
	      let
		(* base, offset, float *)
		val (GPR r1, GPR r2, arg3, ord) = walk3i (kids, order)
		val fp = fregFromArg arg3
		val sw = if ord = M.LR then (fn x => x) else (fn (a,b) => (b,a))
	      in
		NOTHING (M.STORED (M.ADD (sw (r1,
					     M.SLL (r2, M.LI 2, M.LR))),
				  fp))
	      end
	  | (Burm.reg_UPDATEF_reg_reg_freg kids, EX(_,_,order)) =>
	      let
		(* base, offset, float *)
		val (GPR r1, GPR r2, arg3, ord) = walk3i (kids, order)
		val fp = fregFromArg arg3
		val sw = if ord = M.LR then (fn x => x) else (fn (a,b) => (b,a))
	      in
		NOTHING (M.STORED (M.SUB (M.ADD (sw (r1,
						    M.SLL (r2, M.LI 2, M.LR))),
					  M.LI 4,
					  M.LR),
				  fp))
	      end


	  | (Burm.reg_SETHDLR_reg reg, _) =>
	      NOTHING (M.MV (C.exnptr, GPRwalk reg))
	  | (Burm.reg_SETVAR_reg reg, _) =>
	      NOTHING (M.MV (C.varptr, GPRwalk reg))
	  | (Burm.reg_SETSPECIAL_reg_ureg kids, EX(_,_,order)) =>
	      let
		val (GPR r1, GPR r2, ord) = walk2 (kids, order)
	      in
 		NOTHING (M.STORE32 (M.SUB (r1, M.LI 4, M.LR),
				    M.ORB (M.SLL (r2,
						  M.LI T.width_tags,
						  M.LR),
					   M.LI T.desc_special)))
	      end
	  | (Burm.alist_ACONS_LABEL_alist al, EX(_,[EX (LABEL l,_,_),_],_)) =>
	      ALIST ((AL_LABEL l)::(ALISTwalk al))
	  | (Burm.alist_ACONS_INT_alist al, EX(_,[EX (INT i,_,_),_],_)) =>
	      ALIST ((AL_INT i)::(ALISTwalk al))
	  | (Burm.alist_ACONS_REAL_alist al, EX(_,[EX (REAL i,_,_),_],_)) =>
	      ALIST ((AL_REAL i)::(ALISTwalk al))
	  | (Burm.alist_ACONS_STRING_alist al, EX(_,[EX (STRING i,_,_),_],_)) =>
	      ALIST ((AL_STRING i)::(ALISTwalk al))
	  | (Burm.alist_ACONS_reg_alist kids, EX(_,_,order)) => let
		val (arg, ALIST alist, _) = walk2 (kids, order)	(* LR, always *)
	      in
		ALIST ((AL_ARG arg)::alist)
	      end
	  | (Burm.alist_ACONSF_freg_alist kids, EX(_,_,order)) => let
	       val (arg, ALIST alist, _) = walk2(kids,order)
	     in 
	       ALIST((AL_ARG arg)::alist)
	     end
	  | (Burm.alist_ANIL, _) =>
	      ALIST []

	  | (Burm.reg_APP_VAR_alist al, EX(APP_VAR(f,cty),_,_)) => let
		val ALIST alist = walk al
		val regfmls as (dst::_) = 
		  if iscont f then standardcont cty else standardescape cty
		val docall = M.CODE
		  (M.SEQ (M.TESTLIMIT (C.allocptr,
				       C.limitptr),
			  M.JMP dst))
		val exitblk = M.ESCAPEBLOCK  (C.exhausted :: regfmls)
	      in
		callSetup(alist, regfmls);
		addtoseq (docall);
		addtoseq (exitblk);
		NEVER
	      end

	  | (Burm.reg_APP_LABEL_alist al, EX(APP_LABEL(f,cty),_,_)) =>
	      let
		val ALIST alist = walk al
	      in
		(case know f of
		   Frag.STANDARD _ =>
		     let
		       val regfmls = 
			 if iscont f then standardcont cty else standardescape cty
		       val docall = M.CODE
			 (M.SEQ (M.TESTLIMIT (C.allocptr,C.limitptr),
				 M.BR (labmap f)))
		     in
		       callSetup(alist, regfmls); 
		       addtoseq (docall);
		       NEVER
		     end
		 | Frag.KNOWNFUN (ref(Frag.GEN (regfmls))) =>
		     let
		       val docall = M.CODE (M.BR (labmap f))
		     in
		       callSetup(alist, regfmls); 
		       addtoseq (docall);
		       NEVER
		     end
		 | Frag.KNOWNCHK (ref(Frag.GEN (regfmls)), _) =>
		     let
		       val docall = M.CODE
			 (M.SEQ (M.TESTLIMIT (C.allocptr,C.limitptr),
				 M.BR (labmap f)))
		     in
		       callSetup(alist, regfmls); 
		       addtoseq (docall);
		       NEVER
		     end
		 | Frag.KNOWNFUN (r as ref(Frag.UNGEN (varfmls,cl,functree))) =>
		     let
		       val regfmls = paramPassingConvention cl
		       val dolabel = M.LABEL (labmap f)
		     in
		       callSetup(alist, regfmls); 
		       addtoseq (dolabel);
		       r := Frag.GEN regfmls;
		       initialRegBindings(varfmls, regfmls);
		       genfunctree functree;
		       NEVER
		     end
		 | Frag.KNOWNCHK(r as ref(Frag.UNGEN (varfmls,cl,functree)), alloc) =>
		     let
		       val lab = labmap f
		       val regfmls = paramPassingConvention cl
		       val docall = M.CODE
			 (M.SEQ (M.TESTLIMIT (C.allocptr,C.limitptr),
				 M.BR lab))
		     in
		       callSetup(alist, regfmls); 
		       addtoseq (docall);
		       addtoseq (M.MARK);
		       addtoseq (M.LABEL lab);
		       knwCheckLimit (alloc*4, regfmls, cl, M.BR lab);
		       r := Frag.GEN regfmls;
		       initialRegBindings(varfmls, regfmls);
		       genfunctree functree;
		       NEVER
		     end
		 | _ => error "gen.APP_LABEL"
		) (* end case *)
	      end

	  | (Burm.reg_SWITCH_i, EX(SWITCH seqlist,[_,EX(INT i,_,_),_],_)) =>
	      let
		val offset = M.LI (i*4)
	      in
		doswitch (offset, seqlist)
	      end
	  | (Burm.reg_SWITCH_ureg ur, EX(SWITCH seqlist,_,_)) =>
	      let
		val GPR reg = walk ur
		val offset = M.SLL (reg, M.LI 2, M.LR)
	      in
		doswitch (offset, seqlist)
	      end
	  | (Burm.reg_SWITCH_sreg sr, EX(SWITCH seqlist,_,_)) =>
	      let
		val GPR reg = walk sr
		val offset = M.SLL (reg, M.LI 1, M.LR)
	      in
		doswitch (offset, seqlist)
	      end
	  | (Burm.reg_BR_BOXED_reg r, EX(BR_BOXED (seq1,seq2),_,_)) =>
	      let
		val arg = walk r
		val reg = regFromArg arg
		val lab = Label.newLabel ""
		val dobr = M.CODE
		  (M.BCC (M.NEQ, M.ANDB (reg, M.LI 1), M.LI 0, lab, M.LR))
	      in
		addtoseq (dobr);
		parallel_gen seq1;
		addtoseq (M.LABEL lab);
		gen seq2;
		NEVER
	      end
	  | (Burm.reg_BR_INT_reg_reg kids, EX(br_kind,_,order)) =>
	      let
		val (arg1,arg2,ord) = walk2 (kids, order)
		val r1 = regFromArg arg1
		val r2 = regFromArg arg2
		val (test, seq1, seq2) = case br_kind of
		  BR_INT (B_EQ, seq1, seq2) => (M.EQ, seq1, seq2)
		| BR_INT (B_LT, seq1, seq2) => (M.LT, seq1, seq2)
		| BR_INT (B_LE, seq1, seq2) => (M.LE, seq1, seq2)
		| BR_INT (B_LTU, seq1, seq2) => (M.LTU, seq1, seq2)
		| BR_INT (B_LEU, seq1, seq2) => (M.LEU, seq1, seq2)
		| BR_WORD32 (B_EQ, seq1, seq2) => (M.EQ, seq1, seq2)
		| BR_WORD32 (B_LT, seq1, seq2) => (M.LTU, seq1, seq2)
		| BR_WORD32 (B_LE, seq1, seq2) => (M.LEU, seq1, seq2)
		| BR_WORD32 (B_LTU, seq1, seq2) => (M.LTU, seq1, seq2)
		| BR_WORD32 (B_LEU, seq1, seq2) => (M.LEU, seq1, seq2)
		| _ => error "Burm.reg_BR_INT_reg_reg"
		val lab = Label.newLabel ""
		val dobr = M.CODE (M.BCC (test, r1, r2, lab, ord))
	      in
		addtoseq (dobr);
		parallel_gen seq2;
		addtoseq (M.LABEL lab);
		gen seq1;
		NEVER
	      end
	  | (Burm.reg_BR_FLOAT_freg_freg kids, EX(BR_FLOAT (tp,seq1,seq2),_,order)) =>
	      let
		val (arg1,arg2,ord) = walk2 (kids, order)
		val fp1 = fregFromArg arg1
		val fp2 = fregFromArg arg2
		val test = case tp of
		  B_EQ => M.EQ
		| B_LT => M.LT
		| B_LE => M.LE
		val lab = Label.newLabel ""
		val dobr = M.CODE (M.FBCC (test, fp1, fp2, lab, ord))
	      in
		addtoseq (dobr);
		parallel_gen seq2;
		addtoseq (M.LABEL lab);
		gen seq1;
		NEVER
	      end
	  (* To avoid repeated computation, it is essential to
	   * ensure that the source and destination strings are in
	   * pseudo registers. Typically, one operand is a string, whose
	   * start address is a label computation. Coalescing comes to
	   * the rescue if they are already in registers.
	   *)
	  | (Burm.reg_BR_STRING_INT_reg_reg kids,EX(BR_STRING(seq1,seq2),
					    [EX(INT n,_,_),_,_],order)) => let
	      val n' = ((n+3) div 4) * 4
	      val (GPR r1', GPR r2') = walk2i kids
	      val false_lab = Label.newLabel ""
	      val r1 = M.REG(C.newReg())
	      val r2 = M.REG(C.newReg())

	      fun whileLoop () = let
		val i = M.REG(C.newReg())
  	        val loopHead = Label.newLabel ""
	      in
		addCode(M.MV(i, M.LI 0));
		addtoseq(M.LABEL loopHead);
		addCode(M.BCC(M.NEQ, M.LOAD32(M.ADD(r1, i)),
				     M.LOAD32(M.ADD(r2, i)),
			      false_lab, M.LR));
		addCode(M.MV(i, M.ADD(i, M.LI 4)));
		addCode(M.BCC(M.NEQ, i, M.LI n', loopHead, M.LR))
	      end

	      fun unroll i = 
		    if i=n' then ()
		    else (addCode(M.BCC(M.NEQ,M.LOAD32(M.ADD(r1, M.LI i)),
					      M.LOAD32(M.ADD(r2, M.LI i)),
					false_lab, M.LR));
			  unroll (i+4))

            in
	      addCode(M.MV(r1, r1'));
	      addCode(M.MV(r2, r2'));
	      (* if n' <= 2 then unroll 0 else whileLoop();*)
	      unroll 0;
              parallel_gen seq1;
	      addtoseq (M.LABEL false_lab);
	      gen seq2;
	      NEVER
	    end
	  | (Burm.reg_dummy,_) => let
	      val tmpR = M.REG(C.newReg())
	    in
	      GPR(M.MV(tmpR,tmpR))
            end
      in
	walk (Burm.reduce ex)
      end (* genexpr *)

      (* gen : seq -> unit *)
      and gen (fstvar,lstvar:int) =
	let
	  fun loop v = 
	    if v>lstvar then () else
	      let
		val {ctree,deptype} = Array.sub (!curinfoarray, v)
	      in
		if deptype = ~1 orelse deptype = 6 then () else
		  let
		    val arg = genexpr ctree
		  in
		    case arg of
		      GPR gp =>
			let
			  val tmpR = M.REG(C.newReg ())
			  val code = M.CODE (M.MV (tmpR, gp))
			in 
			  addtoseq (code);
			  addRegBinding (v, GPR_K tmpR)
			end
		    | FPR fp => 
			let val tmpF = M.FREG(C.newFreg ())
			in
			  addtoseq (M.CODE(M.FMV(tmpF, fp)));
			  addRegBinding (v, FPR_K tmpF)
			end
		    | NOTHING gp =>
			addtoseq (M.CODE gp)
		    | NEVER => ()
		  end;
		loop (v+1)
	      end
	in
	  curseq := (fstvar,lstvar);			    (* -XXX DEBUG *)
	  loop fstvar
	end

      (* parallel_gen : like gen, but keeps the reg binding context.
       * I don't know how inefficient this is, I'm not even sure that
       * saving only the free variables would be cheaper.
       *)
      and parallel_gen seq = let
	  val bindings = Intmap.intMapToList regtable
	in
	  gen seq;
	  app addRegBinding bindings
	end

      (* genfunctree : functree -> unit *)
      and genfunctree (seq,infoarray) =	let 
          val oldinfoarray = !curinfoarray
	in
	  curinfoarray := infoarray;
	  gen seq;
	  curinfoarray := oldinfoarray
	end

      val regfmls = 
	if iscont funVar then standardcont regtyps else standardescape regtyps
      val linkreg = hd regfmls

    in
      addtoseq (M.MARK);
      addtoseq (M.LABEL lab);
      addtoseq (M.CODE (M.LBASE (linkreg, lab)));
      stdCheckLimit (alloc*4, regfmls, regtyps, M.JMP linkreg);
      initialRegBindings(varfmls, regfmls);
      genfunctree functree;
      app (callGc addtoseq) (!knwGcBlocks);
      !stdGcBlocks
    end (* mltreeGen *)

end (* functor MLTREEgen *)

