(* mltree.sml
 *
 * COPYRIGHT (c) 1994 AT&T Bell Laboratories.
 *)

structure MLTree : MLTREE = struct

  datatype bcond = LT | LTU | LE | LEU | EQ | NEQ | GEU

  datatype order = LR | RL

  datatype ovflType = Overflow | DivByZero | Ovfl_DivByZero

  datatype mlrisc = 
      REG    of int
    | FREG   of int

    | LI     of int
    | LI32   of Word32.word
    | MV     of mlrisc * mlrisc					(* dest, src *)
    | FMV    of mlrisc * mlrisc

    | ADD    of mlrisc * mlrisc
    | SUB    of mlrisc * mlrisc * order
    | MULU   of mlrisc * mlrisc
    | DIVU   of mlrisc * mlrisc * order

    | ADDT   of mlrisc * mlrisc 
    | MULT   of mlrisc * mlrisc
    | SUBT   of mlrisc * mlrisc * order
    | DIVT   of mlrisc * mlrisc * order

    | FADDD  of mlrisc * mlrisc
    | FMULD  of mlrisc * mlrisc
    | FSUBD  of mlrisc * mlrisc * order
    | FDIVD  of mlrisc * mlrisc * order
    | FABSD  of mlrisc 
    | FNEGD  of mlrisc

    | LADDR  of Label.label * int
    | LBASE  of mlrisc * Label.label

    | BR     of Label.label
    | JMP    of mlrisc
    | GOTO   of mlrisc * Label.label list
    | CALL   of mlrisc * mlrisc list * mlrisc list
    | RET

    | LOAD8  of mlrisc
    | LOAD32 of mlrisc
    | LOADD  of mlrisc
    | STORE8  of mlrisc * mlrisc			    (* address, data *)
    | STORE32 of mlrisc * mlrisc			    (* address, data *)
    | STORED  of mlrisc * mlrisc			    (* address, data *)

    | ANDB   of mlrisc * mlrisc
    | ORB    of mlrisc * mlrisc
    | XORB   of mlrisc * mlrisc

    | SRA   of mlrisc * mlrisc * order			     (* value, shift *)
    | SRL   of mlrisc * mlrisc * order
    | SLL   of mlrisc * mlrisc * order

    | CVTI2D of mlrisc

    | TESTLIMIT  of mlrisc * mlrisc
    | CHECKLIMIT of Label.label
    | LDREGMASK  of mlrisc list 

    | BCC       of bcond * mlrisc * mlrisc * Label.label * order
    | FBCC      of bcond * mlrisc * mlrisc * Label.label * order

    | SEQ of mlrisc * mlrisc
    | BARRIER of ovflType

  datatype mltree = 
      MARK
    | LABEL of Label.label
    | JMPTABLE of {base:Label.label,targets:Label.label list}
    | REALCONST of Label.label * string
    | STRINGCONST of Label.label * string
    | CODE of mlrisc
    | BEGINCLUSTER
    | ENDCLUSTER
    | ESCAPEBLOCK of mlrisc list

  fun prTree(MARK)  = print "MARK\n"
    | prTree(LABEL lab) = print ("LABEL: "^Label.nameOf lab^"\n")
    | prTree(JMPTABLE _) = print ("JUMPTABLE...\n")
    | prTree(REALCONST(_,f)) = print ("REAL: "^f^"\n")
    | prTree(STRINGCONST(_,s)) = print ("STRING: "^s^"\n")
    | prTree(BEGINCLUSTER) = print "BEGINCLUSTER\n"
    | prTree(ENDCLUSTER) = print "ENDCLUSTER\n"
    | prTree(ESCAPEBLOCK _) = print "ESCAPE\n"
    | prTree(CODE code) = let
        fun binary(oper,t1,t2) = 
	      "("^oper ^ code2string t1 ^ "," ^ code2string t2 ^ ")"
	and unary(oper,t) = "("^oper^code2string t ^")"
	  
 	and bcond LT = "LT "
 	  | bcond LE = "LE "
 	  | bcond EQ = "EQ "
 	  | bcond NEQ= "NEQ "
 	  | bcond GEU= "GEU "
	  | bcond LEU= "LEU "
	  | bcond LTU= "LTU "

	and code2string(REG r) 		= " REG."^Int.toString r
 	  | code2string(FREG f) 	= " FREG."^Int.toString f
 	  | code2string(LI n) 		= " LI."^Int.toString n
	  | code2string(LI32 n) 	= "LI32."
	  | code2string(MV(t1,t2)) 	= binary("MV",t1,t2)
	  | code2string(FMV(t1,t2))	= binary("FMV",t1,t2)
	  | code2string(ADD(t1,t2)) 	= binary("ADD",t1,t2)
	  | code2string(SUB(t1,t2,_)) 	= binary("SUB",t1,t2)
	  | code2string(MULU(t1,t2))   	= binary("MULU",t1,t2) 
	  | code2string(DIVU(t1,t2,_)) 	= binary("DIVU",t1,t2)
	  | code2string(ADDT(t1,t2)) 	= binary("ADDT",t1,t2)
	  | code2string(MULT(t1,t2)) 	= binary("MULT",t1,t2)
	  | code2string(SUBT(t1,t2,_)) 	= binary("SUBT",t1,t2)
	  | code2string(DIVT(t1,t2,_)) 	= binary("DIVT",t1,t2)
	  | code2string(FADDD(t1,t2)) 	= binary("FADDD",t1,t2)
	  | code2string(FMULD(t1,t2)) 	= binary("FMULD",t1,t2)
	  | code2string(FSUBD(t1,t2,_)) 	= binary("FSUBD",t1,t2)
	  | code2string(FDIVD(t1,t2,_)) 	= binary("FDIVD",t1,t2)
	  | code2string(FABSD(t))     	= unary("FABSD",t)
	  | code2string(FNEGD(t))     	= unary("FNEGD",t)
	  | code2string(LADDR _)      	= "(LADDR)"
	  | code2string(LBASE(t,_))   	= unary("LBASE",t)
	  | code2string(BR _)         	= "(BR)"
	  | code2string(JMP t)	      	= unary("JMP",t)
	  | code2string(GOTO(t,_))    	= unary("GOTO",t)
	  | code2string(CALL(t,_,_))  	= unary("CALL",t)
	  | code2string(RET)	      	= "(RET)"
	  | code2string(LOAD8 t)      	= unary("LOAD8",t)
	  | code2string(LOAD32 t)     	= unary("LOAD32",t)
	  | code2string(LOADD t)      	= unary("LOADD",t)
	  | code2string(STORE8(t1,t2))	= binary("STORE8",t1,t2)
	  | code2string(STORE32(t1,t2))	= binary("STORE32",t1,t2)
	  | code2string(STORED(t1,t2))	= binary("STORED",t1,t2)
	  | code2string(ANDB(t1,t2))	= binary("ANDB",t1,t2)
	  | code2string(ORB(t1,t2))	= binary("ORB",t1,t2)
	  | code2string(XORB(t1,t2))	= binary("XORB",t1,t2)
	  | code2string(SRA(t1,t2,_))	= binary("SRA",t1,t2)
	  | code2string(SRL(t1,t2,_))	= binary("SRL",t1,t2)
	  | code2string(SLL(t1,t2,_))	= binary("SLL",t1,t2)
	  | code2string(CVTI2D t) 	= unary("CVTI2D",t)
	  | code2string(TESTLIMIT(t1,t2))= binary("TESTLIMIT",t1,t2)
 	  | code2string(CHECKLIMIT lab) = "(CHECKLIMIT->"^Label.nameOf lab^")"
  	  | code2string(LDREGMASK _) 	= "(LDREGMASK)"
 	  | code2string(BCC(b,t1,t2,lab,_)) =
  	      binary("BCC-"^ bcond b ^"->"^Label.nameOf lab,t1,t2)
 	  | code2string(FBCC(b,t1,t2,lab,_)) = 
	      binary("FBCC-"^ bcond b ^"->"^Label.nameOf lab,t1,t2)
 	  | code2string(SEQ(t1,t2))	=  code2string t1^"\n\t"^code2string t2
  	  | code2string(BARRIER _) 	= "(BARRIER)"
      in
	print "CODE:\n\t";
	print(code2string code);
	print "\n"
      end
end (* MLTree *)
